/* Bluefish HTML Editor
 * document.c - this file contains the document code
 *
 * Copyright (C) 2000 Olivier Sessink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "default_include.h"

#include <locale.h>

#include <stdio.h>				/* fopen */
#include <string.h>				/*strchr */
#include <stdlib.h>				/* atoi */
#include <unistd.h>  			/* unlink() */

#include "bluefish.h"
#include "document.h"
#include "gtk_easy.h"			/* error dialog */
#include "interface.h"			/* statusbar message */
#include "undo.h"				/*undo_list_add */
#include "highlight.h"			/*line_hi_cb */
#include "rpopup.h"				/* rpopup_eventh */
#include "stringlist.h"			/* free_stringlist */
#include "snr2.h" 			/* find_filenames in test_link_management() */

/*******************************************************************/
void file_to_textbox(gchar * filename);
void textbox_to_file(gchar * filename);
void doc_insert_text_cb(GtkEditable * editable, const char *text, int len, int
						*pos, gpointer d);
void doc_delete_text_cb(GtkEditable * editable, int start, int end, gpointer d);
gboolean highlight_once(Tdocument * whichdoc);
void document_need_highlighting(Tdocument * whichdoc);
void destroy_current_document(void);
Ttext_positions get_positions(GtkWidget * textbox);
void restore_positions(Ttext_positions positions, GtkWidget * textbox);
void document_set_fonts(gchar * textfontstring, gchar * labelfontstring, Tdocument * document);
void insert_text(const gchar * before, const gchar * after);
Tdocument *new_document(void);
void file_save_cb(GtkWidget * widget, gpointer data);
void file_save_as_cb(GtkWidget * widget, gpointer data);
void file_open_cb(GtkWidget * widget, gpointer data);
void file_open(gchar * filename);
void file_insert_cb(GtkWidget * widget, gpointer data);
void file_new_cb(GtkWidget * widget, gpointer data);
void file_close_cb(GtkWidget * widget, gpointer data);
void file_close_all_cb(GtkWidget * widget, gpointer data);
void file_save_all_cb(GtkWidget * widget, gpointer data);
void go_to_line_win_cb(GtkWidget * widget, gpointer data);
static void scrollbar_update(GtkText * txt, gint tl, gint ln);
static void go_to_line_lcb(GtkWidget * widget, gpointer data);

/*******************************************************************/
/* set several document settings */

void document_set_fonts(gchar * textfontstring, gchar * labelfontstring, Tdocument * document)
{
	if (labelfontstring) {
		apply_font_style(document->textbox, textfontstring);
	}
	if (labelfontstring) {
		apply_font_style(document->tab_label, labelfontstring);
	}
}

void document_set_wrap(gint word_wrap, gint line_wrap, Tdocument * document)
{
	gtk_text_set_line_wrap(GTK_TEXT(document->textbox), line_wrap);
	gtk_text_set_word_wrap(GTK_TEXT(document->textbox), word_wrap);
}


/*******************************************************************/



void file_to_textbox(gchar * filename)
{
	FILE *fd;
	gchar *errmessage, line[512];

	statusbar_message(_("Opening file"), 1000);
	/* This opens the contents of a file to a textbox */
	change_dir(filename);
	fd = fopen(filename, "r");
	if (fd == NULL) {
		DEBUG_MSG("file_to_textbox, cannot open file %s\n", filename);
		errmessage = g_strconcat(_("Could not open file:\n"), filename, NULL);
		error_dialog(_("Error"), errmessage);	/* 7 */
		g_free(errmessage);
		return;
	} else {

		gtk_text_freeze(GTK_TEXT(main_v->current_document->textbox));
		while (fgets(line, 512, fd) != NULL) {
			gtk_text_insert(GTK_TEXT(main_v->current_document->textbox), NULL, NULL, NULL, line, -1);
		}
		gtk_text_thaw(GTK_TEXT(main_v->current_document->textbox));
		fclose(fd);
		main_v->current_document->modified = FALSE;
		if (main_v->current_document->highlightstate) {
			highlight_once(main_v->current_document);
			DEBUG_MSG("file_to_textbox, %p needs highlighting\n", main_v->current_document);
		}
	}
}

void textbox_to_file(gchar * filename)
{
	FILE *fd;
	gchar *tmpchar, *errmessage = NULL;

	DEBUG_MSG("textbox_to_file, started file %s\n", filename);
	statusbar_message(_("Saving file"), 1000);
	/* This writes the contents of a textbox to a file */
	change_dir(filename);
	fd = fopen(filename, "w");
	if (fd == NULL) {
		DEBUG_MSG("textbox_to_file, cannot open file %s\n", filename);
		errmessage = g_strconcat(_("Could not write file:\n"), filename, NULL);
		error_dialog(_("Error"), errmessage);
		g_free(errmessage);
		return;
	} else {
		DEBUG_MSG("textbox_to_file, lenght=%d\n", gtk_text_get_length(GTK_TEXT(main_v->current_document->textbox)));
		DEBUG_MSG("textbox_to_file, fd=%p\n", fd);
		DEBUG_MSG("textbox_to_file, filename=%s\n", filename);

		tmpchar = gtk_editable_get_chars(GTK_EDITABLE(main_v->current_document->textbox), 0, -1);
		fputs(tmpchar, fd);
		g_free(tmpchar);
		fclose(fd);
		main_v->current_document->modified = 0;
		DEBUG_MSG("textbox_to file, text saved to: %s\n", filename);
	}
}


/*
 * PUBLIC: doc_insert_text_cb, copied from doc.c in the gnotepad code
 *
 * callback for text widget for text insertion from keyboard typing.  note that
 * this routine and the next one (doc_delete_text_cb()), gets invoked for EVERY
 * key typed by the user.
 */
void doc_insert_text_cb(GtkEditable * editable, const char *text, int len, int
						*pos, gpointer d)
{
	char *buf;

	if (main_v->props.cont_highlight_update) {
		if ((strchr(text, '"') != NULL) || (strchr(text, '<') != NULL) || (strchr(text, '>') != NULL)) {
			document_need_highlighting(main_v->current_document);
		}
	}
	buf = g_strndup(text, len);
	/* does this introduce a memory leak? where is this buf free-ed? 
	I hope undo_list_add() uses it to store the change, but this is not verified */
	
	DEBUG_MSG("doc_insert_text_cb, buf=%s\n", buf);
	undo_list_add(d, buf, *pos, *pos + len, UndoInsert);
	main_v->current_document->modified = 1;
	DEBUG_MSG("doc_insert_text_cb, finished, modified = 1, need_highlighting=%d\n", main_v->current_document->need_highlighting);
}


								/* doc_insert_text_cb */


/*
 * PUBLIC: doc_delete_text_cb, copied from doc.c in the gnotepad code
 *
 * callback for text widget for text deletion from keyboard typing
 */
void doc_delete_text_cb(GtkEditable * editable, int start, int end, gpointer d)
{
	char *text;
	DEBUG_MSG("doc_delete_text_cb, started, editable=%p, start=%d, end=%d, document=%p\n", editable, start, end, d);
	text = gtk_editable_get_chars(editable, start, end);
	DEBUG_MSG("doc_delete_text_cb, text=%p, document->textbox=%p\n", text, ((Tdocument *)d)->textbox);
	undo_list_add(d, text, start, end, UndoDelete);
	main_v->current_document->modified = 1;
	DEBUG_MSG("doc_delete_text_cb, finished, modified=1\n");
}								/* doc_delete_text_cb */

gboolean highlight_line_once(Tdocument * whichdoc) {
	DEBUG_MSG("highlight_line_once, need_highlighting=%d\n", whichdoc->need_highlighting);
	whichdoc->need_highlighting = 0;
	line_hi_cb(NULL, whichdoc);
	return FALSE;	
}

gboolean highlight_once(Tdocument * whichdoc)
{
	DEBUG_MSG("highlight_once, need_highlighting=%d\n", whichdoc->need_highlighting);
	whichdoc->need_highlighting = 0;
	refresh_hi_cb(NULL, whichdoc);
	return FALSE;
}

void document_need_highlighting(Tdocument * whichdoc)
{
	DEBUG_MSG("document_need_highlighting, need_highlighting=%d, cont_highlight_full=%d\n", whichdoc->need_highlighting, main_v->props.cont_highlight_full);
	if (whichdoc->highlightstate && main_v->props.cont_highlight_update) {
		if (main_v->props.cont_highlight_full) {
			gtk_timeout_add(100, (GtkFunction) highlight_once, whichdoc);
		} else {
			gtk_timeout_add(100, (GtkFunction) highlight_line_once, whichdoc);
		}
	}
}


Ttext_positions get_positions(GtkWidget * textbox)
{

	Ttext_positions positions;
	gint tmp;

	positions.currentp = gtk_editable_get_position(GTK_EDITABLE(textbox));
	positions.adj = GTK_TEXT(textbox)->vadj->value;
	positions.selection = GTK_EDITABLE(textbox)->has_selection;
	if (positions.selection) {
		positions.startp = GTK_EDITABLE(textbox)->selection_start_pos;
		positions.endp = GTK_EDITABLE(textbox)->selection_end_pos;
		if (positions.endp < positions.startp) {
			tmp = positions.endp;
			positions.endp = positions.startp;
			positions.startp = tmp;
		}
	} else {
		positions.endp = positions.currentp;
		positions.startp = positions.currentp;
	}
	return positions;
}

void restore_positions(Ttext_positions positions, GtkWidget * textbox)
{

	gtk_editable_set_position(GTK_EDITABLE(textbox), positions.currentp);
	gtk_adjustment_set_value(GTK_TEXT(textbox)->vadj, positions.adj);
	if (positions.selection) {
		DEBUG_MSG("restore_positions, startp=%d, endp=%d\n", positions.startp, positions.endp);
		gtk_editable_select_region(GTK_EDITABLE(textbox), positions.startp, positions.endp);
	}
}


void doc_replace_text(const gchar * newstring, gint pos, gint end, Tdocument *doc) {
#ifdef DEBUG
	gchar *tmpstring;
	gint oldlen, newlen;
#endif
	gtk_text_freeze(GTK_TEXT(doc->textbox));
	DEBUG_MSG("doc_replace_text, started, pos=%d, end=%d, newstring=%s, strlen(newstring)=%d\n", pos, end, newstring, strlen(newstring));
#ifdef DEBUG
	tmpstring = gtk_editable_get_chars(GTK_EDITABLE(GTK_TEXT(doc->textbox)), pos, end);
	if (!tmpstring) {
		DEBUG_MSG("*** doc_replace_text, PROBLEM: tmpstring=NULL ***\nnewstring=%s, pos=%d, end=%d, doc=%p\n", newstring, pos, end, doc);
		return;
	}
	DEBUG_MSG("doc_replace_text, about to replace %s\n", tmpstring);
	oldlen = strlen(tmpstring);
	newlen = strlen(newstring);
	DEBUG_MSG("doc_replace_text, oldlen=%d, newlen=%d, change=%d\n", oldlen, newlen, newlen-oldlen);
#endif	
	gtk_editable_delete_text(GTK_EDITABLE(GTK_TEXT(doc->textbox)), pos, end);
	DEBUG_MSG("doc_replace_text, text deleted\n");
	gtk_editable_insert_text(GTK_EDITABLE(GTK_TEXT(doc->textbox)), newstring, strlen(newstring), &pos);
	gtk_text_thaw(GTK_TEXT(doc->textbox));
	doc->modified = 1;
	document_need_highlighting(doc);
}

void replace_text(const gchar * newstring, gint pos, gint end) {
	DEBUG_MSG("replace_text, calling doc_replace_text on current_document\n");
	doc_replace_text(newstring, pos, end, main_v->current_document);
}

void insert_text(const gchar * before, const gchar * after)
{
	guint pos;
	Ttext_positions positions;
	/* find the current cursor position and selection position */

	positions = get_positions(main_v->current_document->textbox);

/* freeze the textbox, this is better when inserting large blocks */
	gtk_text_freeze(GTK_TEXT(main_v->current_document->textbox));

	DEBUG_MSG("insert_text, A currentp = %d, startp=%d, endp=%d\n", positions.currentp, positions.startp, positions.endp);

	gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox), positions.endp);
	gtk_text_insert(GTK_TEXT(main_v->current_document->textbox), NULL, NULL, NULL, after, -1);
	/* strlen() is unable to find the lenght of multibyte characters (japanese)
	so we use the new position in the text box to find out*/
	pos = gtk_text_get_point(GTK_TEXT(main_v->current_document->textbox));
	undo_list_add(main_v->current_document, g_strdup(after), positions.endp, pos, UndoInsert);

	gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox), positions.startp);
	gtk_text_insert(GTK_TEXT(main_v->current_document->textbox), NULL, NULL, NULL, before, -1);
	pos = gtk_text_get_point(GTK_TEXT(main_v->current_document->textbox));
	undo_list_add(main_v->current_document, g_strdup(before), positions.startp, pos, UndoInsert);

/* thaw textbox, we're finished */
	DEBUG_MSG("insert_text, ->thaw\n");
	gtk_text_thaw(GTK_TEXT(main_v->current_document->textbox));
 	positions.endp += pos - positions.startp;
 	positions.currentp += pos - positions.startp;
 	positions.startp = pos;
	restore_positions(positions, main_v->current_document->textbox);

	/* grab focus so we can continue working */
	DEBUG_MSG("insert_text, about to grab focus\n");
	gtk_widget_grab_focus(GTK_WIDGET(main_v->current_document->textbox));
	DEBUG_MSG("insert_text, set modified=1\n");
	main_v->current_document->modified = 1;

	document_need_highlighting(main_v->current_document);
}

static void scroll_textbox(gint direction_up) {
	gfloat newvalue;

	newvalue = GTK_ADJUSTMENT(GTK_TEXT(main_v->current_document->textbox)->vadj)->value;
	if (direction_up) {
		newvalue += (GTK_ADJUSTMENT(GTK_TEXT(main_v->current_document->textbox)->vadj)->page_increment / 2);
	} else {
		newvalue -= (GTK_ADJUSTMENT(GTK_TEXT(main_v->current_document->textbox)->vadj)->page_increment / 2);
	}
	gtk_adjustment_set_value(GTK_ADJUSTMENT(GTK_TEXT(main_v->current_document->textbox)->vadj),newvalue);
}

static gint textbox_event_lcb(GtkWidget *widget , GdkEvent * event) {

	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event;
		DEBUG_MSG("rpopup_eventh, bevent->button=%d\n", bevent->button);
		if (bevent->button == 3) {
			rpopup_eventh(bevent);
			return TRUE;
		} else if (bevent->button == 4) {
			scroll_textbox(0);
			return TRUE;
		} else if (bevent->button == 5) {
			scroll_textbox(1);
			return TRUE;
		}
	}
	/* Tell calling code that we have not handled this event; pass it on. */
	return FALSE;
}

Tdocument *new_document(void)
{
	GtkWidget *tmptable, *tmpscrollbar;
	Tdocument *document;

	/* Here we create a new document, and add it to the GList */
	DEBUG_MSG("new_document, started\n");
	if (main_v->current_document != NULL) {
		DEBUG_MSG("new_document, modified=%d, filename=%p, list_lenght=%d\n", main_v->current_document->modified,
				  main_v->current_document->filename, g_list_length(main_v->documentlist));

		if ((main_v->current_document->modified == 0)
			&& (main_v->current_document->filename == NULL)
			&& (g_list_length(main_v->documentlist) == 1)
			&& (gtk_text_get_length(GTK_TEXT(main_v->current_document->textbox)) ==0)) {
			DEBUG_MSG("new_document, no new needed, the current one is empty\n");
			return main_v->current_document;
		}
	}

	document = g_malloc0(sizeof(Tdocument));
	DEBUG_MSG("new_document, malloced at %p\n", document);
	tmptable = gtk_table_new(2, 1, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(tmptable), 0);
	gtk_table_set_col_spacings(GTK_TABLE(tmptable), 0);
	document->textbox = gtk_text_new(NULL, NULL);
	DEBUG_MSG("new_document, document->textbox=%p\n", document->textbox);
	gtk_table_attach_defaults(GTK_TABLE(tmptable), document->textbox, 0, 1, 0, 1);

	tmpscrollbar = gtk_vscrollbar_new(GTK_TEXT(document->textbox)->vadj);
	GTK_WIDGET_UNSET_FLAGS(tmpscrollbar, GTK_CAN_FOCUS);
	gtk_table_attach(GTK_TABLE(tmptable), tmpscrollbar, 1, 2, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
	gtk_widget_show(tmpscrollbar);

	document->tab_label = gtk_label_new(_("Untitled"));
	GTK_WIDGET_UNSET_FLAGS(document->tab_label, GTK_CAN_FOCUS);

	apply_font_style(GTK_WIDGET(document->tab_label), main_v->props.cfg_tab_font);

	/* I did the table like this because it is more flexable than */
	/* a hard coded limit with gtk_widget_set_usize(), and I plan */
	/* on making the window size configurable someday */
	DEBUG_MSG("new_document, before notebook_append_page\n");
	gtk_notebook_append_page(GTK_NOTEBOOK(main_v->notebook), tmptable, document->tab_label);
	gtk_widget_show(tmptable);
	DEBUG_MSG("new_document, before realize on textbox\n");
	gtk_widget_realize(document->textbox);
	DEBUG_MSG("new_document, after realize on textbox\n");
	gtk_text_set_editable(GTK_TEXT(document->textbox), TRUE);
	document_set_wrap(main_v->props.word_wrap, main_v->props.line_wrap, document);

	gtk_signal_connect_object(GTK_OBJECT(document->textbox), "event", GTK_SIGNAL_FUNC(textbox_event_lcb), NULL);

	apply_font_style(GTK_WIDGET(document->textbox), main_v->props.cfg_editor_font);
	GTK_TEXT(document->textbox)->default_tab_width = main_v->props.cfg_editor_tabwidth;
	GTK_TEXT(document->textbox)->tab_stops = g_list_remove(GTK_TEXT
														   (document->textbox)->tab_stops, GTK_TEXT(document->textbox)->tab_stops->data);
	GTK_TEXT(document->textbox)->tab_stops = g_list_remove(GTK_TEXT
														   (document->textbox)->tab_stops, GTK_TEXT(document->textbox)->tab_stops->data);
	GTK_TEXT(document->textbox)->tab_stops = NULL;
	GTK_TEXT(document->textbox)->tab_stops =
		g_list_prepend(GTK_TEXT(document->textbox)->tab_stops, (void *) main_v->props.cfg_editor_tabwidth);
	GTK_TEXT(document->textbox)->tab_stops =
		g_list_prepend(GTK_TEXT(document->textbox)->tab_stops, (void *) main_v->props.cfg_editor_tabwidth);

	gtk_widget_show(document->textbox);

	document->modified = FALSE;
	document->filename = NULL;
	document->need_highlighting = 0;
	document->highlightstate = main_v->props.defaulthighlight;

	main_v->documentlist = g_list_append(main_v->documentlist, document);
	gtk_widget_grab_focus(document->textbox);

	/* this part is copied from gnotepad code 
	   Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com> */

	document->ins_txt_id = gtk_signal_connect(GTK_OBJECT(document->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), document);
	document->del_txt_id = gtk_signal_connect(GTK_OBJECT(document->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), document);

	/* set this new document as active notebook page */
	flush_queue();
	DEBUG_MSG("new_document, before notebook_set_page, after flush_queue\n");
	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), g_list_length(main_v->documentlist) - 1);
	notebook_changed();
	DEBUG_MSG("new_document, ended\n");

	return document;
}

/* this function destroys the current_document */
void destroy_current_document(void)
{
	if (main_v->current_document) {
		DEBUG_MSG("destroy_current_document, started, notebook=%p, document=%p\n", main_v->notebook, main_v->current_document);
		DEBUG_MSG("destroy_current_document, textbox=%p\n", main_v->current_document->textbox);
#ifdef DEBUG
		if (main_v->current_document->filename) {
			g_print("destroy_current_document, filename=%s, not modified, destroy it\n",main_v->current_document->filename );
		} else {
			g_print("destroy_current_document, filename=NULL, not modified, destroy it\n");
		}
#endif
		gtk_notebook_remove_page(GTK_NOTEBOOK(main_v->notebook), gtk_notebook_current_page(GTK_NOTEBOOK(main_v->notebook)));
		DEBUG_MSG("destroy_current_document, g_list_length(documentlist)=%d\n", g_list_length(main_v->documentlist));

		if (g_list_length(main_v->documentlist) > 1) {
			main_v->documentlist = g_list_remove(main_v->documentlist, main_v->current_document);
		} else {
			main_v->documentlist = g_list_remove(main_v->documentlist, main_v->current_document);
			DEBUG_MSG("destroy_current_document, last document removed from documentlist\n");
			g_list_free(main_v->documentlist);
			DEBUG_MSG("destroy_current_document, freed documentlist\n");
			main_v->documentlist = NULL;
			DEBUG_MSG("destroy_current_document, documentlist = NULL\n");
		}
		DEBUG_MSG("destroy_current_document, g_list_length(documentlist)=%d\n", g_list_length(main_v->documentlist));
		if (main_v->current_document->filename) {
			DEBUG_MSG("destroy_current_document, main_v->current_document->filename=%s\n", main_v->current_document->filename);
			g_free(main_v->current_document->filename);
		}
		g_free(main_v->current_document);
		gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), ((gint) g_list_length(main_v->documentlist) - 1));
		notebook_changed();
	} else {
		DEBUG_MSG("destroy_current_document, cannot close a NULL current_document\n");
	}
}



static void scrollbar_update(GtkText * txt, gint tl, gint ln)
{
	gfloat value;

	if (tl < 3 || ln > tl)
		return;

	value = (ln * GTK_ADJUSTMENT(txt->vadj)->upper) / tl - GTK_ADJUSTMENT(txt->vadj)->page_increment;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(txt->vadj), value);
}

gint position_to_linenum(gint position)
{
	gint i, lines;
	gchar *c;					/* = g_malloc0(3); */

	lines = 0;
	i = gtk_text_get_length(GTK_TEXT(main_v->current_document->textbox));
	if (position > i) {
		position = i;
	}
	c = gtk_editable_get_chars(GTK_EDITABLE(main_v->current_document->textbox), 0, position);	
	DEBUG_MSG("position_to_linenum, strlen(c)=%d, position=%d\n", strlen(c), position);
	for (i = position; i > 1; i--) {
		if (c[i] == '\n')
			lines++;
	}
	g_free(c);
	return lines;
}

void doc_go_to_line(gint linenum, gint select_line)
{

/* Thanks to Andy Kahn for the amazing non-leaking go to line function
   It dances, it does tricks, AND you have memory left after... :) */

	gchar *buf, *haystack, *needle;
	gint numlines;
	gint a, b, len;

	numlines = 1;

	len = gtk_text_get_length(GTK_TEXT(main_v->current_document->textbox));
	buf = gtk_editable_get_chars(GTK_EDITABLE(main_v->current_document->textbox), 1, len);

/*  If index is past the end, or 0 (Usually because line number is -) */

	if ((linenum > (len - 1)) || (linenum == 0))
		return;

	a = 1;
	b = len;
	haystack = buf;
	do {
		needle = strchr(haystack, '\n');
		if (needle) {
			haystack = needle + 1;
			if (linenum == numlines)
				b = needle - buf + 1;
			numlines++;
			if (linenum == numlines)
				a = needle - buf + 1;
		}
	}
	while (needle != NULL);

	g_free(buf);
	if (select_line) {
		gtk_editable_select_region(GTK_EDITABLE(main_v->current_document->textbox), a, b);
	}
	scrollbar_update(GTK_TEXT(main_v->current_document->textbox), numlines, linenum);
	gtk_signal_disconnect(GTK_OBJECT(main_v->current_document->textbox), main_v->current_document->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(main_v->current_document->textbox), main_v->current_document->del_txt_id);
	gtk_editable_insert_text(GTK_EDITABLE((GtkText *)
										  main_v->current_document->textbox), " ", 1, &b);
	gtk_editable_delete_text(GTK_EDITABLE((GtkText *)
										  main_v->current_document->textbox), b - 1, b);
	main_v->current_document->ins_txt_id = gtk_signal_connect(GTK_OBJECT(main_v->current_document->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), main_v->current_document);
	main_v->current_document->del_txt_id = gtk_signal_connect(GTK_OBJECT(main_v->current_document->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), main_v->current_document);
}


/*****************************************************/

void copy_cb(GtkWidget * w, gpointer data)
{
	gtk_editable_copy_clipboard(GTK_EDITABLE(main_v->current_document->textbox));
}

/*****************************************************/

void paste_cb(GtkWidget * w, gpointer data)
{
	gtk_editable_paste_clipboard(GTK_EDITABLE(main_v->current_document->textbox));
	main_v->current_document->modified = 1;
	if (main_v->current_document->highlightstate) {
		document_need_highlighting(main_v->current_document);
	}
}


/*****************************************************/

void cut_cb(GtkWidget * w, gpointer data)
{
	gtk_editable_cut_clipboard(GTK_EDITABLE(main_v->current_document->textbox));
	main_v->current_document->modified = 1;
	if (main_v->current_document->highlightstate) {
		document_need_highlighting(main_v->current_document);
	}
}

/*****************************************************/

void sel_all_cb(GtkWidget * w, gpointer data)
{
	gtk_editable_select_region(GTK_EDITABLE(main_v->current_document->textbox), 0,
							   gtk_text_get_length(GTK_TEXT(main_v->current_document->textbox)));
}

/*****************************************************/

void all_documents_update_links(Tdocument *curdoc, gchar *oldfilename, gchar *newfilename) {
	gchar *eff_newfilename, *eff_oldfilename;
	GList *tmplist;
	gint count = bf_statusbar_message(_("Updating Links in all Documents"));

	eff_newfilename = most_efficient_filename(g_strdup(newfilename));
	eff_oldfilename = most_efficient_filename(g_strdup(oldfilename));
	tmplist = g_list_first(main_v->documentlist);
	while (tmplist) {
		if ((Tdocument *)tmplist->data == curdoc) {
		
		} else {
			update_filenames_in_file((Tdocument *)tmplist->data, eff_oldfilename, eff_newfilename, 0);
		}
		tmplist = g_list_next(tmplist);
	}
	g_free(eff_newfilename);
	g_free(eff_oldfilename);
	statusbar_remove(GINT_TO_POINTER(count));
}

/**********************************************/

/*
 * Function: file_save_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	This call back function is check and dispath if need
 * 	to save or save as
 */

void file_save_cb(GtkWidget * widget, gpointer data)
{
	gint count;
	/* This is the callback function for a save button */
	DEBUG_MSG("file_save_cb, file save started\n");
	if (main_v->current_document->modified) {
		if (main_v->current_document->filename == NULL) {
			DEBUG_MSG("file_save_cb, No filename known --> save as..\n");
      	count = bf_statusbar_message(_("Save as..."));
			file_save_as_cb(widget, data);
		} else {
			count = bf_statusbar_message(_("Save in progress"));
			DEBUG_MSG("file_save_cb, Saving to %s\n", main_v->current_document->filename);
			textbox_to_file(main_v->current_document->filename);
		}
		statusbar_remove(GINT_TO_POINTER(count));
	} else {
		statusbar_message(_("File not modified, not saving"), 2);
	}
}

/*
 * Function: file_save_as_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Get new file name and small check
 */
void file_save_as_cb(GtkWidget * widget, gpointer data)
{
	gchar *oldfilename;

	oldfilename = main_v->current_document->filename;
	main_v->current_document->filename = return_file(NULL);
	if (main_v->current_document->filename == NULL) {
		main_v->current_document->filename = oldfilename;
		DEBUG_MSG("file_save_as_cb, returned file == NULL\n");
		error_dialog(_("Bluefish error"), _("No filename known"));
		DEBUG_MSG("file_save_as_cb, error dialog has happened? about to return;\n");
		return;
	} else {
		DEBUG_MSG("file_save_as_cb, returned file %s\n", main_v->current_document->filename);
		if (main_v->props.link_management) {
			update_filenames_in_file(main_v->current_document, oldfilename, main_v->current_document->filename, 1);
		}
		textbox_to_file(main_v->current_document->filename);
		gtk_label_set(GTK_LABEL(main_v->current_document->tab_label), strip_filename(main_v->current_document->filename));
		g_free(oldfilename);
	}
}

void file_move_to_cb(GtkWidget * widget, gpointer data) {
	gchar *oldfilename;

	oldfilename = main_v->current_document->filename;
	main_v->current_document->filename = return_file(NULL);
	if (main_v->current_document->filename == NULL) {
		main_v->current_document->filename = oldfilename;
		DEBUG_MSG("file_move_to_cb, returned file == NULL\n");
		error_dialog(_("Bluefish error"), _("No filename known"));
		DEBUG_MSG("file_save_as_cb, error dialog has happened? about to return;\n");
		return;
	} else {
		DEBUG_MSG("file_move_to_cb, returned file %s\n", main_v->current_document->filename);
		if (main_v->props.link_management) {
			update_filenames_in_file(main_v->current_document, oldfilename, main_v->current_document->filename,1 );
		}
		textbox_to_file(main_v->current_document->filename);
		gtk_label_set(GTK_LABEL(main_v->current_document->tab_label), strip_filename(main_v->current_document->filename));
		unlink(oldfilename);
		all_documents_update_links(main_v->current_document, oldfilename, main_v->current_document->filename);
		g_free(oldfilename);
	}
}

/*
 * Function: file_open_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Display multiple file open dialog and load multiple files (if selected)
 */
void file_open_cb(GtkWidget * widget, gpointer data)
{
	gint count;
	GList *tmplist;
	DEBUG_MSG("file_open_cb, started\n");
	count = bf_statusbar_message(_("starting file dialog"));
	tmplist = return_files(NULL);
	statusbar_remove(GINT_TO_POINTER(count));
	if (!tmplist) {
		return;
	}
	tmplist = g_list_first(tmplist);
	while (tmplist) {
		file_open((gchar *) tmplist->data);
		tmplist = g_list_next(tmplist);
	}
	free_stringlist(tmplist);
}


/*
 * Function: file_open
 * Arguments:
 * 	filename	- the filename to open
 * Return value:
 * 	void
 * Description:
 * 	Load the file
 */
void file_open(gchar * filename)
{
	if ((filename == NULL) || (!file_exists_and_readable(filename))) {
		return;
	} else {
		DEBUG_MSG("file_open, filename=%s exists\n", filename);
		main_v->current_document = new_document();
		main_v->current_document->filename = g_strdup(filename);
		file_to_textbox(main_v->current_document->filename);
		gtk_label_set(GTK_LABEL(main_v->current_document->tab_label), strip_filename(main_v->current_document->filename));
	}
}

/*
 * Function: file_insert_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Display dialog for file name and insert entered file in editor
 */
void file_insert_cb(GtkWidget * widget, gpointer data)
{
	gchar *tmpfilename;
	gint currentp;

	tmpfilename = return_file(NULL);
	if (tmpfilename == NULL) {
		error_dialog(_("Bluefish error"), _("No filename known"));
		return;
	} else {
		currentp = gtk_editable_get_position(GTK_EDITABLE(main_v->current_document->textbox));
		gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox), currentp);
		file_to_textbox(tmpfilename);
		g_free(tmpfilename);
	}
}

/*
 * Function: file_new_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Create new editor notebook
 */
void file_new_cb(GtkWidget * widget, gpointer data)
{

	/* This is the callback function for a file new button */
	DEBUG_MSG("file_new_cb, started\n");
	main_v->current_document = new_document();
	if (file_exists_and_readable(main_v->current_project.template) == 1) {
		file_to_textbox(main_v->current_project.template);
	}
	DEBUG_MSG("file_new_cb, end, current_document=%p, current_document->filename=%p\n",main_v->current_document ,main_v->current_document->filename);
}

/*
 * Function: file_close_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Close current editor notebook, make some checking
 */
void file_close_cb(GtkWidget * widget, gpointer data)
{
	gchar *text;
	gint retval;
	
	/* This is the callback function for a file close button */
	DEBUG_MSG("file_close_cb, begin, current_document=%p\n", main_v->current_document);

	if (main_v->current_document != NULL) {
		if (!main_v->current_document->modified) {
#ifdef DEBUG
			if (main_v->current_document->filename) {
				g_print("file_close_cb, filename=%s, not modified, destroy it\n",main_v->current_document->filename );
			} else {
				g_print("file_close_cb, filename=NULL, not modified, destroy it\n");
			}
#endif
			destroy_current_document();
		} else {
			DEBUG_MSG("file_close_cb, modified\n");
			if (main_v->current_document->filename) {
				text = g_strdup_printf(_("Are you sure you want to close\n%s ?"), main_v->current_document->filename);
			} else {
				text = g_strdup("Are you sure you want to close\nthis untitled file ?");
			}
			retval = close_save_cancel_dialog(_("Bluefish warning: file is modified!"), text);
			g_free(text);
			switch (retval) {
			case 3:
				return;
				DEBUG_MSG("file_close_cb, retval=3, returning\n");
				break;
			case 2:
				file_save_cb(NULL, NULL);
				if (main_v->current_document->modified == 1) {
					/* something went wrong it's still not saved */
					return;
				}
				destroy_current_document();
				break;
			case 1:
				destroy_current_document();
				break;
			default:
				return;			/* something went wrong */
				break;
			}
		}
	}
#ifdef DEBUG
	else {
		DEBUG_MSG("file_close_cb, cannot close NULL document\n");
	}
#endif
	notebook_changed();

	if (main_v->current_document == NULL) {
		DEBUG_MSG("file_close_cb, current_document=NULL -> start new document\n");
		file_new_cb(NULL, NULL);
	}
	DEBUG_MSG("file_close_cb, finished\n");
}

/*
 * Function: file_close_all_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Close all editor notebooks
 */
void file_close_all_cb(GtkWidget * widget, gpointer data)
{

	gint count;
	GList *tmplist;
	gchar *text;
	gint retval;

	DEBUG_MSG("file_close_all_cb, started\n");
	/* this can be made far more efficient !!!!!!!! */
	count = g_list_length(main_v->documentlist) - 1;
	while (count > 0) {
		main_v->current_document = NULL;
		main_v->current_document = g_list_nth_data(main_v->documentlist, count);
		if (main_v->current_document == NULL) {
			g_print("file_close_all, a NULL document in the documentlist\n");
		} else {
			if (main_v->current_document->modified == 1) {
#ifdef DEBUG
				if (main_v->current_document->filename) {
					g_print("file_close_all, filename=%s, not modified, destroy it\n",main_v->current_document->filename );
				} else {
					g_print("file_close_all, filename=NULL, not modified, destroy it\n");
				}
#endif
				if (main_v->current_document->filename) {
					text = g_strdup_printf(_("Are you sure you want to close\n%s ?"), main_v->current_document->filename);
				} else {
					text = g_strdup("Are you sure you want to close\nthis untitled file ?");
				}
				retval = close_save_cancel_dialog(_("Bluefish warning: file is modified!"), text);
				g_free(text);			
				switch (retval) {
				case 3:
					DEBUG_MSG("file_close_all_cb, retval=3, returning!!\n");
					return;
					break;
				case 2:
					file_save_cb(NULL, NULL);
					if (main_v->current_document->modified == 1) {
						/* something went wrong it's still not saved */
						return;
					}
					break;
				case 1:
					/* just continue */
					break;
				default:
					return;		/* something went wrong */
					break;
				}
			}
		}
		count--;
	}
	DEBUG_MSG("file_close_all_cb, after the warnings, now close all the windows\n");

	tmplist = g_list_first(main_v->documentlist);
	while (tmplist) {
		main_v->current_document = (Tdocument *) tmplist->data;
		if (main_v->current_document != NULL) {
			destroy_current_document();
		}
#ifdef DEBUG
		else {
			DEBUG_MSG("file_close_all_cb, ***WARNING*** **THIS SHOULD NEVER HAPPEN** main_v->current_document==NULL\n");
			g_list_remove(main_v->documentlist, tmplist);
			DEBUG_MSG("file_close_all_cb, ***WARNING*** **THIS SHOULD NEVER HAPPEN**\n");
		}
#endif
		tmplist = g_list_first(main_v->documentlist);
	}
	notebook_changed();
	if (main_v->current_document == NULL) {
		/* File close all, keep one document open */
		file_new_cb(NULL, NULL);
	}
	DEBUG_MSG("file_close_all_cb, finished\n");
}


/*
 * Function: file_save_all_cb
 * Arguments:
 * 	widget	- callback widget
 * 	data	- data for callback function
 * Return value:
 * 	void
 * Description:
 * 	Save all editor notebooks
 */
void file_save_all_cb(GtkWidget * widget, gpointer data)
{

	gint count;

	count = g_list_length(main_v->documentlist) - 1;
	while (count > 0) {
		main_v->current_document = NULL;
		main_v->current_document = g_list_nth_data(main_v->documentlist, count);
		if (main_v->current_document != NULL) {
			if (main_v->current_document->modified) {
				if (main_v->current_document->filename == NULL) {
					DEBUG_MSG("file_save_all, No filename known --> save as..\n");
					file_save_as_cb(widget, data);
				} else {
					DEBUG_MSG("file_save_all, Saving to %s\n", main_v->current_document->filename);
					textbox_to_file(main_v->current_document->filename);
				}
			}
		} else {
			DEBUG_MSG("file_save_all, main_v->current_document==NULL, count=%d\n", count);
		}
		count--;
	}
}


static void go_to_line_lcb(GtkWidget * widget, gpointer data)
{
	gchar *linestr;
	gint linenum;
	GtkWidget *entry;

	if (data) {
		entry = (GtkWidget *) data;
		linestr = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
		linenum = atoi(linestr);
		doc_go_to_line(linenum, 1);
		DEBUG_MSG("go_to_line_cb, going to line %d (string:%s)\n", linenum, linestr);
	}
	gtk_widget_destroy(GTK_WIDGET(gtk_widget_get_toplevel(widget)));
}


void go_to_line_win_cb(GtkWidget * widget, gpointer data)
{
	GtkWidget *win, *table, *entry;
	GtkWidget *but1, *but2;
	win = window_with_title(_("Goto line"), GTK_WIN_POS_MOUSE, GTK_WINDOW_DIALOG, 5);

	table = gtk_table_new(3, 2, 0);
	gtk_container_add(GTK_CONTAINER(win), table);

	entry = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(_("Line number: ")), 0, 2, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 0, 2, 1, 2);
	but1 = bf_stock_ok_button(go_to_line_lcb, entry);
	but2 = bf_stock_cancel_button(go_to_line_lcb, NULL);
	gtk_table_attach_defaults(GTK_TABLE(table), but1, 0, 1, 2, 3);
	gtk_table_attach_defaults(GTK_TABLE(table), but2, 1, 2, 2, 3);
	gtk_widget_show_all(win);
}

typedef struct {
	char id;
	char *entity;
} Tchar_entity;

static Tchar_entity ascii_charset[] = {
	{34, "&quot;"}, {38, "&amp;"}, {60, "&lt;"}, {62, "&gt;"}, {0, NULL}
};

static Tchar_entity iso8859_1_charset[] = {
	{34, "&quot;"}, {38, "&amp;"}, {60, "&lt;"}, {62, "&gt;"}
	, {160, "&nbsp;"}, {161, "&iexcl;"}, {162, "&cent;"}, {163, "&pound;"},
		{164, "&curren;"}, {165, "&yen;"}, {166, "&brvbar;"}, {167, "&sect;"},
		{168, "&uml;"}, {169, "&copy;"}, {170, "&ordf;"}, {171, "&laquo;"},
		{172, "&not;"}, {173, "&shy;"}, {174, "&reg;"}, {175, "&macr;"},
		{176, "&deg;"}, {177, "&plusmn;"}, {178, "&sup2;"}, {179, "&sup3;"},
		{180, "&acute;"}, {181, "&micro;"}, {182, "&para;"}, {183, "&middot;"},
		{184, "&cedil;"}, {185, "&sup1;"}, {186, "&ordm;"}, {187,"&raquo;"},
		{188, "&frac14;"}, {189, "&frac12;"}, {190, "&frac34;"}, {191,"&iquest;"},
		{192, "&Agrave;"}, {193, "&Aacute;"}, {194, "&Acirc;"}, {195, "&Atilde;"},
		{196, "&Auml;"}, {197, "&Aring;"}, {198, "&AElig;"}, {199,  "&Ccedil;"},
		{200, "&Egrave;"}, {201, "&Eacute;"}, {202, "&Ecirc;"}, {203,
																 "&Euml;"},
		{204, "&Igrave;"}, {205, "&Iacute;"}, {206, "&Icirc;"}, {207,
																 "&Iuml;"},
		{208, "&ETH;"}, {209, "&Ntilde;"}, {210, "&Ograve;"}, {211,
															   "&Oacute;"},
		{212, "&Ocirc;"}, {213, "&Otilde;"}, {214, "&Ouml;"}, {215,
															   "&times;"},
		{216, "&Oslash;"}, {217, "&Ugrave;"}, {218, "&Uacute;"}, {219,
																  "&Ucirc;"},
		{220, "&Uuml;"}, {221, "&Yacute;"}, {222, "&THORN;"}, {223,
															   "&szlig;"},
		{224, "&agrave;"}, {225, "&aacute;"}, {226, "&acirc;"}, {227,
																 "&atilde;"},
		{228, "&auml;"}, {229, "&aring;"}, {230, "&aelig;"}, {231,
															  "&ccedil;"},
		{232, "&egrave;"}, {233, "&eacute;"}, {234, "&ecirc;"}, {235,
																 "&euml;"},
		{236, "&igrave;"}, {237, "&iacute;"}, {238, "&icirc;"}, {239,
																 "&iuml;"},
		{240, "&eth;"}, {241, "&ntilde;"}, {242, "&ograve;"}, {243,
															   "&oacute;"},
		{244, "&ocirc;"}, {245, "&otilde;"}, {246, "&ouml;"}, {247,
															   "&divide;"},
		{248, "&oslash;"}, {249, "&ugrave;"}, {250, "&uacute;"}, {251,
																  "&ucirc;"},
		{252, "&uuml;"}, {253, "&yacute;"}, {254, "&thorn;"}, {255,
															   "&yuml;"}
	, {0, NULL}
};

/* charset 0 = ascii, 1 = iso8859-1 */

static void doc_special_char_replace(Tdocument * document, gint startpos,
							  gint endpos, gint charset)
{
	gint i, j, contj;
	char curchar;
	Tchar_entity *set;

	if (endpos == -1) {
		endpos = gtk_text_get_length(GTK_TEXT(document->textbox));
	}

	if (endpos < startpos) {
		i = startpos;
		startpos = endpos;
		endpos = i;
	}
	if (charset == 1) {
		set = &iso8859_1_charset[0];
	} else {
		set = &ascii_charset[0];
	}
	DEBUG_MSG("doc_special_char_replace, startpos=%d, endpos=%d, length=%d\n", startpos, endpos, gtk_text_get_length(GTK_TEXT(document->textbox)));
	gtk_text_freeze(GTK_TEXT(document->textbox));
	for (i = startpos; i <= endpos; i++) {
		curchar = (GTK_TEXT_INDEX(GTK_TEXT(document->textbox), i));
		DEBUG_MSG("doc_special_char_replace, i=%d, curchar=%d\n", i, curchar);
		if (!curchar) {
			DEBUG_MSG("doc_special_char_replace, char %d doesn exist\n", i);
			gtk_text_thaw(GTK_TEXT(document->textbox));
			return;
		}
		j = 0;
		contj = (set[j].entity != NULL);
		while (contj) {
			if (set[j].id == curchar) {
				gtk_text_set_point(GTK_TEXT(document->textbox), i);
				gtk_text_forward_delete(GTK_TEXT(document->textbox), 1);
				gtk_text_insert(GTK_TEXT(document->textbox), NULL, NULL,
								NULL, set[j].entity,
								strlen(set[j].entity));
				endpos += strlen(set[j].entity) -1;
				contj = 0;
			} else {
				j++;
				contj = (set[j].entity != NULL);
			}
		}
	}
	gtk_text_thaw(GTK_TEXT(document->textbox));
	DEBUG_MSG("doc_special_char_replace, finished\n");
}


static void special_char_replace(gint charset)
{
	gint start, end, count;
	GtkWidget *textbox = main_v->current_document->textbox;

  count = bf_statusbar_message(_("Charset conversion running..."));

	if (GTK_EDITABLE(textbox)->has_selection) {
		start = GTK_EDITABLE(textbox)->selection_start_pos;
		end = GTK_EDITABLE(textbox)->selection_end_pos;
	} else {
		start = 0;
		end = -1;
	}
	doc_special_char_replace(main_v->current_document, start, end,
							 charset);
	if (main_v->current_document->highlightstate) {
			highlight_once(main_v->current_document);
	}
	statusbar_remove(GINT_TO_POINTER(count));
}

void iso8859_1_replace_cb(GtkWidget * widget, gpointer data)
{
	special_char_replace(1);
}

void ascii_replace_cb(GtkWidget * widget, gpointer data)
{
	special_char_replace(0);
}
