/* Bluefish HTML Editor
 * bluefish.c - main source file
 * contains main function and some general often used functions
 *
 * Copyright (C) 1998 Olivier Sessink and Chris Mazuc
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "default_include.h"

#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <locale.h>

#if HAVE_LIBGDK_IMLIB
#include <gdk_imlib.h>
#endif

#include "bluefish.h"
#include "init.h"				/* parse commandline */
#include "gtk_easy.h"			/* window_destroy */
#include "interface.h"
#include "callbacks.h"

Tmain *main_v;

/****************************************************************************/

void cya_later(GtkWidget * widget, gpointer data);
int main(gint argc, gchar * argv[]);
gboolean change_dir(gchar * filename);
gchar *trunc_on_char(gchar * string, gchar which_char);
char *strip_filename(gchar * string);
gchar *strip_common_path(char *image_fn, char *html_fn);
void window_close(GtkWidget * button, gpointer windowname);
gint file_exists_and_readable(gchar * filename);
void flush_queue(void);
gchar *return_file(gchar * set_file);
GList *return_files(gchar * setfile);
static void modal_multiplefs_ok_clicked_lcb(GtkWidget * widget, gpointer data);
static void modal_fs_ok_clicked_lcb(GtkWidget * widget, gpointer window);
static void close_modal_window_lcb(GtkWidget * widget, gpointer window);




/****************************************************************************/
void cya_later(GtkWidget * widget, gpointer data)
{
	DEBUG_MSG("cya_later, calling project_close_cb(NULL, NULL);\n");
	project_close_cb(NULL, NULL);
	DEBUG_MSG("cya_later, calling file_close_all_cb(NULL, NULL);\n");
	file_close_all_cb(NULL, NULL);
	if (g_list_length(main_v->documentlist) > 0) {
		if ((g_list_length(main_v->documentlist) == 1)
			&& (main_v->current_document->modified == 0)) {
			DEBUG_MSG("cya_later, doesn't matter, 1 file and not modified\n");
		} else {
			DEBUG_MSG("cya_later, not all documents are closed\n");
			return;
		}
	}
#ifdef DEVELOPMENT
/* for memory leak debugging */
	gtk_widget_destroy(main_v->main_window);
	flush_queue();
	g_free(main_v);
#endif
	gtk_main_quit();
}

int main(gint argc, gchar * argv[])
{

	GList *tmplist;

	/* init locale before any *_init */
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	/* for XIM support we need to call gtk_set_locale() before gdk_init() */
	gtk_set_locale();
	gdk_init(&argc, &argv);

#if HAVE_LIBGDK_IMLIB
	/* Immediately after initialising GDK, Initialise Imlib */
	gdk_imlib_init();
#endif

	gtk_init(&argc, &argv);

	parse_commandline(argc, argv);

	/* see if we're running as root, if so, exit with an error unless overriden */
	if (root_override != 1) {
		if (geteuid() == 0) {
			g_print(_("Error: cannot run as root (for security reasons)\n"));
			g_print(_("use -s to override\n"));
			/* effective uid == 0 and the real uid != 0 means it's suid root */
			/* give an additional message if this is the case */
			if (getuid() != 0) {
				g_print(_("Bluefish is SUID root, this is a security risk!\n"));
			}
			exit(EXIT_FAILURE);
		}
	}

	DEBUG_MSG("main, 1 g_get_home_dir=%s\n", g_get_home_dir());

	/*initialise the main struct */
	main_v = g_malloc0(sizeof(Tmain));

	/* display a splash screen with progress bar
	 * also creates the main window, reads the config, 
	 * checks directories etc... 
	 */
	startup();
	DEBUG_MSG("main, startup() finished\n");
	if (load_filenames) {
		tmplist = g_list_first(load_filenames);
		while (tmplist) {
			file_open((gchar *) tmplist->data);
			tmplist = g_list_next(tmplist);
		}
		/* the strings don't need to be freed, they're needed in the project anyway */
		g_list_free(load_filenames);
	}
	DEBUG_MSG("main, load file from commandline finished\n");
	if ((load_projectname) && (file_exists_and_readable(load_projectname))) {
		main_v->current_project.filename = load_projectname;
		/* no need to free anymore */
		project_from_file();
	}
	DEBUG_MSG("main, going into gtk_main()\n");
	gtk_main();
	gtk_exit(0);
	return (0);
}

/*
 * Function: change_dir
 * Arguments:
 *      filename - name of file or directory
 * Return value:
 *      FALSE - if error
 *      TRUE - if successful
 * Description:
 *      Will search directory in current path and try change
 *      current path to that directory
 */
gboolean change_dir(gchar * filename)
{
	gchar *tmpstring;
	gboolean returncode;

	if (!filename)
		return FALSE;
	tmpstring = g_dirname(filename);
	DEBUG_MSG("change_dir, filename = %s, tmpstring = %s in %p\n", filename, tmpstring, tmpstring);
	returncode = chdir(tmpstring) == 0 ? TRUE : FALSE;
	g_free(tmpstring);
	DEBUG_MSG("change_dir, after gfree(tmpstring) \n");
	return returncode;
};

/*
 * Function: trunc_on_char
 * Arguments:
 *      string - text string
 *      which_char      - termination character
 * Return value:
 *      pointer same as string
 * Description:
 *      Find which_char and truncate string 'string' on it
 */
gchar *trunc_on_char(gchar * string, gchar which_char)
{
	gint count;
	for (count = 0; count <= strlen(string); count++) {
		if (string[count] == which_char) {
			string[count] = '\0';
			return string;
		};
	};
	return string;
};

/*
 * Function: strip_filename
 * Arguments:
 *      string - file name with full path
 * Return value:
 *      new allocated string with file name only
 * Description:
 *      select basename
 */
char *strip_filename(gchar * string)
{
	return g_basename(string);
}

/* this function is useful in changing absolute paths to relative paths */
gchar *strip_common_path(char *image_fn, char *html_fn)
{
	gchar *tempstr;
	int count, count2, dir_length;

	count = 0;
	tempstr = strrchr(image_fn, '/');
	dir_length = strlen(image_fn) - strlen(tempstr);
	dir_length += 1;
	DEBUG_MSG("strip_common_path, dir_lenght=%d\n", dir_length);
	while ((strncmp(image_fn, html_fn, count + 1)) == 0) {
		count++;
		if (count > dir_length) {
			count = dir_length;
			break;
		}
	}

	/* contributed by Tony Kemp <tony.kemp@studentmail.newcastle.edu.au> */
	while (image_fn[count - 1] != '/')
		count--;


	DEBUG_MSG("strip_common_path, equal count = %d\n", count);
	count2 = strlen(image_fn);
	tempstr = g_malloc(count2 - count + 2);
	memcpy(tempstr, &image_fn[count], count2 - count + 2);
	DEBUG_MSG("strip_common_path, tempstr= %s, should be %d long\n", tempstr, count2 - count);
	return tempstr;
	/* the function that calls this one must use gfree() somewhere!! */
}
/*
 * Function: most_efficient_filename
 * Arguments:
 *      filename: the string which will be modified to remove dir/../
 * Return value:
 *      the same string, but modified this time
 * Description:
 *      remove not wanted dir/../ from filenames
 */
gchar *most_efficient_filename(gchar *filename) {
	gint i,j, len;

	DEBUG_MSG("most_efficient_filename, 1 filename=%s\n", filename);
	len = strlen(filename);
	for (i=0; i < len-4; i++) {
/*		DEBUG_MSG("most_efficient_filename, i=%d\n", i); */
		if (strncmp(&filename[i], "/../", 4) == 0) {
			for (j=1; j < i; j++) {
				if ((filename[i - j] == '/') || (i==j)) {
					DEBUG_MSG("most_efficient_filename, &filename[%d-%d]=%s, &filename[%d+3]=%s, %d-%d-4=%d\n", i,j,&filename[i-j],i, &filename[i+3], len, j, len-j-4);
					memmove(&filename[i-j], &filename[i+3], len-i);
					j=i;
					i--;
				}
			}
		}
	}
	DEBUG_MSG("most_efficient_filename, 3 filename=%s\n", filename);
	return filename;
}

/*
 * Function: create_relative_link_to
 * Arguments:
 * 	current_filepath: the file to link FROM, a full path
 * 	link_to_filepath: the file to link TO, a full path
 * Return value:
 *      a newly allocated string containing a relative path
 * Description:
 *      create a relative link
 */
gchar *create_relative_link_to(gchar * current_filepath, gchar * link_to_filepath)
{
/* feature of this function: existing dir/../ are not removed,
 but they should be removed! */

	gchar *returnstring = NULL;
	gchar *eff_current_filepath, *eff_link_to_filepath;
	gint common_lenght, maxcommonlen;
	gint current_filename_length, link_to_filename_length, current_dirname_length, link_to_dirname_length;
	gint count, deeper_dirs;

	if (!current_filepath){
		returnstring = most_efficient_filename(g_strdup(link_to_filepath));
		return returnstring;
	}
	if (!link_to_filepath) {
		return NULL;
	}
	eff_current_filepath = most_efficient_filename(g_strdup(current_filepath));
	eff_link_to_filepath = most_efficient_filename(g_strdup(link_to_filepath));

	/* get the size of the directory    of the current_filename */

	
	current_filename_length = strlen(strrchr(eff_current_filepath, '/'))-1;
	link_to_filename_length = strlen(strrchr(eff_link_to_filepath, '/'))-1;
	DEBUG_MSG("create_relative_link_to, filenames: current: %d, link_to:%d\n", current_filename_length,link_to_filename_length); 
	current_dirname_length = strlen(eff_current_filepath) - current_filename_length;
	link_to_dirname_length = strlen(eff_link_to_filepath) - link_to_filename_length;
	DEBUG_MSG("create_relative_link_to, dir's: current: %d, link_to:%d\n", current_dirname_length, link_to_dirname_length); 

	if (current_dirname_length < link_to_dirname_length) {
		maxcommonlen = current_dirname_length;
	} else {
		maxcommonlen = link_to_dirname_length;
	}
	
	/* first lets get the common basedir for both dir+file  by comparing the
	   common path in the directories */
	common_lenght = 0;
	while ((strncmp(eff_current_filepath, eff_link_to_filepath, common_lenght + 1)) == 0) {
		common_lenght++;
		if (common_lenght >= maxcommonlen) {
			common_lenght = maxcommonlen;
			break;
		}
	}
	DEBUG_MSG("create_relative_link_to, common_lenght=%d\n", common_lenght);
	/* now we need to count how much deeper (in directories) the current_filename
	   is compared to the link_to_file, that is the amount of ../ we need to add */
	deeper_dirs = 0;
	for (count = common_lenght; count <= current_dirname_length; count++) {
		if (eff_current_filepath[count] == '/') {
			deeper_dirs++;
			DEBUG_MSG("create_relative_link_to, on count=%d, deeper_dirs=%d\n", count, deeper_dirs);
		}
	}
	DEBUG_MSG("create_relative_link_to, deeper_dirs=%d\n", deeper_dirs);

	/* now we know everything we need to know we can create the relative link */
	returnstring = g_malloc0(strlen(link_to_filepath) - common_lenght + 3 * deeper_dirs + 1);
	count = deeper_dirs;
	while (count) {
		strcat(returnstring, "../");
		count--;
	}
	strcat(returnstring, &eff_link_to_filepath[common_lenght]);
	DEBUG_MSG("create_relative_link_to, returnstring=%s\n", returnstring);
	g_free(eff_current_filepath);
	g_free(eff_link_to_filepath);
	return returnstring;
}

/*
 * Function: create_full_path
 * Arguments:
 * 	filename: the relative filename
 * 	basedir: a dir or file in the dir used as basedir for the filename
 * 		if this is NULL the current working directory is used
 * Return value:
 *      a newly allocated string containing a absolute path
 * Description:
 *      create a absolute path
 */
gchar *create_full_path(gchar * filename, gchar *basedir)
{
	gchar *absolute_filename;
	gchar *tmpcdir;

	DEBUG_MSG("create_full_path, filename=%s, basedir=%p\n", filename, basedir);
	if (g_path_is_absolute(filename)) {
		absolute_filename = g_strdup(filename);
	} else {
		if (basedir) {
			tmpcdir = g_dirname(basedir);
		} else {
			tmpcdir = g_get_current_dir();
		}
		absolute_filename = g_malloc0(strlen(tmpcdir) + strlen(filename) + 2);
		absolute_filename = strcat(strcat(strcat(absolute_filename, tmpcdir), "/"), filename);
		g_free(tmpcdir);
	}
	absolute_filename = most_efficient_filename(absolute_filename);
	return absolute_filename;
}

gint file_exists_and_readable(gchar * filename)
{
	struct stat naamstat;

	if (filename == NULL) {
		DEBUG_MSG("file_exists_and_readable, filename == NULL!!!!\n");
		return 0;
	}

	if (strlen(filename) < 2) {
		DEBUG_MSG("file_exists_and_readable, strlen(filename) < 2!!!!\n");
		return 0;
	}
	DEBUG_MSG("file_exists_and_readable, filename(%p)=\"%s\", strlen(filename)=%d\n", filename, filename, strlen(filename));
	if ((stat(filename, &naamstat) == -1) && (errno == ENOENT)) {
/*              if(naamstat.st_mode & S_IREAD) { */
		return (0);
/*              }
   return(0); */
	} else {
		return (1);
	}
}


/* forces everything to be flushed */
void flush_queue(void)
{
	while (gtk_events_pending()) {
		gtk_main_iteration();
	}
	while (gdk_events_pending()) {
		gdk_flush();
	}
}

/*
 * These are the functions to return a file using modal windows,
 * just give a call to return_file() and you get a filename, or
 * you get NULL if non selected
 */

/* just for the modal file selection, a glocal gchar and GList pointer */
static gchar *filename_to_return;
static GList *filenames_to_return;

gchar *return_file(gchar * set_file)
{

	GtkWidget *fs;
	gchar *tmp;
	DEBUG_MSG("return file, started\n");
	fs = gtk_file_selection_new(_("Select file"));

	gtk_signal_connect(GTK_OBJECT(fs), "destroy", close_modal_window_lcb, fs);
	/* Connect the ok_button to file_ok_sel function */
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", (GtkSignalFunc) modal_fs_ok_clicked_lcb, fs);
	/* Connect the cancel_button to destroy the widget */
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
					   "clicked", (GtkSignalFunc) close_modal_window_lcb, GTK_OBJECT(fs));
	if (set_file != NULL) {
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), set_file);
	}
	gtk_widget_show(fs);
	gtk_grab_add(GTK_WIDGET(fs));
	gtk_widget_realize(GTK_WIDGET(fs));
	gtk_window_set_transient_for(GTK_WINDOW(fs), GTK_WINDOW(main_v->main_window));
	gtk_main();
	tmp = filename_to_return;
	filename_to_return = NULL;
	return tmp;
}

/*
 * These are the functions to return multiple file using modal windows,
 * just give a call to return_file() and you get the filenames, or
 * you get NULL if non selected
 */

GList *return_files(gchar * setfile)
{
	GtkWidget *fs;
	GList *tmp;

	DEBUG_MSG("return files2, started\n");
	filenames_to_return = NULL;
	fs = gtk_file_selection_new(_("Select files"));

	gtk_signal_connect(GTK_OBJECT(fs), "destroy", close_modal_window_lcb, fs);
	/* Connect the ok_button to file_ok_sel function */
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", (GtkSignalFunc) modal_multiplefs_ok_clicked_lcb, fs);
	/* Connect the cancel_button to destroy the widget */
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
					   "clicked", (GtkSignalFunc) modal_multiplefs_ok_clicked_lcb, GTK_OBJECT(fs));
	gtk_clist_set_selection_mode(GTK_CLIST(GTK_FILE_SELECTION(fs)->file_list), GTK_SELECTION_EXTENDED);
	gtk_widget_show(fs);
	gtk_grab_add(GTK_WIDGET(fs));
	gtk_widget_realize(GTK_WIDGET(fs));
	gtk_window_set_transient_for(GTK_WINDOW(fs), GTK_WINDOW(main_v->main_window));
	gtk_main();
	tmp = filenames_to_return;
	filenames_to_return = NULL;
	return tmp;
}

static void modal_multiplefs_ok_clicked_lcb(GtkWidget * widget, gpointer data)
{
	GtkFileSelection *fs;
	gchar *dirname;

	fs = GTK_FILE_SELECTION(data);
	dirname = g_dirname(gtk_file_selection_get_filename(fs));
	{
		gint rownum = 0;
		gchar *temp, *totalfilename;
		GList *row = GTK_CLIST(fs->file_list)->row_list;
		while (row) {
			if (GTK_CLIST_ROW(row)->state == GTK_STATE_SELECTED) {
				gtk_clist_get_text(GTK_CLIST(fs->file_list), rownum, 0, &temp);
				DEBUG_MSG("modal_multifs_ok_clicked_lcb, temp=%s\n", temp);
				totalfilename = g_strconcat(dirname, "/", temp, NULL);
				filenames_to_return = g_list_append(filenames_to_return, totalfilename);
			}
			DEBUG_MSG("modal_multifs_ok_clicked_lcb, rownum=%d\n", rownum);
			rownum++;
			row = g_list_next(row);
		}
	}
	g_free(dirname);
	gtk_main_quit();
	window_destroy(data);
}


static void modal_fs_ok_clicked_lcb(GtkWidget * widget, gpointer window)
{
	filename_to_return = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(window)));
	gtk_main_quit();
	window_destroy(window);
}

static void close_modal_window_lcb(GtkWidget * widget, gpointer window)
{
	if (filename_to_return != NULL) {
		g_free(filename_to_return);
		filename_to_return = NULL;
	}
	gtk_main_quit();
	window_destroy(window);
}
