/* Bluefish HTML Editor
 * menu.c - the home of the pulldowns
 *
 * Copyright (C) 1998-1999 Olivier Sessink, Chris Mazuc and Roland Steinbach
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "default_include.h"

#include <locale.h>

#include <string.h>
#include <dirent.h>
#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>

#include "bluefish.h"
#include "callbacks.h"
#include "html.h"
#include "html2.h"
#include "document.h"			/* insert_text() */
#include "interface.h"			/* show_about_window_cb() */
#include "stringlist.h" 		/* edit_stringlist() */
#include "init.h" 				/* save_config_files() */
#include "snr2.h" 				/* the new search and replace */
#include "wml.h"
#include "images.h"
#include "wizards.h"
#include "ref_dialog.h"
#include "configure.h"
#include "undo.h"
#include "gtk_easy.h"
#include "highlight.h"
#include "toolbars.h"


typedef struct {
	gchar *name;
	GList *entry_list;
	GtkWidget *item;
	GtkWidget *subtree;
} cust_main_menu_struct;

typedef struct {
	gchar *name;
	gchar *bef_html;
	gchar *aft_html;
	GtkWidget *item;
	GtkWidget *subtree;
	cust_main_menu_struct *main_entry;
} cust_menu_struct;

/***************************************************************************/
void save_menu_shortcuts();
void load_menu_shortcuts();
static void edit_some_list_lcb(GtkWidget * widget, gpointer data);
gchar *menu_translate(const gchar * path, gpointer data);
static void insert_char_cb(GtkWidget * widget, gint data);
void get_main_menu(GtkWidget * window, GtkWidget ** menubar);
static void toggle_cust_menubar_lcb(GtkWidget * w, gpointer data);
static void hide_cust_menubar_cb(GtkWidget * w, gpointer data);
void show_cust_menubar_cb(GtkWidget * w, gpointer data);
static void remake_cust_menubar(void);
static void make_cust_menubar(void);
static void item_response(cust_menu_struct * entry);
static GList *parse_cust_menu_file(char *filename);
static void save_cust_menu(void);
static void edit_custom_menu(GtkWidget * w, gpointer data);
static void main_entry_new(GtkWidget * w, gpointer * tree);
static void entry_new(GtkWidget * w, gpointer * tree);
static void main_entry_update(GtkWidget * w, gpointer * tree);
static void entry_update(GtkWidget * w, gpointer * tree);
static void main_entry_del(GtkWidget * w, gpointer * tree);
static void entry_del(GtkWidget * w, gpointer * tree);
static void entry_save(GtkWidget * w, gpointer * window);
static void entry_selected(GtkWidget * item, cust_menu_struct * entry);
static void main_entry_selected(GtkWidget * item, cust_main_menu_struct * main_entry);



/***************************************************************************/

/* I wish there was a better way to show he names of the different chars */
/* as in actually show them in parenthesis */
static GtkItemFactoryEntry menu_items[] = {
	{N_("/_File"), NULL, NULL, 0, "<Branch>"},
	{N_("/File/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/File/_New"), "<control>N", file_new_cb, 0, NULL},
	{N_("/File/_Open"), "<control>O", file_open_cb, 0, NULL},
	{N_("/File/Open from the web"), NULL, open_from_web_cb, 0, NULL},
	{N_("/File/Insert"), NULL, file_insert_cb, 0, NULL},
	{N_("/File/sep2"), NULL, NULL, 0, "<Separator>"},
	{N_("/File/_Save"), "<control>S", file_save_cb, 0, NULL},
	{N_("/File/Save _as"), NULL, file_save_as_cb, 0, NULL},
	{N_("/File/Move to"), NULL, file_move_to_cb, 0, NULL},
	{N_("/File/Save all"), NULL, file_save_all_cb, 0, NULL},
	{N_("/File/sep3"), NULL, NULL, 0, "<Separator>"},
	{N_("/File/Close"), "<control>W", file_close_cb, 0, NULL},
	{N_("/File/Close all"), NULL, file_close_all_cb, 0, NULL},
	{N_("/File/sep4"), NULL, NULL, 0, "<Separator>"},
	{N_("/File/Quit"), "<control>Q", cya_later, 0, NULL},
	{N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
	{N_("/_Edit/Tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Edit/Cut"), "<control>x", cut_cb, 0, NULL},
	{N_("/Edit/Copy"), "<control>c", copy_cb, 0, NULL},
	{N_("/Edit/Paste"), "<control>v", paste_cb, 0, NULL},
	{N_("/Edit/sep2"), NULL, NULL, 0, "<Separator>"},
	{N_("/Edit/Select all"), "<control>a", sel_all_cb, 0, NULL},
	{N_("/Edit/Go to line"), NULL, go_to_line_win_cb, 0, NULL},
	{N_("/Edit/sep3"), NULL, NULL, 0, "<Separator>"},
	{N_("/Edit/Find"), "F6", search_cb, 0, NULL},
	{N_("/Edit/New find"), NULL, new_search_cb, 0, NULL},
	{N_("/Edit/Find again"), "<shift>F6", search_again_cb, 0, NULL},
	{N_("/Edit/Replace"), "F7", replace_cb, 0, NULL},
	{N_("/Edit/New replace"), NULL, new_replace_cb, 0, NULL},
	{N_("/Edit/Replace again"), "<shift>F7", replace_again_cb, 0, NULL},
	{N_("/Edit/sep3"), NULL, NULL, 0, "<Separator>"},
	{N_("/Edit/Undo"), "<control>z", undo_cb, 0, NULL},
	{N_("/Edit/Redo"), "<control>r", redo_cb, 0, NULL},
	{N_("/Edit/Undo all"), NULL, undo_all_cb, 0, NULL},
	{N_("/Edit/Redo all"), NULL, redo_all_cb, 0, NULL},
	{N_("/_View"), NULL, NULL, 0, "<Branch>"},
	{N_("/_View/Tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/View/View in _Netscape"), NULL, view_in_netscape_cb, 0, NULL},
	{N_("/View/sep1"), NULL, NULL, 0, "<Separator>"},
	{N_("/View/Next document"), "F1", notebook_next_document_cb, 0, NULL},
	{N_("/View/Previous document"), "F2", notebook_prev_document_cb, 0, NULL},
	{N_("/View/sep3"), NULL, NULL, 0, "<Separator>"},
	{N_("/View/Highlight syntax"), NULL, toggle_highlight_syntax_cb, 0, "<ToggleItem>"},
	{N_("/View/Refresh highlighting"), "F5", refresh_hi_cb, 0, NULL},
	{N_("/_Project"), NULL, NULL, 0, "<Branch>"},
	{N_("/Project/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Project/Open"), "<control>P", project_open_cb, 0, NULL},
	{N_("/Project/Close"), NULL, project_close_cb, 0, NULL},
	{N_("/Project/Save"), NULL, project_save_cb, 0, NULL},
	{N_("/Project/Save as"), NULL, project_save_as_cb, 0, NULL},
	{N_("/Project/sep1"), NULL, NULL, 0, "<Separator>"},
	{N_("/Project/Edit"), NULL, project_edit_cb, 0, NULL},
	{N_("/Project/sep2"), NULL, NULL, 0, "<Separator>"},
	{N_("/Project/Add current document"), NULL, project_add_document_cb, 0, NULL},
	{N_("/Project/Add all open documents"), NULL, project_add_document_cb, 1, NULL},
	{N_("/_Project/Files"), NULL, NULL, 0, "<Branch>"},
	{N_("/Project/Files/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/_Tags"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Headings"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Headings/Tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Headings/H1"), "<alt>F1", general_html_cb, 18, NULL},
	{N_("/Tags/Headings/H2"), "<alt>F2", general_html_cb, 19, NULL},
	{N_("/Tags/Headings/H3"), "<alt>F3", general_html_cb, 20, NULL},
	{N_("/Tags/Headings/H4"), "<alt>F4", general_html_cb, 21, NULL},
	{N_("/Tags/Headings/H5"), "<alt>F5", general_html_cb, 22, NULL},
	{N_("/Tags/Headings/H6"), "<alt>F6", general_html_cb, 23, NULL},
	{N_("/Tags/Special"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Accent/Uppercase"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Uppercase/A-I"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Uppercase/A-I/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A grave"), NULL, insert_char_cb, 0, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A acute"), NULL, insert_char_cb, 1, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A circumflex"), NULL, insert_char_cb, 2, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A tilde"), NULL, insert_char_cb, 3, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/AE ligature"), NULL, insert_char_cb, 4, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A diaeresis"), NULL, insert_char_cb, 5, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/A ring"), NULL, insert_char_cb, 6, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/C cedilla"), NULL, insert_char_cb, 7, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/E grave"), NULL, insert_char_cb, 8, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/E acute"), NULL, insert_char_cb, 9, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/E circumflex"), NULL, insert_char_cb, 10, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/E diaeresis"), NULL, insert_char_cb, 11, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/I grave"), NULL, insert_char_cb, 12, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/I acute"), NULL, insert_char_cb, 13, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/I circumflex"), NULL, insert_char_cb, 14, NULL},
	{N_("/Tags/Special/Accent/Uppercase/A-I/I diaeresis"), NULL, insert_char_cb, 15, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/N tilde"), NULL, insert_char_cb, 16, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O grave"), NULL, insert_char_cb, 17, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O acute"), NULL, insert_char_cb, 18, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O circumflex"), NULL, insert_char_cb, 19, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O tilde"), NULL, insert_char_cb, 20, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O diaeresis"), NULL, insert_char_cb, 21, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/O slash"), NULL, insert_char_cb, 22, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/U grave"), NULL, insert_char_cb, 23, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/U acute"), NULL, insert_char_cb, 24, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/U circumflex"), NULL, insert_char_cb, 25, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/U diaeresis"), NULL,
	 insert_char_cb, 26, NULL},
	{N_("/Tags/Special/Accent/Uppercase/J-Z/Y acute"), NULL, insert_char_cb,
	 27, NULL},
	{N_("/Tags/Special/Accent/Lowercase"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Lowercase/A-I"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Lowercase/A-I/tearoff1"), NULL, NULL, 0,
	 "<Tearoff>"},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a grave"), NULL, insert_char_cb,
	 28, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a acute"), NULL, insert_char_cb,
	 29, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a circumflex"), NULL,
	 insert_char_cb, 30, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a tilde"), NULL, insert_char_cb,
	 31, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a ring"), NULL, insert_char_cb,
	 32, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/ae ligature"), NULL,
	 insert_char_cb, 33, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/a diaeresis"), NULL,
	 insert_char_cb, 34, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/c cedilla"), NULL, insert_char_cb,
	 35, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/e grave"), NULL, insert_char_cb,
	 36, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/e acute"), NULL, insert_char_cb,
	 37, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/e circumflex"), NULL,
	 insert_char_cb, 38, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/e diaeresis"), NULL,
	 insert_char_cb, 39, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/i grave"), NULL, insert_char_cb,
	 40, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/i acute"), NULL, insert_char_cb,
	 41, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/i circumflex"), NULL,
	 insert_char_cb, 42, NULL},
	{N_("/Tags/Special/Accent/Lowercase/A-I/i diaeresis"), NULL,
	 insert_char_cb, 43, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/tearoff1"), NULL, NULL, 0,
	 "<Tearoff>"},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/n tilde"), NULL, insert_char_cb,
	 44, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o grave"), NULL, insert_char_cb,
	 45, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o acute"), NULL, insert_char_cb,
	 46, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o circumflex"), NULL,
	 insert_char_cb, 47, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o tilde"), NULL, insert_char_cb,
	 48, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o diaeresis"), NULL,
	 insert_char_cb, 49, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/o slash"), NULL, insert_char_cb,
	 50, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/u grave"), NULL, insert_char_cb,
	 51, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/u acute"), NULL, insert_char_cb,
	 52, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/u circumflex"), NULL,
	 insert_char_cb, 53, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/u diaeresis"), NULL,
	 insert_char_cb, 54, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/y acute"), NULL, insert_char_cb,
	 55, NULL},
	{N_("/Tags/Special/Accent/Lowercase/J-Z/y diaeresis"), NULL,
	 insert_char_cb, 56, NULL},
	{N_("/Tags/Special/Accent/separator"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Special/Accent/Umlaut"), NULL, insert_char_cb, 57, NULL},
	{N_("/Tags/Special/Accent/Acute"), NULL, insert_char_cb, 58, NULL},
	{N_("/Tags/Special/Accent/Cedilla"), NULL, insert_char_cb, 59, NULL},
	{N_("/Tags/Special/Currency"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Currency/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Currency/Cent sign"), NULL, insert_char_cb,
	 60, NULL},
	{N_("/Tags/Special/Currency/Pound sterling"), NULL, insert_char_cb,
	 61, NULL},
	{N_("/Tags/Special/Currency/Currency sign"), NULL, insert_char_cb,
	 62, NULL},
	{N_("/Tags/Special/Currency/Yen sign"), NULL, insert_char_cb, 63, NULL},
	{N_("/Tags/Special/Math-Science"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Math-Science/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Math-Science/Logical not sign"), "<alt>1", insert_char_cb,
	 64, NULL},
	{N_("/Tags/Special/Math-Science/Multiplication sign"), "<alt>8",
	 insert_char_cb, 65, NULL},
	{N_("/Tags/Special/Math-Science/Division sign"), "<alt>slash", insert_char_cb,
	 66, NULL},
	{N_("/Tags/Special/Math-Science/Plus-minus sign"), NULL, insert_char_cb,
	 67, NULL},
	{N_("/Tags/Special/Math-Science/Less-than sign"), "<alt>comma", insert_char_cb,
	 68, NULL},
	{N_("/Tags/Special/Math-Science/Greater-than sign"), "<alt>period", insert_char_cb,
	 69, NULL},
	{N_("/Tags/Special/Math-Science/Superscript 1"), NULL, insert_char_cb,
	 70, NULL},
	{N_("/Tags/Special/Math-Science/Superscript 2"), NULL, insert_char_cb,
	 71, NULL},
	{N_("/Tags/Special/Math-Science/Superscript 3"), NULL, insert_char_cb,
	 72, NULL},
	{N_("/Tags/Special/Math-Science/One quarter"), NULL, insert_char_cb,
	 73, NULL},
	{N_("/Tags/Special/Math-Science/One half"), NULL, insert_char_cb,
	 74, NULL},
	{N_("/Tags/Special/Math-Science/Three quarters"), NULL, insert_char_cb,
	 75, NULL},
	{N_("/Tags/Special/Math-Science/Degree sign"), NULL, insert_char_cb,
	 76, NULL},
	{N_("/Tags/Special/Non Latin"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Non Latin/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Non Latin/ETH (Icelandic)"), NULL, insert_char_cb,
	 77, NULL},
	{N_("/Tags/Special/Non Latin/THORN (Icelandic)"), NULL, insert_char_cb,
	 78, NULL},
	{N_("/Tags/Special/Non Latin/eth (Icelandic)"), NULL, insert_char_cb,
	 79, NULL},
	{N_("/Tags/Special/Non Latin/thorn"), NULL, insert_char_cb, 80, NULL},
	{N_("/Tags/Special/Non Latin/Sharp s"), NULL, insert_char_cb, 81, NULL},
	{N_("/Tags/Special/Non Latin/Micro sign"), NULL, insert_char_cb,
	 82, NULL},
	{N_("/Tags/Special/Other"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Special/Other/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Special/Other/Non-breaking space"), NULL, insert_char_cb,
	 83, NULL},
	{N_("/Tags/Special/Other/Section sign"), NULL, insert_char_cb,
	 84, NULL},
	{N_("/Tags/Special/Other/Copyright sign"), NULL, insert_char_cb,
	 85, NULL},
	{N_("/Tags/Special/Other/Left angle quotes"), NULL, insert_char_cb,
	 86, NULL},
	{N_("/Tags/Special/Other/Right angle quotes"), NULL, insert_char_cb,
	 87, NULL},
	{N_("/Tags/Special/Other/Registered trademark"), NULL, insert_char_cb,
	 88, NULL},
	{N_("/Tags/Special/Other/Inverted exclamation"), NULL, insert_char_cb,
	 89, NULL},
	{N_("/Tags/Special/Other/Inverted question mark"), NULL, insert_char_cb,
	 90, NULL},
	{N_("/Tags/Special/Other/Feminine ordinal"), NULL, insert_char_cb,
	 91, NULL},
	{N_("/Tags/Special/Other/Masculine ordinal"), NULL, insert_char_cb,
	 92, NULL},
	{N_("/Tags/Special/Other/Pilcrow (paragraph sign)"), NULL, insert_char_cb,
	 93, NULL},
	{N_("/Tags/Special/Other/Broken bar"), NULL, insert_char_cb, 94, NULL},
	{N_("/Tags/Special/Other/Soft hypen"), "<alt>minus", insert_char_cb, 95, NULL},
	{N_("/Tags/Special/Other/Spacing macron"), NULL, insert_char_cb,
	 96, NULL},
	{N_("/Tags/Special/Other/Middle dot"), NULL, insert_char_cb,
	 97, NULL},
	{N_("/Tags/Formatting"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Formatting/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Formatting/Bold"), "<alt>b", general_html_cb, 1, NULL},
	{N_("/Tags/Formatting/Italic"), "<alt>i", general_html_cb, 2, NULL},
	{N_("/Tags/Formatting/Underline"), "<alt>u", general_html_cb, 3, NULL},
	{N_("/Tags/Formatting/Strikeout"), "<alt>s", general_html_cb, 4, NULL},
	{N_("/Tags/Formatting/Strong"), NULL, general_html_cb, 16, NULL},
	{N_("/Tags/Formatting/Emphasis"), NULL, general_html_cb, 17, NULL},
	{N_("/Tags/Formatting/<separator>"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Formatting/Paragraph"), "<alt>P", general_html_cb, 5, NULL},
	{N_("/Tags/Formatting/Break"),  "<Control><Alt>r", general_html_cb, 6, NULL},
	{N_("/Tags/Formatting/Break clear all"), NULL, general_html_cb, 41, NULL},
	{N_("/Tags/Formatting/Non-Breaking Space"), "<alt>n", general_html_cb, 7, NULL},
	{N_("/Tags/Formatting/<separator>"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Formatting/Font Size +1"), "<control>equal", general_html_cb, 11, NULL},
	{N_("/Tags/Formatting/Font Size -1"), "<control>minus", general_html_cb, 12, NULL},
	{N_("/Tags/Formatting/<separator>"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Formatting/Preformatted Text"), "<Control><Alt>p", general_html_cb, 13, NULL},
	{N_("/Tags/Formatting/Subscript"), "<alt><shift>s", general_html_cb, 14, NULL},
	{N_("/Tags/Formatting/Superscript"), "<alt><control>s", general_html_cb, 15, NULL},
	{N_("/Tags/Formatting/<separator>"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Formatting/Center"), "<alt>c", general_html_cb, 8, NULL},
	{N_("/Tags/Formatting/Align right"), "<Alt>r", general_html_cb, 9, NULL},
	{N_("/Tags/Formatting/<separator>"), NULL, NULL, 0, "<Separator>"},
	{N_("/Tags/Formatting/Comment"), NULL, general_html_cb, 10, NULL},
	{N_("/Tags/Table"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/Table/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/Table/Table"), NULL, general_html_cb, 24, NULL},
	{N_("/Tags/Table/Tablerow"), NULL, general_html_cb, 25, NULL},
	{N_("/Tags/Table/Tableheader"), NULL, general_html_cb, 26, NULL},
	{N_("/Tags/Table/Tabledata"), NULL, general_html_cb, 27, NULL},
	{N_("/Tags/Table/Tablecaption"), NULL, general_html_cb, 28, NULL},
	{N_("/Tags/List"), NULL, NULL, 0, "<Branch>"},
	{N_("/Tags/List/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Tags/List/Unordered list"), NULL, general_html_cb, 33, NULL},
	{N_("/Tags/List/Ordered list"), NULL, general_html_cb, 34, NULL},
	{N_("/Tags/List/List item"), "<alt>l", general_html_cb, 35, NULL},
	{N_("/Tags/List/Definition list"), NULL, general_html_cb, 36, NULL},
	{N_("/Tags/List/Definition term"), "<alt>d", general_html_cb, 37, NULL},
	{N_("/Tags/List/Definition"), NULL, general_html_cb, 38, NULL},
	{N_("/Tags/List/menu"), NULL, general_html_cb, 39, NULL},
	{N_("/Tags/Replace iso 8859-1 chars"), NULL, iso8859_1_replace_cb, 39, NULL},
	{N_("/Tags/Replace ascii chars"), NULL, ascii_replace_cb, 39, NULL},
	{N_("/_Dialogs"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/General"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/General/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/General/Quickstart"), "<alt>q", quickstart, 0, NULL},
	{N_("/Dialogs/General/DTD"), NULL, dtd_cb, 0, NULL},
	{N_("/Dialogs/General/Head"), NULL, head_cb, 0, NULL},
	{N_("/Dialogs/General/Body"), NULL, body_cb, 0, NULL},
	{N_("/Dialogs/General/Anchor"), "<alt>a", quickanchor_cb, 0, NULL},
	{N_("/Dialogs/General/Email"), NULL, email_cb, 0, NULL},
	{N_("/Dialogs/General/Rule"), "<alt>h", quickrule_cb, 0, NULL},
	{N_("/Dialogs/General/Font"), "<alt>f", fontdialog_cb, 0, NULL},
	{N_("/Dialogs/General/Basefont"), NULL, basefont_cb, 0, NULL},
	{N_("/Dialogs/General/Quicklist"), NULL, quicklist_cb, 0, NULL},
	{N_("/Dialogs/General/Meta"), "<alt>m", meta_cb, 0, NULL},
	{N_("/Dialogs/General/Embed"), NULL, embed_cb, 0, NULL},
	{N_("/Dialogs/General/Select colour"), NULL, sel_colour_cb, 0, NULL},
	{N_("/Dialogs/General/Insert time"), NULL, insert_time_cb, 0, NULL},
	{N_("/Dialogs/General/Insert image"), "<control>i", image_insert_dialog_cb, 0, NULL},
	{N_("/Dialogs/General/Insert thumbnail"), "<control>t", image_thumbnail_dialog_cb, 0, NULL},
	{N_("/Dialogs/Table"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/Table/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/Table/Table"), NULL, tabledialog_cb, 0, NULL},
	{N_("/Dialogs/Table/Tablerow"), NULL, tablerowdialog_cb, 0, NULL},
	{N_("/Dialogs/Table/Tablehead_cber"), NULL, tableheaddialog_cb, 0, NULL},
	{N_("/Dialogs/Table/Tabledata"), NULL, tabledatadialog_cb, 0, NULL},
	{N_("/Dialogs/CSS"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/CSS/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/CSS/Create style"), NULL, insert_in_stylesheet, 0, NULL},
	{N_("/Dialogs/CSS/Span"), NULL, insert_span, 0, NULL},
	{N_("/Dialogs/CSS/Div"), NULL, insert_div, 0, NULL},
	{N_("/Dialogs/CSS/Add class"), NULL, insert_class, 0, NULL},
	{N_("/Dialogs/CSS/Style"), NULL, general_html_cb, 42, NULL},
	{N_("/Dialogs/CSS/Link to stylesheet"), NULL, general_html_cb, 46, NULL},
	{N_("/Dialogs/Frame"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/Frame/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/Frame/Frameset"), NULL, framesetdialog_cb, 0, NULL},
	{N_("/Dialogs/Frame/Frame"), NULL, framedialog_cb, 0, NULL},
	{N_("/Dialogs/Form"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/Form/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/Form/Form"), NULL, formdialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Submit"), NULL, submitdialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Reset"), NULL, resetdialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Text"), NULL, textdialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Hidden"), NULL, hiddendialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Textarea"), NULL, textareadialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Radio buttons"), NULL, radiodialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Check buttons"), NULL, checkdialog_cb, 0, NULL},
	{N_("/Dialogs/Form/Select"), NULL, selectdialog_cb, 0, NULL},
	{N_("/Dialogs/_WML"), NULL, NULL, 0, "<Branch>"},
	{N_("/Dialogs/WML/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Dialogs/WML/Standard Document"), NULL, general_wml_cb, 6, NULL},
	{N_("/Dialogs/WML/Card"), NULL, carddialog_cb, 0, NULL},
	{N_("/Dialogs/WML/Postfield"), NULL, postfielddialog_cb, 0, NULL},
	{N_("/Dialogs/WML/Go"), NULL, godialog_cb, 0, NULL},
	{N_("/Dialogs/WML/Do"), NULL, dodialog_cb, 0, NULL},
	{N_("/Dialogs/WML/Anchor"), NULL, anchordialog_cb, 0, NULL},
	{N_("/Dialogs/WML/Access"), NULL, accessdialog_cb, 0, NULL},
	{N_("/Dialogs/WML/sep11"), NULL, NULL, 0, "<Separator>"},
	{N_("/Dialogs/WML/Paragraph"), NULL, general_wml_cb, 1, NULL},
	{N_("/Dialogs/WML/Line Break"), NULL, general_wml_cb, 2, NULL},
	{N_("/Dialogs/WML/Italic"), NULL, general_wml_cb, 3, NULL},
	{N_("/Dialogs/WML/Non-Breaking Space"), NULL, general_wml_cb, 4, NULL},
	{N_("/Dialogs/WML/Bold"), NULL, general_wml_cb, 5, NULL},
	{N_("/Dialogs/WML/sep12"), NULL, NULL, 0, "<Separator>"},
	{N_("/Dialogs/WML/Prev"), NULL, general_wml_cb, 7, NULL},
	{N_("/Dialogs/WML/Refresh"), NULL, general_wml_cb, 8, NULL},
	{N_("/Dialogs/WML/Noop"), NULL, general_wml_cb, 9, NULL},
	{N_("/Dialogs/WML/sep13"), NULL, NULL, 0, "<Separator>"},
	{N_("/Dialogs/WML/Set Variable"), NULL, vardialog_cb, 0, NULL},
	{N_("/_External"), NULL, NULL, 0, "<Branch>"},
	{N_("/External/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/External/Weblint"), NULL, run_weblint_cb, 0, NULL},
	{N_("/External/sep1"), NULL, NULL, 0, "<Separator>"},
	{N_("/External/Filters"), NULL, NULL, 0, "<Branch>"},
	{N_("/External/Filters/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/External/Commands"), NULL, NULL, 0, "<Branch>"},
	{N_("/External/Commands/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/_Options"), NULL, NULL, 0, "<Branch>"},
	{N_("/Options/tearoff1"), NULL, NULL, 0, "<Tearoff>"},
	{N_("/Options/Preferences"), NULL, configure_cb, 0, NULL},
	{N_("/Options/View Main toolbar"), NULL, toggle_main_toolbar_lcb, 0, "<ToggleItem>"},
	{N_("/Options/View HTML toolbar"), NULL, toggle_html_toolbar_lcb, 0, "<ToggleItem>"},
	{N_("/Options/View Custom toolbar"), NULL, toggle_cust_menubar_lcb, 0, "<ToggleItem>"},
	{N_("/Options/sep2"), NULL, NULL, 0, "<Separator>"},
	{N_("/Options/_Lists"), NULL, NULL, 0, "<Branch>"},
	{N_("/Options/Lists/External filters"), NULL, edit_some_list_lcb, 1, NULL},
	{N_("/Options/Lists/External programs"), NULL, edit_some_list_lcb, 2, NULL},
	{N_("/Options/Lists/Syntax highlighting"), NULL, edit_some_list_lcb, 3, NULL},
	{N_("/Options/Lists/Reset syntax highlighting"), NULL, reset_syntax_highlighting_lcb, 0, NULL},
	{N_("/Options/sep1"), NULL, NULL, 0, "<Separator>"},
	{N_("/Options/Save config"), NULL, save_config_files, 0, NULL},
	{N_("/Options/Save shortcut key settings"), NULL, save_menu_shortcuts, 0, NULL},
	{N_("/_Help"), NULL, NULL, 0, "<Branch>"},
	{N_("/Help/_About"), NULL, show_about_window_cb, 0, NULL}
};

static gchar *sp_chars[] = {
	"&Agrave;", "&Aacute;", "&Acirc;", "&Atilde;", "&AElig;",
	"&Auml;", "&Aring;", "&Ccedil;", "&Egrave;", "&Eacute;",
	"&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", "&Icirc;",
	"&Iuml;", "&Ntilde;", "&Ograve;", "&Oacute;", "&Ocirc;",
	"&Otilde;", "&Ouml;", "&Oslash;", "&Ugrave;", "&Uacute;",
	"&Ucirc;", "&Uuml;", "&Yacute;", "&agrave;", "&aacute;",
	"&acirc;", "&atilde;", "&aring;", "&aelig;", "&auml;",
	"&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;",
	"&igrave;", "&iacute;", "&icircumflex;", "&iuml;", "&ntilde;",
	"&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;",
	"&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;",
	"&yacute;", "&yuml;", "&uml;", "&acute;", "&cedil;", "&cent;",
	"&pound;", "&curren;", "&yen;", "&not;", "&mult;", "&divide;",
	"&plusmn;", "&lt;", "&gt;", "&sup1;", "&sup2;", "&sup3;",
	"&frac14;", "&frac12;", "&frac34;", "&deg;", "&ETH;", "&THORN;",
	"&eth;", "&thorn;", "&szlig;", "&micro;", "&nbsp;", "&sect;",
	"&copy;", "&laquo;", "&raquo;", "&reg;", "&iexcl;", "&iquest;",
	"&ordf;", "&ordm;", "&para;", "&brvbar;", "&shy;", "&macr;",
	"&middot;"
};
#ifndef PATH_MAX
#define PATH_MAX 1024
#endif

void save_menu_shortcuts() {
	gchar *filename;
	
	filename = g_malloc(PATH_MAX);
	filename = strncat(strncpy(filename, g_get_home_dir(), PATH_MAX), "/.bluefish/menudump", PATH_MAX);
	gtk_item_factory_dump_rc(filename, NULL, TRUE);
	g_free(filename);
}

void load_menu_shortcuts() {
	gchar *filename;
	
	filename = g_malloc(PATH_MAX);
	filename = strncat(strncpy(filename, g_get_home_dir(), PATH_MAX), "/.bluefish/menudump", PATH_MAX);
	gtk_item_factory_parse_rc(filename);
	g_free(filename);
}


static void edit_some_list_lcb(GtkWidget * widget, gpointer data)
{
	switch (GPOINTER_TO_INT(data)) {
	case 1:
		main_v->props.external_filters = edit_stringlist(main_v->props.external_filters, _("Bluefish configuration - external filters"), 7);
		break;
	case 2:
		main_v->props.external_commands =
			edit_stringlist(main_v->props.external_commands, _("Bluefish configuration - external commands"), 7);
		break;
	case 3:
		main_v->props.syntax_configstrings =
			edit_stringlist(main_v->props.syntax_configstrings, _("Bluefish configuration - syntax highlighting"), 8);
		break;
	default:

		break;
	}
}


/* 
 * these widgets are NULL if the bars are not initialised yet,
 * they're !NULL after initialisation 
 */

GList *entry_list_list;
GtkWidget *entry1, *entry2, *entry3, *entry4;
extern GtkWidget *cust_widget;
cust_menu_struct *last_entry = NULL;
cust_main_menu_struct *last_main_entry = NULL;



/***************************************************************************/
/* takes care of inserting the special characters */
static void insert_char_cb(GtkWidget * widget, gint data)
{
	insert_text(sp_chars[data], "");
}

/* 
 * menu factory crap, thanks to the gtk tutorial for this
 * both the 1.0 and the 1.2 code is directly from the tutorial
 */
void get_main_menu(GtkWidget * window, GtkWidget ** menubar)
{
	GtkItemFactory *item_factory;
	GtkAccelGroup *accel_group;
	gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);
	accel_group = gtk_accel_group_new();
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);
#ifdef ENABLE_NLS
	gtk_item_factory_set_translate_func(item_factory, menu_translate, "<main>", NULL);
#endif
	gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
	gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);
	if (menubar)
		*menubar = gtk_item_factory_get_widget(item_factory, "<main>");
	setup_toggle_item(item_factory, "/Options/View Main toolbar", main_v->props.v_main_tb);
	setup_toggle_item(item_factory, "/Options/View HTML toolbar", main_v->props.v_html_tb);
	setup_toggle_item(item_factory, "/Options/View Custom toolbar", main_v->props.v_custom_tb);
	setup_toggle_item(item_factory, "/View/Highlight syntax", main_v->props.cont_highlight_update);
}


static void toggle_cust_menubar_lcb(GtkWidget * w, gpointer data)
{
	if (GTK_WIDGET_VISIBLE(cust_handle_box)) {
		hide_cust_menubar_cb(NULL, NULL);
	} else {
		show_cust_menubar_cb(NULL, NULL);
	}
}

static void hide_cust_menubar_cb(GtkWidget * w, gpointer data)
{
	gtk_widget_hide(cust_handle_box);
}

void show_cust_menubar_cb(GtkWidget * w, gpointer data)
{
	if (!cust_widget) {
		make_cust_menubar();
	}
	gtk_widget_show(cust_handle_box);
}

static void remake_cust_menubar(void)
{
	int count, count2;
	cust_main_menu_struct *main_entry;
	cust_menu_struct *entry;

	count = g_list_length(entry_list_list) - 1;
	DEBUG_MSG("remake_cust_menubar, count = %d\n", count);
	while (count > 0) {
		main_entry = g_list_nth_data(entry_list_list, count);
		count2 = g_list_length(main_entry->entry_list) - 1;
		while (count2 > 0) {
			entry = g_list_nth_data(main_entry->entry_list, count2);
			DEBUG_MSG("remake_cust_menubar, pos1, count=%d\n", count);
			g_free(entry->name);
			g_free(entry->bef_html);
			g_free(entry->aft_html);
			DEBUG_MSG("remake_cust_menubar, pos2\n");
			main_entry->entry_list = g_list_remove(main_entry->entry_list, entry);
			g_free(entry);
			DEBUG_MSG("remake_cust_menubar, pos3\n");
			count2--;
		}
		entry_list_list = g_list_remove(entry_list_list, main_entry);
		g_list_free(main_entry->entry_list);
		g_free(main_entry->name);
		g_free(main_entry);
		DEBUG_MSG("remake_cust_menubar, entry_list_list lenght = %d\n", g_list_length(entry_list_list));
		count--;
	}
	g_list_free(entry_list_list);
	gtk_widget_destroy(cust_widget);
	make_cust_menubar();
	DEBUG_MSG("remake_cust_menubar, finished\n");
}

static void make_cust_menubar(void)
{
	GtkWidget *menu, *item;
	int count;
	cust_menu_struct *entry;
	cust_main_menu_struct *main_entry;
	gchar dirname[256];
	gchar *tmpstring;
	DIR *dp;
	struct dirent *ep;

	DEBUG_MSG("make_cust_menubar, begin\n");
	entry_list_list = g_list_alloc();
	cust_widget = gtk_menu_bar_new();
	gtk_container_add(GTK_CONTAINER(cust_handle_box), cust_widget);
	gtk_widget_show(cust_widget);

	menu = gtk_menu_new();
	item = gtk_menu_item_new_with_label(_("edit"));
	gtk_menu_append(GTK_MENU(menu), item);
	gtk_signal_connect_object(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(edit_custom_menu), NULL);
	gtk_widget_show(item);
	item = gtk_menu_item_new_with_label(_("Edit menu"));
	gtk_widget_show(item);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);
	gtk_menu_bar_append(GTK_MENU_BAR(cust_widget), item);

	strncat(strcpy(dirname, getenv("HOME")), "/.bluefish/menu", 255);
	dp = opendir(dirname);
	if (dp != NULL) {
		while ((ep = readdir(dp))) {
			DEBUG_MSG("make_cust_menubar, filename=%s\n", ep->d_name);
			if ((strcmp(ep->d_name, "..") == 0) || (strcmp(ep->d_name, ".") == 0)) {
				DEBUG_MSG("make_cust_menubar, %s is no usable file\n", ep->d_name);
			} else {
				main_entry = g_malloc0(sizeof(cust_main_menu_struct));
				menu = gtk_menu_new();
				/* we don't need to show the menu */
				tmpstring = g_malloc0(270);
				main_entry->entry_list = parse_cust_menu_file(strcat(strcat(strcat(tmpstring, dirname), "/"), ep->d_name));
				entry_list_list = g_list_append(entry_list_list, main_entry);
				count = g_list_length(main_entry->entry_list) - 1;
#ifdef DEBUG
				if (main_entry->entry_list == NULL)
					DEBUG_MSG("make_cust_menubar, entry_list is NULL!!\n");
				DEBUG_MSG("make_cust_menubar, count=%d\n", count);
#endif
				while (count > 0) {
					DEBUG_MSG("make_cust_menubar, adding an entry, count=%d\n", count);
					entry = g_list_nth_data(main_entry->entry_list, count);
#ifdef DEBUG
					if (entry == NULL)
						DEBUG_MSG("make_cust_menubar, entry %d is NULL!!\n", count);
					DEBUG_MSG("make_cust_menubar, entry->name = %s\n", entry->name);
#endif
					item = gtk_menu_item_new_with_label(entry->name);
					gtk_menu_append(GTK_MENU(menu), item);
					gtk_signal_connect_object(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(item_response), (gpointer) entry);
					gtk_widget_show(item);
					DEBUG_MSG("make_cust_menubar, entry done\n");
					count--;
				}
				strcpy(tmpstring, strip_filename(ep->d_name));
				main_entry->name = g_malloc(strlen(tmpstring + 1));
				strcpy(main_entry->name, tmpstring);
				item = gtk_menu_item_new_with_label(tmpstring);
				g_free(tmpstring);
				gtk_widget_show(item);
				gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);
				gtk_menu_bar_append(GTK_MENU_BAR(cust_widget), item);
			}
		}
		closedir(dp);
	} else {
		DEBUG_MSG("make_cust_menubar, couldn't open the directory %s\n", dirname);
		return;
	}
	DEBUG_MSG("make_cust_menubar, finished\n");
}

static void item_response(cust_menu_struct * entry)
{
	DEBUG_MSG("item_response, entry->bef_html=%s, entry->aft_html=%s \n", entry->bef_html, entry->aft_html);
	insert_text(entry->bef_html, entry->aft_html);
}

static GList *parse_cust_menu_file(char *filename)
{

	GList *return_list;
	FILE *fd;
	gchar *tmpbuf, *tmpbuf2;
	cust_menu_struct *item_set;


	fd = fopen(filename, "r");
	if (fd == NULL) {
		DEBUG_MSG("parse_cust_menu_file, cannot open file %s\n", filename);
	}
	return_list = g_list_alloc();
	tmpbuf = g_malloc0(2045);
	DEBUG_MSG("parse_cust_menu_file, list lenght at start = %d\n", g_list_length(return_list));
	while (fgets(tmpbuf, 2044, fd) != NULL) {
		item_set = g_malloc0(sizeof(cust_menu_struct));
		tmpbuf2 = strrchr(tmpbuf, '\n');
		strcpy(tmpbuf2, "\0");
		tmpbuf2 = strrchr(tmpbuf, ':');
		item_set->aft_html = g_malloc0(strlen(tmpbuf2) + 1);
		DEBUG_MSG("parse_cust_menu_file, tmpbuf2 = %s\n", tmpbuf2);
		strcpy(item_set->aft_html, tmpbuf2 + 1);
		strcpy(tmpbuf2, "\0");
		tmpbuf2 = strrchr(tmpbuf, ':');
		item_set->bef_html = g_malloc0(strlen(tmpbuf2) + 1);
		DEBUG_MSG("parse_cust_menu_file, tmpbuf2 = %s\n", tmpbuf2);
		strcpy(item_set->bef_html, tmpbuf2 + 1);
		strcpy(tmpbuf2, "\0");
		item_set->name = g_malloc0(strlen(tmpbuf) + 1);
		DEBUG_MSG("parse_cust_menu_file, tmpbuf = %s\n", tmpbuf);
		strcpy(item_set->name, tmpbuf);
		return_list = g_list_append(return_list, item_set);
#ifdef DEBUG
		if (item_set == NULL)
			DEBUG_MSG("parse_cust_menu_file, item_set is NULL !!\n");
		DEBUG_MSG
			("parse_cust_menu_file, item_set->name = %s, item_set->bef_html = %s, item_set->aft_html = %s\n",
			 item_set->name, item_set->bef_html, item_set->aft_html);
		DEBUG_MSG("parse_cust_menu_file, list lenght is now = %d\n", g_list_length(return_list));
#endif
	}
	g_free(tmpbuf);
	return return_list;
}

static void save_cust_menu(void)
{
	gchar *filename, *tmpstring;
	gint count, count2;
	FILE *fd;
	cust_main_menu_struct *main_entry;
	cust_menu_struct *entry;

	DEBUG_MSG("save_cust_menu, started\n");
	count = g_list_length(entry_list_list) - 1;
	DEBUG_MSG("save_cust_menu, %d entries in entry_list_list\n", g_list_length(entry_list_list) - 1);
	filename = g_malloc(PATH_MAX + 1);
	tmpstring = g_malloc(2044);
	while (count > 0) {
		main_entry = g_list_nth_data(entry_list_list, count);
		strncat(strncat(strncpy(filename, getenv("HOME"), PATH_MAX), "/.bluefish/menu/", PATH_MAX), main_entry->name, PATH_MAX);
		fd = fopen(filename, "w");
		if (fd == NULL) {
			DEBUG_MSG("save_cust_menu, cannot open file %s\n", filename);
			return;
		}
		DEBUG_MSG("save_cust_menu, filename=%s\n", filename);
		DEBUG_MSG("save_cust_menu, fd=%p\n", fd);
		DEBUG_MSG("save_cust_menu, list lenght in main_entry %s = %d\n", filename, g_list_length(main_entry->entry_list));
		count2 = (g_list_length(main_entry->entry_list) - 1);
		while (count2 > 0) {
			entry = g_list_nth_data(main_entry->entry_list, count2);
			g_snprintf(tmpstring, 2044, "%s:%s:%s\n", entry->name, entry->bef_html, entry->aft_html);
			fputs(tmpstring, fd);
			DEBUG_MSG("save_cust_menu, entry %d, wrote %s", count2, tmpstring);
			count2--;
		}
		fclose(fd);
		count--;
	}
	g_free(filename);
	g_free(tmpstring);
	DEBUG_MSG("save_cust_menu, finished\n");
}


static void edit_custom_menu(GtkWidget * w, gpointer data)
{
/*
 * the custom menu is build out of a list, the entry_list_list
 * this list contains main_entry's, and every main_entry contains
 * again a list with entry's.
 * the items in the lists are crosslinked by pointers, pointing to their
 * item in the tree, the subtree they are connected to, and their main_entry.
 * 
 * The editor keeps track of the last selected entries with the pointers
 * last_entry and last_main_entry. This is very convenient when deleting
 * updating etc., you only need to check if that pointer != NULL and then
 * you can directly access all the values, and the part in the tree
 */


	GtkWidget *menu_window, *scrolled_win, *tree, *hbox, *hbox2, *vbox, *vbox2, *frame, *table;
	gint count, count2 = 0;
	cust_main_menu_struct *main_entry;
	cust_menu_struct *entry;
	GtkWidget *label, *but_cancel, *but_save;
	GtkWidget *but1, *but2, *but3, *but4, *but5, *but6;

	/* a generic toplevel window */
	menu_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(menu_window), "delete_event", GTK_SIGNAL_FUNC(gtk_widget_destroy), menu_window);
	gtk_container_border_width(GTK_CONTAINER(menu_window), 5);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(menu_window), vbox);
	gtk_widget_show(vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	/* A generic scrolled window */
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize(scrolled_win, 150, 200);
	gtk_box_pack_start(GTK_BOX(hbox), scrolled_win, FALSE, FALSE, 5);
	gtk_widget_show(scrolled_win);

	vbox2 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 5);
	gtk_widget_show(vbox2);

	/* make main menu frame + content */
	frame = gtk_frame_new(_("Menu"));
	gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 5);
	/* gtk_widget_set_usize(frame, 250, 200); */
	gtk_widget_show(frame);
	table = gtk_table_new(3, 2, TRUE);
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_container_add(GTK_CONTAINER(frame), table);
	label = gtk_label_new(_("Name"));
	entry1 = gtk_entry_new_with_max_length(200);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry1, 1, 3, 0, 1);
	gtk_widget_show(label);
	gtk_widget_show(entry1);
	/* buttons for main menu */
	but1 = gtk_button_new_with_label(_("Delete"));
	gtk_table_attach_defaults(GTK_TABLE(table), but1, 0, 1, 1, 2);
	gtk_widget_show(but1);
	but2 = gtk_button_new_with_label(_("Update"));
	gtk_table_attach_defaults(GTK_TABLE(table), but2, 1, 2, 1, 2);
	gtk_widget_show(but2);
	but3 = gtk_button_new_with_label(_("Add"));
	gtk_table_attach_defaults(GTK_TABLE(table), but3, 2, 3, 1, 2);
	gtk_widget_show(but3);
	gtk_widget_show(table);

	/* make menu entry frame + content */
	frame = gtk_frame_new(_("Entry"));
	gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 5);
	/* gtk_widget_set_usize(frame, 250, 200); */
	gtk_widget_show(frame);
	table = gtk_table_new(3, 4, TRUE);
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_container_add(GTK_CONTAINER(frame), table);
	label = gtk_label_new(_("Name"));
	entry2 = gtk_entry_new_with_max_length(200);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry2, 1, 3, 0, 1);
	gtk_widget_show(label);
	gtk_widget_show(entry2);
	label = gtk_label_new(_("Before"));
	entry3 = gtk_entry_new_with_max_length(200);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), entry3, 1, 3, 1, 2);
	gtk_widget_show(label);
	gtk_widget_show(entry3);
	label = gtk_label_new(_("After"));
	entry4 = gtk_entry_new_with_max_length(200);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	gtk_table_attach_defaults(GTK_TABLE(table), entry4, 1, 3, 2, 3);
	gtk_widget_show(label);
	gtk_widget_show(entry4);
	/* buttons for menu item */
	but4 = gtk_button_new_with_label(_("Delete"));
	gtk_table_attach_defaults(GTK_TABLE(table), but4, 0, 1, 3, 4);
	gtk_widget_show(but4);
	but5 = gtk_button_new_with_label(_("Update"));
	gtk_table_attach_defaults(GTK_TABLE(table), but5, 1, 2, 3, 4);
	gtk_widget_show(but5);
	but6 = gtk_button_new_with_label(_("Add"));
	gtk_table_attach_defaults(GTK_TABLE(table), but6, 2, 3, 3, 4);
	gtk_widget_show(but6);
	gtk_widget_show(table);

	/* Create the root tree */
	tree = gtk_tree_new();

	/* connect all buttons */
	gtk_signal_connect(GTK_OBJECT(but1), "clicked", GTK_SIGNAL_FUNC(main_entry_del), (gpointer) tree);
	gtk_signal_connect(GTK_OBJECT(but2), "clicked", GTK_SIGNAL_FUNC(main_entry_update), (gpointer) tree);
	gtk_signal_connect(GTK_OBJECT(but3), "clicked", GTK_SIGNAL_FUNC(main_entry_new), (gpointer) tree);
	gtk_signal_connect(GTK_OBJECT(but4), "clicked", GTK_SIGNAL_FUNC(entry_del), (gpointer) tree);
	gtk_signal_connect(GTK_OBJECT(but5), "clicked", GTK_SIGNAL_FUNC(entry_update), (gpointer) tree);
	gtk_signal_connect(GTK_OBJECT(but6), "clicked", GTK_SIGNAL_FUNC(entry_new), (gpointer) tree);


	/* Add it to the scrolled window */
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_win), tree);
	/* Set the selection mode */
	gtk_tree_set_selection_mode(GTK_TREE(tree), GTK_SELECTION_SINGLE);
	/* Show it */
	gtk_widget_show(tree);


	count = g_list_length(entry_list_list) - 1;
	while (count > 0) {
		/* loop trough the entry_list_list */
		main_entry = g_list_nth_data(entry_list_list, count);
		main_entry->item = gtk_tree_item_new_with_label(main_entry->name);
		/* Add it to the parent tree */
		gtk_tree_append(GTK_TREE(tree), main_entry->item);
		gtk_signal_connect(GTK_OBJECT(main_entry->item), "select", GTK_SIGNAL_FUNC(main_entry_selected), (gpointer) main_entry);
		/* Show it - this can be done at any time */
		gtk_widget_show(main_entry->item);
		/* Create this item's subtree */
		main_entry->subtree = gtk_tree_new();
		DEBUG_MSG("edit_custom_menu, main_entry->subtree=%p,main_entry->item=%p \n", main_entry->subtree, main_entry->item);
		gtk_tree_item_set_subtree(GTK_TREE_ITEM(main_entry->item), main_entry->subtree);
		count2 = g_list_length(main_entry->entry_list) - 1;
		while (count2 > 0) {
			/* loop trough list of submenu's, and add them to the tree */
			entry = g_list_nth_data(main_entry->entry_list, count2);
			entry->item = gtk_tree_item_new_with_label(entry->name);
			gtk_tree_append(GTK_TREE(main_entry->subtree), entry->item);
			entry->subtree = main_entry->subtree;
			entry->main_entry = main_entry;
			gtk_signal_connect(GTK_OBJECT(entry->item), "select", GTK_SIGNAL_FUNC(entry_selected), (gpointer) entry);
			gtk_widget_show(entry->item);
			count2--;
		}
		count--;
	}

	hbox2 = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
	gtk_widget_show(hbox2);

	but_save = bf_stock_ok_button(GTK_SIGNAL_FUNC(entry_save), (gpointer) menu_window);
	but_cancel = bf_stock_cancel_button(GTK_SIGNAL_FUNC(window_close_by_data_cb), (gpointer) menu_window);
	gtk_box_pack_start(GTK_BOX(hbox2), but_save, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), but_cancel, TRUE, TRUE, 0);
	gtk_widget_show(but_save);
	gtk_widget_show(but_cancel);
	DEBUG_MSG("edit_custom_menu, window = %p\n", menu_window);
	gtk_widget_show(menu_window);
}

static void main_entry_new(GtkWidget * w, gpointer * tree)
{
	gchar *tmpstring;
	cust_main_menu_struct *main_entry;

	DEBUG_MSG("main_entry_new, entry_list_list length=%d\n", g_list_length(entry_list_list));
	tmpstring = g_malloc(254);
	strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry1)), 253);
	main_entry = (cust_main_menu_struct *) g_malloc0(sizeof(cust_main_menu_struct));
	main_entry->name = g_malloc(strlen(tmpstring + 1));
	main_entry->entry_list = g_list_alloc();
	strcpy(main_entry->name, tmpstring);
	main_entry->item = gtk_tree_item_new_with_label(main_entry->name);
	/* Add it to the parent tree */
	gtk_tree_append(GTK_TREE(tree), main_entry->item);
	gtk_signal_connect(GTK_OBJECT(main_entry->item), "select", GTK_SIGNAL_FUNC(main_entry_selected), (gpointer) main_entry);
	/* Show it - this can be done at any time */
	gtk_widget_show(main_entry->item);
	/* Create this item's subtree */
	main_entry->subtree = gtk_tree_new();
	gtk_tree_item_set_subtree(GTK_TREE_ITEM(main_entry->item), main_entry->subtree);
	g_free(tmpstring);
	entry_list_list = g_list_append(entry_list_list, main_entry);
	DEBUG_MSG("main_entry_new, entry_list_list length=%d\n", g_list_length(entry_list_list));
}

static void entry_new(GtkWidget * w, gpointer * tree)
{
	gchar *tmpstring;
	cust_main_menu_struct *main_entry = NULL;
	cust_menu_struct *entry;
	GtkWidget *item, *add_to_item;
	GList *i;
	gint count;

	i = GTK_TREE_SELECTION(tree);
	item = GTK_WIDGET(i->data);
	DEBUG_MSG("entry_new, (item) i->data=%p, &i->data=%p\n", i->data, &i->data);
	g_list_free(i);
	gtk_tree_item_deselect(GTK_TREE_ITEM(item));

	/* this whole part can be replaced by using the last_entry and 
	   last_main_entry pointers */
	DEBUG_MSG("entry_new, last_entry=%p, last_main_entry=%p\n", last_entry, last_main_entry);
	if (GTK_TREE(item->parent)->level == 0) {
		add_to_item = item;
		DEBUG_MSG("entry_new, have the main entry in the tree add_to_item=%p\n", add_to_item);
	} else {
		add_to_item = GTK_TREE(item->parent)->tree_owner;
		DEBUG_MSG("entry_new, need to use the parent as the main entry in the tree, add_to_item=%p\n", add_to_item);
	}
	/* till here replacable */

	tmpstring = g_malloc(600);
	entry = (cust_menu_struct *) g_malloc0(sizeof(cust_menu_struct));
	strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry2)), 599);
	entry->name = g_malloc(strlen(tmpstring + 1));
	strcpy(entry->name, tmpstring);
	strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry3)), 599);
	entry->bef_html = g_malloc(strlen(tmpstring + 1));
	strcpy(entry->bef_html, tmpstring);
	strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry4)), 599);
	entry->aft_html = g_malloc(strlen(tmpstring + 1));
	strcpy(entry->aft_html, tmpstring);
	g_free(tmpstring);
	count = g_list_length(entry_list_list) - 1;
	while (count > 0) {
		main_entry = g_list_nth_data(entry_list_list, count);
		if (main_entry->item == add_to_item) {
			DEBUG_MSG
				("entry_new, lenght main_entry->entry_list=%d last_main_entry=%p\n",
				 g_list_length(main_entry->entry_list), last_main_entry);
			entry->main_entry = main_entry;
			main_entry->entry_list = g_list_append(main_entry->entry_list, entry);
			count = 0;
			DEBUG_MSG("entry_new, added to main entry %p with name = %s\n", main_entry, main_entry->name);
			DEBUG_MSG("entry_new, lenght main_entry->entry_list %d\n", g_list_length(main_entry->entry_list));
		}
		count--;
	}

	entry->item = gtk_tree_item_new_with_label(entry->name);
#ifdef DEBUG
	DEBUG_MSG("entry_new, main_entry=%p, last_main_entry=%p\n", last_main_entry, last_main_entry);
	if (GTK_IS_TREE(main_entry->subtree)) {
		DEBUG_MSG("entry_new, main_entry->subtree is a valid tree\n");
	}
#endif
	gtk_tree_append(GTK_TREE(main_entry->subtree), GTK_WIDGET(entry->item));
	entry->subtree = main_entry->subtree;
#ifdef DEBUG
	if (GTK_WIDGET_REALIZED(entry->item)) {
		DEBUG_MSG("entry_new, entry->item is realized\n");
	}
#endif
	gtk_signal_connect(GTK_OBJECT(entry->item), "select", GTK_SIGNAL_FUNC(entry_selected), (gpointer) entry);
	gtk_widget_show(entry->item);
	DEBUG_MSG("entry_new, child position=%d\n", gtk_tree_child_position(GTK_TREE(main_entry->subtree), entry->item));
	DEBUG_MSG("entry_new, lenght main_entry->entry_list %d\n", g_list_length(main_entry->entry_list));
	DEBUG_MSG("entry_new, finished, on subtree=%p, tree=%p\n", entry->subtree, tree);
}

static void main_entry_update(GtkWidget * w, gpointer * tree)
{
	gchar *tmpstring;

	if (last_main_entry != NULL) {
		/* free the memory of the old values */
		DEBUG_MSG("main_entry_update, start free-ing last_main_entry=%p\n", last_main_entry);
		g_free(last_main_entry->name);
		/* allocate the memory of the new values and insert the values */
		tmpstring = g_malloc(601);
		strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry1)), 600);
		last_main_entry->name = g_malloc(strlen(tmpstring) + 1);
		strcpy(last_main_entry->name, tmpstring);
		g_free(tmpstring);
		/* update the label on the tree */
		DEBUG_MSG("main_entry_update, update label in item=%p\n", last_main_entry->item);
		gtk_label_set(GTK_LABEL(GTK_BIN(last_main_entry->item)->child), last_main_entry->name);
	}
}

static void entry_update(GtkWidget * w, gpointer * tree)
{
	gchar *tmpstring;

	if (last_entry != NULL) {
		/* free the memory of the old values */
		DEBUG_MSG("entry_update, start free-ing last_entry=%p\n", last_entry);
		g_free(last_entry->name);
		g_free(last_entry->bef_html);
		g_free(last_entry->aft_html);
		/* allocate the memory of the new values and insert the values */
		tmpstring = g_malloc(601);
		strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry2)), 600);
		last_entry->name = g_malloc(strlen(tmpstring) + 1);
		strcpy(last_entry->name, tmpstring);
		strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry3)), 600);
		last_entry->bef_html = g_malloc(strlen(tmpstring) + 1);
		strcpy(last_entry->bef_html, tmpstring);
		strncpy(tmpstring, gtk_entry_get_text(GTK_ENTRY(entry4)), 600);
		last_entry->aft_html = g_malloc(strlen(tmpstring) + 1);
		strcpy(last_entry->aft_html, tmpstring);
		g_free(tmpstring);
		/* update the label on the tree */
		DEBUG_MSG("entry_update, update label in item=%p\n", last_entry->item);
		gtk_label_set(GTK_LABEL(GTK_BIN(last_entry->item)->child), last_entry->name);
	}
}

static void main_entry_del(GtkWidget * w, gpointer * tree)
{
	GList *i;
	gint count2;
	cust_menu_struct *cur_last_entry;

	if (last_main_entry != NULL) {
		DEBUG_MSG("main_entry_del, last_main_entry=%p\n", last_main_entry);
		g_free(last_main_entry->name);
		/* remove sub menu */
		cur_last_entry = last_entry;
		count2 = g_list_length(last_main_entry->entry_list) - 1;
		while (count2 > 0) {
			last_entry = g_list_nth_data(last_main_entry->entry_list, count2);
			entry_del(w, tree);
			count2--;
		}
		last_entry = cur_last_entry;
		entry_list_list = g_list_remove(entry_list_list, last_main_entry);
		g_list_free(last_main_entry->entry_list);
/*      gtk_tree_select_child(tree, last_main_entry->item); */
		i = GTK_TREE_SELECTION(tree);
		gtk_tree_remove_items(GTK_TREE(tree), i);
		g_free(last_main_entry);
		last_main_entry = NULL;
	}
}

static void entry_del(GtkWidget * w, gpointer * tree)
{
	GList *i;

	if (last_entry != NULL) {
		gtk_tree_item_select(GTK_TREE_ITEM(last_entry->item));
		g_free(last_entry->name);
		g_free(last_entry->bef_html);
		g_free(last_entry->aft_html);
		last_entry->main_entry->entry_list = g_list_remove(last_entry->main_entry->entry_list, last_entry);
		i = GTK_TREE_SELECTION(tree);
		gtk_tree_remove_items(GTK_TREE(tree), i);
		g_free(last_entry);
		last_entry = NULL;
	}
}

static void entry_save(GtkWidget * w, gpointer * window)
{
	save_cust_menu();
	remake_cust_menubar();
	DEBUG_MSG("entry_save, removing signal handlers from %p\n", window);
	gtk_signal_handlers_destroy(GTK_OBJECT(window));
	DEBUG_MSG("entry_save, about to destroy window %p\n", window);
	gtk_object_destroy(GTK_OBJECT(window));
	DEBUG_MSG("entry_save, finished\n");
}

static void entry_selected(GtkWidget * item, cust_menu_struct * entry)
{
	gtk_entry_set_text(GTK_ENTRY(entry2), entry->name);
	gtk_entry_set_text(GTK_ENTRY(entry3), entry->bef_html);
	gtk_entry_set_text(GTK_ENTRY(entry4), entry->aft_html);
	last_entry = entry;
	DEBUG_MSG("entry_selected, entry->name=%s, last_entry=%p\n", entry->name, entry);
}

static void main_entry_selected(GtkWidget * item, cust_main_menu_struct * main_entry)
{
	gtk_entry_set_text(GTK_ENTRY(entry1), main_entry->name);
	last_main_entry = main_entry;
	DEBUG_MSG("main_entry_selected, main_entry->name=%s, last_main_entry=%p\n", main_entry->name, main_entry);
}

#ifdef ENABLE_NLS

gchar *menu_translate(const gchar * path, gpointer data)
{
	static gchar *menupath = NULL;

	gchar *retval;
	gchar *factory;

	factory = (gchar *) data;

	if (menupath)
		g_free(menupath);

	menupath = g_strdup(path);

	if ((strstr(path, "/tearoff1") != NULL) || (strstr(path, "/---") != NULL) || (strstr(path, "/MRU") != NULL))
		return menupath;

	retval = gettext(menupath);

	return retval;
}

#endif							/*  ENABLE_NLS  */
