/* Bluefish HTML Editor
 * rpopup.c - this file contains the right mouseclick popup
 *
 * Copyright (C) 1999-2000 Olivier Sessink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <string.h>
#include <ctype.h>

#include "intl.h"
#include "config.h"

#include "bluefish.h"
#include "html.h"
#include "images.h"
#include "debug.h"
#include "snr.h"
#include "undo.h"
#include "highlight.h"

/**************************************************************************/

gint in_tag(gchar * text, gint pos);
gint rpopup_eventh(GtkWidget * widget, GdkEvent * event);
static gint xy_to_cursor_pos(gdouble x, gdouble y);
static void edit_tag_cb(gint pos);
static void parse_tagstring(gchar * tagstring, gint pos, gint len);
static GtkWidget *rpopup_create_menu(GdkEventButton * pevent);
static void input_tag_splitter(gpointer nothing, gpointer data);


/**************************************************************************/

/**************************************************************************/
/* returns FALSE, or the start of the tag + 1 */

/* original function Copyright (C) David A Knight */
/* changes (C) Olivier Sessink                */
gint in_tag(gchar * text, gint pos)
{
	gint ipos = pos;

	DEBUG_MSG("in_tag, started\n");
	/* go back until start of text, or we hit a '<' or '>' */
	while ((--pos) >= 0) {
		if (text[pos] == '<')
			break;
		else if (text[pos] == '>')
			return FALSE;
	}
	if (pos < 0)
		return FALSE;

	/* we have an opening, but do we have a closing? */
	while (ipos < strlen(text)) {
		if (text[ipos] == '<')
			return FALSE;
		if (text[ipos] == '>')
			return pos + 1;
		ipos++;
	}
	return FALSE;
}

/*************************************************************************/

/* original function Copyright (C) David A Knight */
/* changes (C) Olivier Sessink                */
static gint xy_to_cursor_pos(gdouble x, gdouble y)
{
	gchar *text;
	gchar *line = NULL;
	gint i;
	gint lineWidth = 0;
	gint editorWidth;
	gint height;
	gint start = 0;
	gint charWidth;
/*	GdkFont *font = gdk_fontset_load(main_v->props.cfg_editor_font); */
	GdkFont *font = main_v->current_document->textbox->style->font;
	GtkText *e = GTK_TEXT(main_v->current_document->textbox);
	const int line_wrap_room = 8;	/* 8 == LINE_WRAP_ROOM in gtktext.c */
	gboolean wrapped = FALSE;
	gint previ;

	DEBUG_MSG("xy_to_cursor_pos, started, e=%p\n", e);
	DEBUG_MSG("xy_to_cursor_pos, x=%d, y=%d\n", (gint) x, (gint) y);
	gdk_window_get_size(e->text_area, &editorWidth, NULL);
	DEBUG_MSG("xy_to_cursor_pos, editorWidth=%d, line_wrap_room=%d\n", editorWidth, line_wrap_room);
	editorWidth -= line_wrap_room;
	DEBUG_MSG("xy_to_cursor_pos, editorWidth=%d\n", editorWidth);
	text = gtk_editable_get_chars(GTK_EDITABLE(e), 0, -1);
	DEBUG_MSG("xy_to_cursor_pos, strlen(text)=%d, font=%p\n", strlen(text), font);
	height = font->ascent + font->descent;
	DEBUG_MSG("xy_to_cursor_pos, height=%d, ascent=%d, descent=%d\n", height, font->ascent, font->descent);
	y += e->first_cut_pixels;
	DEBUG_MSG("xy_to_cursor_pos, e->first_line_start_index=%d, strlen(text)=%d\n", e->first_line_start_index, strlen(text));
	for (i = e->first_line_start_index; i < strlen(text); i++) {
		if (text[i] != '\n') {
			charWidth = gdk_char_width(font, text[i]);
			if (text[i] == '\t') {
				charWidth = gdk_char_width(font, ' ');
				charWidth *= main_v->props.cfg_editor_tabwidth;
			}
			lineWidth += charWidth;
		}
		if ((text[i] == '\n') || (lineWidth > editorWidth)) {
			lineWidth = 0;
			if ((y -= height) <= 0)
				break;
			wrapped = (text[i] != '\n');
			previ = i;
			while (!isspace(text[i]))
				i--;
			if (i == start && wrapped)
				i = previ;
			start = i;
		}
	}
	line = g_strndup(&text[start], i - start);
	DEBUG_MSG("xy_to_cursor_pos, line=%s\n", line);
	/* wasn't inside the text area */
	if (y <= 0) {
		for (start = 0; x > 0; start++) {
			charWidth = gdk_char_width(font, line[start]);
			if (line[start] == '\t') {
				charWidth = gdk_char_width(font, ' ');
				charWidth *= main_v->props.cfg_editor_tabwidth;
			}
			x -= charWidth;
		}
		if ((start = (strlen(line) - start)) > 0)
			i -= start;
	}

	g_free(line);
	g_free(text);
	DEBUG_MSG("xy_to_cursor_pos, return %d\n", i);
	return i;
}

/*************************************************************************/

static void edit_tag_cb(gint pos)
{
	gchar *text, *tagstring;
	gint len;


	DEBUG_MSG("edit_tag_cb, started\n");
	text = gtk_editable_get_chars(GTK_EDITABLE(GTK_TEXT(main_v->current_document->textbox)), 0, -1);
	/* gtk_editable_set_position( GTK_EDITABLE( main_v->current_document->textbox ), pos ); */
	pos = in_tag(text, pos);
	if (pos) {
		DEBUG_MSG("edit_tag_cb, in_tag!\n");
		for (len = (--pos); text[len] != '>'; len++) {
		}
		gtk_editable_select_region(GTK_EDITABLE(main_v->current_document->textbox), pos, ++len);
		DEBUG_MSG("edit_tag_cb, making tagstring\n");
		tagstring = g_strndup(&text[pos + 1], len - pos - 2);
		DEBUG_MSG("edit_tag_cb, tagstring=%s\n", tagstring);
		parse_tagstring(tagstring, pos, len);
		g_free(tagstring);
	} else {
		DEBUG_MSG("edit_tag_cb, not in_tag\n");
	}
	g_free(text);

}


/*************************************************************************/

static void parse_tagstring(gchar * tagstring, gint pos, gint len)
{
	GList *tmplist = NULL;
	gchar *tmpstring, *item, *value;
	gint count, prevtag, item_value_delimiter;
	Ttagitem *tag_item;
	Ttagpopup *tag_popup;
	gboolean in_quote, has_quotes;

	DEBUG_MSG("parse_tagstring, started, tagstring=%s\n", tagstring);

	/* parsing the values from this tag */
	tmpstring = g_strdup(tagstring);
	item_value_delimiter = prevtag = count = 0;
	has_quotes = in_quote = FALSE;
	while (tmpstring[count] != '\0') {

		/* spaces (delimiters) are allowed within quotes, so we have to keep track of quotes */
		if (tmpstring[count] == '"') {
			has_quotes = TRUE;
			if (in_quote) {
				in_quote = FALSE;
			} else {
				in_quote = TRUE;
			}
		}
		/* to split the item and the value we have to keep track of '=' characters */
		if (tmpstring[count] == '=') {
			item_value_delimiter = count;
		}
		/* it is a delimiter if it a space (or tab, newline), outside a quote or the last character of the string */
		if ((isspace(tmpstring[count]) && (in_quote == FALSE)) || (tmpstring[count + 1] == '\0')) {
			if (prevtag == (count - 1)) {
				DEBUG_MSG("parse_tagstring, two spaces!\n");
				prevtag = count;
			} else if (prevtag == 0) {
				DEBUG_MSG("parse_tagstring, this is the name of the tag itself\n");
				prevtag = count;
			} else {
				DEBUG_MSG("parse_tagstring, making split, count=%d, prevtag=%d\n", count, prevtag);
				if (item_value_delimiter > prevtag) {
					item = g_strndup(&tmpstring[prevtag + 1], item_value_delimiter - prevtag - 1);
					if (has_quotes == TRUE) {
						value = g_strndup(&tmpstring[item_value_delimiter + 2], count - item_value_delimiter - 2);
						value = trunc_on_char(value, '"');
					} else {
						value = g_strndup(&tmpstring[item_value_delimiter + 1], count - item_value_delimiter);
						g_strstrip(value);
					}
				} else {
					item = g_strndup(&tmpstring[prevtag + 1], count - prevtag);
					value = g_strdup("");
				}
				g_strdown(item);
				g_strstrip(item);
				tag_item = g_malloc(sizeof(Ttagitem));
				tag_item->item = item;
				tag_item->value = value;
				tmplist = g_list_append(tmplist, tag_item);
				DEBUG_MSG("parse_tagstring, item=%s with value=%s appended to list %p\n", item, value, tmplist);
				prevtag = count;
				has_quotes = FALSE;
			}
		}
		count++;
	}
	g_free(tmpstring);

	tag_popup = g_malloc(sizeof(Ttagpopup));
	tag_popup->taglist = tmplist;
	tag_popup->pos = pos;
	tag_popup->len = len;
	DEBUG_MSG("parse_tagstring, tag_popup->pos=%d, tag_popup->len=%d\n", tag_popup->pos, tag_popup->len);

	tmpstring = g_strdup(tagstring);
	tmpstring = trunc_on_char(tmpstring, ' ');
	g_strdown(tmpstring);
	/* identifying which tag we have */

	if (strcmp(tmpstring, "body") == 0) {
		DEBUG_MSG("parse_tagstring, identified as body_cb tag\n");
		body_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "a") == 0) {
		DEBUG_MSG("parse_tagstring, identified as anchor tag\n");
		quickanchor_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "hr") == 0) {
		quickrule_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "font") == 0) {
		fontdialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "basefont") == 0) {
		basefont_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "table") == 0) {
		tabledialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "tr") == 0) {
		tablerowdialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "th") == 0) {
		tableheaddialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "td") == 0) {
		tabledatadialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "frameset") == 0) {
		framesetdialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "frame") == 0) {
		framedialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "img") == 0) {
		image_insert_dialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "form") == 0) {
		formdialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "textarea") == 0) {
		textareadialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "select") == 0) {
		selectdialog_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "meta") == 0) {
		meta_cb(NULL, tag_popup);
	}
	if (strcmp(tmpstring, "input") == 0) {
		DEBUG_MSG("parse_tagstring, identified as INPUT tag, splitting tag!\n");
		input_tag_splitter(NULL, tag_popup);
	}
	tmplist = g_list_first(tmplist);
	while (tmplist) {
		g_free(((Ttagitem *) tmplist->data)->item);
		g_free(((Ttagitem *) tmplist->data)->value);
		g_free(tmplist->data);
		tmplist = g_list_next(tmplist);
	}
	g_list_free(tmplist);
	g_free(tag_popup);
	g_free(tmpstring);
}

/*************************************************************************/


static GtkWidget *rpopup_create_menu(GdkEventButton * pevent)
{
	GtkWidget *menu_item;
	GtkWidget *menu;
	gint pos;

	DEBUG_MSG("rpopup_create_menu, started\n");
	menu = gtk_menu_new();
	menu_item = gtk_menu_item_new_with_label(_("cut"));
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cut_cb), NULL);
	gtk_menu_append(GTK_MENU(menu), menu_item);
	menu_item = gtk_menu_item_new_with_label(_("copy"));
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(copy_cb), NULL);
	gtk_menu_append(GTK_MENU(menu), menu_item);
	menu_item = gtk_menu_item_new_with_label(_("paste"));
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(paste_cb), NULL);
	gtk_menu_append(GTK_MENU(menu), menu_item);
	menu_item = gtk_menu_item_new_with_label(_("undo"));
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(undo_cb), NULL);
	gtk_menu_append(GTK_MENU(menu), menu_item);
	menu_item = gtk_menu_item_new_with_label(_("redo"));
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(redo_cb), NULL);
	gtk_menu_append(GTK_MENU(menu), menu_item);

	menu_item = gtk_menu_item_new_with_label(_("edit tag"));
	pos = xy_to_cursor_pos(pevent->x, pevent->y);
	gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(edit_tag_cb), GINT_TO_POINTER(pos));
	gtk_menu_append(GTK_MENU(menu), menu_item);
	gtk_widget_show_all(menu);
	return menu;
}

/*************************************************************************/



gint rpopup_eventh(GtkWidget * widget, GdkEvent * event)
{
	GtkWidget *menu;

	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event;
		if (bevent->button == 3) {
			DEBUG_MSG("rpopup_eventh, bevent->button=%d\n", bevent->button);
			menu = rpopup_create_menu(bevent);
			gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
			/* Tell calling code that we have handled this event; the buck
			 * stops here. */
			return TRUE;
		}
	}

	/* Tell calling code that we have not handled this event; pass it on. */
	return FALSE;
}


/*************************************************************************/

static void input_tag_splitter(gpointer nothing, gpointer data)
{
	Ttagpopup *tag_popup;
	GList *tmplist;
	gint counter = 0;
	gchar *itemvalue = NULL;

	tag_popup = (Ttagpopup *) data;
	DEBUG_MSG("input_tag_splitter, started\n");

	tmplist = g_list_first(tag_popup->taglist);
	while (tmplist) {
		if (strcasecmp((gchar *) ((Ttagitem *) (tmplist->data))->item, "type") == 0) {
			DEBUG_MSG("input_tag_splitter, found!! tmplist->data->item=%s, counter=%d\n", (gchar *) ((Ttagitem *) (tmplist->data))->item,
					  counter);
			itemvalue = g_strdup((gchar *) ((Ttagitem *) (tmplist->data))->value);
			break;
		} else {
			DEBUG_MSG("input_tag_splitter, nope.., tmplist->data->item=%s, counter=%d\n", (gchar *) ((Ttagitem *) (tmplist->data))->item,
					  counter);
		}
		counter++;
		tmplist = g_list_next(tmplist);
	}
	if (tmplist) {
		g_strdown(itemvalue);
		DEBUG_MSG("input_tag_splitter, itemvalue=%s, counter=%d\n", itemvalue, counter);
		if (strcmp(itemvalue, "submit") == 0) {
			DEBUG_MSG("input_tag_splitter, identified as submit type\n");
			submitdialog_cb(NULL, data);
		} else if (strcmp(itemvalue, "reset") == 0) {
			resetdialog_cb(NULL, data);
		} else if (strcmp(itemvalue, "text") == 0) {
			textdialog_cb(NULL, data);
		} else if (strcmp(itemvalue, "hidden") == 0) {
			hiddendialog_cb(NULL, data);
		} else if (strcmp(itemvalue, "radio") == 0) {
			radiodialog_cb(NULL, data);
		} else if (strcmp(itemvalue, "checkbox") == 0) {
			checkdialog_cb(NULL, data);
		}
		g_free(itemvalue);
	} else {
		DEBUG_MSG("input_tag_splitter, not found ?\n");
	}


}
