/* Bluefish HTML Editor
 * interface.c - this file contains the UI code and some init code
 *
 * Copyright (C) 1998-1999 Olivier Sessink and Chris Mazuc
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* Changes by Antti-Juhani Kaijanaho <gaia@iki.fi> on 1999-10-20  */

#include <unistd.h>
#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>

#include <locale.h>
#include <libintl.h>
#include "intl.h"

#include "config.h"

#include "bluefish.h"
#include "interface.h"
#include "init.h"
#include "undo.h"
#include "snr.h"
#include "toolbars.h"
#include "callbacks.h"
#include "rpopup.h"
#include "debug.h"
#include "highlight.h"
#include "menu.h"
#include "gtk_easy.h"

#include "pixmaps/bluefish_icon.xpm"
#include "pixmaps/bluefish.xpm"

GtkWidget *cust_handle_box, *handle_box, *html_handle_box;

/********************************************************************/



/********************************************************************/


/* local "global" variables */
static gint context_id = 0;

/* 
 * start of the functions
 * 
 */

static gint statusbar_remove(gpointer message_id)
{
	gtk_statusbar_remove(GTK_STATUSBAR(main_v->statusbar), context_id, GPOINTER_TO_INT(message_id));
	return 0;
}


void statusbar_message(gpointer message, gint time)
{

	gint count;

	count = gtk_statusbar_push(GTK_STATUSBAR(main_v->statusbar), context_id, (gchar *) message);
	gtk_timeout_add(time, statusbar_remove, GINT_TO_POINTER(count));
	flush_queue();
}


void switch_to_document(gpointer data)
{
	gint index;

	index = g_list_index(main_v->documentlist, data);
	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), (index - 1));

}

void notebook_changed(void)
{
	gint cur;

	/* This one is called when you click on the notebook
	   it _should_ be called also when you use the keys to change the page */
	cur = 1 + (gint) gtk_notebook_current_page(GTK_NOTEBOOK(main_v->notebook));
	if (main_v->last_notebook_page == cur) {
		DEBUG_MSG("notebook_changed, it didn't change to a new document (cur=%d)\n", cur);
		return;
	} else {
		DEBUG_MSG("notebook_changed, it did really change to a new document, cur=%d\n", cur);
		main_v->last_notebook_page = cur;
		main_v->current_document = NULL;
		main_v->current_document = g_list_nth_data(main_v->documentlist, cur);
		if (main_v->current_document == NULL) {
#ifdef DEBUG
			DEBUG_MSG("notebook_changed, finished, main_v->current_document == NULL\n");
#endif
		} else {
			gtk_widget_grab_focus(GTK_WIDGET(main_v->current_document->textbox));
			setup_toggle_item(gtk_item_factory_from_widget(main_v->menubar), N_("/View/Highlight syntax"),
							  main_v->current_document->highlightstate);
		}
	}
}

gboolean highlight_once(filestruct * whichdoc)
{
	DEBUG_MSG("highlight_once, need_highlighting=%d\n", whichdoc->need_highlighting);
	whichdoc->need_highlighting = 0;
	refresh_hi_cb(NULL, whichdoc);
	return FALSE;
}

void document_need_highlighting(filestruct * whichdoc)
{
	DEBUG_MSG("document_need_highlighting, need_highlighting=%d\n", whichdoc->need_highlighting);
	if (whichdoc->highlightstate && main_v->props.cont_highlight_update) {
		gtk_timeout_add(100, (GtkFunction) highlight_once, whichdoc);
	}
}

/* this function destroys the current_document */
void destroy_current_document(void)
{
	if (main_v->current_document != NULL) {
		DEBUG_MSG("destroy_current_document, started, notebook=%p\n", main_v->notebook);
		/* this introduces a memleak, but at least the gtkeditor widget
		   doesn't crash anymore
		   gtk_widget_ref(GTK_WIDGET(current_document->textbox)); */
		DEBUG_MSG("destroy_current_document, textbox=%p\n", main_v->current_document->textbox);
		gtk_notebook_remove_page(GTK_NOTEBOOK(main_v->notebook), gtk_notebook_current_page(GTK_NOTEBOOK(main_v->notebook)));
		DEBUG_MSG("destroy_current_document, g_list_length(documentlist)=%d\n", g_list_length(main_v->documentlist));
		if (g_list_length(main_v->documentlist) > 1) {
			g_list_remove(main_v->documentlist, main_v->current_document);
		} else {
			g_list_remove(main_v->documentlist, main_v->current_document);
			DEBUG_MSG("destroy_current_document, removed from documentlist\n");
			g_list_free(main_v->documentlist);
			DEBUG_MSG("destroy_current_document, freed documentlist\n");
			main_v->documentlist = NULL;
			DEBUG_MSG("destroy_current_document, documentlist = NULL\n");
		}
		DEBUG_MSG("destroy_current_document, g_list_length(documentlist)=%d\n", g_list_length(main_v->documentlist));
		if (main_v->current_document->filename != NULL) {
			g_free(main_v->current_document->filename);
		}
		g_free(main_v->current_document);
		gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), ((gint) g_list_length(main_v->documentlist) - 1));
		notebook_changed();
	} else {
		DEBUG_MSG("destroy_current_document, cannot close a NULL current_document\n");
	}
}


Ttext_positions get_positions(GtkWidget * textbox)
{

	Ttext_positions positions;
	gint tmp;

	positions.currentp = gtk_editable_get_position(GTK_EDITABLE(textbox));
	positions.adj = GTK_TEXT(textbox)->vadj->value;
	positions.selection = GTK_EDITABLE(textbox)->has_selection;
	if (positions.selection) {
		positions.startp = GTK_EDITABLE(textbox)->selection_start_pos;
		positions.endp = GTK_EDITABLE(textbox)->selection_end_pos;
		if (positions.endp < positions.startp) {
			tmp = positions.endp;
			positions.endp = positions.startp;
			positions.startp = tmp;
		}
	} else {
		positions.endp = positions.currentp;
		positions.startp = positions.currentp;
	}
	return positions;
}

void restore_positions(Ttext_positions positions, GtkWidget * textbox)
{

	gtk_editable_set_position(GTK_EDITABLE(textbox), positions.currentp);
	gtk_adjustment_set_value(GTK_TEXT(textbox)->vadj, positions.adj);
	if (positions.selection) {
		gtk_editable_select_region(GTK_EDITABLE(textbox), positions.startp, positions.endp);
	}
}

void document_set_fonts(gchar *textfontstring, gchar *labelfontstring, filestruct *document) {
	if (labelfontstring) {
		apply_font_style(document->textbox, textfontstring);
	}	
	if (labelfontstring) {
		apply_font_style(document->tab_label, labelfontstring);
	}
}

void notebook_set_tab_pos(gchar *position) {
	if (strcmp(position, "left") == 0) {
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_LEFT);
	} else if (strcmp(position, "right") == 0) {
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_RIGHT);
	} else if (strcmp(position, "top") == 0) {
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_TOP);
	} else {
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_BOTTOM);
	}
}

void replace_text(const gchar * newstring, gint pos, gint len)
{
	gtk_text_freeze(GTK_TEXT(main_v->current_document->textbox));
	DEBUG_MSG("replace_text, started, pos=%d, len=%d, newstring=%s, strlen(newstring)=%d\n", pos, len, newstring, strlen(newstring));
	gtk_editable_delete_text(GTK_EDITABLE(GTK_TEXT(main_v->current_document->textbox)), pos, len);
	DEBUG_MSG("replace_text, text deleted\n");
	gtk_editable_insert_text(GTK_EDITABLE(GTK_TEXT(main_v->current_document->textbox)), newstring, strlen(newstring), &pos);

	gtk_text_thaw(GTK_TEXT(main_v->current_document->textbox));
	main_v->current_document->modified = 1;
	document_need_highlighting(main_v->current_document);
}

void insert_text(const gchar * before, const gchar * after)
{
	Ttext_positions positions;
	/* find the current cursor position and selection position */

	positions = get_positions(main_v->current_document->textbox);

/* freeze the textbox, this is better when inserting large blocks */
	gtk_text_freeze(GTK_TEXT(main_v->current_document->textbox));

	DEBUG_MSG("insert_text, A currentp = %d, startp=%d, endp=%d\n", positions.currentp, positions.startp, positions.endp);

	gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox), positions.endp);
	gtk_text_insert(GTK_TEXT(main_v->current_document->textbox), NULL, NULL, NULL, after, -1);
	undo_list_add(main_v->current_document, g_strdup(after), positions.endp, positions.endp + strlen(after), UndoInsert);

	gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox), positions.startp);
	gtk_text_insert(GTK_TEXT(main_v->current_document->textbox), NULL, NULL, NULL, before, -1);
	undo_list_add(main_v->current_document, g_strdup(before), positions.startp, positions.startp + strlen(before), UndoInsert);

/* thaw textbox, we're finished */
	DEBUG_MSG("insert_text, ->thaw\n");
	gtk_text_thaw(GTK_TEXT(main_v->current_document->textbox));
	positions.currentp += strlen(before);
	restore_positions(positions, main_v->current_document->textbox);

	/* grab focus so we can continue working */
	DEBUG_MSG("insert_text, about to grab focus\n");
	gtk_widget_grab_focus(GTK_WIDGET(main_v->current_document->textbox));
	DEBUG_MSG("insert_text, set modified=1\n");
	main_v->current_document->modified = 1;

	document_need_highlighting(main_v->current_document);
}


filestruct *new_document(void)
{
	GtkWidget *tmptable, *tmpscrollbar;
	filestruct *document;

	/* Here we create a new document, and add it to the GList */
	DEBUG_MSG("new_document, started\n");
	if (main_v->current_document != NULL) {
		if ((main_v->current_document->modified == 0)
			&& (main_v->current_document->filename == NULL)
			&& (g_list_length(main_v->documentlist) == 2)) {

			DEBUG_MSG("new_document, no new needed, the current one is empty\n");
			return main_v->current_document;
		}
	}

	document = g_malloc0(sizeof(filestruct));
	DEBUG_MSG("new_document, malloced at %p\n", document);
	tmptable = gtk_table_new(1, 100, FALSE);	/* yes this 100 has a purpose */
	gtk_table_set_row_spacings(GTK_TABLE(tmptable), 0);
	gtk_table_set_col_spacings(GTK_TABLE(tmptable), 0);
	document->textbox = gtk_text_new(NULL, NULL);

	tmpscrollbar = gtk_vscrollbar_new(GTK_TEXT(document->textbox)->vadj);
	gtk_table_attach_defaults(GTK_TABLE(tmptable), document->textbox, 0, 99, 0, 1);
	gtk_table_attach(GTK_TABLE(tmptable), tmpscrollbar, 99, 100, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
	GTK_WIDGET_UNSET_FLAGS(tmpscrollbar, GTK_CAN_FOCUS);
	gtk_widget_show(tmpscrollbar);

	document->tab_label = gtk_label_new(_("Untitled"));
	GTK_WIDGET_UNSET_FLAGS(document->tab_label, GTK_CAN_FOCUS);

	apply_font_style(GTK_WIDGET(document->tab_label), main_v->props.cfg_tab_font);

	/* I did the table like this because it is more flexable than */
	/* a hard coded limit with gtk_widget_set_usize(), and I plan */
	/* on making the window size configurable someday */
	DEBUG_MSG("new_document, before notebook_append_page\n");
	gtk_notebook_append_page(GTK_NOTEBOOK(main_v->notebook), tmptable, document->tab_label);
	gtk_widget_show(tmptable);

	gtk_widget_realize(document->textbox);

	gtk_text_set_editable(GTK_TEXT(document->textbox), TRUE);
	gtk_text_set_word_wrap(GTK_TEXT(document->textbox), FALSE);

	gtk_signal_connect_object(GTK_OBJECT(document->textbox), "event", GTK_SIGNAL_FUNC(rpopup_eventh), NULL);

	apply_font_style(GTK_WIDGET(document->textbox), main_v->props.cfg_editor_font);
	GTK_TEXT(document->textbox)->default_tab_width = main_v->props.cfg_editor_tabwidth;
	GTK_TEXT(document->textbox)->tab_stops = g_list_remove(GTK_TEXT
														   (document->textbox)->tab_stops, GTK_TEXT(document->textbox)->tab_stops->data);
	GTK_TEXT(document->textbox)->tab_stops = g_list_remove(GTK_TEXT
														   (document->textbox)->tab_stops, GTK_TEXT(document->textbox)->tab_stops->data);
	GTK_TEXT(document->textbox)->tab_stops = NULL;
	GTK_TEXT(document->textbox)->tab_stops =
		g_list_prepend(GTK_TEXT(document->textbox)->tab_stops, (void *) main_v->props.cfg_editor_tabwidth);
	GTK_TEXT(document->textbox)->tab_stops =
		g_list_prepend(GTK_TEXT(document->textbox)->tab_stops, (void *) main_v->props.cfg_editor_tabwidth);

	gtk_widget_show(document->textbox);

	document->modified = FALSE;
	document->filename = NULL;
	document->need_highlighting = 0;
	document->highlightstate = main_v->props.cont_highlight_update;

	main_v->documentlist = g_list_append(main_v->documentlist, document);
	gtk_widget_grab_focus(document->textbox);

	/*
	 * this used to be the signal, but I'll put this in doc_insert_text_cb
	 * and doc_delete_text_cb 
	 */
	/* gtk_signal_connect(GTK_OBJECT(document->textbox), "key_press_event",
	   textbox_key_press, NULL);   */
	/* this part is copied from gnotepad code 
	   Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com> */

	document->ins_txt_id = gtk_signal_connect(GTK_OBJECT(document->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), document);
	document->del_txt_id = gtk_signal_connect(GTK_OBJECT(document->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), document);

	/* set this new document as active notebook page */
	flush_queue();
	DEBUG_MSG("new_document, before notebook_set_page, after flush_queue\n");
	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), g_list_length(main_v->documentlist) - 2);
	notebook_changed();
	DEBUG_MSG("new_document, ended\n");

	return document;
}

static void make_main_window(void)
{
	GtkWidget *vbox;
	
	/* main window */
	DEBUG_MSG("make_main_window, started\n");
	main_v->main_window = window_with_title(CURRENT_VERSION_NAME, GTK_WIN_POS_CENTER, GTK_WINDOW_TOPLEVEL, 0);
	DEBUG_MSG("make_main_window, main_v->main_window(%p)\n", main_v->main_window);
	gtk_signal_connect(GTK_OBJECT(main_v->main_window), "delete_event", GTK_SIGNAL_FUNC(cya_later), NULL);
	gtk_signal_connect(GTK_OBJECT(main_v->main_window), "destroy", GTK_SIGNAL_FUNC(cya_later), NULL);
	if ((main_v->props.main_window_h < gdk_screen_height() ) && (main_v->props.main_window_w < gdk_screen_width() )) {
		gtk_window_set_default_size(GTK_WINDOW(main_v->main_window), main_v->props.main_window_w, main_v->props.main_window_h);
	}

	gtk_window_set_wmclass(GTK_WINDOW(main_v->main_window), "Bluefish", "bluefish");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(main_v->main_window), vbox);
	gtk_widget_show(vbox);
	/* menu bar */
	get_main_menu(main_v->main_window, &main_v->menubar);
	gtk_box_pack_start(GTK_BOX(vbox), main_v->menubar, FALSE, TRUE, 0);
	gtk_widget_show(main_v->menubar);

	/* the pixmaps can't be created without a realise on the window, the
	   error Gdk-WARNING **: Creating pixmap from xpm with NULL window and 
	   colormap is removed by adding this line */
	gtk_widget_realize(main_v->main_window);

	/* pack the main toolbar handle box on the right place */
	handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), handle_box, FALSE, FALSE, 0);
	if (main_v->props.v_main_tb)
		show_main_toolbar_cb(NULL, NULL);

	/* pack the HTML toolbar handle box on the right place */
	html_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(html_handle_box), FALSE, FALSE, 0);
	if (main_v->props.v_html_tb)
		show_html_toolbar_cb(NULL, NULL);

	/* pack the custom toolbar handle box on the right place */
	cust_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), cust_handle_box, FALSE, FALSE, 0);
	if (main_v->props.v_custom_tb)
		show_cust_menubar_cb(NULL, NULL);

	/* notebook with the text widget in there */
	main_v->notebook = gtk_notebook_new();
	notebook_set_tab_pos(main_v->props.cfg_tab_pos);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_notebook_set_tab_border(GTK_NOTEBOOK(main_v->notebook), 1);

	/* We have to know when the notebook changes */
	gtk_signal_connect(GTK_OBJECT(main_v->notebook), "button_release_event", notebook_changed, NULL);
	gtk_signal_connect_after(GTK_OBJECT(main_v->notebook), "key_press_event", notebook_changed, NULL);
/*  gtk_signal_connect_after (GTK_OBJECT (main_v->notebook), "switch_page",
   notebook_changed, NULL); */
	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), 0);
	gtk_box_pack_start(GTK_BOX(vbox), main_v->notebook, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_widget_show(main_v->notebook);

	/* status bar */
	main_v->statusbar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(vbox), main_v->statusbar, FALSE, TRUE, 0);
	gtk_widget_show(main_v->statusbar);

	DEBUG_MSG("make_main_window, end\n");
}

void check_font_type() {
	GtkWidget *textwidget;

	textwidget = gtk_text_new(NULL, NULL);                                         
	gtk_widget_ensure_style(textwidget);
	if (textwidget->style->font->type == GDK_FONT_FONT) {
		main_v->fontset = 0;
	} else {
		main_v->fontset = 1;
	}
	gtk_widget_destroy(textwidget);
}


void startup(void)
{
	GdkPixmap *icon = NULL;
#ifndef NOSPLASH
	GtkWidget *vbox, *splash_window;
	GtkWidget *wpixmap;
	GdkPixmap *pixm;
#endif

	/* splash screen */
#ifndef NOSPLASH
	splash_window = gtk_window_new(GTK_WINDOW_DIALOG);
/*    gtk_widget_set_usize(GTK_WIDGET(splash_window), 394, 162); */
	gtk_window_set_title(GTK_WINDOW(splash_window), CURRENT_VERSION_NAME);
	gtk_window_position(GTK_WINDOW(splash_window), GTK_WIN_POS_CENTER);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(splash_window), vbox);
	gtk_widget_show(vbox);
	gtk_widget_realize(splash_window);
	pixm = gdk_pixmap_create_from_xpm_d(splash_window->window, NULL, NULL, bluefish_xpm);
	wpixmap = gtk_pixmap_new(pixm, NULL);
	gdk_pixmap_unref(pixm);
	gtk_box_pack_start(GTK_BOX(vbox), wpixmap, FALSE, FALSE, 0);
	gtk_widget_show(wpixmap);
	gtk_widget_realize(wpixmap);
	gtk_widget_show(splash_window);
	flush_queue();
#endif
	check_font_type();
	check_directories();
	parse_config_files();
	main_v->documentlist = g_list_alloc();
	snr_init();
	make_main_window();
	gtk_widget_realize(main_v->main_window);
	flush_queue();
	/* make external commands and filters ready */
	make_external_menu_entries(1);
	make_external_menu_entries(2);
	context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(main_v->statusbar), "Bluefish");
	DEBUG_MSG("startup, main_window is ready\n");
	/* add icon to the window when in minimized state */
	icon = gdk_pixmap_create_from_xpm_d(main_v->main_window->window, NULL, NULL, (gchar **) bluefish_icon_xpm);
	gdk_window_set_icon(main_v->main_window->window, NULL, icon, NULL);
	gdk_window_set_icon_name(main_v->main_window->window, CURRENT_VERSION_NAME);
	DEBUG_MSG("startup, icon is there\n");
#ifndef NOSPLASH
	/* kill the splash window when the main window is going up */
	flush_queue();
	sleep(1);
	gtk_signal_handlers_destroy(GTK_OBJECT(splash_window));
	gtk_widget_destroy(GTK_WIDGET(splash_window));
#endif
	gtk_widget_show(main_v->main_window);
	if (g_list_length(main_v->documentlist) <= 1) {
		main_v->current_document = new_document();
		gtk_label_set(GTK_LABEL(main_v->current_document->tab_label), _("Untitled"));
	}
	gtk_window_set_policy(GTK_WINDOW(main_v->main_window), TRUE, TRUE, FALSE);
}

void show_about_window_cb(GtkWidget * widget, gpointer data)
{
	GtkWidget *wpixmap, *window, *vbox, *button;
	GdkPixmap *pixm;
	GtkStyle *style;

	DEBUG_MSG("about, started\n");
	window = window_with_title("About Bluefish", GTK_WIN_POS_MOUSE, GTK_WINDOW_DIALOG, 10);

	style = gtk_style_new();
	style->bg->red = 65535;
	style->bg->blue = 65535;
	style->bg->green = 65535;
	gtk_widget_push_style(style);
	gtk_widget_set_style(GTK_WIDGET(window), style);
	gtk_widget_pop_style();

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_realize(window);
	pixm = gdk_pixmap_create_from_xpm_d(window->window, NULL, NULL, bluefish_xpm);
	wpixmap = gtk_pixmap_new(pixm, NULL);
	gdk_pixmap_unref(pixm);
	gtk_box_pack_start(GTK_BOX(vbox), wpixmap, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(CURRENT_VERSION_NAME), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox),
					   gtk_label_new(_
									 ("(C) Olivier Sessink\n\ndeveloped by the bluefish development team\nsee AUTHORS for more information\nhttp://bluefish.linuxbox.com")),
					   FALSE, FALSE, 0);
	button = gtk_button_new_with_label(_("Dismiss"));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", gtk_widget_destroy, GTK_OBJECT(window));
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show_all(window);
	DEBUG_MSG("about, finished\n");
}
