/*
			(c) Copyright 1998-2000 - Tord Jansson
			======================================

		This file is part of the BladeEnc MP3 Encoder, based on
		ISO's reference code for MPEG Layer 3 compression, and might
		contain smaller or larger sections that are directly taken
		from ISO's reference code.

		All changes to the ISO reference code herein are either
		copyrighted by Tord Jansson (tord.jansson@swipnet.se)
		or sublicensed to Tord Jansson by a third party.

	BladeEnc is free software; you can redistribute this file
	and/or modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	2000-03-12  Andr Piotrowski
*/



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "system.h"
#include "l3side.h"
#include "loop.h"
#include "huffman.h"
#include "l3bitstream.h"
#include "reservoir.h"
#include "loop-pvt.h"





#define		infinity		99999999





/*  ========================================================================================  */
/*      the scale factor band partition                                                       */
/*  ========================================================================================  */
/*
	Here are MPEG1 Table B.8 and MPEG2 Table B.1
	-- Layer III scalefactor bands.
	Index into this using a method such as:
	idx  = fr_ps->header->sampling_frequency
	       + (fr_ps->header->version * 3)

	The commented numbers show the threefolds of the short band entries to make visible
	which place is covered in case of short block types.
		Especially you can see, that the first eight long bands share the same memory as
	the first three short bands, non-depending on the sampling frequency.
		This attribute can be used to build mixed blocks (first coming eight long blocks,
	succeeded by ten short blocks), a feature that is not installed at the time.
*/

struct	scalefac_struct sfBandIndex[3] =
{
	{ /* Table B.8.b: 44.1 kHz */
		{  0,  4,  8, 12, 16, 20, 24, 30, 36, 44, 52, 62, 74, 90,110,134,162,196,238,288,342,418,576},
		{  0,          4,          8,     12,   16,     22,   30,   40, 52,     66, 84,106,136,  192}
	/* *3  0          12          24      36    48      66    90   120 156     198 252 318 408   576 */
	},
	{ /* Table B.8.c: 48 kHz */
		{  0,  4,  8, 12, 16, 20, 24, 30, 36, 42, 50, 60, 72, 88,106,128,156,190,230,276,330,384,576},
		{  0,          4,          8,     12,   16,     22, 28,     38, 50,     64, 80,100,126,  192}
	/* *3  0          12          24      36    48      66  84     114 150     192 240 300 378   576 */
	},
	{ /* Table B.8.a: 32 kHz */
		{  0,  4,  8, 12, 16, 20, 24, 30, 36, 44, 54, 66, 82,102,126,156,194,240,296,364,448,550,576},
		{  0,          4,          8,     12,   16,   22,   30,     42, 58, 78,    104,138,180,  192}
	/* *3  0          12          24      36    48    66    90     126 174 234     312 414 540   576 */
	}
};

int		*scalefac_band_long  = &sfBandIndex[0].l[0];
int		*scalefac_band_short = &sfBandIndex[0].s[0];





/*  ========================================================================================  */
/*      MPEG-2 scale factor partitioning                                                      */
/*  ========================================================================================  */
/*
	The following table is used to implement the scalefactor
	partitioning for MPEG2 as described in section
	2.4.3.2 of the IS. The indexing corresponds to the
	way the tables are presented in the IS:

	[table_number][row_in_table][column of nr_of_sfb]

	>>>>    Cant say much about it --- we dont use it at the time...
*/

#if 0
unsigned				nr_of_sfb_block[6][3][4] =
{
	{
		{ 6, 5, 5, 5},
		{ 9, 9, 9, 9},
		{ 6, 9, 9, 9}
	},
	{
		{6, 5,  7, 3},
		{9, 9, 12, 6},
		{6, 9, 12, 6}
	},
	{
		{11,10, 0, 0},
		{18,18, 0, 0},
		{15,18, 0, 0}
	},
	{
		{ 7, 7, 7, 0},
		{12,12,12, 0},
		{ 6,15,12, 0}
	},
	{
		{ 6, 6, 6, 3},
		{12, 9, 9, 6},
		{ 6,12, 9, 6}
	},
	{
		{ 8, 8, 5, 0},
		{15,12, 9, 0},
		{ 6,18, 9, 0}
	}
};
#endif





/*  ========================================================================================  */
/*      MPEG-3 scale factor select index partitioning                                         */
/*  ========================================================================================  */
/*
	This is the scfsi_band table from 2.4.2.7 of the IS 

		We believe, the medium values are computed the following way:
			( 0+21 + 1) / 2 = 11
			( 0+11 + 1) / 2 =  6
			(11+21 + 1) / 2 = 16
		By the way, you will see the band widthes again a few times: 6, 5, 5, 5

		We can do the same for short bands:
			( 0+12 + 1) / 2 =  6
			( 0+ 6 + 1) / 2 =  3
			( 6+12 + 1) / 2 =  9
		You will see the band widthes again a few times, but multiplied by three,
		the number of windows in case of short blocks: 9, 9, 9, 9

	The table is unused at the time, but has a documentation effect!
*/

int		scfsi_band_long[5] = { 0, 6, 11, 16, SBMAX_l };





/*  ========================================================================================  */
/*      MPEG-3 preemphasis table                                                              */
/*  ========================================================================================  */
/*
	Table B.6: layer3 preemphasis

	Cant say too much about it --- is used in preemphasis() ...
*/

int  pretab[SBMAX_l] =
{
	0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 1, 1, 1, 
	1, 2, 2, 3, 3, 3, 2
};





/*  ========================================================================================  */
/*      (get) the power                                                   [Snap! Attack]      */
/*  ========================================================================================  */
/*
	For more details see file "Loop01_get_the_power" in the "docs" folder.

	Meanwhile, the pow216 table works with negative indices, too!
	But they are needed only in strange cases, when the quantizer step size comes next to
	its absolute maximum, some subblock_gain is enabled and the scale factors are zero.
	So, we choose only a relative small negative area.

	Additionally, we compute the noise power table.
*/

#define BOT						200		/* the limit used in original bin_search_StepSize() */
#define POW216_MAX				(BOT/2 * 16)
#define POW216_MIN				(-25 * 16)

static	double					pow216_space[POW216_MAX-POW216_MIN];
static	double					*pow216 = pow216_space - POW216_MIN;
static	double					noisePowTab[8191+15];



static	void get_the_power (void)
{
	static	int				got_the_power = FALSE;
	int						i;

	if (!got_the_power)
	{
		for (i = POW216_MIN;  i < POW216_MAX;  i++)
			pow216[i] = pow (2.0, (double)i/16.0);

		for (i = 0;  i < 8191+15;  i++)
			noisePowTab[i] = pow ((double)i, 4.0/3.0);

		got_the_power = TRUE;
	}
}





/*  ========================================================================================  */
/*      modules instance variables                                                           */
/*  ========================================================================================  */





static	int						gr;						/* the current granule */
static	int						ch;						/* the current channel */



static	III_side_info_t			*side_info;				/* the current side information   */
static	gr_info					*cod_info;				/* the current coding information */



static	double				 *xr_org_l;					/* magnitudes of the spectral values     */
static	double			        xr34_l[576];			/* the initial magnitudes powered by 3/4 */
static	int				         *ix_l;					/* vector of quantized values ix(0..575) */

static	double				(*xr_org_s)[3];				/* some short block versions */
static	double			      (*xr34_s)[3] = (double (*)[3]) xr34_l;
static	int				        (*ix_s)[3];


static	int				        expo_l[SFB_LMAX];		/* xr == |xr_org| * (2^^(1/16))^^expo */
static	double			      energy_l[SFB_LMAX];
static	double			     l3_xmin_l[SBMAX_l];		/* the initial allowed distortion of the scalefactor (* bandwidth) */
static	double				    xfsf_l[SFB_LMAX];		/* current distortion (* bandwidth) relative to the initial        */

static	int				   *scalefac_l;					/* the current scalefactors   */
#if 0 /* not needed without SCFSI */
static	int				 *scalefac_0_l;					/* scalefactors for granule 0 */
#endif

static	int				        expo_s[SFB_SMAX][3];	/* some more short block versions */
static	double			      energy_s[SFB_SMAX][3];
static	double			     l3_xmin_s[SBMAX_s][3];
static	double					xfsf_s[SFB_SMAX][3];

static	int				  (*scalefac_s)[3];



static	double				    xmax_l[SFB_LMAX];		/* The initial (absolute) maximum magnitude */
static	int				   xmax_line_l[SFB_LMAX];		/* of the long bands and their line indices */

static	double				    xmax_s[SFB_SMAX][3];	/* Guess ... */
static	int				   xmax_line_s[SFB_SMAX][3];



static	int						quant_step;				/* the current quantizing step size */



static	int						mark_idx_l;				/* speed up - partial quantizing */
static	int						mark_tab_l[SFB_LMAX];	/* changed sfb-s                 */ 

static	int						mark_idx_s;
static	int						mark_tab_s[SFB_SMAX*3*2];	/* changed (sfb,b)-s         */





/*  ========================================================================================  */
/*      do you see the light                                              [Snap! Attack]      */
/*  ========================================================================================  */
/*
	The formerly function gr_deco() seemed to be the most underrated, although it gives you
	(slightly expanded) THAT recipe to handle the scale factor band partition.

	Unfortunately, some of this work should have already been done in "l3psy.c".
*/

static	int						max_used_sfb_l;
static	int						min_used_sfb_s;

static	int						end_sfb_l;
static	int						end_sfb_s;



static	void do_you_see_the_light (void)
{
	if (cod_info->window_switching_flag && (cod_info->block_type == SHORT_TYPE))
	{
#ifdef MIXED_BLOCKS
		if (cod_info->mixed_block_flag)
		{
			/*
				In mixed blocks there come first 8 long scale factor band areas covering
				the place normally used by the first 3 short scale factor band areas.
			*/
			max_used_sfb_l = /*cod_info->sfb_lmax =*/ 8;
			min_used_sfb_s = /*cod_info->sfb_smax =*/ 3;

			/* The following values dont need to be set again and again ... */
			cod_info->region0_count =  7;   /* scalefac_band_long[7+1   ] =  36               */
			cod_info->region1_count = 13;   /* scalefac_band_long[7+13+2] = 576  (no region2) */
		}
		else
#endif
		{
			max_used_sfb_l = /*cod_info->sfb_lmax =*/ 0;  /* No long blocks */
			min_used_sfb_s = /*cod_info->sfb_smax =*/ 0;

			/* The following values dont need to be set again and again ... */
			cod_info->region0_count =  8;   /*                   scalefac_band_short[(8+1   )/3] =  12  ( 12*3 =  36) */
			cod_info->region1_count = 36;   /* 36? should be 29: scalefac_band_short[(8+29+2)/3] = 192  (192*3 = 576) */
										    /* probably meant  : scalefac_band_short[36/3 + 1  ] = 192  (192*3 = 576) */
		                                    /* 2000-02-27 AP      no effect on output because block_type != NORM_TYPE */
		}

		/* to access the entire array we need the last scalefac_band_short area */
		end_sfb_l = max_used_sfb_l; /*cod_info->sfb_lmax;*/
		end_sfb_s = SFB_SMAX;


		/* The following values dont need to be set again and again ... */
		cod_info->count1     =   0;         /* (zero_region-bigv_region) / 4; */
		cod_info->big_values = 288;         /*              bigv_region  / 2; */

		cod_info->count1table_select = 1;   /* sum0 == sum1 == 0 */

		cod_info->address1 =  36;           /* choose one of the region0_count formulas above */
		cod_info->address2 = 576;           /* bigv_region; */
		cod_info->address3 =   0;
	}
	else
	{
		max_used_sfb_l = /*cod_info->sfb_lmax =*/ SBMAX_l;
		min_used_sfb_s = /*cod_info->sfb_smax =*/ SBMAX_s;  /* No short blocks */

		/* to access the entire array we need the last scalefac_band_long area */
		end_sfb_l = SFB_LMAX;
		end_sfb_s = min_used_sfb_s; /*cod_info->sfb_smax;*/

		if (cod_info->window_switching_flag)
		{
			/* The following values dont need to be set again and again ... */
			cod_info->region0_count =  7;   /* scalefac_band_long[7+1   ] =  36               */
			cod_info->region1_count = 13;   /* scalefac_band_long[7+13+2] = 576  (no region2) */

			cod_info->address1 =  36;       /* choose one of the region0_count formulas above */
			cod_info->address3 =   0;       /* address 2 will be set to current bigv_region   */
		}
	}
}





/*  ========================================================================================  */
/*      while the earth sleeps                          [Deep Forest with Peter Gabriel]      */
/*  ========================================================================================  */
/*
	This function is dedicated to the people who decided to calculate the maximum of the
	(absolute) magnitudes for the one reason to test whether or not all the spectral values
	are zero, although they computed the energy of the granule (even multiple times).

	Once the energy is computed, the function will return 0.0 if all spectral values are
	zero, otherwise the total energy of the granule!
*/

static	double					total_energy;



static	double while_the_earth_sleeps (void)
{
	return total_energy;
}





/*  ========================================================================================  */
/*      radioactivity                                                        [Kraftwerk]      */
/*  ========================================================================================  */
/*
	We will compute the total energy of the granule using the scale factor band partition.

	Since we access all scale factor bands, windows and lines, we will initialize some of
	the needed values en passant!
*/

static	void radioactivity (III_psy_ratio *ratio)
{
	double					*ratio_l = ratio->l[gr][ch];
	double					(*ratio_s)[3] = (double (*)[3]) ratio->s[gr][ch];

	int						i, b, sfb;
	int						start, end;
	double					temp, x;


	/*  ========    calc_xmin(), start of quantanf_init() and calc_scfsi()    ========  */
	/*
		Calculate the allowed distortion for each scalefactor band,
		as determined by the psychoacoustic model.
		xmin(sb) = ratio(sb) * energy(sb) / bandwidth(sb)
	*/

	total_energy = 0.0;
 
	for (sfb = 0;  sfb < end_sfb_l;  sfb++)
	{
		start = scalefac_band_long[sfb];
		end   = scalefac_band_long[sfb+1];

		expo_l[sfb] = 0;

		xmax_l[sfb] = -1.0;   /* -1 causes a setting! */
		temp = 0.0;
		for (i = start;  i < end;  i++)
		{
			if ((x = fabs(xr_org_l[i])) != 0.0)
			{
				xr34_l[i] = sqrt(x * sqrt(x));

				temp += x*x;
				if (x > xmax_l[sfb])
				{
					xmax_l[sfb] = x;
					xmax_line_l[sfb] = i;
				}
			}
			else
				xr34_l[i] = 0.0;
		}
		total_energy += energy_l[sfb] = temp;

		if (sfb < max_used_sfb_l)
			l3_xmin_l[sfb] = ratio_l[sfb] * temp;
	}

	for (sfb = min_used_sfb_s;  sfb < end_sfb_s;  sfb++)
	{
		start = scalefac_band_short[sfb];
		end   = scalefac_band_short[sfb+1];

		for (b = 0;  b < 3;  b++)
		{
			expo_s[sfb][b] = 0;

			xmax_s[sfb][b] = -1.0;   /* -1 causes a setting! */
			temp = 0.0;
			for (i = start;  i < end;  i++)
			{
				if ((x = fabs(xr_org_s[i][b])) != 0.0)
				{
					xr34_s[i][b] = sqrt(x * sqrt(x));
					temp += x*x;

					if (x > xmax_s[sfb][b])
					{
						xmax_s[sfb][b] = x;
						xmax_line_s[sfb][b] = i;
					}
				}
				else
					xr34_s[i][b] = 0.0;
			}
			total_energy += energy_s[sfb][b] = temp;

			if (sfb < SBMAX_s)
				l3_xmin_s[sfb][b] = ratio_s[sfb][b] * temp;
		}
	}
}





/*  ========================================================================================  */
/*      kiss that frog                                                   [Peter Gabriel]      */
/*  ========================================================================================  */

static	int						lo_quant_l [SFB_LMAX];
static	int						hi_quant_l [SBMAX_l];

static	int						lo_quant_s [SFB_SMAX][3];
static	int						hi_quant_s [SFB_SMAX][3];

static	int						the_lo_quant;
static	int						the_hi_quant;

static	double					log_2, cc, dd;



static	void init_kissing (void)   /* called by iteration_loop() */
{
	log_2 = log(2.0);
	cc = 4.0/3.0 * log(8205.0 - 0.5 + 0.0946) / log_2;
	dd = 4.0/3.0 * log(   1.0 - 0.5 + 0.0946) / log_2;
}



static	void kiss_that_frog (void)
{
	int						sfb, b;
	double					xmax, the_xmax;


	the_lo_quant = -infinity;   /* "-infinity" */
	the_hi_quant = -infinity;   /* the real maximum for high_quant is about +4 ! */


	the_xmax = -1.0;

	for (sfb = 0;  sfb < end_sfb_l;  sfb++)
	{
		xmax = xmax_l[sfb];
		if (xmax == 0.0)
		{
			lo_quant_l[sfb] = -infinity;
			hi_quant_l[sfb] = -infinity;
		}
		else
		{
			lo_quant_l[sfb] = floor (4.0 * (log(xmax)/log_2 - cc)) + 1;
			hi_quant_l[sfb] = floor (4.0 * (log(xmax)/log_2 - dd)) + 1;

			if (xmax > the_xmax)
			{
				the_xmax = xmax;
				the_lo_quant = lo_quant_l[sfb];
				the_hi_quant = hi_quant_l[sfb];
			}
		}
	}

	for (sfb = min_used_sfb_s;  sfb < end_sfb_s;  sfb++)
	{
		for (b = 0;  b < 3;  b++)
		{
			xmax = xmax_s[sfb][b];
			if (xmax == 0.0)
			{
				lo_quant_s[sfb][b] = -infinity;
				hi_quant_s[sfb][b] = -infinity;
			}
			else
			{
				lo_quant_s[sfb][b] = floor (4.0 * (log(xmax)/log_2 - cc) - 8 * cod_info->subblock_gain[b]) + 1;
				hi_quant_s[sfb][b] = floor (4.0 * (log(xmax)/log_2 - dd) - 8 * cod_info->subblock_gain[b]) + 1;

				if (xmax > the_xmax)
				{
					the_xmax = xmax;
					the_lo_quant = lo_quant_s[sfb][b];
					the_hi_quant = hi_quant_s[sfb][b];
				}
			}
		}		
	}


	/*
		Try the power table at its least boundary
		Ive never reached this deep before!
	*/
	assert (the_lo_quant > POW216_MIN);
}





/*  ========================================================================================  */
/*      iteration_loop                                                                        */
/*  ========================================================================================  */

void iteration_loop
(
	double					pe[][2],
	double					xr_org[2][2][576],
	III_psy_ratio			*ratio,
	III_side_info_t			*l3_side,
	int						l3_enc[2][2][576],
	int						mean_bits,
	int						stereo,
	III_scalefac_t			*scalefac,
	frame_params			*fr_ps,
	int						ancillary_pad,
	int						bitsPerFrame
)
{
	static	int				fInit_iteration_loop = 0;

	int						max_bits;
	int						sfb, b;
#if 0   /* MPEG-2 */
	int						scfsi_band;
#endif
	int						mode_gr;
	int						*main_data_begin;

	layer					*info = fr_ps->header;


	side_info = l3_side;
	side_info->resvDrain = 0;
	main_data_begin = &side_info->main_data_begin;

	if (!fInit_iteration_loop)
	{
		*main_data_begin = 0;
		get_the_power ();
		read_huffcodetab ();
		init_kissing ();

		fInit_iteration_loop++;
	}
	mode_gr = 2;

	scalefac_band_long  = &sfBandIndex[info->sampling_frequency].l[0];
	scalefac_band_short = &sfBandIndex[info->sampling_frequency].s[0];


	ResvFrameBegin (fr_ps, side_info, mean_bits, bitsPerFrame);

	for (gr = 0;  gr < mode_gr;  gr++)
	{
		for (ch = 0;  ch < stereo;  ch++)
		{
			xr_org_l = xr_org[gr][ch];
			xr_org_s = (double (*)[3]) xr_org_l;

			ix_l = l3_enc[gr][ch];
			ix_s = (int (*)[3]) ix_l;

			cod_info = &side_info->gr[gr].ch[ch].tt;

#if 0   /* not needed without SCFSI */
			if (gr == 1)
				scalefac_0_l = scalefac->l[0][ch];
#endif
			scalefac_l = scalefac->l[gr][ch];
			scalefac_s = scalefac->s[gr][ch];



			/*
				reset of iteration variables

				Its no fun to control exactly which value is really used and WHEN is it used,
				so we will simply use the complete reset. But the following values can be reset
				without any special knowledge about the granules type at this point, where they
				dont overwrite some useful settings done by do_you_see_the_light().
			*/

#if 0   /* MPEG-2 */
			cod_info->sfb_partition_table = &nr_of_sfb_block[0][0][0];

			for (scfsi_band = 0;  scfsi_band < 4;  scfsi_band++)
				cod_info->slen[scfsi_band] = 0;
#endif
			cod_info->subblock_gain[0]    = 0;
			cod_info->subblock_gain[1]    = 0;
			cod_info->subblock_gain[2]    = 0;

			cod_info->count1              = 0;
			cod_info->big_values          = 0;
			cod_info->region0_count       = 0;
			cod_info->region1_count       = 0;
			cod_info->address1            = 0;
			cod_info->address2            = 0;
			cod_info->address3            = 0;
			cod_info->count1table_select  = 0;
			cod_info->table_select[0]     = 0;
			cod_info->table_select[1]     = 0;
			cod_info->table_select[2]     = 0;

			cod_info->scalefac_scale      = 0;
			cod_info->scalefac_compress   = 0;
			cod_info->preflag             = 0;
			cod_info->part2_length        = 0;
			cod_info->part2_3_length      = 0;

			for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
				scalefac_l[sfb] = 0;
			for (sfb = min_used_sfb_s;  sfb < SBMAX_s;  sfb++)
				for (b = 0;  b < 3;  b++)
					scalefac_s[sfb][b] = 0;



			do_you_see_the_light ();

			radioactivity (ratio);

			riding_on_a_dead_horse ();



			/* calculation of number of available bit( per granule ) */
			max_bits = ResvMaxBits (fr_ps, side_info, &pe[gr][ch], mean_bits);

			/* all spectral values zero ? */
			if (while_the_earth_sleeps())   /* KEEP THIS FUNCTION ALIVE */
			{
				kiss_that_frog ();
				sooner_or_later (max_bits, fr_ps);
			}

			ResvAdjust (fr_ps, cod_info, side_info, mean_bits);

			cod_info->global_gain = quant_step + 210;
/*			assert (cod_info->global_gain < 256); */
		}	/* for ch */
	}	/* for gr */

	ResvFrameEnd (fr_ps, side_info, mean_bits);
}





/*  ========================================================================================  */
/*      riding on a dead horse                              [Fury In The Slaughterhouse]      */
/*  ========================================================================================  */
/*
	A man to a woman: `Unbelievable --- you are really ugly!
	       The woman: `Oh my god! You are drunken!
	         The man: `Yes! But tomorrow, I will be sober! And you will still be ugly!

																			  [Otto Waalkes]
	----------------------------------------------------------------------------------------

	The original function calc_scfsi() is full of bugs, like multiple gr-ch-arrangement
	array access, definitely unwanted pointer subtracting, inting results away, ...
	--- but fixing the worst things doesnt produce somewhat you really want to here.

	----------------------------------------------------------------------------------------

	we killed that cat --- 2000-02-25 AP
*/

static	void riding_on_a_dead_horse (void)
{
	int						scfsi_band;

	/* None of the granules contains short blocks */
	if (!cod_info->window_switching_flag || (cod_info->block_type != SHORT_TYPE))
	{
		if (gr == 1)
		{
			for (scfsi_band = 0;  scfsi_band < 4;  scfsi_band++)
				side_info->scfsi[ch][scfsi_band] = 0;
		}
	}
}





/*  ========================================================================================  */
/*      sooner or later                                                       Supertramp      */
/*  ========================================================================================  */
/*
	The outer iteration loop controls the masking conditions of all scale
	factor bands. It computes the best scale factors and global gain.
	Meanwhile, this function contains the inner iteration loop.
*/

static	int sooner_or_later
(
	int						max_bits,
	frame_params			*fr_ps
)
{
	int						scalesave_l[SBMAX_l];
	int						scalesave_s[SBMAX_s][3];
	int						save_preflag, save_compress, save_part2len;
	int						bits, huff_bits;
	int						sfb, b, iteration;
	int						top, bot, next;


#define		TAKE_THE_LONG_WAY_HOME		0                       /* Supertramp */


	/* reset the pointers of our changed sfb [(sfb,b)] indices list */
	mark_idx_l = mark_idx_s = 0;


#if 0
	cod_info->preflag           = 0;   /* assignments are all done in iteration_loop()                      */
	cod_info->scalefac_compress = 0;   /* just to show whats going on ...                                  */
	cod_info->part2_length      = 0;   /* == part2_length(fr_ps) because of slen1_tab[0] = slen2_tab[0] = 0 */
#endif

	huff_bits = max_bits /* - cod_info->part2_length */;   /* try first without scaling */

	iteration = 1;


	bits = 0;   /* avoid compiler warning */
	/*
		A nice binary search to find a good quantizer step size
	*/
	top = the_lo_quant;
	bot = the_hi_quant;

	while (top < bot)
	{
		next = top + (bot-top)/2;

		quant_step = next;
		quantize ();
		bits = count_bits ();

		if (bits > huff_bits)
			top = next+1;
		else
			bot = next;
	}

	if (bits > huff_bits)
	{
		quant_step = top;
		quantize ();
		bits = count_bits ();
	}


	while (1) 
	{
		/*
			We assume the actual values may be the best we will find,
			so we prevent them from becoming overwritten.
		*/
		for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
			scalesave_l[sfb] = scalefac_l[sfb];

		for (sfb = min_used_sfb_s;  sfb < SBMAX_s;  sfb++)
			for (b = 0;  b < 3;  b++)
				scalesave_s[sfb][b] = scalefac_s[sfb][b];

		save_preflag  = cod_info->preflag;
		save_compress = cod_info->scalefac_compress;
		save_part2len = cod_info->part2_length;

#if TAKE_THE_LONG_WAY_HOME
		if (chart_breaker())
			break;   /* no more bands to amplify */
#endif

		silent_running ();   /* calculate the current distortion */
		if (big_time (iteration))
			break;   /* no more need to amplify bands */
		if ((huff_bits = max_bits - missing (fr_ps)) < 0)
			break;   /* not enough space to store the scale factors */

		iteration++;

		/*
			Most of the times, only a few bands will be changed,
			so why quantize the whole area?
		*/
		partial_quantize ();
		bits = count_bits ();

		while (bits > huff_bits)
		{
			quant_step++;
			quantize ();
			bits = count_bits ();
		}

#if !TAKE_THE_LONG_WAY_HOME
		if (chart_breaker())   /* no more bands to amplify */
			goto take_that_and_party;
#endif
	}


	/*
		The loop normally gets left if we arent satisfied with the result of
		the current iteration step. We use the last saved values instead.
	*/

	for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
		scalefac_l[sfb] = scalesave_l[sfb];    

	for (sfb = min_used_sfb_s;  sfb < SBMAX_s;  sfb++)
		for (b = 0;  b < 3;  b++)
			scalefac_s[sfb][b] = scalesave_s[sfb][b];

	cod_info->preflag           = save_preflag;
	cod_info->scalefac_compress = save_compress;
	cod_info->part2_length      = save_part2len;

take_that_and_party:                                 /* Take That -- CD-title */

	cod_info->part2_3_length    = cod_info->part2_length + bits;
	return cod_info->part2_3_length;
}





/*  ========================================================================================  */
/*      chart breaker                                                     [Massive Tne]      */
/*  ========================================================================================  */
/*
	There had been a lot of changes to original loop_break(), which had been
	the reason for choosing a new name for this function. Alas, the changes
	were faulty and so the (more or less) original code is still number one,
	hence, the chart breaker!

	Unfortunately, the logic of this function has a tiny lack:
		You simply cant amplify bands containing only zeroes!
	But repairing doesnt pay!
*/

static	int chart_breaker (void)
{
	int						sfb, b;

	for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
		if (scalefac_l[sfb] == 0)
			return 0;

	for (sfb = min_used_sfb_s;  sfb < SBMAX_s;  sfb++)
		for (b = 0;  b < 3;  b++)
			if (scalefac_s[sfb][b] == 0)
				return 0;

	return 1;
}





/*  ========================================================================================  */
/*      big time                                                         [Peter Gabriel]      */
/*  ========================================================================================  */
/*
	We dont use scale factor select index at the moment,
	so the parameter iteration is unused.
	Let it there as a reminder, where we have to make
	changes in case of a reanimation of scfsi.
*/

static	int big_time (int iteration)
{
	int						sfb, b;
	int						expo;
	int						over = 0;

#if SMOOTH_OPERATOR == 0
	int						stop_at;
#else
#define		stop_at					max_used_sfb_l
#endif

	if (cod_info->window_switching_flag && (cod_info->block_type == SHORT_TYPE))
	{
		expo = 16/2 /* * (cod_info->scalefac_scale + 1) */;

#ifdef MIXED_BLOCKS
		if (cod_info->mixed_block_flag)
			for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
				if (xfsf_l[sfb] > l3_xmin_l[sfb])
				{
					expo_l[sfb] += expo;
					scalefac_l[sfb]++;
					over++;
					mark_tab_l[mark_idx_l++] = sfb;
				}
#endif

		for (sfb = min_used_sfb_s;  sfb < SBMAX_s;  sfb++)
			for (b = 0;  b < 3;  b++)
				if (xfsf_s[sfb][b] > l3_xmin_s[sfb][b])
				{
					expo_s[sfb][b] += expo;
					scalefac_s[sfb][b]++;
					over++;
					mark_tab_s[mark_idx_s++] = sfb;
					mark_tab_s[mark_idx_s++] = b;
				}
	}
	else
	{



#if 0
		/*  ============    some old code as a reminder - let it in    ============  */

			/*  --------    from preemphasis()    --------  */
				if (gr == 1)
					for (scfsi_band = 0;  scfsi_band < 4;  scfsi_band++)
						if (side_info->scfsi[ch][scfsi_band])
						{
							cod_info->preflag = side_info->gr[0].ch[ch].tt.preflag;
							return;
						}

			/*  --------    from amp_scalefac_bands()    --------  */
				gr_info					*gr0 = &side_info->gr[0].ch[ch].tt;
				int						copy_or_preventSF = 0;
				int						over = 0;

				if (gr == 1)
					for (scfsi_band = 0;  scfsi_band < 4;  scfsi_band++)
						if (side_info->scfsi[ch][scfsi_band])
						{
							copy_or_preventSF = iteration;
							break;
						}

				if (copy_or_preventSF)
					expo = 16/2 /* * (gr0->scalefac_scale + 1) */;
				else
					expo = 16/2 /* * (cod_info->scalefac_scale + 1) */;

				scfsi_band = 0;
				for (sfb = 0;  sfb < max_used_sfb_l;  sfb++)
				{
					if (copy_or_preventSF)
					{
						if (sfb == scfsi_band_long[scfsi_band+1])
							scfsi_band++;
						if (side_info->scfsi[ch][scfsi_band])
						{
							if (copy_or_preventSF == 1)   /* copySF */
								scalefac_l[sfb] = scalefac_0_l[sfb];
							continue;
						}
					}	    

					if (xfsf_l[sfb] > l3_xmin_l[sfb])
					{
						expo_l[sfb] += expo;
						scalefac_l[sfb]++;
						mark_tab_l[mark_idx_l++] = sfb;
						over++;
					}
				}

		/*  =======================================================================  */
#endif



#if SMOOTH_OPERATOR <= 2
#if SMOOTH_OPERATOR == 0
		stop_at = max_used_sfb_l;
#endif

		if (!cod_info->preflag)
		{
			/*
				Lets have a break, if there is at least one of the upper four bands
				where the actual distortion does not exceed the threshold...
			*/	
			for (sfb = SBMAX_l-4;  sfb < SBMAX_l;  sfb++)
				if (xfsf_l[sfb] <= l3_xmin_l[sfb])
					goto heaven;   /* dont complain about the goto --- or go to hell ... */

			cod_info->preflag = 1;

			for (sfb = 11;  sfb < max_used_sfb_l;  sfb++)   /* pretab[sfb] = 0  for  sfb < 11 !!! */
			{
				expo = 16/2 /* * (cod_info->scalefac_scale + 1) */ * pretab[sfb];
				expo_l[sfb] += expo;
				mark_tab_l[mark_idx_l++] = sfb;
			}

#if SMOOTH_OPERATOR == 0
			over++;
			stop_at = 11;  /* we try to amplify the non-preemphased bands */
#else
			return 0;  /* we wont amplify bands because weve just turned on preemphasis */
#endif
		}

heaven:                                                       /* Eric Clapton */
#endif

		expo = 16/2 /* * (cod_info->scalefac_scale + 1) */;

		for (sfb = 0;  sfb < stop_at;  sfb++)
			if (xfsf_l[sfb] > l3_xmin_l[sfb])
			{
				expo_l[sfb] += expo;
				scalefac_l[sfb]++;
				mark_tab_l[mark_idx_l++] = sfb;
				over++;
			}
	}


	return (over == 0);
}





/*  ========================================================================================  */
/*      missing                                                [Everything But The Girl]      */
/*  ========================================================================================  */
/*
	The relation of the functions name to that what the function does is
	missing...   ;->                                     [not really   ;->>]

	It tells you whether there is no more scale factor band to amplify: it
	returns "infinity" in that case.

	If not, it counts the bits needed to code the scale factors (cod_info->
	part2_length) and the compression index (cod_info->scalefac_compress).

	If there is no suitable index, it returns "infinity", too.


	Differing from original scale_bitcount(), we will choose the best scale
	factor compression.

		Special thanks to Takehiro Tominaga [LAME] for the tip.

	We use an deranged access to the table, so we can decide a little bit
	earlier when there is no more chance to find a suitable compression. The
	algorithm doesnt work with calc_scfsi() without some changes, but we
	wont reanimate the horse at the moment.
*/

static	int missing (frame_params *fr_ps)
{
	/* 2^^slen1[k] */
	static	int    pow2_slen1[16] = {  1,  1,  1,  1,  8,  2,  2,  2,  4,  4,  4,  8,  8,  8, 16, 16};
	/* 2^^slen2[k] */
	static	int    pow2_slen2[16] = {  1,  2,  4,  8,  1,  2,  4,  8,  2,  4,  8,  2,  4,  8,  4,  8};

#ifdef MIXED_BLOCKS
	/* (8+9) * pow2_slen1[k] + (9+9) * pow2_slen2[k] */
	static	int mixed_pickles[16] = {  0, 18, 36, 54, 51, 35, 53, 71, 52, 70, 88, 69, 87,105,104,122};
#endif
	/* (9+9) * pow2_slen1[k] + (9+9) * pow2_slen2[k] */
	static	int  catch_shorty[16] = {  0, 18, 36, 54, 54, 36, 54, 72, 54, 72, 90, 72, 90,108,108,126};
	/* (6+5) * pow2_slen1[k] + (5+5) * pow2_slen2[k] */
	static	int   extra_large[16] = {  0, 10, 20, 30, 33, 21, 31, 41, 32, 42, 52, 43, 53, 63, 64, 74};

	static	int      deranged[16] = { 15, 14, 13, 12, 11,  4, 10,  9,  8,  7,  6,  5,  3,  2,  1,  0};

	int						sfb, b, k, q;
	int						max_slen1, max_slen2, *table;


	max_slen1 = max_slen2 = 0;

	if (cod_info->window_switching_flag && (cod_info->block_type == SHORT_TYPE))
	{
#ifdef MIXED_BLOCKS
		if (cod_info->mixed_block_flag)
		{
			table = mixed_pickles;

			for (sfb = 0;  sfb < 8;  sfb++)
				if (scalefac_l[sfb] > max_slen1)
					max_slen1 = scalefac_l[sfb];

			for (sfb = 3;  sfb < 6;  sfb++)
				for (b = 0;  b < 3;  b++)
					if (scalefac_s[sfb][b] > max_slen1)
						max_slen1 = scalefac_s[sfb][b];

			for (sfb = 6;  sfb < SBMAX_s;  sfb++)
				for (b = 0;  b < 3;  b++)
					if (scalefac_s[sfb][b] > max_slen2)
						max_slen2 = scalefac_s[sfb][b];
		}
		else
#endif
		{
			table = catch_shorty;

			for (sfb = 0;  sfb < 6;  sfb++)
				for (b = 0;  b < 3;  b++)
					if (scalefac_s[sfb][b] > max_slen1)
						max_slen1 = scalefac_s[sfb][b];

			for (sfb = 6;  sfb < SBMAX_s;  sfb++)
				for (b = 0;  b < 3;  b++)
					if (scalefac_s[sfb][b] > max_slen2)
						max_slen2 = scalefac_s[sfb][b];
		}
	}
	else
	{
		table = extra_large;

		for (sfb = 0;  sfb < 11;  sfb++)
			if (scalefac_l[sfb] > max_slen1)
				max_slen1 = scalefac_l[sfb];

#if SMOOTH_OPERATOR >= 2
		/* This was seen in LAME */
		if (!cod_info->preflag)
		{
			for (sfb = 11;  sfb < SBMAX_l;  sfb++)
				if (scalefac_l[sfb] < /* (cod_info->scalefac_scale + 1) * */ pretab[sfb])
					break;

			if (sfb == SBMAX_l)
			{
				for (sfb = 11;  sfb < SBMAX_l;  sfb++)
					scalefac_l[sfb] -= /* (cod_info->scalefac_scale + 1) * */ pretab[sfb];
				cod_info->preflag = 1;
			}
		}
#endif

		for (sfb = 11;  sfb < SBMAX_l;  sfb++)
			if (scalefac_l[sfb] > max_slen2)
				max_slen2 = scalefac_l[sfb];
	}


	cod_info->part2_length = infinity;

	for (k = 0;  k < 16;  k++)
	{
		q = deranged[k];

		if (max_slen1 >= pow2_slen1[q])  break;   /* it doesnt get better */

		if ((max_slen2 < pow2_slen2[q]) && (cod_info->part2_length > table[q]))
		{
			cod_info->scalefac_compress = q;
			cod_info->part2_length      = table[q];
		}
	}


	return cod_info->part2_length;
}





/*  ========================================================================================  */
/*      silent running                                          [Mike And The Mechanics]      */
/*  ========================================================================================  */
/*
	Calculate the current distortion introduced by the quantization in each
	scale factor band.
*/

static	void silent_running (void)
{
	int						i, b, sfb, start, end;
	double					f, sum, temp;

	for (sfb=0; sfb<max_used_sfb_l; sfb++)
	{
		if (ix_l[xmax_line_l[sfb]] == 0)
		{
			xfsf_l[sfb] = energy_l[sfb];
		}
		else
		{
			start = scalefac_band_long[sfb];
			end   = scalefac_band_long[sfb+1];

			sum = 0.0;

			f = pow216[expo_l[sfb] - 4 * quant_step];

			for (i=start; i<end; i++)
			{
				temp = fabs(xr_org_l[i]) - noisePowTab[ix_l[i]] / f;
				sum += temp * temp;
			}

			xfsf_l[sfb] = sum;
		}
	}

	for (b=0; b<3; b++)
	{
		for (sfb=min_used_sfb_s; sfb<SBMAX_s; sfb++)
		{
			if (ix_s[xmax_line_s[sfb][b]] == 0)
			{
				xfsf_s[sfb][b] = energy_s[sfb][b];
			}
			else
			{
				start = scalefac_band_short[sfb];
				end   = scalefac_band_short[sfb+1];

				sum = 0.0;

				f = pow216[expo_s[sfb][b] - 4 * (quant_step + 8 * cod_info->subblock_gain[b])];

				for (i=start; i<end; i++)
				{
	 				temp = fabs(xr_org_s[i][b]) - noisePowTab[ix_s[i][b]] / f;
					sum += temp * temp;
				}       

				xfsf_s[sfb][b] = sum;
			}
		}
	}
}





/*  ========================================================================================  */
/*      quantize                                                                              */
/*  ========================================================================================  */
/*
	Quantization of the vector xr ( -> ix)
*/

#define	cutting_crew(hair) (hair + 0.4054)			/* Inline just caused us troubles... */
								/* 0.5 - 0.0946 */
/*
static	int INLINE cutting_crew (double hair)
{
	return  hair + 0.4054;
}
*/


static	void quantize (void)
{
	int						sfb, i, b, start, end;
	double					f, z, y;

	for (sfb=0; sfb<end_sfb_l; sfb++)
	{
		start = scalefac_band_long[sfb];
		end   = scalefac_band_long[sfb+1];

		if (quant_step - expo_l[sfb]/4 >= hi_quant_l[sfb])
		{
			for (i=start; i<end; )
			{
				ix_l[i++] = 0;
				ix_l[i++] = 0;
			}
		}
		else
		{
			        /* (expo_l[sfb] - 16/4 * quant_step) * 3/4 */
			f = pow216[(expo_l[sfb]/4 - quant_step) * 3];

			for (i=start; i<end; i+=2)
			{
				z = xr34_l[i  ] * f;
				y = xr34_l[i+1] * f;
				ix_l[i  ] = cutting_crew (z);
				ix_l[i+1] = cutting_crew (y);
			}
		}
	}

	for (sfb=min_used_sfb_s; sfb<end_sfb_s; sfb++)
	{
		start = scalefac_band_short[sfb];
		end   = scalefac_band_short[sfb+1];

		for (b=0; b<3; b++)
		{
			if (quant_step - expo_s[sfb][b]/4 >= hi_quant_s[sfb][b])
			{
				for (i=start; i<end; )
				{
					ix_s[i++][b] = 0;
					ix_s[i++][b] = 0;
				}
			}
			else
			{
				        /* (expo_s[sfb][b] - 16/4 * (quant_step + 8 * cod_info->subblock_gain[b])) * 3/4 */
				f = pow216[(expo_s[sfb][b] / 4 - quant_step - 8 * cod_info->subblock_gain[b]) * 3];

				for (i=start; i<end; i+=2)
				{
					z = xr34_s[i  ][b] * f;
					y = xr34_s[i+1][b] * f;
					ix_s[i  ][b] = cutting_crew (z);
					ix_s[i+1][b] = cutting_crew (y);
				}
			}
		}
	}
}



static	void partial_quantize (void)
{
	int						sfb, i, b, start, end;
	double					f, z, y;

	while (mark_idx_l)
	{
		sfb = mark_tab_l[--mark_idx_l];

		start = scalefac_band_long[sfb];
		end   = scalefac_band_long[sfb+1];

		if (quant_step - expo_l[sfb]/4 >= hi_quant_l[sfb])
		{
			for (i=start; i<end; )
			{
				ix_l[i++] = 0;
				ix_l[i++] = 0;
			}
		}
		else
		{
			        /* (expo_l[sfb] - 16/4 * quant_step) * 3/4 */
			f = pow216[(expo_l[sfb]/4 - quant_step) * 3];

			for (i=start; i<end; i+=2)
			{
				z = xr34_l[i  ] * f;
				y = xr34_l[i+1] * f;
				ix_l[i  ] = cutting_crew (z);
				ix_l[i+1] = cutting_crew (y);
			}
		}
	}

	while (mark_idx_s)
	{
		b   = mark_tab_s[--mark_idx_s];
		sfb = mark_tab_s[--mark_idx_s];

		start = scalefac_band_short[sfb];
		end   = scalefac_band_short[sfb+1];

		if (quant_step - expo_s[sfb][b]/4 >= hi_quant_s[sfb][b])
		{
			for (i=start; i<end; )
			{
				ix_s[i++][b] = 0;
				ix_s[i++][b] = 0;
			}
		}
		else
		{
			        /* (expo_s[sfb][b] - 16/4 * (quant_step + 8 * cod_info->subblock_gain[b])) * 3/4 */
			f = pow216[(expo_s[sfb][b] / 4 - quant_step - 8 * cod_info->subblock_gain[b]) * 3];

			for (i=start; i<end; i+=2)
			{
				z = xr34_s[i  ][b] * f;
				y = xr34_s[i+1][b] * f;
				ix_s[i  ][b] = cutting_crew (z);
				ix_s[i+1][b] = cutting_crew (y);
			}
		}
	}
}





/*  ========================================================================================  */
/*      count_bits                                                                            */
/*  ========================================================================================  */

struct
{
	unsigned region0_count;
	unsigned region1_count;
} subdv_table[ 23 ] =
{
	{0, 0}, /*  0 bands */
	{0, 0}, /*  1 bands */
	{0, 0}, /*  2 bands */
	{0, 0}, /*  3 bands */
	{0, 0}, /*  4 bands */
	{0, 1}, /*  5 bands */
	{1, 1}, /*  6 bands */
	{1, 1}, /*  7 bands */
	{1, 2}, /*  8 bands */
	{2, 2}, /*  9 bands */
	{2, 3}, /* 10 bands */
	{2, 3}, /* 11 bands */
	{3, 4}, /* 12 bands */
	{3, 4}, /* 13 bands */
	{3, 4}, /* 14 bands */
	{4, 5}, /* 15 bands */
	{4, 5}, /* 16 bands */
	{4, 6}, /* 17 bands */
	{5, 6}, /* 18 bands */
	{5, 6}, /* 19 bands */
	{5, 7}, /* 20 bands */
	{6, 7}, /* 21 bands */
	{6, 7}, /* 22 bands */
};



/*
	Calculation of rzero, count1, big_values
	(Partitions ix into big values, quadruples and zeros).

	Determines the number of bits to encode the quadruples.

	Presumable subdivides the bigvalue region which will
	use separate Huffman tables.

	Select huffman code tables for bigvalues regions

	Count the number of bits necessary to code the bigvalues region.
*/

static	int count_bits (void)
{
	cod_info->table_select[0] = 0;
	cod_info->table_select[1] = 0;
	cod_info->table_select[2] = 0;

	if (cod_info->window_switching_flag && (cod_info->block_type == SHORT_TYPE))
		return count_bits_short ();
	else
		return count_bits_long ();
}





static	int count_bits_short (void)
{
	/*
		Within each scalefactor band, data is given for successive
		time windows, beginning with window 0 and ending with window 2.
		Within each window, the quantized values are then arranged in
		order of increasing frequency...
	*/
	int						sfb, window;
	int						temp, max;

	unsigned				bits = 0;


	/*
		the first part --- 8 long blocks or 3 short blocks
	*/

	max = 0;
#ifdef MIXED_BLOCKS
	if (cod_info->mixed_block_flag)
	{
		for (sfb=0; sfb<8; sfb++)
			if ((temp = ix_l[xmax_line_l[sfb]]) > max)
				max = temp;
		everything_counts (0, 36, max, &cod_info->table_select[0], &bits);
	}
	else
#endif
	{
		for (sfb=0; sfb<3; sfb++)
			for (window=0; window<3; window++)
				if ((temp = ix_s[xmax_line_s[sfb][window]][window]) > max)
					max = temp;
		step_by_step (0, 3, max, &cod_info->table_select[0], &bits);
	}


	/*
		the second part --- short blocks only
	*/

	max = 0;
	for (sfb=3; sfb<SFB_SMAX; sfb++)
		for (window=0; window<3; window++)
			if ((temp = ix_s[xmax_line_s[sfb][window]][window]) > max)
				max = temp;
	step_by_step (3, SFB_SMAX, max, &cod_info->table_select[1], &bits);


	return bits;
}





static	int count_bits_long (void)
{
	int						zero_region, zero_sfb;
	int						bigv_region;

	unsigned				bits = 0;
	unsigned				sum0 = 0;
	unsigned				sum1/* = 0*/;

	int						sfb_anz, index0, index1, sfb, i;
	unsigned				max, temp;
	int						p;

	for (zero_sfb=SFB_LMAX; zero_sfb>0; zero_sfb--)
		if (ix_l[xmax_line_l[zero_sfb-1]])
			break;
	zero_region = scalefac_band_long[zero_sfb];

	for (; zero_region>1; zero_region-= 2)
		     if (ix_l[zero_region-1])  break;
		else if (ix_l[zero_region-2])  break;

	for (bigv_region=zero_region; bigv_region>3; bigv_region-= 4)
	{
		     if (ix_l[bigv_region-1] > 1)  break;
		else if (ix_l[bigv_region-2] > 1)  break;
		else if (ix_l[bigv_region-3] > 1)  break;
		else if (ix_l[bigv_region-4] > 1)  break;

		p = 0;
		if (ix_l[bigv_region-1])  bits++, p |= 8;
		if (ix_l[bigv_region-2])  bits++, p |= 4;
		if (ix_l[bigv_region-3])  bits++, p |= 2;
		if (ix_l[bigv_region-4])  bits++, p |= 1;

		sum0 += ht[32].hlen[p];
/*		sum1 += ht[33].hlen[p];*/
	}

	cod_info->count1     = (sum1 = zero_region-bigv_region) / 4;
	cod_info->big_values =                     bigv_region  / 2;



	cod_info->count1table_select = (sum0 < sum1) ? (bits += sum0, 0) : (bits += sum1, 1);



	if (bigv_region)
	{
		sfb_anz = 1;   /* setting at this place isnt needed, but avoids a compiler warning */

		if (cod_info->window_switching_flag)   /* START_TYPE, STOP_TYPE */
		{
			index0 = /* cod_info->region0_count + 1 = */  8;
			index1 = /* cod_info->region1_count + 1 = */ 14;

			cod_info->address2 = bigv_region;
		}
		else   /* NORM_TYPE */
		{
			while (scalefac_band_long[sfb_anz] < bigv_region)
				sfb_anz++;

			index0 = (cod_info->region0_count = subdv_table[sfb_anz].region0_count) + 1;
			index1 = (cod_info->region1_count = subdv_table[sfb_anz].region1_count) + 1;

			cod_info->address1 = scalefac_band_long[index0];
			cod_info->address2 = scalefac_band_long[index0 + index1];
			cod_info->address3 = bigv_region;
		}

		/* if (cod_info->address1 > 0) --- always, because of index0 > 0 AND cod_info->address1 == scalefac_band_long[index0] */
		{
			max = 0;
			for (sfb=0; sfb<index0; sfb++)
				if ((temp = ix_l[xmax_line_l[sfb]]) > max)
					max = temp;
			everything_counts (0, cod_info->address1, max, &cod_info->table_select[0], &bits);
		}

		if (cod_info->address2 > cod_info->address1) /* index1 > 0, but cod_info->address2 ... [in case of cod_info->window_switching_flag] */
		{
			max = 0;
			for (sfb=index0; sfb<index0+index1; sfb++)
				if ((temp = ix_l[xmax_line_l[sfb]]) > max)
					max = temp;
			everything_counts (cod_info->address1, cod_info->address2, max, &cod_info->table_select[1], &bits);
		}

		if (bigv_region > cod_info->address2)   /* especially is sfb_anz valid */
		{
			max = 0;
			for (sfb=index0+index1; sfb<sfb_anz; sfb++)   /* here we can use the maxima of the scale factor bands */
				if ((temp = ix_l[xmax_line_l[sfb]]) > max)
					max = temp;
			for (i=scalefac_band_long[sfb_anz-1]; i<bigv_region; i++)   /* but this last part is not the complete band */
				if (ix_l[i] > max)
					max = ix_l[i];
			everything_counts (cod_info->address2, bigv_region, max, &cod_info->table_select[2], &bits);
		}
	}
	else   /* no big_values region --- under rare circumstances */
	{
		cod_info->region0_count = 0;
		cod_info->region1_count = 0;

		cod_info->address1 = 0;   /* but avoid at least superfluous calculations elsewhere, e.g. "l3bitstream.c" */
		cod_info->address2 = 0;
		cod_info->address3 = 0;
	}


	return bits;
}





/*  ========================================================================================  */
/*      everything counts                                                 [Depeche Mode]      */
/*  ========================================================================================  */
/*
	Choose the Huffman table that will encode ix[start..end] with the fewest
	bits and increases the bit_sum by the amount of these bits.

	Note: This code contains knowledge about the sizes and characteristics
	of the Huffman tables as defined in the IS (Table B.7), and will not work
	with any arbitrary tables.
*/
static	void everything_counts
(
	unsigned				start,
	unsigned				end,
	unsigned				max,
	unsigned				*table,
	unsigned				*bit_sum
)
{
	unsigned				choice0, choice1;


	if (max == 0)
		return;

	if (max < 15)
	{
		choice0 = 1;  /* we can start with 1 because ht[0].xlen == 0 <= max */
		while (ht[choice0].xlen <= max)
			choice0++;

		switch (choice0)
		{
			case  1:  single_Huffman (start, end,/* 1 */      table, bit_sum);  break;
			case  2:  double_Huffman (start, end,  2,  3,     table, bit_sum);  break;
			case  5:  double_Huffman (start, end,  5,  6,     table, bit_sum);  break;
			case  7:  triple_Huffman (start, end,  7,  8,  9, table, bit_sum);  break;
			case 10:  triple_Huffman (start, end, 10, 11, 12, table, bit_sum);  break;
			case 13:  double_Huffman (start, end, 13, 15,     table, bit_sum);  break;
		}
	}
	else if (max == 15)
	{
		triple_Huffman_2 (start, end,/* 13, 15, 24, */ table, bit_sum);
	}
	else
	{
		max -= 15;
		choice0 = 16;  while (ht[choice0].linmax < max)  choice0++;
assert(choice0 < 24);
		choice1 = 24;  while (ht[choice1].linmax < max)  choice1++;
assert(choice1 < 32);
		double_Huffman_2 (start, end, choice0, choice1, table, bit_sum);
	}
}





/*  ========================================================================================  */
/*      step by step                                                   [Whitney Houston]      */
/*  ========================================================================================  */
/*
	Choose the Huffman table that will encode ix[start_sfb..end_sfb][0..2]
	with the fewest bits and increases the bit_sum by the amount of these bits.

	Note: This code contains knowledge about the sizes and characteristics
	of the Huffman tables as defined in the IS (Table B.7), and will not work
	with any arbitrary tables.
*/

static	void step_by_step
(
	unsigned				start_sfb,
	unsigned				end_sfb,
	unsigned				max,
	unsigned				*table,
	unsigned				*bit_sum
)
{
	unsigned				choice0, choice1;
	int						start, end;

	start = 3 * scalefac_band_short[start_sfb];
	end   = 3 * scalefac_band_short[  end_sfb];

	if (max == 0)
		return;

	if (max < 15)
	{
		choice0 = 1;  /* we can start with 1 because ht[0].xlen == 0 <= max */
		while (ht[choice0].xlen <= max)
			choice0++;

		switch (choice0)
		{
			case  1:  tiny_single_Huffman (start, end,/* 1 */      table, bit_sum);  break;
			case  2:  tiny_double_Huffman (start, end,  2,  3,     table, bit_sum);  break;
			case  5:  tiny_double_Huffman (start, end,  5,  6,     table, bit_sum);  break;
			case  7:  tiny_triple_Huffman (start, end,  7,  8,  9, table, bit_sum);  break;
			case 10:  tiny_triple_Huffman (start, end, 10, 11, 12, table, bit_sum);  break;
			case 13:  tiny_double_Huffman (start, end, 13, 15,     table, bit_sum);  break;
		}
	}
	else if (max == 15)
	{
		tiny_triple_Huffman_2 (start, end,/* 13, 15, 24, */ table, bit_sum);
	}
	else
	{
		max -= 15;
		choice0 = 16;  while (ht[choice0].linmax < max)  choice0++;
assert(choice0 < 24);
		choice1 = 24;  while (ht[choice1].linmax < max)  choice1++;
assert(choice1 < 32);
		tiny_double_Huffman_2 (start, end, choice0, choice1, table, bit_sum);
	}
}





/*  ========================================================================================  */
/*      Huffmania                                                                             */
/*  ========================================================================================  */



/*
	That case, we dont need to decide which is the best table.
*/

static	void single_Huffman
(
	unsigned				start,
	unsigned				end,
/*	unsigned				table0, == 1 */
	unsigned				*choice,
	unsigned				*sum
)
{
/*
	int						v;
*/
	unsigned				bits0, signs, idx;

	static	struct huffcodetab		*h0 = ht + /* table0 */ 1;   /* static because of the constant!!! */

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == 2 */
#endif

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = signs = 0;

	while (pos < fin)
	{
		idx = 0;
#if 0
		v = *pos++;  if (v)  {signs++;  idx = v /* * ylen */ + v;}
		v = *pos++;  if (v)  {signs++;  idx += v;}
#else
		if (*pos++)  {signs++;  idx = 2;}
		if (*pos++)  {signs++;  idx++;}
#endif
		bits0 += h0->hlen[idx];
	}

	*choice = /* table0 */ 1;
	*sum += bits0 + signs;
}



static	void tiny_single_Huffman
(
	unsigned				start,
	unsigned				end,
/*	unsigned				table0, == 1 */
	unsigned				*choice,
	unsigned				*sum
)
{
/*
	int						v0, v1, v2;
*/
	unsigned				bits0, signs, idx0, idx1, idx2;

	static	struct huffcodetab		*h0 = ht + /* table0 */ 1;   /* static because of the constant!!! */

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == 2 */
#endif

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = signs = 0;

	while (pos < fin)
	{
		idx0 = idx1 = idx2 = 0;
#if 0
		v0 = *pos++;  if (v0)  {signs++;  idx0 = v0 /* * ylen */ + v0;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 = v1 /* * ylen */ + v1;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 = v2 /* * ylen */ + v2;}
		v0 = *pos++;  if (v0)  {signs++;  idx0 += v0;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 += v1;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 += v2;}
#else
		if (*pos++)  {signs++;  idx0 = 2;}
		if (*pos++)  {signs++;  idx1 = 2;}
		if (*pos++)  {signs++;  idx2 = 2;}
		if (*pos++)  {signs++;  idx0++;}
		if (*pos++)  {signs++;  idx1++;}
		if (*pos++)  {signs++;  idx2++;}
#endif
		bits0 += h0->hlen[idx0] + h0->hlen[idx1] + h0->hlen[idx2];
	}

	*choice = /* table0 */ 1;
	*sum += bits0 + signs;
}





/*
	The following function is called for the most maximum values below 16 (respectively 15)
*/

static	void double_Huffman
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 2, 5, 13 */
	unsigned				table1,   /* 3, 6, 15 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v;
	unsigned				bits0, bits1, signs, idx;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;

	unsigned				ylen = h0->ylen;   /* == h1->ylen */

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = signs = 0;

	while (pos < fin)
	{
		idx = 0;
		v = *pos++;  if (v)  {signs++;  idx = v * ylen;}
		v = *pos++;  if (v)  {signs++;  idx += v;}
		bits0 += h0->hlen[idx];
		bits1 += h1->hlen[idx];
	}

	if (bits0 < bits1)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
}



static	void tiny_double_Huffman
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 2, 5, 13 */
	unsigned				table1,   /* 3, 6, 15 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v0, v1, v2;
	unsigned				bits0, bits1, signs, idx0, idx1, idx2;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;

	unsigned				ylen = h0->ylen;   /* == h1->ylen */

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = signs = 0;

	while (pos < fin)
	{
		idx0 = idx1 = idx2 = 0;
		v0 = *pos++;  if (v0)  {signs++;  idx0 = v0 * ylen;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 = v1 * ylen;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 = v2 * ylen;}
		v0 = *pos++;  if (v0)  {signs++;  idx0 += v0;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 += v1;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 += v2;}
		bits0 += h0->hlen[idx0] + h0->hlen[idx1] + h0->hlen[idx2];
		bits1 += h1->hlen[idx0] + h1->hlen[idx1] + h1->hlen[idx2];
	}

	if (bits0 < bits1)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
}



/*
	poor mens brave tailor --- only three at a blow...
*/

static	void triple_Huffman
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 7, 10 */
	unsigned				table1,   /* 8, 11 */
	unsigned				table2,   /* 9, 12 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v;
	unsigned				bits0, bits1, bits2, signs, idx;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;
	struct huffcodetab		*h2 = ht + table2;

	unsigned				ylen = h0->ylen;   /* == h1->ylen == h2->ylen */

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = bits2 = signs = 0;

	while (pos < fin)
	{
		idx = 0;
		v = *pos++;  if (v)  {signs++;  idx = v * ylen;}
		v = *pos++;  if (v)  {signs++;  idx += v;}
		bits0 += h0->hlen[idx];
		bits1 += h1->hlen[idx];
		bits2 += h2->hlen[idx];
	}

	if (bits0 < bits1  &&  bits0 < bits2)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else if (bits1 < bits2)
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
	else
	{
		*choice = table2;
		*sum += bits2 + signs;
	}
}



static	void tiny_triple_Huffman
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 7, 10 */
	unsigned				table1,   /* 8, 11 */
	unsigned				table2,   /* 9, 12 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v0, v1, v2;
	unsigned				bits0, bits1, bits2, signs, idx0, idx1, idx2;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;
	struct huffcodetab		*h2 = ht + table2;

	unsigned				ylen = h0->ylen;   /* == h1->ylen == h2->ylen */

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = bits2 = signs = 0;

	while (pos < fin)
	{
		idx0 = idx1 = idx2 = 0;
		v0 = *pos++;  if (v0)  {signs++;  idx0 = v0 * ylen;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 = v1 * ylen;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 = v2 * ylen;}
		v0 = *pos++;  if (v0)  {signs++;  idx0 += v0;}
		v1 = *pos++;  if (v1)  {signs++;  idx1 += v1;}
		v2 = *pos++;  if (v2)  {signs++;  idx2 += v2;}
		bits0 += h0->hlen[idx0] + h0->hlen[idx1] + h0->hlen[idx2];
		bits1 += h1->hlen[idx0] + h1->hlen[idx1] + h1->hlen[idx2];
		bits2 += h2->hlen[idx0] + h2->hlen[idx1] + h2->hlen[idx2];
	}

	if (bits0 < bits1  &&  bits0 < bits2)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else if (bits1 < bits2)
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
	else
	{
		*choice = table2;
		*sum += bits2 + signs;
	}
}





/*
	The escape table 24 deals with linbits=4 instead of linbits=0 in case of table 13 and 15.
	Nevertheless, sometimes it produces the better result...
	Furthermore we take advantage because of the constant table numbers.
*/

static	void triple_Huffman_2
(
	unsigned				start,
	unsigned				end,
/*	unsigned				table0,   == 13 */
/*	unsigned				table1,   == 15 */
/*	unsigned				table2,   == 24 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v;
	unsigned				bits0, bits1, bits2, signs, idx;

	static	struct huffcodetab		*h0 = ht + /* table0 */ 13;   /* all static declarations because of the constant values!!! */
	static	struct huffcodetab		*h1 = ht + /* table1 */ 15;
	static	struct huffcodetab		*h2 = ht + /* table2 */ 24;

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == h1->ylen == h2->ylen */   /* == 16 */
#endif

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = bits2 = signs = 0;

	while (pos < fin)
	{
		idx = 0;
		v = *pos++;  if (v)  {if (v == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx = v /* * ylen */ << 4;}
		v = *pos++;  if (v)  {if (v == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx += v;}
		bits0 += h0->hlen[idx];
		bits1 += h1->hlen[idx];
		bits2 += h2->hlen[idx];
	}

	if (bits0 < bits1  &&  bits0 < bits2)
	{
		*choice = /* table0 */ 13;
		*sum += bits0 + signs;
	}
	else if (bits1 < bits2)
	{
		*choice = /* table1 */ 15;
		*sum += bits1 + signs;
	}
	else
	{
		*choice = /* table2 */ 24;
		*sum += bits2 + signs;
	}
}



static	void tiny_triple_Huffman_2
(
	unsigned				start,
	unsigned				end,
/*	unsigned				table0,   == 13 */
/*	unsigned				table1,   == 15 */
/*	unsigned				table2,   == 24 */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v0, v1, v2;
	unsigned				bits0, bits1, bits2, signs, idx0, idx1, idx2;

	static	struct huffcodetab		*h0 = ht + /* table0 */ 13;   /* all static declarations because of the constant values!!! */
	static	struct huffcodetab		*h1 = ht + /* table1 */ 15;
	static	struct huffcodetab		*h2 = ht + /* table2 */ 24;

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == h1->ylen == h2->ylen */   /* == 16 */
#endif

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = bits2 = signs = 0;

	while (pos < fin)
	{
		idx0 = idx1 = idx2 = 0;
		v0 = *pos++;  if (v0)  {if (v0 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx0 = v0 /* * ylen */ << 4;}
		v1 = *pos++;  if (v1)  {if (v1 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx1 = v1 /* * ylen */ << 4;}
		v2 = *pos++;  if (v2)  {if (v2 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx2 = v2 /* * ylen */ << 4;}
		v0 = *pos++;  if (v0)  {if (v0 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx0 += v0;}
		v1 = *pos++;  if (v1)  {if (v1 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx1 += v1;}
		v2 = *pos++;  if (v2)  {if (v2 == 15)  bits2 += /* h2->linbits */ 4;  signs++;  idx2 += v2;}
		bits0 += h0->hlen[idx0] + h0->hlen[idx1] + h0->hlen[idx2];
		bits1 += h1->hlen[idx0] + h1->hlen[idx1] + h1->hlen[idx2];
		bits2 += h2->hlen[idx0] + h2->hlen[idx1] + h2->hlen[idx2];
	}

	if (bits0 < bits1  &&  bits0 < bits2)
	{
		*choice = /* table0 */ 13;
		*sum += bits0 + signs;
	}
	else if (bits1 < bits2)
	{
		*choice = /* table1 */ 15;
		*sum += bits1 + signs;
	}
	else
	{
		*choice = /* table2 */ 24;
		*sum += bits2 + signs;
	}
}





/*
	In case of two escape tables, we esepecially have to take care for
	the possibly different linbits values...
*/

static	void double_Huffman_2   /* Escape tables */
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 16... */
	unsigned				table1,   /* 24... */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v;
	unsigned				bits0, bits1, signs, xbits, idx;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == h1->ylen */   /* == 16 */
#endif
	unsigned				linbits0 = h0->linbits;
	unsigned				linbits1 = h1->linbits;

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = signs = xbits = 0;

	while (pos < fin)
	{
		idx = 0;
		v = *pos++;  if (v)  {if (v > 14)  {v = 15;  xbits++;/*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx = v /* * ylen */ << 4;}
		v = *pos++;  if (v)  {if (v > 14)  {v = 15;  xbits++;/*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx += v;}
		bits0 += h0->hlen[idx];
		bits1 += h1->hlen[idx];
	}
	bits0 += xbits * linbits0;
	bits1 += xbits * linbits1;

	if (bits0 < bits1)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
}



static	void tiny_double_Huffman_2   /* Escape tables */
(
	unsigned				start,
	unsigned				end,
	unsigned				table0,   /* 16... */
	unsigned				table1,   /* 24... */
	unsigned				*choice,
	unsigned				*sum
)
{
	int						v0, v1, v2;
	unsigned				bits0, bits1, signs, xbits, idx0, idx1, idx2;

	struct huffcodetab		*h0 = ht + table0;
	struct huffcodetab		*h1 = ht + table1;

#if 0   /* not needed */
	static	unsigned				ylen = h0->ylen;   /* == h1->ylen */   /* == 16 */
#endif
	unsigned				linbits0 = h0->linbits;
	unsigned				linbits1 = h1->linbits;

	int						*pos = ix_l + start;
	int						*fin = ix_l + end;

	bits0 = bits1 = signs = xbits = 0;

	while (pos < fin)
	{
		idx0 = idx1 = idx2 = 0;
		v0 = *pos++;  if (v0)  {if (v0 > 14)  {v0 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx0 = v0 /* * ylen */ << 4;}
		v1 = *pos++;  if (v1)  {if (v1 > 14)  {v1 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx1 = v1 /* * ylen */ << 4;}
		v2 = *pos++;  if (v2)  {if (v2 > 14)  {v2 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx2 = v2 /* * ylen */ << 4;}
		v0 = *pos++;  if (v0)  {if (v0 > 14)  {v0 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx0 += v0;}
		v1 = *pos++;  if (v1)  {if (v1 > 14)  {v1 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx1 += v1;}
		v2 = *pos++;  if (v2)  {if (v2 > 14)  {v2 = 15;  xbits++; /*bits0 += linbits0;  bits1 += linbits1;*/}  signs++;  idx2 += v2;}
		bits0 += h0->hlen[idx0] + h0->hlen[idx1] + h0->hlen[idx2];
		bits1 += h1->hlen[idx0] + h1->hlen[idx1] + h1->hlen[idx2];
	}
	bits0 += xbits * linbits0;
	bits1 += xbits * linbits1;

	if (bits0 < bits1)
	{
		*choice = table0;
		*sum += bits0 + signs;
	}
	else
	{
		*choice = table1;
		*sum += bits1 + signs;
	}
}





/*  ========================================================================================  */
/*      This is the end, ...                                                 [The Doors]      */
/*  ========================================================================================  */


