/*
			(c) Copyright 1998-2000 - Tord Jansson
			=======================================

		This file is part of the BladeEnc MP3 Encoder, based on
		ISO's reference code for MPEG Layer 3 compression, and might
		contain smaller or larger sections that are directly taken
		from ISO's reference code.

		All changes to the ISO reference code herein are either
		copyrighted by Tord Jansson (tord.jansson@swipnet.se)
		or sublicensed to Tord Jansson by a third party.

	BladeEnc is free software; you can redistribute this file
	and/or modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	2000-03-21  Andr Piotrowski
*/

#include "common.h"
#include "encoder.h"





/*  ========================================================================================  */
/*      rebuffer_audio                                                                        */
/*  ========================================================================================  */

#define		FLUSH					1152
#define		DELAY					 768

void rebuffer_audio
(
	const short				*insamp,
	FLOAT					buffer[2][2048],
	int						*buffer_idx,
	unsigned int			samples_read,
	int						stereo
)
{
	int						idx, med, fin;

	/* flush the last two read granules */
	*buffer_idx = (*buffer_idx + FLUSH) & 2047;
	idx = (*buffer_idx + DELAY) & 2047;
	fin = (idx + FLUSH) & 2047;

	if (stereo == 2)
	{
		med = (idx + samples_read/2) & 2047;
		if (idx >= med)
		{
			while (idx < 2048)
			{
				buffer[0][idx] = *insamp++;
				buffer[1][idx] = *insamp++;
				idx++;
			}
			idx = 0;
		}
		while (idx < med)
		{
			buffer[0][idx] = *insamp++;
			buffer[1][idx] = *insamp++;
			idx++;
		}
	}
	else
	{		
		med = (idx + samples_read) & 2047;
		if (idx >= med)
		{
			while (idx < 2048)
			{
				buffer[0][idx] = *insamp++;
				buffer[1][idx] = 0;
				idx++;
			}
			idx = 0;
		}
		while (idx < med)
		{
			buffer[0][idx] = *insamp++;
			buffer[1][idx] = 0;
			idx++;
		}
	}

	if (idx != fin)
	{
		if (idx > fin)
		{
			while (idx < 2048)
			{
				buffer[0][idx] = 0;
				buffer[1][idx] = 0;
				idx++;
			}
			idx = 0;
		}
		while (idx < fin)
		{
			buffer[0][idx] = 0;
			buffer[1][idx] = 0;
			idx++;
		}
	}
}





typedef double 			filter_matrix[SBLIMIT/4][32];

static	filter_matrix	m;

/*  ========================================================================================  */
/*      create_ana_filter                                                                     */
/*  ========================================================================================  */
/*
	Calculates the analysis filterbank coefficients and rounds to the 9th decimal place
	accuracy of the filterbank tables in the ISO document.
	The coefficients are stored in #new_filter#


	Originally was computed

		filter[i][j]  :=  cos (PI64 * ((2*i+1) * (16-j)))

	for i = 0..SBLIMIT-1 and j = 0..63 .


	But for j = 0..15 is 16-j = -(16-(32-j)) which implies

	(1)		filter[i][j]  =  filter[i][32-j] .

	(We know that filter[i][16] = cos(0) = 1 , but that is not so interesting.)

	Furthermore, for j = 33..48 we get

			   filter[i][96-j]
			=  cos (PI64 * (2*i+1) * (j-80))
			=  cos (PI * (2*i+1) + PI64 * (2*i+1) * (16-j))
	(2)		=  cos (PI * (2*i+1)) * cos (PI64 * (2*i+1) * (16-j))   // sin (PI * (2*i+1)) = 0
			=  -cos (PI64 * (2*i+1) * (16-j))
			=  -filter[i][j] ,

	especially is filter[i][48] = 0 .


	On the other hand there is

			   filter[31-i][j]
			=  cos (PI64 * (2*(31-i)+1) * (16-j))
			=  cos (PI64 * (64-(2*i+1)) * (16-j))
			=  cos (PI * (16-j) - PI64 * (2*i+1) * (16-j))
			=  cos (PI * (16-j)) * cos (PI64 * (2*i+1) * (16-j))   // sin (PI * (16-j)) = 0
			=  (-1)^^j * cos (PI64 * (2*i+1) * (16-j))
			=  (-1)^^j * filter[i][j] .


	There is a third somewhat more complication "symmetry":

			   filter[15-i][j]
			=  cos (PI64 * (2*(15-i)+1) * (16-j))
			=  cos (PI64 * (32-(2*i+1)) * (16-j))
			=  cos (PI/2 * (16-j) - PI64 * (2*i+1) * (16-j))
			=  cos (PI/2 * (16-j)) * cos (PI64 * (2*i+1) * (16-j)) + sin (PI/2 * (16-j)) * sin (PI64 * (2*i+1) * (16-j))
			=  cos (PI/2 * (16-j)) * cos (PI64 * (2*i+1) * (16-j)) + sin (PI/2 * (16-j)) * cos (PI64 * (2*i+1) * (-1)^^i * (16-(j+32)))

			=  (-1)^^(  j  /2    ) * filter[i][j   ]   for even j
			=  (-1)^^((j+1)/2 + i) * filter[i][j+32]   for odd  j  with  j <  32
			=  (-1)^^((j-1)/2 + i) * filter[i][j-32]   for odd  j  with  j >= 32


	For these reasons we create a filter of the eighth size only and set

		new_filter[i][j]  :=  filter[i][j]       for i = 0..SBLIMIT/4-1 and j =  0..16
		new_filter[i][j]  :=  filter[i][j+16]    for i = 0..SBLIMIT/4-1 and j = 17..31
*/
#define NewEnwXX
#ifdef NewEnw
#include <stdio.h>
#include <stdlib.h>
#endif

static	void create_ana_filter (double new_filter[SBLIMIT/4][32])
{
	register int			i, j;
	double					t;

	for (i=0; i<SBLIMIT/4; i++)
	{
		for (j=0; j<=16; j++)
		{
			t = 1e9 * cos (PI64 * (double) ((2*i+1) * (16-j)));
			if (t >= 0)
				modf (t + 0.5, &t);
			else
				modf (t - 0.5, &t);
			new_filter[i][j] = t * 1e-9;
		}
		for (j=17; j<32; j++)
		{
			t = 1e9 * cos (PI64 * (double) ((2*i+1) * j));   /* (16-(j+16)) */
			if (t >= 0)
				modf (t + 0.5, &t);
			else
				modf (t - 0.5, &t);
			new_filter[i][j] = t * 1e-9;
		}
	}

#ifdef NewEnw
FILE *f;
f = fopen("Macintosh HD:Desktop Folder:Downloads:enwindow.c","w");
if (f == NULL)
{
	printf("Da ist was schiefgegangen!");
}
else
{
	for (i=0; i<64; i++)
	{
		for (j=0; j<8; j++)
		{
			fprintf(f,"%20.17f,",enwindow[i+j*64]/SCALE);
		}
		fprintf(f,"\n");
	}
	fclose(f);
}
exit(0);
#endif

}

/*  ========================================================================================  */
/*      windowFilterSubband                                                                   */
/*  ========================================================================================  */
/*
	Calculates the analysis filter bank coefficients

	The windowed samples #z# is filtered by the digital filter matrix #m# to produce the
	subband samples #s#. This done by first selectively picking out values from the windowed
	samples, and then multiplying them by the filter matrix, producing 32 subband samples.
*/

void  windowFilterSubband
(
	const FLOAT				*buffer,
	int						buffer_idx,
	double					s[SBLIMIT]
)
{
	int						i, k;

	double					t, u, v, w;

	double					z[32];
	double					*p;
	double					*pEnw;

	pEnw = enwindow;

	for (k=0; k<=16; k++)
	{
		t =  buffer[(buffer_idx+511) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+447) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+383) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+319) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+255) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+191) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+127) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+ 63) & 2047] * *pEnw++;
		z[k] = t;

		buffer_idx--;
	}			

	for (k=15; k>=0; k--)
	{
		t =  buffer[(buffer_idx+511) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+447) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+383) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+319) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+255) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+191) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+127) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+ 63) & 2047] * *pEnw++;
		z[k] += t;

		buffer_idx--;
	}			

	for (k=17; k<32; k++)
	{
		t =  buffer[(buffer_idx+511) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+447) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+383) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+319) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+255) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+191) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+127) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+ 63) & 2047] * *pEnw++;
		z[k] = t;

		buffer_idx--;
	}			

	/* normally y[48] was computed like the other entries, */
	/* but this entry is not needed because m[i][48] = 0.  */
	buffer_idx--; /* But we have to increase the pointers. */
	pEnw += 8;

	for (k=31; k>16; k--)
	{
		t =  buffer[(buffer_idx+511) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+447) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+383) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+319) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+255) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+191) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+127) & 2047] * *pEnw++;
		t += buffer[(buffer_idx+ 63) & 2047] * *pEnw++;
		z[k] -= t;

		buffer_idx--;
	}			

	for (i=0; i<SBLIMIT/4; i++)
	{
		p = m[i];

		t = u = v = w = 0.0;
		for (k=0; k<16; k+=4)
		{
			t += p[k   ] * z[k  ];
			v += p[k+ 2] * z[k+2];
			u += p[k+ 1] * z[k+1] + p[k+ 3] * z[k+3];
			w -= p[k+17] * z[k+1] - p[k+19] * z[k+3];
		}
		for (   ; k<32; k+=4)
		{
			t += p[k   ] * z[k  ];
			v += p[k+ 2] * z[k+2];
			u += p[k+ 1] * z[k+1] + p[k+ 3] * z[k+3];
			w += p[k-15] * z[k+1] - p[k-13] * z[k+3];
		}

		s[          i] = (t + v) + u;
		s[SBLIMIT-1-i] = (t + v) - u;
		if (i & 1)
		{
			s[SBLIMIT/2-1-i] = (t - v) - w;
			s[SBLIMIT/2  +i] = (t - v) + w;
		}
		else
		{
			s[SBLIMIT/2-1-i] = (t - v) + w;
			s[SBLIMIT/2  +i] = (t - v) - w;
		}
	}	 
}





/*  ========================================================================================  */
/*      initWindowFilterSubband                                                               */
/*  ========================================================================================  */

void initWindowFilterSubband (void)
{
	int						i;

	for (i=0; i<512; i++)
		enwindow[i] /= SCALE;

	create_ana_filter (m);
}





