/* bfbtester -- Brute Force Binary Tester
 *
 * Copyright 2000	Mike Heffner <spock@techfour.net>
 * Use it and abuse it, but send patches and/or money to me. =)
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>

#include "datatypes.h"
#include "exec.h"
#include "main.h"

extern	THREAD_DATA	*q;

/* Maybe later ... currently broken
 *
#define USE_STDIN
#define USE_PIPES
*/

int
execute_program(char ** args, char ** env)
{
	EXEC_DATA	*new_exec;

	if( q->current_execs >= q->max_execs ){
		debugmsg(0, "Hit maximum execs\n");
		return (1);
	}

	if( args == NULL )
		return (1);
	new_exec=Malloc(sizeof(EXEC_DATA));

#ifdef USE_PIPES
	if( pipe(new_exec->sout) == -1 ){
		perror("pipe1");
		exec_destroy(new_exec);
		return (1);
	}
#endif

#ifdef USE_STDIN
	if( pipe(new_exec->sin) == -1 ){
		close(new_exec->sout[0]);
		close(new_exec->sout[1]);
		perror("pipe2");
		exec_destroy(new_exec);
		return (1);
	}
#endif

	if( args[0][0] != '/'){
		if( (new_exec->progpath = FindInPath(args[0])) == NULL ){
			debugmsg(0, "Can't find executable '%s' in path\n", args[0]);
			exec_destroy(new_exec);
			return (1);
		}
	}
	else
		new_exec->progpath = strdup(args[0]);

				 

	/* copy environment variables */
	exec_create_env(env, new_exec);
	exec_env_to_str(env, new_exec);

	new_exec->args = args;
	exec_args_to_str(new_exec);
	switch( new_exec->pid = fork() ){
	case -1:

		/* uh oh! close pipes */
#ifdef USE_PIPES
		close(new_exec->sout[0]);
		close(new_exec->sout[1]);
#endif
#ifdef USE_STDIN
		close(new_exec->sin[0]);
		close(new_exec->sin[1]);
#endif
		debugmsg(1, "Couldn't fork process\n");
		exec_destroy(new_exec);
		return (1);

	case 0:
		
		/* we're the kiddie */
	{
		pid_t	pgid = getpid();
		setpgid(pgid, pgid);
	}
	
#ifdef USE_STDIN
		close(new_exec->sin[1]);
		if( dup2(new_exec->sin[0], STDIN_FILENO) == -1 ){
			perror("dup1");
			_exit(2);
		}
#endif
#ifndef USE_PIPES

		close(0);
		close(1);
		close(2);

#endif
#ifdef USE_PIPES
		close(new_exec->sout[0]);
		if( dup2(new_exec->sout[1], STDOUT_FILENO) == -1 ){
			perror("dup2");
			_exit(2);
		}
		if( dup2(new_exec->sout[1], STDERR_FILENO) == -1 ){
			perror("dup3");
			_exit(2);
		}
#endif
		
		execve(new_exec->progpath, new_exec->args, new_exec->env);

		/* if we get here then the execve() didn't work */
		debugmsg(0, "execve-%s: %s", new_exec->progpath, strerror(errno));
		_exit(2);

	default:
		
		/* we're the big daddy */
		pthread_mutex_lock(q->mut);
		q->current_execs++;
		gettimeofday(&(new_exec->tm), NULL);
		exec_add_entry(new_exec);
		pthread_mutex_unlock(q->mut);
#ifdef USE_PIPES
	{
		int flags;
		flags = fcntl( new_exec->sout[0], F_GETFL, 0) | O_NONBLOCK;
		fcntl(new_exec->sout[0], F_SETFL, flags);
		close(new_exec->sout[1]);
	}
#endif
#ifdef USE_STDIN
	close(new_exec->sin[0]);
#endif
	}
	return (0);
}

static EXEC_DATA	*list_top=NULL; /* lowest pid */
static EXEC_DATA	*list_bot=NULL; /* highest pid */

void
exec_add_entry(EXEC_DATA	*new)
{
	EXEC_DATA	*temp=list_top;

	assert(new);
	assert(new->pid);

	if(list_top == NULL || list_bot == NULL){
		new->prev = NULL;
		new->next = NULL;
		list_top = new;
		list_bot = new;
	} else if(new->pid < list_top->pid){
		list_top->prev = new;
		new->next = list_top;
		new->prev = NULL;
		list_top = new;
	} else if(new->pid > list_bot->pid){
		list_bot->next = new;
		new->prev = list_bot;
		new->next = NULL;
		list_bot = new;
	} else {
		while( new->pid > temp->pid )
			temp = temp->next;
		if( new->pid == temp->pid ){
			debugmsg(0, "ERROR: pid collision %s = %s\n", 
					 new->progpath, temp->progpath);
			exit(1);
		}
		new->prev = temp->prev;
		new->next = temp;
		temp->prev->next = new;
		temp->prev = new;
	}
}

/* does _not_ free() node */
void
exec_remove_entry(EXEC_DATA		*old)
{
	assert(old);
	assert(old->pid);

	if(list_top == NULL || list_bot == NULL)
		return; /* empty list */
	if(list_top->pid == old->pid)
		list_top = old->next;
	if(list_bot->pid == old->pid)
		list_bot = old->prev;
	if(old->prev)
		old->prev->next = old->next;
	if(old->next)
		old->next->prev = old->prev;
}

/* polling function...reads pipes and kills off terminated
 * processes
 */

void
exec_poll(void)
{
	EXEC_DATA * temp=NULL, *dead;
#ifdef USE_PIPES
	fd_set		rdfs;
	int			ret;
	struct timeval	tv = { 0, 0 };
#endif
	int			i = 0;
	
	pthread_mutex_lock(q->mut);
	for( temp = list_top; temp ; i++){
#ifdef USE_PIPES
		FD_ZERO(&rdfs);
		FD_SET(temp->sout[0], &rdfs);
		ret = select(temp->sout[0]+1, &rdfs, NULL, NULL, &tv);
		printf("pid: %d ", temp->pid);
		if(ret > 0){
			; /* we should read from sout */
		}
#endif
		if( exec_IsDone(temp)){
			exec_process_close(temp);
			dead = temp;
			temp = temp->next;
			exec_remove_entry(dead);
			exec_destroy(dead);
		}
		else
			temp = temp->next;
	}
	pthread_mutex_unlock(q->mut);
}

char *
FindInPath(char *prog)
{
	char * path = getenv("PATH");
	char * str=NULL, *buf;
	char filename[2*FILENAME_MAX];

	if(path){
		buf = strdup(path);
		str = strtok(buf, ":");
		while(str){
			snprintf(filename, sizeof(filename), "%s/%s", str, prog);
			if(! access(filename, F_OK | X_OK) ){
				free(buf);
				return strdup(filename);
			}
			str = strtok(NULL, ":");
		}
		/* we can't find it if we get here */
		free(buf);
	}
	return NULL;
}

int
exec_IsDone(EXEC_DATA *temp)
{
	struct timeval	tm;
	if( waitpid(temp->pid, &(temp->status), WNOHANG) > 0 )
		return TRUE;
	else{
		gettimeofday(&tm, NULL);
		if( tm.tv_sec - temp->tm.tv_sec > EXEC_STALE )
			debugmsg(1, "Stale: %ld secs '%s'\n", tm.tv_sec - temp->tm.tv_sec, 
					 temp->progpath);
		if( tm.tv_sec - temp->tm.tv_sec > EXEC_TIMEOUT ){
			exec_kill(temp);
			if( waitpid(temp->pid, &(temp->status), WNOHANG) > 0 )
				return TRUE;
		}
		return FALSE;
	}
}

static void
exec_record_failure(int		pid,
					char	*progpath,
					char	*args,
					char	*envs,
					char	*sout_buf,
					char	*sin_buf,
					int		signal,
					int		didCore)
{

	printf("\n" \
		   "<>Crash<>\n" \
		   "** Binary:     %s\n" \
		   "** Signal:     %d (%s)\n" \
		   "** Core?:      %s\n" \
		   "** Args:       %s\n" \
		   "** Longenvs:   %s\n\n",
		   progpath, signal, (char *)strsignal(signal),
		   (didCore ? "Yes" : "No"), args, envs);

}
void
exec_process_close(EXEC_DATA *temp)
{
	q->current_execs--;
	if( WIFEXITED(temp->status) )
		; /* darn, nothing bad happened to it */
	else if( WIFSIGNALED(temp->status) ){
		/* we ignore SIGKILL (we use it to remove timeouts) */
		if( WTERMSIG(temp->status) == SIGKILL || WTERMSIG(temp->status) == 13){
			return;
		}
		/* we need to log, record this app */
		exec_record_failure(temp->pid,
						   temp->progpath,
						   temp->cargs,
						   temp->cenv,
						   temp->sout_buf,
						   temp->sin_buf,
						   (int)WTERMSIG(temp->status),
						   (int)WCOREDUMP(temp->status) );
	} else
		debugmsg(1, "%s ended in unknown condition, status=%d\n", 
				 temp->progpath, temp->status);
}

void
exec_kill(EXEC_DATA		*temp)
{
	kill(-(temp->pid), SIGKILL);
}

#define		LONG_ARG_FMT			"[%5.5d]"
#define		LONG_ARG_LEN			(sizeof("[12345]")-1)
#define		MAX_VALID_ARG_LEN		100

void
exec_args_to_str(EXEC_DATA *temp)
{
	char	*str, **temp_args;
	int		j;

	if(*(temp->args+1) == NULL){
		temp->cargs = NULL;
		return;
	}
	for( j=0, temp_args=(temp->args+1); *temp_args; temp_args++ )
		j += (strlen(*temp_args) > MAX_VALID_ARG_LEN ?
			  LONG_ARG_LEN : strlen(*temp_args)) + 1;
	str = Malloc(j);
	for( temp_args=(temp->args+1); *temp_args; temp_args++ ){
		if( strlen(*temp_args) > MAX_VALID_ARG_LEN ){
			char long_str[LONG_ARG_LEN+1];
			sprintf(long_str, LONG_ARG_FMT, strlen(*temp_args));
			strcat(str, long_str);
		}
		else
			strcat(str, *temp_args);
		if( *(temp_args+1) )
			strcat(str, " ");
	}
	temp->cargs = str;
}

void
exec_destroy(EXEC_DATA *dead)
{
#ifdef USE_PIPES
	close(dead->sout[0]);
	close(dead->sout[1]);
#endif
#ifdef USE_STDIN
	close(dead->sin[0]); 
	close(dead->sin[1]); 
#endif
	if( dead->sout_buf ) free(dead->sout_buf);
	if( dead->sin_buf ) free(dead->sin_buf);
	if( dead->cargs ) free(dead->cargs);
	if( dead->env && dead->using_environ == FALSE) free(dead->env);
	if( dead->cenv ) free(dead->cenv);
	if( dead->progpath ) free(dead->progpath);
	free(dead);
}

extern	char ** environ;

void
exec_create_env(char **env, EXEC_DATA *to)
{
	int		i, j;
	char	**temp, **temp2;
	if( env == NULL ){ /* no need to copy */
		to->using_environ = TRUE;
		to->env = environ;
		return;
	}
	to->using_environ = FALSE;
	/* NOTE: to save CPU cycles we alloc the sum of all entries in env and environ
	 * even if there might be duplicates in env and environ
	 * this will create a temporary N*sizeof(char *) bloat, but N will probably 
	 * only be about 1-2
	 */
	for(i=0, temp2=env; *temp2; temp2++, i++);
	for(temp2=environ; *temp2; temp2++, i++);
	temp = Malloc((i+1) * sizeof(*temp));
	for(j=0, temp2=env; *temp2; j++, temp2++)
		temp[j] = *temp2;
	for(temp2=environ; *temp2; temp2++){
		if( exec_env_not_used(env, *temp2) ){
			temp[j] = *temp2;
			j++;
		}
	}


	to->env = temp;
}

/* tests if the env. variable _str_ is in _env_ 
 * Returns: TRUE if _str_ is not in _env_, else FALSE
 */
int
exec_env_not_used(char **env, char *str)
{

	int len = strcspn(str, "=");
	char	**temp;
	for(temp=env;*temp;temp++)
		if( strncmp(*temp, str, len) == 0 )
			return FALSE;

	return TRUE;
}

#define		LONG_ENV_FMT			"[%5.5d]"
#define		LONG_ENV_LEN			(sizeof("[12345]")-1)
#define		MAX_VALID_ENV_LEN		100

void
exec_env_to_str(char **env, EXEC_DATA *to)
{
	int		j;
	char 	**temp;
	char	long_env[LONG_ENV_LEN+1];

	if( env == NULL ){
		to->cenv = NULL;
		return;
	}
	for(j=0, temp=env; *temp; temp++)
		j += ( (strlen(*temp) > MAX_VALID_ENV_LEN) ? (strcspn(*temp, "=") + 1 + LONG_ENV_LEN) :
		(strlen(*temp)) ) + 1;
	to->cenv = Malloc(j);
	for(temp=env; *temp; temp++){
		if( strlen(*temp) > MAX_VALID_ENV_LEN ){
			int i = strcspn(*temp, "=");
			strncat(to->cenv, *temp, i);
			strcat(to->cenv, "=");
			snprintf(long_env, LONG_ENV_LEN+1, LONG_ENV_FMT, strlen(*temp+i+1));
			strcat(to->cenv, long_env);
		}
		else
			strcat(to->cenv, *temp);
		if( *(temp+1) )
			strcat(to->cenv, " ");
	}
}
