/*
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include <glib.h>
#include <glib/gprintf.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

#include "def.h"
#include "alloc.h"


void options_names(GKeyFile *keys, struct _prog *prog_data)
{
	gsize length;
	
	if(prog_data->player_names)
		g_strfreev(prog_data->player_names);
	
	prog_data->player_names = g_key_file_get_string_list(keys, 
		"main", "player_names", &length, NULL);
	
	if(prog_data->player_names && prog_data->output)
	{
		int i=0;
		while(prog_data->player_names[i] != NULL)
			fprintf(prog_data->output, 
				_("Player name: %s\n"), 
				prog_data->player_names[i++]);
	}
	
}

void options_bool(GKeyFile *keys, struct _prog *prog_data)
{
	prog_data->animation = g_key_file_get_boolean(keys, 
		"main", "animations", NULL);
	
	prog_data->no_drag = g_key_file_get_boolean(keys, 
		"main", "click_drop", NULL);
	
	prog_data->withdeclaration = g_key_file_get_boolean(keys, 
		"main", "declarations", NULL);
	
}

void options_background(GKeyFile *keys, struct _prog *prog_data)
{
}

void options_cards(GKeyFile *keys, struct _prog *prog_data)
{
}
/*
void options_size(GKeyFile *keys, struct _prog *prog_data)
{
	prog_data->space = (float) g_key_file_get_integer(keys, 
		"main", "space_multiplier", NULL);
	
	if(prog_data->output)
	{
		fprintf(prog_data->output,
			_("space size: %f\n"), prog_data->space);
	}
}
*/
void load_all_keys(GKeyFile *keys, struct _prog *prog_data)
{
	options_names(keys, prog_data);
	
	options_bool(keys, prog_data);
	
	options_background(keys, prog_data);
	
	options_cards(keys, prog_data);
	
}

gboolean check_allocated_names(struct _prog *prog_data)
{
	if(!prog_data->player_names)
		prog_data->player_names = alloc_default_names();
			
	return (prog_data->player_names != NULL);
}

gboolean properties_create_default(const gchar *filename, 
	struct _prog *prog_data, const gchar *home_dir)
{
	gboolean done = FALSE;
	gchar *key_file_content = NULL;
	
	if(properties_create_directory(prog_data, home_dir))
	{
		GKeyFile *keys = g_key_file_new();
		gsize length;
	
		if(keys)
		{
			g_key_file_set_boolean(keys, "main",
				"animations", prog_data->animation);
			
			g_key_file_set_boolean(keys, "main",
				"click_drop", prog_data->no_drag);
			
			g_key_file_set_boolean(keys, "main",
				"declarations", prog_data->withdeclaration);
			
			if( check_allocated_names(prog_data) )
				g_key_file_set_string_list(keys, "main",
					"player_names", prog_data->player_names, 4);
			
			key_file_content = g_key_file_to_data(keys, 
				&length, NULL);
				
			g_key_file_free(keys);
		}
	}

	if(key_file_content)
	{
		done = g_file_set_contents(filename,
			key_file_content, -1, NULL);
		
		g_free(key_file_content);
	}
	
	return done;
}

gboolean properties_create_directory(struct _prog *prog_data, 
	const gchar *home_dir)
{
	gboolean direxist = FALSE;
	gboolean done = FALSE;
	 
	gchar *belooted_dir = g_strconcat(home_dir, 
		"/.belooted", NULL);
	
	if(belooted_dir)
	{
		direxist = g_file_test(belooted_dir, G_FILE_TEST_EXISTS);
		
		if(!direxist && g_mkdir(belooted_dir, 0755) != 0)
		{
			if(prog_data->output)
				fprintf(prog_data->output, 
					_("Unable to create belooted directory:\n%s\n"),
					belooted_dir);
		}
		else
			done = TRUE;
		
		g_free(belooted_dir);
	}
	
	return done;
}


gboolean load_properties(struct _prog *prog_data)
{
	gboolean done = FALSE;
	
	gchar *filename;
	
	const gchar *home_dir = g_getenv("HOME");
	
	GKeyFile *keys = g_key_file_new();
	
	if(keys)
	{
		filename = g_strconcat(home_dir, "/.belooted/conf", NULL);
		
		if(filename)
		{
			
			if(g_file_test(filename, G_FILE_TEST_EXISTS) )
			{
				GError *error = NULL;
				
				if(prog_data->output)
					fprintf(prog_data->output,
						_("Loading config file\n") );

				done = g_key_file_load_from_file(keys,
					filename, G_KEY_FILE_NONE, &error);
				
				if(error)
				{
					if(prog_data->output)
						fprintf(prog_data->output, "%s\n", 
							error->message);
					
					g_clear_error(&error);
				}
			}
			else
			{
				if(prog_data->output)
					fprintf(prog_data->output,
						_("Creating default configuration file:\n") );
		

				if(properties_create_default(filename, prog_data, 
					home_dir) )
				{
					if(prog_data->output)
						fprintf(prog_data->output,
							_("Done\n") );
				}
				else if(prog_data->output)
						fprintf(prog_data->output,
							_("Failed\n") );
		
			}
				
			g_free(filename);
		}
		
		if(done)
			load_all_keys(keys, prog_data);
		
		g_key_file_free(keys);
	}
	
	return done;
}
