/*
	Audio File Library
	Copyright (C) 1998-2000, Michael Pruett <michael@68k.org>

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the 
	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
	Boston, MA  02111-1307  USA.
*/

/*
	aiff.c

	This file contains routines for parsing AIFF and AIFF-C sound
	files.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>

#include "extended.h"
#include "audiofile.h"
#include "util.h"
#include "afinternal.h"
#include "byteorder.h"
#include "aiff.h"

static status ParseFVER (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);
static status ParseAESD (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);
static status ParseMiscellaneous (AFfilehandle file, AF_VirtualFile *fh,
	u_int32_t type, size_t size);
static status ParseINST (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);
static status ParseMARK (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);
static status ParseCOMM (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);
static status ParseSSND (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size);

_InstParamInfo _af_aiff_inst_params[_AF_AIFF_NUM_INSTPARAMS] =
{
	{ AF_INST_MIDI_BASENOTE, AU_PVTYPE_LONG, "MIDI base note", {60} },
	{ AF_INST_NUMCENTS_DETUNE, AU_PVTYPE_LONG, "Detune in cents", {0} },
	{ AF_INST_MIDI_LOVELOCITY, AU_PVTYPE_LONG, "Low velocity", {1} },
	{ AF_INST_MIDI_HIVELOCITY, AU_PVTYPE_LONG, "High velocity", {127} },
	{ AF_INST_MIDI_LONOTE, AU_PVTYPE_LONG, "Low note", {0} },
	{ AF_INST_MIDI_HINOTE, AU_PVTYPE_LONG, "High note", {127} },
	{ AF_INST_NUMDBS_GAIN, AU_PVTYPE_LONG, "Gain in dB", {0} },
	{ AF_INST_SUSLOOPID, AU_PVTYPE_LONG, "Sustain loop id", {0} },
	{ AF_INST_RELLOOPID, AU_PVTYPE_LONG, "Release loop id", {0} }
};

int _af_aiffc_compression_types[_AF_AIFF_NUM_COMPTYPES] =
{
	AF_COMPRESSION_G711_ULAW,
	AF_COMPRESSION_G711_ALAW
};

/*
	FVER chunks are only present in AIFF-C files.
*/
static status ParseFVER (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type, size_t size)
{
	u_int32_t	timestamp;

	assert(!memcmp(&type, "FVER", 4));

	af_fread(&timestamp, sizeof (u_int32_t), 1, fh);
	timestamp = BENDIAN_TO_HOST_INT32(timestamp);
	/* timestamp holds the number of seconds since January 1, 1904. */

	return AF_SUCCEED;
}

/*
	Parse AES recording data.
*/
static status ParseAESD (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type, size_t size)
{
	unsigned char	aesChannelStatusData[24];

	assert(!memcmp(&type, "AESD", 4));
	assert(size == 24);

	file->aesDataPresent = 1;
	af_fread(aesChannelStatusData, 1, 24, fh);
	memcpy(file->aesData, aesChannelStatusData, 24);

	return AF_SUCCEED;
}

/*
	Parse miscellaneous data chunks such as name, author, copyright,
	and annotation chunks.
*/
static status ParseMiscellaneous (AFfilehandle file, AF_VirtualFile *fh,
	u_int32_t type, size_t size)
{
	int	misctype = 0;

	assert(!memcmp(&type, "NAME", 4) || !memcmp(&type, "AUTH", 4) ||
		!memcmp(&type, "(c) ", 4) || !memcmp(&type, "ANNO", 4) ||
		!memcmp(&type, "APPL", 4) || !memcmp(&type, "MIDI", 4));
	assert(size >= 0);

	if (file->miscellaneousCount == 0)
	{
		assert(file->miscellaneous == NULL);
		file->miscellaneousCount++;
		file->miscellaneous = _af_malloc(sizeof (struct _Miscellaneous));
	}
	else
	{
		file->miscellaneousCount++;
		file->miscellaneous = _af_realloc(file->miscellaneous,
			file->miscellaneousCount * sizeof (struct _Miscellaneous));
	}

	if (!memcmp(&type, "NAME", 4))
		misctype = AF_MISC_NAME;
	else if (!memcmp(&type, "AUTH", 4))
		misctype = AF_MISC_AUTH;
	else if (!memcmp(&type, "(c) ", 4))
		misctype = AF_MISC_COPY;
	else if (!memcmp(&type, "ANNO", 4))
		misctype = AF_MISC_ANNO;
	else if (!memcmp(&type, "APPL", 4))
		misctype = AF_MISC_APPL;
	else if (!memcmp(&type, "MIDI", 4))
		misctype = AF_MISC_MIDI;

	file->miscellaneous[file->miscellaneousCount - 1].id = file->miscellaneousCount;
	file->miscellaneous[file->miscellaneousCount - 1].type = misctype;
	file->miscellaneous[file->miscellaneousCount - 1].size = size;
	/*
		ftell should probably be replaced by ftell64 on systems such
		as Irix, but until a 64-bit audio file format comes out, I'm not
		going to worry about it.
	*/
	file->miscellaneous[file->miscellaneousCount - 1].offset = af_ftell(fh);
	file->miscellaneous[file->miscellaneousCount - 1].position = 0;

	return AF_SUCCEED;
}

/*
	Parse instrument chunks, which contain information about using
	sound data as a sampled instrument.
*/
static status ParseINST (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size)
{
	u_int8_t	baseNote, detune, lowNote, highNote, lowVelocity, highVelocity;
	u_int16_t	gain;

	u_int16_t	sustainLoopPlayMode, sustainLoopBegin, sustainLoopEnd;
	u_int16_t	releaseLoopPlayMode, releaseLoopBegin, releaseLoopEnd;

	assert(!memcmp(&type, "INST", 4));

	file->instrumentCount = 1;
	file->instruments =
		(struct _Instrument *) _af_calloc(1, sizeof (struct _Instrument));

	af_fread(&baseNote, sizeof (u_int8_t), 1, fh);
	af_fread(&detune, sizeof (u_int8_t), 1, fh);
	af_fread(&lowNote, sizeof (u_int8_t), 1, fh);
	af_fread(&highNote, sizeof (u_int8_t), 1, fh);
	af_fread(&lowVelocity, sizeof (u_int8_t), 1, fh);
	af_fread(&highVelocity, sizeof (u_int8_t), 1, fh);
	af_fread(&gain, sizeof (u_int16_t), 1, fh);
	gain = BENDIAN_TO_HOST_INT16(gain);

	file->instruments[0].id = AF_DEFAULT_INST;
	file->instruments[0].midiBaseNote = baseNote;
	file->instruments[0].detune = detune;
	file->instruments[0].midiLowNote = lowNote;
	file->instruments[0].midiHighNote = highNote;
	file->instruments[0].midiLowVelocity = lowVelocity;
	file->instruments[0].midiHighVelocity = highVelocity;
	file->instruments[0].gain = gain;
	file->instruments[0].sustainLoopID = 1;
	file->instruments[0].releaseLoopID = 2;

#ifdef DEBUG
	printf(" baseNote/detune/lowNote/highNote/lowVelocity/highVelocity/gain:\n"
		" %d %d %d %d %d %d %d\n",
		baseNote, detune, lowNote, highNote, lowVelocity, highVelocity, gain);
#endif

	af_fread(&sustainLoopPlayMode, sizeof (u_int16_t), 1, fh);
	sustainLoopPlayMode = BENDIAN_TO_HOST_INT16(sustainLoopPlayMode);
	af_fread(&sustainLoopBegin, sizeof (u_int16_t), 1, fh);
	sustainLoopBegin = BENDIAN_TO_HOST_INT16(sustainLoopBegin);
	af_fread(&sustainLoopEnd, sizeof (u_int16_t), 1, fh);
	sustainLoopEnd = BENDIAN_TO_HOST_INT16(sustainLoopEnd);

	af_fread(&releaseLoopPlayMode, sizeof (u_int16_t), 1, fh);
	releaseLoopPlayMode = BENDIAN_TO_HOST_INT16(releaseLoopPlayMode);
	af_fread(&releaseLoopBegin, sizeof (u_int16_t), 1, fh);
	releaseLoopBegin = BENDIAN_TO_HOST_INT16(releaseLoopBegin);
	af_fread(&releaseLoopEnd, sizeof (u_int16_t), 1, fh);
	releaseLoopEnd = BENDIAN_TO_HOST_INT16(releaseLoopEnd);

#ifdef DEBUG
	printf("sustain loop: %d %d %d\n", sustainLoopPlayMode,
		sustainLoopBegin, sustainLoopEnd);

	printf("release loop: %d %d %d\n", releaseLoopPlayMode,
		releaseLoopBegin, releaseLoopEnd);
#endif

	file->loops = _af_calloc(2, sizeof (struct _Loop));
	file->loopCount = 2;

	file->loops[0].id = 1;
	file->loops[0].playMode = sustainLoopPlayMode;
	file->loops[0].beginLoop = sustainLoopBegin;
	file->loops[0].endLoop = sustainLoopEnd;

	file->loops[1].id = 2;
	file->loops[1].playMode = releaseLoopPlayMode;
	file->loops[1].beginLoop = releaseLoopBegin;
	file->loops[1].endLoop = releaseLoopEnd;

	return AF_SUCCEED;
}

/*
	Parse marker chunks, which contain the positions and names of loop markers.
*/
static status ParseMARK (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size)
{
	int			i;
	u_int16_t	numMarkers;

	assert(!memcmp(&type, "MARK", 4));

	af_fread(&numMarkers, sizeof (u_int16_t), 1, fh);
	numMarkers = BENDIAN_TO_HOST_INT16(numMarkers);

	file->markerCount = numMarkers;
	file->markers = _af_calloc(numMarkers, sizeof (struct _Marker));

	for (i=0; i<numMarkers; i++)
	{
		u_int16_t	markerID = 0;
		u_int32_t	markerPosition = 0;
		unsigned char	sizeByte = 0;
		char		*markerName = NULL;

		af_fread(&markerID, sizeof (u_int16_t), 1, fh);
		markerID = BENDIAN_TO_HOST_INT16(markerID);
		af_fread(&markerPosition, sizeof (u_int32_t), 1, fh);
		markerPosition = BENDIAN_TO_HOST_INT32(markerPosition);
		af_fread(&sizeByte, sizeof (unsigned char), 1, fh);
		markerName = _af_malloc(sizeByte + 1);
		af_fread(markerName, sizeof (unsigned char), sizeByte, fh);

		markerName[sizeByte] = '\0';

#ifdef DEBUG
		printf("marker id: %d, position: %d, name: %s\n", markerID,
			markerPosition, markerName);

		printf("size byte: %d\n", sizeByte);
#endif

		if ((sizeByte % 2) == 0)
			af_fseek(fh, 1, SEEK_CUR);

		file->markers[i].id = markerID;
		file->markers[i].position = markerPosition;
		file->markers[i].name = markerName;
	}

	return AF_SUCCEED;
}

/*
	Parse common data chunks, which contain information regarding the
	sampling rate, the number of sample frames, and the number of
	sound channels.
*/
static status ParseCOMM (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size)
{
	u_int16_t		numChannels;
	u_int32_t		numSampleFrames;
	u_int16_t		sampleSize;
	unsigned char	sampleRate[10];

	assert(!memcmp(&type, "COMM", 4));

	af_fread(&numChannels, sizeof (u_int16_t), 1, fh);
	file->channelCount = BENDIAN_TO_HOST_INT16(numChannels);

	af_fread(&numSampleFrames, sizeof (u_int32_t), 1, fh);
	file->frameCount = BENDIAN_TO_HOST_INT32(numSampleFrames);

	af_fread(&sampleSize, sizeof (u_int16_t), 1, fh);
	file->sampleWidth = BENDIAN_TO_HOST_INT16(sampleSize);

	af_fread(sampleRate, 10, 1, fh);
	file->sampleRate = ConvertFromIeeeExtended(sampleRate);

	return AF_SUCCEED;
}

/*
	Parse the stored sound chunk, which usually contains little more
	than the sound data.
*/
static status ParseSSND (AFfilehandle file, AF_VirtualFile *fh, u_int32_t type,
	size_t size)
{
	u_int32_t	offset, blockSize;

	assert(!memcmp(&type, "SSND", 4));

	af_fread(&offset, sizeof (u_int32_t), 1, fh);
	offset = BENDIAN_TO_HOST_INT32(offset);
	af_fread(&blockSize, sizeof (u_int32_t), 1, fh);
	blockSize = BENDIAN_TO_HOST_INT32(blockSize);

	/*
		This seems like a reasonable way to calculate the number of
		bytes in an SSND chunk.
	*/
	file->trackBytes = size - 8 - offset;

#ifdef DEBUG
	printf("offset: %d\n", offset);
	printf("block size: %d\n", blockSize);
#endif

	file->dataStart = af_ftell(fh) + offset;

#ifdef DEBUG
	printf("data start: %d\n", file->dataStart);
#endif

	/* Sound data follows. */

	return AF_SUCCEED;
}

status _af_aiff_read_init (AFfilesetup setup, AFfilehandle file)
{
	u_int32_t	type, size, formtype;
	size_t		index = 0;
	bool		hasCOMM, hasFVER, hasSSND, hasMARK, hasINST;
	bool		hasAESD, hasNAME, hasAUTH, hasCOPY;

	hasCOMM = AF_FALSE;
	hasFVER = AF_FALSE;
	hasSSND = AF_FALSE;
	hasMARK = AF_FALSE;
	hasINST = AF_FALSE;
	hasAESD = AF_FALSE;
	hasNAME = AF_FALSE;
	hasAUTH = AF_FALSE;
	hasCOPY = AF_FALSE;

	assert(file != NULL);
	assert(file->fh != NULL);

	af_fseek(file->fh, 0, SEEK_SET);

	af_fread(&type, 4, 1, file->fh);
	af_fread(&size, 4, 1, file->fh);
	size = BENDIAN_TO_HOST_INT32(size);
	af_fread(&formtype, 4, 1, file->fh);

	assert(!memcmp(&type, "FORM", 4));
	assert(!memcmp(&formtype, "AIFF", 4) || !memcmp(&formtype, "AIFC", 4));

	/*
		I'm not sure if this is the proper error to throw here. It
		shouldn't ever be true because of the way file types are
		distinguished in audiofile.c, but it might be good to have this
		code anyway.
	*/

	if (memcmp(&type, "FORM", 4) ||
		(memcmp(&formtype, "AIFF", 4) && memcmp(&formtype, "AIFC", 4)))
		_af_error(AF_BAD_AIFF_HEADER, "bad AIFF header");
	
#ifdef DEBUG
	printf("size: %d\n", size);
#endif

	file->sampleFormat = AF_SAMPFMT_TWOSCOMP;
	file->byteOrder = AF_BYTEORDER_BIGENDIAN;

	/* Include the offset of the form type. */
	index += 4;

	while (index < size)
	{
		u_int32_t	chunkid = 0, chunksize = 0;

#ifdef DEBUG
		printf("index: %d\n", index);
#endif
		af_fread(&chunkid, 4, 1, file->fh);
/*		chunkid = BENDIAN_TO_HOST_INT32(chunkid); */
		af_fread(&chunksize, 4, 1, file->fh);
		chunksize = BENDIAN_TO_HOST_INT32(chunksize);

#ifdef DEBUG
		_af_printid(chunkid);
		printf(" size: %d\n", chunksize);
#endif

		if (!memcmp("COMM", &chunkid, 4))
		{
			hasCOMM = AF_TRUE;
			ParseCOMM(file, file->fh, chunkid, chunksize);
		}
		else if (!memcmp("FVER", &chunkid, 4))
		{
			hasFVER = AF_TRUE;
			ParseFVER(file, file->fh, chunkid, chunksize);
		}
		else if (!memcmp("INST", &chunkid, 4))
		{
			hasINST = AF_TRUE;
			ParseINST(file, file->fh, chunkid, chunksize);
		}
		else if (!memcmp("MARK", &chunkid, 4))
		{
			hasMARK = AF_TRUE;
			ParseMARK(file, file->fh, chunkid, chunksize);
		}
		else if (!memcmp("AESD", &chunkid, 4))
		{
			hasAESD = AF_TRUE;
			ParseAESD(file, file->fh, chunkid, chunksize);
		}
		else if (!memcmp("NAME", &chunkid, 4) ||
			!memcmp("AUTH", &chunkid, 4) ||
			!memcmp("(c) ", &chunkid, 4) ||
			!memcmp("ANNO", &chunkid, 4) ||
			!memcmp("APPL", &chunkid, 4) ||
			!memcmp("MIDI", &chunkid, 4))
		{
			ParseMiscellaneous(file, file->fh, chunkid, chunksize);
		}
		/*
			The sound data chunk is required if there are more than
			zero sample frames.
		*/
		else if (!memcmp("SSND", &chunkid, 4))
		{
			hasSSND = AF_TRUE;
			ParseSSND(file, file->fh, chunkid, chunksize);
		}

		index += chunksize + 8;

		/* all chunks must be aligned on an even number of bytes */
		if ((index % 2) != 0)
			index++;

		af_fseek(file->fh, index + 8, SEEK_SET);
	}

	if (!hasCOMM)
	{
		_af_error(AF_BAD_AIFF_COMM, "bad AIFF COMM chunk");
	}

	/* A zero return value indicates successful parsing. */
	return 0;
}

bool _af_aiff_recognize (AFvirtualfile *fh)
{
	u_int8_t	buffer[8];

	af_fseek(fh, 0, SEEK_SET);

	if (af_fread(buffer, 1, 8, fh) != 8 || memcmp(buffer, "FORM", 4) != 0)
		return AF_FALSE;
	if (af_fread(buffer, 1, 4, fh) != 4 || memcmp(buffer, "AIFF", 4) != 0)
		return AF_FALSE;

	return AF_TRUE;
}

bool _af_aifc_recognize (AFvirtualfile *fh)
{
	u_int8_t	buffer[8];

	af_fseek(fh, 0, SEEK_SET);

	if (af_fread(buffer, 1, 8, fh) != 8 || memcmp(buffer, "FORM", 4) != 0)
		return AF_FALSE;
	if (af_fread(buffer, 1, 4, fh) != 4 || memcmp(buffer, "AIFC", 4) != 0)
		return AF_FALSE;

	return AF_TRUE;
}

AFfilesetup _af_aiff_complete_setup (AFfilesetup setup)
{
	return AF_NULL_FILESETUP;
}

bool _af_aiff_instparam_valid (AFfilehandle filehandle, AUpvlist list, int i)
{
	int	param, type, lval;

	AUpvgetparam(list, i, &param);
	AUpvgetvaltype(list, i, &type);
	if (type != AU_PVTYPE_LONG)
		return AF_FALSE;

	AUpvgetval(list, i, &lval);

	switch (param)
	{
		case AF_INST_MIDI_BASENOTE:
			return ((lval >= 0) && (lval <= 127));

		case AF_INST_NUMCENTS_DETUNE:
			return ((lval >= -50) && (lval <= 50));

		case AF_INST_MIDI_LOVELOCITY:
			return ((lval >= 1) && (lval <= 127));

		case AF_INST_MIDI_HIVELOCITY:
			return ((lval >= 1) && (lval <= 127));

		case AF_INST_MIDI_LONOTE:
			return ((lval >= 0) && (lval <= 127));

		case AF_INST_MIDI_HINOTE:
			return ((lval >= 0) && (lval <= 127));

		case AF_INST_NUMDBS_GAIN:
		case AF_INST_SUSLOOPID:
		case AF_INST_RELLOOPID:
			return AF_TRUE;

		default:
			return AF_FALSE;
			break;
	}

	return AF_TRUE;
}

int _af_aifc_get_version (AFfilehandle file)
{
	return AIFC_VERSION_1;
}
