/* Stolen almost verbatim from 
 * "C Interfaces and Implementations"
 * David R. Hanson, Addison Wesley, 1996
 */

#include "table.h"
#include <assert.h>

int primes[] = { 509, 509, 1021, 2053, 4093, 8191, 16381, 
			32771, 65521, INT_MAX };

struct table *t_new(hint, compare, hash)
int hint;
int compare(const void *x, const void *y);
unsigned hash(const void *key);
{
  struct table *table;
  int i;
  
  assert(hint >= 0);
  
  /* find adequate table size */
  for(i = 1; primes[i] < hint; i++);
  i--;

  /* allocate the table structure */
  table = (struct table *) malloc(sizeof(struct table));
  if(!table) 
    return(NULL);

  /* initialize the table */
  table->size    = primes[i];
  table->compare = compare ? compare : tableCompareStrings;
  table->hash    = hash    ? hash    : tableHashStrings;
  
  /* allocate the hash table */
  table->buckets = (struct bucket **) malloc(primes[i] *
					      sizeof(struct bucket));
  if(!table->buckets)
    return(NULL);
  
  /* initialize the hash table */
  for(i = 0; i < table->size; i++) 
    table->buckets[i] = NULL;
  
  table->length = 0;
  table->timestamp = 0;
  
  return(table);
}

void *t_get(struct table *table, const void *key) {
  unsigned i;
  struct bucket *b;
  
  assert(table);
  assert(key);
  
  /* hash */
  i = table->hash(key) % table->size;

  /* check the bucket, check for collisions */
  for(b = table->buckets[i]; b; b = b->next)
    if(!(table->compare(key, b->key))) 
      break;
  
  return(b ? b->value : NULL);
}

void *t_put(struct table *table, const void *key, void *value) {
  unsigned i;
  struct bucket *b;
  void *prev;
  
  assert(table);
  assert(key);
  
  /* hash */
  i = table->hash(key) % table->size;
  
  /* find matching key */
  for(b = table->buckets[i]; b; b = b->next) 
    if(!(table->compare(key, b->key)))
      break;
    
  /* no such entry exists in the table */
  if(!b) {
    b = (struct bucket *) malloc(sizeof(struct bucket));
    assert(b);
    
    /* initialize new entry */
    b->key  = key;
    b->next = table->buckets[i];
    table->buckets[i] = b;
    table->length++;
    prev = NULL;
  
  } else
    prev = b->value;
  
  b->value = value;
  table->timestamp++;
  
  /* if we updated a key, return the old value so we know */
  return(prev);
}

/* e-z */
int t_length(struct table *table) {
  assert(table);
  
  return(table->length);
}

void t_map(table, apply, closure) 
struct table *table;
void apply(const void *key, void **value, void *closure);
void *closure;
{
  int i;

  /* to verify that the table hasn't been modified */
  unsigned stamp;

  struct bucket *b;
  
  assert(table);
  assert(apply);
  
  stamp = table->timestamp;
  
  for(i = 0; i < table->size; i++) {
    for(b = table->buckets[i]; b; b = b->next) {
      apply(b->key, &b->value, closure);

      /* apply function cannot modify the table */
      assert(table->timestamp == stamp);
    }
  }

  return;
}

void *t_remove(struct table *table, const void *key) {
  unsigned i;
  struct bucket **bb;
  
  assert(table);
  assert(key);
  
  table->timestamp++;
  
  i = table->hash(key) % table->size;
  
  for(bb = &table->buckets[i]; *bb; bb = &(*bb)->next) {
    if(!(table->compare(key, (*bb)->key))) {
     struct bucket *b = *bb;
     void *value = b->value;
     *bb = b->next;
     free(b);
     table->length--;
     return(value);
    }
  }
  
  return(NULL);
}

void **t_2array(struct table *table, void *end) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  /* allocate a pointer to key and value for each entry */
  array = malloc( (2 * (table->length + 1) ) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) {
    for(b = table->buckets[i]; b; b = b->next) {
      array[j++] = (void *) b->key;
      array[j++] = (void *) b->value;
    }
  }
   
  array[j] = end;
  
  return(array);
}

void **t_keys(struct table *table) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  array = malloc( (table->length + 1) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) 
    for(b = table->buckets[i]; b; b = b->next) 
      array[j++] = (void *) b->key;
   
  array[j] = NULL;
  
  return(array);
}

void **t_values(struct table *table) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  array = malloc( (2 * (table->length + 1) ) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) 
    for(b = table->buckets[i]; b; b = b->next) 
        array[j++] = (void *) b->value;
       
  array[j] = NULL;
  
  return(array);
}

void t_free(struct table **table) {
  assert(table);
  assert(*table);
  
  if((*table)->length) {
    int i;
    struct bucket *p, *q;
    for(i = 0; i < (*table)->size; i++) {
      for(p = (*table)->buckets[i]; p; p = q) {
	q = p->next;
	free(p);
      }
    }
    
    free(*table);
  }
  
  return;
}

/* silly */

int tableCompareStrings(const void *x, const void *y) {
	return(strcmp((char *)x, (char *)y));
}

/* i hope this doesn't look familiar... */

unsigned tableHashStrings(const void *key) {
	unsigned h = 0;
	unsigned g;

	char *name = (char *) key;

	while(*name) {
		h = (h << 4) + *name++;
		if((g = (h & 0xf0000000)))
			h ^= g >> 24;
	
		h &= ~g;
	}

	return(h);
}	
