#include "arpcatch.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <net/ethernet.h>

static list_t *add_host(char *buffer, list_t *hosts);
static int add_proxy(char *buffer, table_t *at);

static void config_error(char *, int);

extern struct ether_addr *ether_aton(char *);

int compare_ip(const void *a, const void *b) {
	struct in_addr *ia, *ib;
	ia = (struct in_addr *) a;
	ib = (struct in_addr *) b;
	
	if(ia->s_addr != ib->s_addr)
		return(1);

	return(0);
}

unsigned hash_ip(const void *a) {
	struct in_addr *ia = (struct in_addr *) a;
	return(ia->s_addr);
}

static list_t *routers = NULL;

table_t *read_config(char *filename) {
	FILE *fp;
	char buffer[1024];
	char *c;
	table_t *at;
	int line = 0;

	at = t_new(509, compare_ip, hash_ip);
	if(!at)
		return(NULL);

	if(!(fp = fopen(filename, "r"))) 
		return(NULL);

	while((c = fgets(buffer, 1024, fp))) {
		line++;

		chop(buffer);
		if(*c == '\0' || *c == '#')
			continue;

		while(isspace(*c))
			c++;

		if(!strncasecmp(c, "router", 6)) {
			list_t *l = add_host(buffer, routers);
			if(l)
				routers = l;
			else
				config_error(buffer, line);

			continue;
		}

		if(!strncasecmp(c, "proxy", 5)) {
			add_proxy(buffer, at);
			continue;
		} 

		config_error(buffer, line);

	}

	return(at);
}

void config_error(char *string, int line) {
	fprintf(stderr, "Configuration file error on line %d:\n%s\n",
		line, string);
	return;
}

list_t *add_host(char *buffer, list_t *hosts) {
	char *cp, *c = buffer;
	char *host;
	struct arp_map *am;

	if(!(cp = strsep(&c, " \t"))) 
		return(NULL);

	if(!(cp = strsep(&c, " \t"))) 
		return(NULL);

	host = cp;

	if(!(cp = strsep(&c, " \t"))) 
		return(NULL);

	am = (struct arp_map *) NEW(sizeof(*am));

	am->name = strdup(host);
	am->mac  = ether_aton(cp);
	if(!am->mac) {
		free(am);
		return(NULL);
	}

	hosts = l_push(hosts, am);

	return(hosts);
}

int add_proxy(char *buffer, table_t *at) {
	char *cp, *c = buffer;
	struct in_addr ia;
	struct ether_addr *mac;

	struct in_addr *iae;
	struct ether_addr *mace;

	if(!(cp = strsep(&c, " \t"))) 
		return(-1);

	if(!(cp = strsep(&c, " \t"))) 
		return(-1);
	
	ia.s_addr = resolve(cp);
	if(ia.s_addr == INADDR_NONE)
		return(-1);

	if(!(cp = strsep(&c, " \t"))) 
		return(-1);

	mac = ether_aton(cp);
	if(!mac) {
		list_t *node;

		for(node = routers; node; node = node->next) {
			struct arp_map *amp = (struct arp_map *) node->data; 
			if(strcasecmp(amp->name, cp))
				continue;

			mac = amp->mac;
			break;
		}	

		if(!node)
			return(-1);
	}

	iae = (struct in_addr *) calloc(1, sizeof(*iae));
	if(!iae)
		return(-1);
	
	mace = (struct ether_addr *) calloc(1, sizeof(*mace));
	if(!mace) {
		free(iae);
		return(-1);
	}
	
	iae->s_addr = ia.s_addr;
	memcpy(mace, mac, sizeof(*mace));

	t_put(at, iae, mace);

	return(0);
}
	
