/* 1997 Thomas Ptacek, EnterAct, L.L.C. */

/* 4.4BSD BPF interface for capturing ARP requests */

#include "arpcatch.h"

struct bpf_insn capture_arp[] = {
	/* load halfword from (packet + 12) into accumulator,
	 * this is the ethernet protocol.
	 */

	BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),

	/* compare accumulator against constant ETHERTYPE_ARP,
	 * exit if not equal, continue if otherwise 
	 */

	BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ETHERTYPE_ARP, 0, 3),

	/* load halfword from (packet + 20) into accumulator,
	 * this is the ARP opcode
	 */

	BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 20),
	
	/* compare the accumulator against ARPOP_REQUEST, exit
	 * if not equal
	 */

	BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ARPOP_REQUEST, 0, 1),

	/* return the size of the ARP packet (this much will
	 * be copied into user space 
	 */

	BPF_STMT(BPF_RET+BPF_K, 256),

	/* return nothing */

	BPF_STMT(BPF_RET+BPF_K, 0)
};

int bpf_open(char *device) {
	int i = 0, fd;
	char file[PATH_MAX];
	struct ifreq ifr;

	do {
		snprintf(file, PATH_MAX, "/dev/bpf%d", i++);
		if((fd = open(file, O_RDWR)) < 0)
			continue;
	} while(errno == EBUSY && fd < 0);
	
	if(fd < 0)
		return(-1);

	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, device, 15);
	
	if(bpf_setbuf(fd, DEFAULT_BUFLEN) < 0)
		return(-2);

	if(ioctl(fd, BIOCSETIF, (char *) &ifr) < 0)
		return(-3);

	return(fd);
}

int bpf_setfilter(int fd, struct bpf_insn *filter, int len) {
	struct bpf_program bp;	

	if(!filter) 
		return(-1);

	bp.bf_len = len;
	bp.bf_insns = filter;

	return(ioctl(fd, BIOCSETF, (char *) &bp));
}

int bpf_setbuf(int fd, int bufsize) {
	u_int bl = bufsize;

	return(ioctl(fd, BIOCSBLEN, (char *) &bl));
}

int bpf_getbuf(int fd) {
	u_int bl;

	if(ioctl(fd, BIOCGBLEN, (char *) &bl) < 0) 
		return(-1);

	return(bl);
}

int bpf_flush(int fd) {
	return(ioctl(fd, BIOCFLUSH, NULL));
}

int bpf_received(int fd) {
	struct bpf_stat bs;

	if(ioctl(fd, BIOCGSTATS, (char *) &bs) < 0)
		return(-1);

	return(bs.bs_recv);
}

int bpf_dropped(int fd) {
	struct bpf_stat bs;

	if(ioctl(fd, BIOCGSTATS, (char *) &bs) < 0)
		return(-1);

	return(bs.bs_drop);
}

int bpf_immediate(int fd, int toggle) {
	u_int t = toggle;

	return(ioctl(fd, BIOCIMMEDIATE, &t));
}

int bpf_promisc(int fd, int toggle) {
	u_int t = toggle;

	return(ioctl(fd, BIOCPROMISC, &t));
}

