/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * ragrep.c  - find regular expressions in argus user data buffers.
 *
 * written by Carter Bullard
 * QoSient, LLC
 */

/* 
 * $Id: //depot/argus/argus-3.0/clients/ragrep/ragrep.c#14 $
 * $DateTime: 2006/03/31 13:25:33 $
 * $Change: 793 $
 */


#include <unistd.h>
#include <stdlib.h>

#include <compat.h>

#include <rabins.h>
#include <argus_util.h>
#include <argus_client.h>
#include <argus_main.h>
#include <argus_filter.h>
#include <signal.h>
#include <ctype.h>


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#if defined(HAVE_MMAP)
# include <sys/mman.h>
#endif
#if defined(HAVE_SETRLIMIT)
# include <sys/time.h>
# include <sys/resource.h>
#endif
#include <stdio.h>
#include "system.h"

/*
#if !defined(__FreeBSD__)
#include "getopt.h"
#endif
*/

#include "getpagesize.h"
#include "ragrep.h"

#undef MAX
#define MAX(A,B) ((A) > (B) ? (A) : (B))


struct stats
{
  struct stats *parent;
  struct stat stat;
};

/* Short options.  */
static char const short_options[] =
"0123456789A:B:C::EFGHIUVX:abcd:e:f:hiLlnqrsuvwxyZz";

/* Non-boolean long options that have no corresponding short equivalents.  */
enum
{
  BINARY_FILES_OPTION = CHAR_MAX + 1
};

/* Long options equivalences. */
/*
static struct option long_options[] =
{
  {"after-context", required_argument, NULL, 'A'},
  {"basic-regexp", no_argument, NULL, 'G'},
  {"before-context", required_argument, NULL, 'B'},
  {"binary-files", required_argument, NULL, BINARY_FILES_OPTION},
  {"byte-offset", no_argument, NULL, 'b'},
  {"context", optional_argument, NULL, 'C'},
  {"count", no_argument, NULL, 'c'},
  {"directories", required_argument, NULL, 'd'},
  {"extended-regexp", no_argument, NULL, 'E'},
  {"file", required_argument, NULL, 'f'},
  {"files-with-matches", no_argument, NULL, 'l'},
  {"files-without-match", no_argument, NULL, 'L'},
  {"fixed-regexp", no_argument, NULL, 'F'},
  {"fixed-strings", no_argument, NULL, 'F'},
  {"help", no_argument, &show_help, 1},
  {"ignore-case", no_argument, NULL, 'i'},
  {"line-number", no_argument, NULL, 'n'},
  {"line-regexp", no_argument, NULL, 'x'},
  {"mmap", no_argument, &mmap_option, 1},
  {"no-filename", no_argument, NULL, 'h'},
  {"no-messages", no_argument, NULL, 's'},
  {"null", no_argument, NULL, 'Z'},
  {"null-data", no_argument, NULL, 'z'},
  {"quiet", no_argument, NULL, 'q'},
  {"recursive", no_argument, NULL, 'r'},
  {"regexp", required_argument, NULL, 'e'},
  {"invert-match", no_argument, NULL, 'v'},
  {"silent", no_argument, NULL, 'q'},
  {"text", no_argument, NULL, 'a'},
  {"binary", no_argument, NULL, 'U'},
  {"unix-byte-offsets", no_argument, NULL, 'u'},
  {"version", no_argument, NULL, 'V'},
  {"with-filename", no_argument, NULL, 'H'},
  {"word-regexp", no_argument, NULL, 'w'},
  {0, 0, 0, 0}
};
*/

/* Define flags declared in grep.h. */
int match_icase;
int match_words;
int match_lines;
unsigned char eolbyte;

/* For error messages. */
static char *prog;
static int errseen;

/* How to handle directories.  */
/*
static enum
  {
    READ_DIRECTORIES,
    RECURSE_DIRECTORIES,
    SKIP_DIRECTORIES
  } directories;
*/

static int  install_matcher PARAMS ((char const *));
static int  grepbuf PARAMS ((char *, char *));

/* Functions we'll use to search. */
static void (*compile) PARAMS ((char *, size_t));
static char *(*execute) PARAMS ((char *, size_t, char **));

/* Flags controlling the style of output. */


static int filename_mask = 0;   /* If zero, output nulls after filenames.  */
static int out_invert = 0;   /* Print nonmatching stuff. */
static int out_before = 0;   /* Lines of leading context. */
static int out_after = 0;   /* Lines of trailing context. */

/* Internal variables to keep track of byte count, context, etc. */
static int done_on_match;      /* Stop scanning file on first match */



void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   char keybuf[MAXSTRLEN], *cptr = NULL, *kptr = NULL;
   int with_filenames;
   int default_context;
   unsigned digit_args_val;
   size_t keycc = 0;
   char *keys;

   parser->RaWriteOut = 1;

   if (!(parser->RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      parser->RaWriteOut = 0;
      parser->RaInitialized++;
      bzero (keybuf, sizeof(keybuf));

      if ((kptr = parser->estr) != NULL) {
         cptr = keybuf;
         while (*kptr) {
            if ((kptr[0] == '\\') && (kptr[1] == '\\'))
               kptr++;
            *cptr++ = *kptr++;  
         }
      } else 
         usage ();

      if (parser->iflag > 0) {
         match_icase = 1;
         parser->idflag = 0;
      } else
         match_icase = 0;

      if ((keys = (char *) ArgusCalloc (1, strlen(keybuf) + 2)) != NULL) {
         bcopy (keybuf, keys, strlen(keybuf));
         keycc = strlen(keys);
      } else
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusCalloc failed %s\n", strerror(errno));

      with_filenames = 0;
      eolbyte = '\n';
      filename_mask = ~0;
/*
      SET_BINARY (desc);
*/
      if (!install_matcher(matcher))
         abort ();

      (*compile)(keys, keycc);
  
/* The value -1 means to use DEFAULT_CONTEXT. */
      out_after = out_before = -1;

/* Default before/after context: chaged by -C/-NUM options */
      default_context = 0;

/* Accumulated value of individual digits in a -NUM option */
      digit_args_val = 0;
  
/* Internationalization. */
#if HAVE_SETLOCALE
      setlocale (LC_ALL, "");
#endif
#if ENABLE_NLS
      bindtextdomain (PACKAGE, LOCALEDIR);
      textdomain (PACKAGE);
#endif

      parser->RaInitialized++;
   }
}

void RaArgusInputComplete (struct ArgusInput *input) {};

void
RaParseComplete (int sig)
{
   if (sig >= 0) {

      if (sig == SIGINT)
         exit(0);
   }
}


void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout()\n");
#endif
}

void
parse_arg (int argc, char**argv)
{}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragrep Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s -e 'regex' -v [raoptions] [- filter-expression]\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -e <regexp>        print record summaries on termination.\n");
   fprintf (stderr, "         -i                 ignore case.\n");
   fprintf (stderr, "         -v                 reverse the expression matching logic.\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusDataStruct *user = NULL;
   int len, retn, found = 0;

   if (((argus->dsrs[ARGUS_SRCUSERDATA_INDEX]) ==  NULL) &&
       ((argus->dsrs[ARGUS_SRCUSERDATA_INDEX]) ==  NULL))
      return;

   if ((user = (struct ArgusDataStruct *)argus->dsrs[ARGUS_SRCUSERDATA_INDEX]) !=  NULL) {
      char *buf = (char *)&user->array;
      if (ArgusGrepSource) {
         if ((user->hdr.type == ARGUS_DATA_DSR) && (user->hdr.subtype & ARGUS_LEN_16BITS)) {
            len = (user->hdr.argus_dsrvl16.len - 2 ) * 4;
         } else 
            len = (user->hdr.argus_dsrvl8.len - 2 ) * 4;

         if ((retn = grepbuf (buf, &buf[len])))
            found++;
      }
   }

   if ((user = (struct ArgusDataStruct *)argus->dsrs[ARGUS_DSTUSERDATA_INDEX]) !=  NULL) {
      char *buf = (char *)&user->array;
      if (ArgusGrepDestination) {
         if ((user->hdr.type == ARGUS_DATA_DSR) && (user->hdr.subtype & ARGUS_LEN_16BITS)) {
            len = (user->hdr.argus_dsrvl16.len - 2 ) * 4;
         } else
            len = (user->hdr.argus_dsrvl8.len - 2 ) * 4;

         if ((retn = grepbuf (buf, &buf[len])))
            found++;
      }
   }

   if ((!parser->vflag && found) || (parser->vflag && !found)) {
      char buf[MAXSTRLEN];

      if (parser->ArgusWfileList != NULL) {
         struct ArgusWfileStruct *wfile = NULL;
         struct ArgusListObjectStruct *lobj = NULL;
         int i, count = parser->ArgusWfileList->count;

         if ((lobj = parser->ArgusWfileList->start) != NULL) {
            for (i = 0; i < count; i++) {
               if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
                  int pass = 1;
                  if (wfile->filterstr) {
                     struct nff_insn *wfcode = wfile->filter.bf_insns;
                     pass = ArgusFilterRecord (wfcode, argus);
                  }

                  if (pass != 0) {
                     if ((parser->exceptfile == NULL) || strcmp(wfile->filename, parser->exceptfile)) {
                        struct ArgusRecord *argusrec = NULL;
                        static char sbuf[0x10000];
                        if ((argusrec = ArgusGenerateRecord (argus, 0L, sbuf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                           ArgusHtoN(argusrec);
#endif
                           ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);
                        }
                     }
                  }
               }

               lobj = lobj->nxt;
            }
         }

      } else {
         if (!parser->qflag) {
            if (parser->Lflag) {
               if (parser->RaLabel == NULL)
                  parser->RaLabel = ArgusGenerateLabel(parser, argus);
    
               if (!(parser->RaLabelCounter++ % parser->Lflag))
                  printf ("%s\n", parser->RaLabel);
    
               if (parser->Lflag < 0)
                  parser->Lflag = 0;
            }

            *(int *)&buf = 0;
            ArgusPrintRecord(parser, buf, argus, MAXSTRLEN);
            fprintf (stdout, "%s\n", buf);
         }
      }
   }
}

int RaSendArgusRecord(struct ArgusRecordStruct *argus) {return 0;}

void ArgusWindowClose(void);

void ArgusWindowClose(void) { 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusWindowClose () returning\n"); 
#endif
}


/* ragrep.c - main driver file for ragrep.
 * Copyright 2000-2004, QoSient, LLC.
 * Completely derived from grep-2.4.2.
 */

/* grep.c - main driver file for grep.
   Copyright 1992, 1997-1999, 2000 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/* Written July 1992 by Mike Haertel.  */

static void
error (const char *mesg, int errnum)
{
   if (errnum)
      fprintf (stderr, "%s: %s: %s\n", prog, mesg, strerror (errnum));
   else
      fprintf (stderr, "%s: %s\n", prog, mesg);
   errseen = 1;
}

/* Like error (), but die horribly after printing. */

void
fatal (const char *mesg, int errnum)
{
   error (mesg, errnum);
   exit (2);
}

/* Interface to handle errors and fix library lossage. */
char *
xmalloc (size_t size)
{
   char *result;

   result = malloc (size);
   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

char * xcalloc (size_t, size_t);

char *
xcalloc (size_t num, size_t size)
{
   char *result;

   result = calloc (num, size);
   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

/* Interface to handle errors and fix some library lossage. */
char *
xrealloc (char *ptr, size_t size)
{
  char *result;

   if (ptr)
      result = realloc (ptr, size);
   else
      result = malloc (size);

   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

/* Convert STR to a positive integer, storing the result in *OUT.
   If STR is not a valid integer, return -1 (otherwise 0). */

/*
static int
ck_atoi (char const *str, int *out)
{
   char const *p;
   for (p = str; *p; p++)
      if (*p < '0' || *p > '9')
         return -1;

   *out = atoi (optarg);

   return 0;
}
*/


/* Hairy buffering mechanism for grep.  The intent is to keep
   all reads aligned on a page boundary and multiples of the
   page size. */

#define PREFERRED_SAVE_FACTOR 5   /* Preferred value of bufalloc / bufsalloc.  */

#if defined(HAVE_MMAP)
static int bufmapped;      /* True if buffer is memory-mapped.  */
static off_t initial_bufoffset;   /* Initial value of bufoffset. */
#endif

/* Return VAL aligned to the next multiple of ALIGNMENT.  VAL can be
   an integer or a pointer.  Both args must be free of side effects.  */

#define ALIGN_TO(val, alignment) \
  ((size_t) (val) % (alignment) == 0 \
   ? (val) \
   : (val) + ((alignment) - (size_t) (val) % (alignment)))

/* Return the address of a page-aligned buffer of size SIZE,
   reallocating it from *UP.  Set *UP to the newly allocated (but
   possibly unaligned) buffer used to build the aligned buffer.  To
   free the buffer, free (*UP).  */

/*
static char *
page_alloc (size_t size, char **up)
{
   size_t asize = size + pagesize - 1;
   if (size <= asize) {
      char *p = *up ? xrealloc (*up, asize) : xmalloc (asize);

      if (p) {
         *up = p;
         return ALIGN_TO (p, pagesize);
      }
   }

   return NULL;
}
*/

#if O_BINARY
/*
#include "dosbuf.c"
*/
#endif

/*
static void
nlscan (char *lim)
{
   char *beg;
   for (beg = lastnl;  (beg = memchr (beg, eolbyte, lim - beg));  beg++)
      totalnl++;

   lastnl = lim;
}
*/


/* Scan the specified portion of the buffer, matching lines (or
   between matching lines if OUT_INVERT is true).  Return a count of
   lines printed. */

static int
grepbuf (char *beg, char *lim)
{
   int nlines;
   register char *p, *b;
   char *endp;
   char eol = eolbyte;

   nlines = 0;
   p = beg;
   while ((b = (*execute)(p, lim - p, &endp)) != 0) {
      /* Avoid matching the empty line at the end of the buffer. */
      if (b == lim && ((b > beg && b[-1] == eol) || b == beg))
         break;
      if (!out_invert) {
         nlines += 1;
         if (done_on_match)
            return nlines;

      } else
         if (p < b) {
            nlines += 1;
         }
      p = endp;
   }

   if (out_invert && (p < lim))
      nlines += 1;

   return nlines;
}


/* Set the matcher to M, reporting any conflicts.  */
/*
static void
setmatcher (char const *m)
{
   if (matcher && strcmp (matcher, m) != 0)
      fatal (_("conflicting matchers specified"), 0);

   matcher = m;
}
*/

/* Go through the matchers vector and look for the specified matcher.
   If we find it, install it in compile and execute, and return 1.  */

static int
install_matcher (char const *name)
{
   int i;
#ifdef HAVE_SETRLIMIT
   struct rlimit rlim;
#endif

   for (i = 0; matchers[i].name; ++i) {
      if (strcmp (name, matchers[i].name) == 0) {
         compile = matchers[i].compile;
         execute = matchers[i].execute;

#if HAVE_SETRLIMIT && defined(RLIMIT_STACK)
         /* I think every platform needs to do this, so that regex.c
            doesn't oveflow the stack.  The default value of
            `re_max_failures' is too large for some platforms: it needs
            more than 3MB-large stack.  The test for HAVE_SETRLIMIT
            should go into `configure'.  */

         if (!getrlimit (RLIMIT_STACK, &rlim)) {
            long newlim;
            extern long int re_max_failures; /* from regex.c */

            /* Approximate the amount regex.c needs, plus some more.  */

            newlim = re_max_failures * 2 * 20 * sizeof (char *);
            if (newlim > rlim.rlim_max) {
               newlim = rlim.rlim_max;
               re_max_failures = newlim / (2 * 20 * sizeof (char *));
            }

            if (rlim.rlim_cur < newlim)
               rlim.rlim_cur = newlim;

            setrlimit (RLIMIT_STACK, &rlim);
         }
#endif
         return 1;
      }
   }
   return 0;
}


/* Find the white-space-separated options specified by OPTIONS, and
   using BUF to store copies of these options, set ARGV[0], ARGV[1],
   etc. to the option copies.  Return the number N of options found.
   Do not set ARGV[N] to NULL.  If ARGV is NULL, do not store ARGV[0]
   etc.  Backslash can be used to escape whitespace (and backslashes).  */

/*
static int
prepend_args (char const *options, char *buf, char **argv)
{
   char const *o = options;
   char *b = buf;
   int n = 0;

   for (;;) {
      while (ISSPACE ((unsigned char) *o))
         o++;

      if (!*o)
         return n;

      if (argv)
         argv[n] = b;
      n++;

      do
         if ((*b++ = *o++) == '\\' && *o)
            b[-1] = *o++;
      while (*o && ! ISSPACE ((unsigned char) *o));

      *b++ = '\0';
   }
}
*/

/* Prepend the whitespace-separated options in OPTIONS to the argument
   vector of a main program with argument count *PARGC and argument
   vector *PARGV.  */

/*
static void
prepend_default_options (char const *options, int *pargc, char ***pargv)
{
   if (options) {
      char *buf = xmalloc (strlen (options) + 1);
      int prepended = prepend_args (options, buf, (char **) NULL);
      int argc = *pargc;
      char * const *argv = *pargv;
      char **pp = (char **) xmalloc ((prepended + argc + 1) * sizeof *pp);
      *pargc = prepended + argc;
      *pargv = pp;
      *pp++ = *argv++;

      pp += prepend_args (options, buf, pp);

      while ((*pp++ = *argv++))
         continue;
    }
}
*/

