/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * argus_main - main routine for parsing argus output.
 *       this module performs all the argus(1) related connection parsing,
 *       selects datum from a set of criteria, and then calls specific
 *       protocol dependant routines, depending on the selected datum.
 *       at the end of processing, argus_parse calls an application
 *       specific finish routine, RaParseComplete(), and when
 *       connected to a remote data source, it supplies a periodic
 *       timeout routine;
 *
 *       this module defines all things, except:
 *
 *   (void) usage ((char *) argv[0]);
 *                    this routine should print the standard usage message
 *                    for the specific application.
 *
 *   (void) ArgusClientInit ();  this is the application specific init
 *                    routine, which is called after all parsing
 *                    initialization is done, prior to reading the
 *                    first monitor(1) datum.
 *
 *   (void) ArgusClientTimeout ();
 *                    this routine is called every second, when
 *                    argus_parse is connected to a remote data source.
 *
 *   (void) RaParseComplete (0);
 *                    this routine will be called after all the
 *                    monitor data has been read.
 *
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: //depot/argus/argus-3.0/clients/common/argus_main.c#63 $
 * $DateTime: 2006/04/10 11:27:11 $
 * $Change: 822 $
 */

#include <sys/types.h>
#include <unistd.h>

#include <pwd.h>
#include <grp.h>

#if defined(ARGUS_THREADS) 
#include <pthread.h>
#endif

#define ArgusMain

#include <compat.h>

#include <argus_def.h>
#include <argus_out.h>

#include <signal.h>

#include <argus_util.h>

#include <argus_client.h>
#include <argus_main.h>
#include <argus_filter.h>
#include <dscodepoints.h>

#include <ctype.h>
#include <strings.h>

#if defined(HAVE_SOLARIS)
#include <string.h>
#endif

#include <sys/wait.h>

#if defined(__NetBSD__)
#include <sys/sched.h>
#else
#include <sched.h>
#endif

void ArgusHandleSig (int);

char *chroot_dir = NULL;
uid_t new_uid;
gid_t new_gid;


int
main (int argc, char **argv)
{
   struct ArgusInput *addr;
   extern char *optarg;
   extern int optind, opterr;
   int i, cc;

#if defined(ARGUS_THREADS)
   int hosts = 0;
   pthread_attr_t attr;
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && !defined(sun) && !defined(CYGWIN)
   int thread_policy;
   struct sched_param thread_param;
#if defined(HAVE_SCHED_GET_PRIORITY_MIN)
   int rr_min_priority, rr_max_priority;
#endif
#endif
   int status;
   size_t stacksize;
#endif

   for (i = 0, cc = 0; i < argc; i++)
      cc += strlen(argv[i]);

   if (strchr (argv[0], '/'))
      argv[0] = strrchr(argv[0], '/') + 1;

   if ((ArgusParser = ArgusNewParser(argv[0])) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewParser failed %s", strerror(errno));

#if defined(ARGUS_THREADS)
#if defined(sun)
   thr_setconcurrency (2);
#endif

   if ((status = pthread_attr_init(&attr)) != 0)
      ArgusLog (LOG_ERR, "pthreads init error");
 
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && !defined(sun) && !defined(CYGWIN)
   if ((status = pthread_attr_getschedpolicy(&attr, &thread_policy)) != 0)
      ArgusLog (LOG_ERR, "pthreads get policy error");
   if ((status = pthread_attr_getschedparam(&attr, &thread_param)) != 0)
      ArgusLog (LOG_ERR, "pthreads get sched params error");
   if ((status = pthread_attr_setschedpolicy(&attr, SCHED_RR)) != 0)
      ArgusLog (LOG_ERR, "pthreads set SCHED_RR error");

#if defined(HAVE_SCHED_GET_PRIORITY_MIN)
   if ((rr_min_priority = sched_get_priority_min(SCHED_RR)) == -1)
      ArgusLog (LOG_ERR, "pthreads get priority min error");
   if ((rr_max_priority = sched_get_priority_max(SCHED_RR)) == -1)
      ArgusLog (LOG_ERR, "pthreads get priority max error");

   thread_param.sched_priority = (rr_max_priority + rr_min_priority)/2 + 1;

   if (thread_param.sched_priority > rr_max_priority)
      thread_param.sched_priority = rr_max_priority;
   if (thread_param.sched_priority < (rr_max_priority - 8))
      thread_param.sched_priority = rr_max_priority - 8;

   if ((status = pthread_attr_setschedparam(&attr, &thread_param)) != 0)
      ArgusLog (LOG_ERR, "pthreads set sched param error");
#endif
#else
   pthread_attr_setschedpolicy(&attr, SCHED_RR);
#endif

#if defined(_POSIX_THREAD_ATTR_STACKSIZE)
#define ARGUS_MIN_STACKSIZE	524288

   if (pthread_attr_getstacksize(&attr, &stacksize))
      ArgusLog (LOG_ERR, "pthreads get stacksize error");

   if (stacksize < ARGUS_MIN_STACKSIZE) {
#ifdef ARGUSDEBUG
      ArgusDebug (1, "setting stacksize from %d to %d", stacksize, ARGUS_MIN_STACKSIZE);
#endif
      if (pthread_attr_setstacksize(&attr, ARGUS_MIN_STACKSIZE))
         ArgusLog (LOG_ERR, "pthreads set stacksize error");
   }
#endif
 
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
#endif

   ArgusMainInit (ArgusParser, argc, argv);
   ArgusClientInit (ArgusParser);
 
   if (!(ArgusParser->Sflag))
      if (ArgusParser->ArgusInputFileList == NULL)
         if (!(ArgusAddFileList (ArgusParser, "-", ARGUS_DATA_SOURCE, -1, -1)))
            ArgusLog(LOG_ERR, "%s: error: file arg %s", *argv, optarg);

/*
   OK now we're ready.  Read in all the files, for as many passes as
   needed, and then attach to any remote sources as a group until
   they close, then we're done.
*/

   if (ArgusParser->ArgusInputFileList != NULL) {
      struct ArgusInput *file; 

      for (i = 0; i < ArgusPassNum; i++) {
         file = ArgusParser->ArgusInputFileList;
         while (file && ArgusParser->eNflag) {
            if (strcmp (file->filename, "-")) {
               if (file->fd < 0) {
                  if ((file->file = fopen(file->filename, "r")) == NULL) {
#ifdef ARGUSDEBUG
                     ArgusDebug (0, "open '%s': %s", file->filename, strerror(errno));
#endif
                  }

               } else {
                  fseek(file->file, 0, SEEK_SET);
               }

               if ((file->file != NULL) && ((ArgusReadConnection (ArgusParser, file, ARGUS_FILE)) >= 0)) {
                  ArgusParser->ArgusTotalMarRecords++;
                  ArgusParser->ArgusTotalRecords++;

                  if (ArgusParser->RaPollMode) {
                      ArgusHandleDatum (ArgusParser, file, &file->ArgusInitCon, &ArgusParser->ArgusFilterCode);
                  } else {
                     if (file->ostart != -1) {
                        file->offset = file->ostart;
                        if (fseek(file->file, file->offset, SEEK_SET) >= 0)
                           ArgusReadFileStream(ArgusParser, file);
                     } else
                        ArgusReadFileStream(ArgusParser, file);
                  }

               } else
                  file->fd = -1;

               if (file->file != NULL) {
                  ArgusCloseInput(ArgusParser, file);  
               }

            } else {
               int flags;
               file->file = stdin;
               file->ostart = -1;
               file->ostop = -1;

               if (((ArgusReadConnection (ArgusParser, file, ARGUS_FILE)) >= 0)) {
                  ArgusParser->ArgusTotalMarRecords++;
                  ArgusParser->ArgusTotalRecords++;

                  if ((flags = fcntl(fileno(stdin), F_GETFL, 0L)) < 0)
                     ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

                  if (fcntl(fileno(stdin), F_SETFL, flags | O_NONBLOCK) < 0)
                     ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

                  ArgusReadFileStream(ArgusParser, file);
               }
            }

#ifdef ARGUSDEBUG
            ArgusDebug (1, "main: ArgusReadFile (%s) done", file->filename);
#endif
            RaArgusInputComplete(file);
            file = (struct ArgusInput *)file->qhdr.nxt;
         }
/*
         RaParseComplete(ArgusPassNum - i);
*/
      }
   }
#ifdef ARGUSDEBUG
   ArgusDebug (1, "main: reading files completed");
#endif

/*
   Now we're going to deal with remote data sources.  To implement
   reliable connections effeciently, we need to put the input blocks
   in a data structure so that our reliable thread can do the right
   thing with them.
   
   The idea is that if they are in the queue we need to get a connection
   with the input.  If they are not in the queue, we have a connection or
   we are going to delete/forget them  because of massive errors.

   So, if we are reliably connected, first we put them all on the queue.
   If not we just connect to them sequentially.
*/

   if (ArgusParser->Sflag) {
      if (ArgusParser->ArgusRemoteHosts && (ArgusParser->ArgusRemoteHosts->count > 0)) {
         struct ArgusQueueStruct *tqueue = ArgusNewQueue();
         int flags;

         ArgusParser->ArgusRemotes = ArgusParser->ArgusRemoteHosts->count;

#if defined(ARGUS_THREADS)
         if (ArgusParser->ArgusReliableConnection) {
            if (ArgusParser->ArgusRemoteHosts && (hosts = ArgusParser->ArgusRemoteHosts->count)) {
               if ((pthread_create(&ArgusParser->remote, &attr, ArgusConnectRemotes, ArgusParser->ArgusRemoteHosts)) != 0)
                  ArgusLog (LOG_ERR, "ArgusNewOutput() pthread_create error %s\n", strerror(errno));
            }

         } else {
#else
         {
#endif
            while ((addr = (void *)ArgusPopQueue(ArgusParser->ArgusRemoteHosts, ARGUS_LOCK)) != NULL) {
               if ((addr->fd = ArgusGetServerSocket (addr, 5)) >= 0) {
                  if ((ArgusReadConnection (ArgusParser, addr, ARGUS_SOCKET)) >= 0) {
                     ArgusParser->ArgusTotalMarRecords++;
                     ArgusParser->ArgusTotalRecords++;

                     if ((flags = fcntl(addr->fd, F_GETFL, 0L)) < 0)
                        ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

                     if (fcntl(addr->fd, F_SETFL, flags | O_NONBLOCK) < 0)
                        ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

                     if (ArgusParser->RaPollMode)
                        ArgusHandleDatum (ArgusParser, addr, &addr->ArgusInitCon, &ArgusParser->ArgusFilterCode);

                     ArgusAddToQueue(ArgusParser->ArgusActiveHosts, &addr->qhdr, ARGUS_LOCK);
                     ArgusParser->ArgusHostsActive++;
                  } else
                     ArgusAddToQueue(tqueue, &addr->qhdr, ARGUS_LOCK);
               } else
                  ArgusAddToQueue(tqueue, &addr->qhdr, ARGUS_LOCK);
#if !defined(ARGUS_THREADS)
            }
#else
            }
#endif
         }

         while ((addr = (void *)ArgusPopQueue(tqueue, ARGUS_LOCK)) != NULL)
            ArgusAddToQueue(ArgusParser->ArgusRemoteHosts, &addr->qhdr, ARGUS_LOCK);

         ArgusDeleteQueue(tqueue);
      }

#if defined(ARGUS_THREADS) 
      if (ArgusParser->ArgusReliableConnection || ArgusParser->ArgusActiveHosts->count)
#else
      if (ArgusParser->ArgusActiveHosts->count)
#endif
         ArgusReadStream(ArgusParser, ArgusParser->ArgusActiveHosts);

   } else {
#if defined(ARGUS_THREADS) 
      ArgusParser->RaDonePending++;
#else
      ArgusParser->RaParseDone++;
#endif
   }

   RaParseComplete(1);
   ArgusShutDown (0);

#if defined(ARGUS_THREADS) 
   if (ArgusParser->Sflag) {
      void *retn = NULL;

      if (ArgusParser->ArgusReliableConnection)
         pthread_attr_destroy(&attr);

      while ((addr = (void *)ArgusPopQueue(ArgusParser->ArgusActiveHosts, ARGUS_LOCK)) != NULL) {
         if (addr->tid != (pthread_t) 0) {
            pthread_join(addr->tid, &retn);
         }
      }
   }

   if (ArgusParser->dns != (pthread_t) 0)
      pthread_join(ArgusParser->dns, NULL);
#endif

   ArgusCloseParser(ArgusParser);
   exit (ArgusExitStatus);
}


#include <dirent.h>
 
int
RaDescend(char *name)
{
   int retn = 0;
   DIR *dir;
   struct dirent *direntry;
   struct stat statbuf;
   char buf[MAXSTRLEN];
 
   if (stat(name, &statbuf) < 0)
      return(0);
 
   if ((dir = opendir(name)) != NULL) {
      while ((direntry = readdir(dir)) != NULL) {
         if (*direntry->d_name != '.') {
            snprintf (buf, MAXSTRLEN, "%s/%s", name, direntry->d_name);
            if (stat(buf, &statbuf) == 0) {
               if ((statbuf.st_mode & S_IFMT) == S_IFDIR) {
                  retn += RaDescend(buf);
 
               } else {
                  if ((statbuf.st_mode & S_IFMT) == S_IFREG) {
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "RaDescend: adding %s\n", buf);
#endif
                     if (!(ArgusAddFileList (ArgusParser, buf, ARGUS_DATA_SOURCE, -1, -1)))
                        ArgusLog (LOG_ERR, "error: -R file arg %s\n", buf);

                     retn++;
                  }
               }
            }
         }
      }
      closedir(dir);

   }
 
   return(retn);
}

int RaSortFileList (const void *, const void *);

int
RaSortFileList (const void *item1, const void *item2)
{
   struct ArgusInput *input1 = *(struct ArgusInput **) item1;
   struct ArgusInput *input2 = *(struct ArgusInput **) item2;

   return (strcmp (input1->filename, input2->filename));
}



void ArgusSortFileList (struct ArgusInput **);

void
ArgusSortFileList (struct ArgusInput **list)
{
   struct ArgusInput *input = NULL;
   void **array = NULL;
   int count = 0, i;

   if ((input = *list) != NULL) {
      while (input != NULL) {
         count++;
         input = (struct ArgusInput *)input->qhdr.nxt;
      }

      if ((array = ArgusCalloc (count, sizeof(input))) == NULL)
         ArgusLog (LOG_ERR, "ArgusSortFileList: ArgusCalloc %s", strerror(errno));

      for (i = 0, input = *list ; i < count; i++) {
          array[i] = input;
          input = (struct ArgusInput *)input->qhdr.nxt;
      }

      qsort (array, count, sizeof(input), RaSortFileList);

      for (i = 0; i < count; i++) {
         ((struct ArgusInput *)array[i])->qhdr.nxt = NULL;
         if (i > 0)
            ((struct ArgusInput *)array[i - 1])->qhdr.nxt = &((struct ArgusInput *)array[i])->qhdr;
      }

      *list = array[0];
      ArgusFree (array);
   }
}


int
RaProcessRecursiveFiles (char *path)
{
   int retn = 1;
   struct stat statbuf;

   if (stat(path, &statbuf) < 0)
      return(0);

   if ((strlen(path) > 1) && ((path[0] == '.') && (path[1] != '/')))
      return (0);

   if ((statbuf.st_mode & S_IFMT) == S_IFDIR) {
      retn = RaDescend (path);
   } else {
      if ((statbuf.st_mode & S_IFMT) == S_IFREG) {
#ifdef ARGUSDEBUG
         ArgusDebug (1, "RaProcessRecursiveFiles: adding %s\n", path);
#endif
         if (!(ArgusAddFileList (ArgusParser, path, ARGUS_DATA_SOURCE, -1, -1)))
            ArgusLog (LOG_ERR, "error: -R file arg %s\n", path);
      }
   }

   ArgusSortFileList (&ArgusParser->ArgusInputFileList);
   return (retn);
}


void
ArgusHandleSig (int sig)
{
   int value = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusHandleSig: received signal %d", sig);
#endif

   switch (sig) {
      case SIGUSR1:
         value = ArgusParser->debugflag;
         ArgusParser->debugflag = (value == 15) ? 15 : value + 1;
         break;

      case SIGUSR2:
         ArgusParser->debugflag = 0;
         break;

      case SIGTERM:
      case SIGQUIT:
      case SIGINT: 
         ArgusParser->RaParseDone++;
         ArgusShutDown(sig);
         break;

      default:
         break;
   }
}

void
ArgusShutDown (int sig)
{
#if defined(ARGUS_THREADS)
   void *retn;
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusShutDown (%d)\n", sig);
#endif

   if (!(ArgusParser->RaShutDown++)) {
      if (sig >= 0)
         RaParseComplete (0);

      if (ArgusParser->ArgusRemoteHosts != NULL) {
         struct ArgusQueueStruct *queue =  ArgusParser->ArgusRemoteHosts;
         struct ArgusInput *input = NULL;
 
         while (queue->count > 0) {
            if ((input = (struct ArgusInput *) ArgusPopQueue(queue, ARGUS_LOCK)) != NULL) {
               ArgusCloseInput(ArgusParser, input);
               if (input->hostname != NULL)
                  free (input->hostname);
               if (input->filename != NULL)
                  free (input->filename);
#if defined(HAVE_GETADDRINFO)
               if (input->host != NULL)
                  freeaddrinfo (input->host);
#endif
               ArgusFree(input);
            }
         }
         ArgusDeleteQueue(queue);
         ArgusParser->ArgusRemoteHosts = NULL;
      }

      if (ArgusParser->ArgusActiveHosts != NULL) {
         struct ArgusQueueStruct *queue =  ArgusParser->ArgusActiveHosts;
         struct ArgusInput *input = NULL;
 
         while ((input = (void *)ArgusPopQueue(queue, ARGUS_LOCK)) != NULL) {
            ArgusCloseInput(ArgusParser, input);
            if (input->hostname != NULL)
               free (input->hostname);
            if (input->filename != NULL)
               free (input->filename);
#if defined(HAVE_GETADDRINFO)
            if (input->host != NULL)
               freeaddrinfo (input->host);
#endif

#if defined(ARGUS_THREADS) 
            if (input->tid != (pthread_t) 0)
               pthread_join(input->tid, &retn);
#endif

            ArgusFree(input);
         }

         ArgusDeleteQueue(queue);
         ArgusParser->ArgusActiveHosts = NULL;
      }

      ArgusWindowClose();

      if (ArgusParser->pidflag)
         if (ArgusParser->ArgusPidFile)
            ArgusDeletePIDFile (ArgusParser);

      if (ArgusParser->ArgusPrintXml && ArgusParser->RaXMLStarted)
         printf("</ArgusDataStream>\n"); 

      switch (sig) {
         case SIGTERM:
         case SIGQUIT:
         case SIGINT: 
            exit (1);
      }

      return;
   }
}

void
ArgusMainInit (struct ArgusParserStruct *parser, int argc, char **argv)
{
   extern char *optarg;
   extern int optind, opterr;
   int i, cc, noconf = 0;
   time_t tsec;

   char *envstr = NULL;
   struct stat statbuf;
   struct timezone tz;
   static char path[MAXPATHNAMELEN];

   (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
   (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
   (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
   (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);
   (void) signal (SIGPIPE,  SIG_IGN);

   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "stime";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "flgs";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "proto";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "saddr";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "sport";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "dir";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "daddr";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "dport";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "pkts";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "bytes";
   ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = "state";

   ArgusProcessSOptions(parser);

   for (i = 0; i < parser->RaSOptionIndex; i++)
      if (parser->RaSOptionStrings[i] != NULL)
         parser->RaSOptionStrings[i] = NULL;

   parser->RaSOptionIndex = 0;

   if (!(strncmp(parser->ArgusProgramName, "radium", 6))) {
      parser->ArgusReliableConnection = 1;
      parser->pflag = 6;
   }

   for (i = 0, cc = 0; i < argc; i++)
      cc += strlen(argv[i]);

   if (cc > 0) {
      int len = cc + (argc + 1); 
                      
      if ((parser->ArgusProgramArgs = (char *) ArgusCalloc (len, sizeof(char))) != NULL) { 
         for (i = 0, *parser->ArgusProgramArgs = '\0'; i < argc; i++) { 
            strncat (parser->ArgusProgramArgs, argv[i], (1024 - strlen(parser->ArgusProgramArgs))); 
            strncat (parser->ArgusProgramArgs, " ", (1024 - strlen(parser->ArgusProgramArgs))); 
         }
      } else             
         ArgusLog (LOG_ERR, "ArgusCalloc(%d, %d) failed %s", len, sizeof(char), strerror(errno));
   } 

   if (gettimeofday(&parser->ArgusRealTime, &tz) < 0)
      ArgusLog (LOG_ERR, "gettimeofday failed %s", strerror(errno));

   parser->ArgusGlobalTime = parser->ArgusRealTime;
   thiszone = tz.tz_minuteswest * -60;

   tsec = parser->ArgusRealTime.tv_sec;
   if ((parser->RaTmStruct = localtime (&tsec))) {
      if (parser->RaTmStruct->tm_isdst)
         thiszone += 3600;

   } else {
      ArgusLog (LOG_ERR, "%s: localtime: error %s", *argv, strerror(errno));
   }

   for (i = 1; i < argc; i++)
      if (strstr (argv[i], "-X"))
         noconf++;

   if (!(noconf)) {
      snprintf (path, MAXPATHNAMELEN - 1, "/etc/ra.conf");

      if (stat (path, &statbuf) == 0)
         ArgusParseResourceFile (parser, path);

      if ((RaHomePath = getenv("ARGUSHOME")) != NULL) {
         snprintf (path, MAXPATHNAMELEN - 1, "%s/ra.conf", RaHomePath);
         if (stat (path, &statbuf) == 0) {
            ArgusParseResourceFile (parser, path);
         }
      }

      if ((envstr = getenv("ARGUSPATH")) != NULL) {
         while ((RaHomePath = strtok(envstr, ":")) != NULL) {
            snprintf (path, MAXPATHNAMELEN - 1, "%s/.rarc", RaHomePath);
            if (stat (path, &statbuf) == 0) {
               ArgusParseResourceFile (parser, path);
               break;
            }
            envstr = NULL;
         }

      } else {
         for (i = 0; i < RAENVITEMS; i++) {
            envstr = RaResourceEnvStr[i];
            if ((RaHomePath = getenv(envstr)) != NULL) {
               snprintf (path, MAXPATHNAMELEN, "%s/.rarc", RaHomePath);
               if (stat (path, &statbuf) == 0) {
                  ArgusParseResourceFile (parser, path);
                  break;
               }
            }
         }
      }
   }

   if (ArgusParser->pidflag)
      ArgusCreatePIDFile (ArgusParser, ArgusParser->ArgusProgramName);

   ArgusParseArgs (ArgusParser, argc, argv);
}

void
ArgusParseArgs (struct ArgusParserStruct *parser, int argc, char **argv)
{
   extern char *optarg;
   extern int optind, opterr;
   int op, retn = 0, rcmdline = 0, Scmdline = 0;
   char *cmdbuf = NULL, *str = NULL;
   char *getoptStr = NULL;
#if defined(HAVE_GETADDRINFO)
   struct addrinfo *host = NULL;
#else
   struct hostent *host = NULL;
#endif

   char *filter = NULL;
   char *tmparg = NULL;

   opterr = 0;

   if ((argv[optind]) != NULL)
      ArgusParser->ArgusProgramOptions = ArgusCopyArgv (&argv[optind]);

   if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6)))
      getoptStr = "a:A:bB:c:C:dD:E:e:f:F:g:GhH:iJlL:m:M:nN:OpP:qr:R:S:s:t:T:u:U:Vvw:xXzZ:";
   else
      getoptStr = "a:AbB:c:C:dD:E:e:f:F:GhH:ilL:m:M:nN:Op:P:qQ:r:R:S:s:t:T:uU:Vvw:xXzZ:%";

   while ((op = getopt (argc, argv, getoptStr)) != EOF) {
      switch (op) {
         case '%': ++ArgusParser->Pctflag; break;
         case 'a': ArgusParser->aflag = atoi (optarg); break;
         case 'A': 
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               setArgusArchive (ArgusParser, optarg);
            } else 
               ++ArgusParser->Aflag;
            break;

         case 'b': ++ArgusParser->bflag; break;
         case 'B': {
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               ArgusParser->ArgusBindAddr = strdup(optarg);
            } else {
               char *ptr;
               ArgusParser->Bflag = strtol(optarg, (char **)&ptr, 10);
               if (ptr == optarg)
                  usage ();
               if (isalpha((int) *ptr)) {
                  switch (*ptr) {
                     case 's': break;
                     case 'm': ArgusParser->Bflag *= 60; break;
                     case 'h': ArgusParser->Bflag *= 60 * 60; break;
                     case 'd': ArgusParser->Bflag *= 60 * 60 * 24; break;
                     case 'w': ArgusParser->Bflag *= 60 * 60 * 24 * 7; break;
                     case 'M': ArgusParser->Bflag *= 60 * 60 * 24 * 7 * 4; break;
                     case 'y': ArgusParser->Bflag *= 60 * 60 * 24 * 7 * 4 * 12; break;
                  }
               }
            }
            break;
         }

         case 'c': 
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               if ((chroot_dir = strdup(optarg)) == NULL)
                   ArgusLog (LOG_ERR, "strdup %s", strerror(errno));
            } else {

               ArgusParser->cflag++;
               if (optarg[0] == '\\') {
                  switch (optarg[1]) {
                     case 't': ArgusParser->RaFieldDelimiter = '\t'; break;
                  }

               } else
                  ArgusParser->RaFieldDelimiter = *optarg;
            }
            break;

         case 'C':
            ++ArgusParser->Cflag;
            ++ArgusParser->Sflag;
            if ((!Scmdline++) && (ArgusParser->ArgusRemoteHostList != NULL))
               ArgusDeleteHostList(ArgusParser);

            if (!(ArgusAddHostList (ArgusParser, optarg, ARGUS_CISCO_DATA_SOURCE)))
               ArgusLog(LOG_ERR, "%s: host %s unknown", *argv, optarg);
            break;

         case 'D': ArgusParser->debugflag = atoi (optarg); break;
         case 'd': ArgusParser->dflag = (ArgusParser->dflag) ? 0 : 1; break;

         case 'e':
            ArgusParser->estr = optarg;
            if (!(strncmp(ArgusParser->ArgusProgramName, "ragrep", 6))) {
               ArgusGrepSource++;
               ArgusGrepDestination++;
 
               if ((ArgusParser->estr[0] == 's') && (ArgusParser->estr[1] == ':')) {
                  ArgusGrepDestination = 0;
                  ArgusParser->estr = &ArgusParser->estr[2];
               }
               if ((ArgusParser->estr[0] == 'd') && (ArgusParser->estr[1] == ':')) {
                  ArgusGrepSource = 0;
                  ArgusParser->estr = &ArgusParser->estr[2];
               }
            } else {
               if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
                  if (optarg && isalnum((int)*optarg)) {
#if defined(HAVE_GETADDRINFO)
                     struct addrinfo *hp = host;
                     if ((retn = getaddrinfo(optarg, NULL, NULL, &host)) == 0) {
                        unsigned int found = 0, addr;
                        while (host && !found) {
                           switch (host->ai_family) {
                              case AF_INET: {
                                 struct sockaddr_in *sa = (struct sockaddr_in *) host->ai_addr;

                                 if (sa != NULL) {
                                    bcopy ((char *)&sa->sin_addr, (char *)&addr, 4);
                                    ArgusParser->ArgusID = (ntohl(addr));
                                    ArgusParser->ArgusIDType = ARGUS_ID_IS_IPADDR;
                                    found++;
                                 } else
                                    ArgusLog (LOG_ERR, "Probe ID %s error %s\n", optarg, strerror(errno));
                                 break;
                              }

                              default:
                                 hp = hp->ai_next;
                                 break;
                           }
                        }
                        freeaddrinfo(hp);
                    } else
#else
                    if ((host = gethostbyname(optarg)) != NULL) {
                       if ((host->h_addrtype == 2) && (host->h_length == 4)) {
                          unsigned int addr;
                          bcopy ((char *) *host->h_addr_list, (char *)&addr, host->h_length);
                          ArgusParser->ArgusID = (ntohl(addr));
                       } else
                          ArgusLog (LOG_ERR, "Probe ID %s error %s\n", optarg, strerror(errno));

                       ArgusParser->ArgusIDType = ARGUS_ID_IS_IPADDR;

                    } else
#endif
                        if (optarg && isdigit((int)*optarg)) {
                           ArgusParser->ArgusID = atoi (optarg);
                        } else
                           ArgusLog (LOG_ERR, "Probe ID value %s is not appropriate (%s)\n", optarg, strerror(errno));
                  } else
                     ArgusLog (LOG_ERR, "Probe ID value %s is not appropriate\n", optarg);

               } else {
                  if (!(strncasecmp(optarg, "hex", 5)))
                     ArgusParser->eflag = ARGUS_HEXDUMP;
                  else
                  if (!(strncasecmp(optarg, "ascii", 5)))
                     ArgusParser->eflag = ARGUS_ENCODE_ASCII;
                  else
                  if (!(strncasecmp(optarg, "encode64", 8)))
                     ArgusParser->eflag = ARGUS_ENCODE_64;
                  else
                  if (!(strncasecmp(optarg, "encode32", 8)))
                     ArgusParser->eflag = ARGUS_ENCODE_32;
                  else
                     usage();
               }
            }
            break;

         case 'E':
            ArgusParser->exceptfile = optarg;
            setArgusWfile (ArgusParser, optarg, NULL);
            break;

         case 'f': ArgusParser->ArgusFlowModelFile = optarg; break;
         case 'F': 
            if (!(ArgusParseResourceFile (ArgusParser, optarg)))
               ArgusLog(LOG_ERR, "%s: %s", optarg, strerror(errno));
            break;

         case 'g': {
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               struct group *gr;
               if ((gr = getgrnam(optarg)) == NULL)
                   ArgusLog (LOG_ERR, "unknown group \"%s\"\n", optarg);
               new_gid = gr->gr_gid;
               endgrent();
            } else {
            }
            break;
         }

	 case 'G': ArgusParser->Gflag++; break;
	 case 'H': {
            char str[1024], Hstr[1024], *Hptr = Hstr;
            bzero (str, 1024);
            bzero (Hstr, 1024);
            do {
               if (*optarg == '"') {
                  if (Hptr[1] != '\0')
                     snprintf (Hstr, 1024, "%s", (&Hptr[1]));

                  while ((Hptr = strchr (Hstr, '"')) == NULL) {
                     if ((optarg = argv[optind]) != NULL) {
                        strncat (Hstr, optarg, (1024 - strlen(Hstr)));
                        optind++;
                     } else
                        break;
                  }
                  optarg = Hstr;
               }

               snprintf (&str[strlen(str)], (1024 - strlen(str)), "%s ", optarg);

               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));

            ArgusParser->Hstr = strdup(str);
            break;
         }

         case 'i': ++ArgusParser->iflag; break;
         case 'I': ++ArgusParser->Iflag; break;
         case 'J': ++ArgusParser->jflag; break;
         case 'l': ++ArgusParser->lflag; break;
         case 'L': 
            ArgusParser->Lflag = atoi(optarg);
            switch (ArgusParser->Lflag) {
               case  0: ArgusParser->Lflag = -1; break;
               case -1: ArgusParser->Lflag =  0; break;
            }
            break;
         case 'm':
            do {
               if (!(ArgusAddMaskList (ArgusParser, optarg))) {
                  ArgusLog(LOG_ERR, "%s: error: file arg %s", *argv, optarg);
               }
               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;

         case 'M': {
            char Mstr[1024], *Mptr = Mstr, *tzptr;
            bzero (Mstr, 1024);
            
            do {
               if (*optarg == '"') {
                  if (Mptr[1] != '\0')
                     snprintf (Mstr, 1024, "%s", (&Mptr[1]));

                  while ((Mptr = strchr (Mstr, '"')) == NULL) {
                     if ((optarg = argv[optind]) != NULL) {
                        strncat (Mstr, optarg, (1024 - strlen(Mstr)));
                        optind++;
                     } else
                        break;
                  }
                  optarg = Mstr;
               }

               if (!(strcmp (optarg, "xml"))) {
                  ArgusParser->ArgusPrintXml++;
               } else
               if (!(strcmp (optarg, "disa"))) {
                  ArgusParser->ArgusDSCodePoints = ARGUS_DISA_DSCODES;
                  RaPrintAlgorithmTable[ARGUSPRINTSRCDSBYTE].length = 8;
                  RaPrintAlgorithmTable[ARGUSPRINTDSTDSBYTE].length = 8;
               } else
               if ((tzptr = strstr (optarg, "TZ="))) {
                  if (ArgusParser->RaTimeZone != NULL)
                     free (ArgusParser->RaTimeZone);
                  ArgusParser->RaTimeZone = strdup(optarg);
#if defined(HAVE_SETENV)
                  setenv("TZ", (ArgusParser->RaTimeZone + 3), 1);
#else
                  putenv(ArgusParser->RaTimeZone);
#endif
                  tzset();
               } else {
#if defined(ARGUS_SASL)
               if ((tzptr = strstr (optarg, "saslmech="))) {
                  extern char *RaSaslMech;
                  if (RaSaslMech)
                     free (RaSaslMech);
                  RaSaslMech=strdup(&optarg[9]);
               }
#endif /* ARGUS_SASL */
               }

               if (!(ArgusAddModeList (ArgusParser, optarg))) {
                  ArgusLog(LOG_ERR, "%s: error: arg %s", *argv, optarg);
               }
               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;
         }

         case 'n': {
            if (++ArgusParser->nflag > 3) 
               ArgusParser->nflag = 0;
            break;
         }
         case 'N': {
            char *ptr = NULL;

            if ((ptr = strchr (optarg, '-')) != NULL) {
               char *eptr = ptr + 1;
               ArgusParser->sNflag = strtol(optarg, (char **)&ptr, 10);
               if (ptr == optarg)
                  usage ();
               ArgusParser->eNflag = strtol(eptr, (char **)&ptr, 10);
               if (ptr == eptr)
                  usage ();

            } else {
               ArgusParser->sNflag = 0;
               ArgusParser->eNflag = strtol(optarg, (char **)&ptr, 10);
               if (ptr == optarg)
                  usage ();
            }
            break;
         }
         case 'O': ArgusParser->Oflag = 0; break;
         case 'p': 
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               ArgusParser->ArgusReliableConnection = 0;
            } else {
               ArgusParser->pflag = atoi (optarg); break;
            }
            break;

         case 'P': 
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               ArgusParser->ArgusPortNum = atoi (optarg);
            } else {
               ArgusParser->Pflag++;
               ArgusParser->projstr = strdup(optarg);
            }
            break;

         case 'q': ++ArgusParser->qflag; break;
         case 'Q': {
            ArgusParser->Qflag = atoi (optarg);
            break;
         }

/* -r file[::ostart:ostop] */

         case 'r': {
            ++ArgusParser->rflag; 
            ArgusParser->Sflag = 0;
            if ((!rcmdline++) && (ArgusParser->ArgusInputFileList != NULL))
               ArgusDeleteFileList(ArgusParser);

            if (optarg == NULL)
               optarg = "-";
            do {
               long long ostart = -1, ostop = -1;
               char *ptr, *eptr;

               if ((ptr = strstr(optarg, "::")) != NULL) {
                  char *endptr;

                  *ptr++ = '\0';
                  ptr++;

                  if ((eptr = strstr(ptr, ":")) != NULL) {
                     ostart = strtol(ptr, (char **)&endptr, 10);
                     if (ptr == optarg)
                        usage ();
                     ostop = strtol((eptr + 1), (char **)&endptr, 10);
                     if (eptr == optarg)
                        usage ();
                  } else
                     usage ();
               }

               if (!(ArgusAddFileList (ArgusParser, optarg, (ArgusParser->Cflag ? ARGUS_CISCO_DATA_SOURCE : ARGUS_DATA_SOURCE), ostart, ostop))) {
                  ArgusLog(LOG_ERR, "%s: error: file arg %s", *argv, optarg);
               }
               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;
         }

         case 'R': {
            ArgusParser->Sflag = 0;
            if ((!rcmdline++) && (ArgusParser->ArgusInputFileList != NULL))
               ArgusDeleteFileList(ArgusParser);

            do {
               RaProcessRecursiveFiles (optarg);
               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));

            break;
         }

         case 's': 
            do {
               if (ArgusParser->RaSOptionIndex < ARGUS_MAX_S_OPTIONS) {
                  char *soptstr = strdup(optarg), *sptr;
                  if ((sptr = soptstr) != NULL) {
                     char *cptr;
                     while ((cptr = strtok(sptr, " ,")) != NULL) {
                        ArgusParser->RaSOptionStrings[ArgusParser->RaSOptionIndex++] = strdup(cptr);
                        sptr = NULL;
                     }
                  }
                  free (soptstr);
               } else
                  ArgusLog (LOG_ERR, "usage: number of -s options exceeds %d", ARGUS_MAX_S_OPTIONS);

               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;

         case 'S':
            ++ArgusParser->Sflag;
            if ((!Scmdline++) && (ArgusParser->ArgusRemoteHostList != NULL))
               ArgusDeleteHostList(ArgusParser);

            if (!(ArgusAddHostList (ArgusParser, optarg, ARGUS_DATA_SOURCE)))
               ArgusLog(LOG_ERR, "%s: host %s unknown", *argv, optarg);
            break;

         case 't': {
            ArgusParser->timearg = strdup(optarg);
            if (ArgusParser->timearg != NULL) {
               if ((retn = ArgusParseTimeArg (&ArgusParser->timearg, argv, optind, ArgusParser->RaTmStruct)) < 0) {
                  usage ();
               } else {
                  ++ArgusParser->tflag; 
                  optind += retn;
               }
            }
            break;
         }

         case 'T': ArgusParser->Tflag = atoi(optarg); break;

         case 'u': { 
            if (!(strncmp(ArgusParser->ArgusProgramName, "radium", 6))) {
               char login[256];
               struct passwd *pw;  
               sprintf (login, "%s", optarg);
               if ((pw = getpwnam(login)) == NULL)  
                  ArgusLog (LOG_ERR, "unknown user \"%s\"\n", optarg);
               new_uid = pw->pw_uid;
               endpwent();
            } else {
               ArgusParser->uflag++;
            }
            break;
         }

         case 'U':
            if (strstr(ArgusParser->ArgusProgramName, "sql") != NULL)
               ArgusParser->dbustr = strdup (optarg);
            else
               ArgusParser->ustr = strdup(optarg);
            break;

         case 'v': ArgusParser->vflag++; break;
         case 'V': ArgusParser->Vflag++; break;
         case 'w':  
            if ((tmparg = optarg) != NULL) {
               if ((*tmparg != '-') || ((*tmparg == '-') &&
                                       (!(strcmp (tmparg, "-"))))) {
                  if (argc == optind)
                     filter = NULL;
                  else {
                     filter = argv[optind];
                     if (*filter == '-') {
                        filter = NULL;
                     } else
                        optind++;
                     }
                  setArgusWfile (ArgusParser, tmparg, filter);
                  break;
               }
            }
            break;

	 case 'x': ++ArgusParser->xflag; break;
         case 'X': RaClearConfiguration (ArgusParser); break;
	 case 'z': ++ArgusParser->zflag; break;
	 case 'Z': ArgusParser->Zflag = *optarg; break;
         case 'h':
            default:  
               usage ();
            /* NOTREACHED */
      }
   }

   if (rcmdline)
      if (ArgusParser->ArgusInputFileList == NULL)
         ArgusLog (LOG_ERR, "no input files");
 
   if ((str = argv[optind]) != NULL) {
      if ((strcmp(str, "-") == 0) || (strcmp(str, "--") == 0))
         optind++;
      cmdbuf = ArgusCopyArgv (&argv[optind]);
   }

   if (cmdbuf) {
      if (ArgusParser->ArgusLocalFilter != NULL)
         free(ArgusParser->ArgusLocalFilter);

      if (ArgusParser->ArgusRemoteFilter != NULL)
         free(ArgusParser->ArgusRemoteFilter);

      if ((str = strstr (cmdbuf, "local ")) != NULL) {
         *str = '\0';
         ArgusParser->ArgusLocalFilter = strdup(&cmdbuf[strlen("local ")]);
      } else 
      if ((str = strstr (cmdbuf, "display ")) != NULL) {
         *str = '\0';
         ArgusParser->ArgusDisplayFilter = strdup(&cmdbuf[strlen("display ")]);
      } else 
      if ((str = strstr (cmdbuf, "remote ")) != NULL) {
         *str = '\0';
         ArgusParser->ArgusRemoteFilter = strdup(&cmdbuf[strlen("remote ")]);
      } else
         ArgusParser->ArgusRemoteFilter = strdup(cmdbuf);

      free(cmdbuf);
   }

   if (ArgusParser->RaSOptionIndex > 0)
      ArgusProcessSOptions(ArgusParser);
 
   if (ArgusParser->ArgusRemoteFilter != NULL)
      if (ArgusFilterCompile (&ArgusParser->ArgusFilterCode, ArgusParser->ArgusRemoteFilter, ArgusParser->Oflag) < 0)
         ArgusLog (LOG_ERR, "%s filter syntax error", ArgusParser->ArgusRemoteFilter);

   if (ArgusParser->ArgusLocalFilter != NULL)
      if (ArgusFilterCompile (&ArgusParser->ArgusFilterCode, ArgusParser->ArgusLocalFilter, ArgusParser->Oflag) < 0)
         ArgusLog (LOG_ERR, "%s filter syntax error", ArgusParser->ArgusLocalFilter);

   if (ArgusParser->ArgusDisplayFilter != NULL)
      if (ArgusFilterCompile (&ArgusParser->ArgusDisplayCode, ArgusParser->ArgusDisplayFilter, ArgusParser->Oflag) < 0)
         ArgusLog (LOG_ERR, "%s filter syntax error", ArgusParser->ArgusDisplayFilter);

   if (ArgusParser->bflag) {
      if ((ArgusParser->ArgusLocalFilter != NULL) || (ArgusParser->ArgusRemoteFilter != NULL)) {
         nff_dump(&ArgusParser->ArgusFilterCode, ArgusParser->bflag);
         exit (0);
      }
   }

   if (ArgusParser->RaParseDone)
      exit (0);
}
