/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * rastream - time based stream processor. 
 *    this routine will take in an argus stream and align it to
 *    to a time array, and hold it for a hold period, and then
 *    output the bin contents as an argus stream, splitting into
 *    an output strategy like rasplit().  when a file is done,
 *    rastream() closes the file, and then forks whatever
 *    program is specified on the commandline.
 *
 *    this is the primary stream block processor for Argus.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: //depot/gargoyle/gargoyle-4.0/clients/clients/rastream.c#23 $
 * $DateTime: 2007/02/09 14:41:23 $
 * $Change: 1034 $
 */

#if defined(CYGWIN)
#define USE_IPV6
#endif

#include <math.h>

#include <unistd.h>
#include <stdlib.h>

#include <argus_util.h>
#include <argus_client.h>
#include <argus_main.h>
#include <argus_filter.h>

#include <rabins.h>
#include <rasplit.h>
#include <argus_sort.h>
#include <argus_cluster.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <signal.h>
#include <ctype.h>

int RaRealTime = 0;
float RaUpdateRate = 1.0;
 
struct timeval ArgusLastRealTime = {0, 0};
struct timeval ArgusLastTime     = {0, 0};
struct timeval ArgusThisTime     = {0, 0};
 
 
struct timeval dLastTime = {0, 0};
struct timeval dRealTime = {0, 0};
struct timeval dThisTime = {0, 0};
struct timeval dTime     = {0, 0};

long long thisUsec = 0;

#define ARGUS_SCRIPT_TIMEOUT            30

struct ArgusScriptStruct {
   struct ArgusListObjectStruct *nxt;
   struct ArgusWfileStruct *file;
   char *script, *filename, *cmd;
   char *args[8];
   struct timeval startime;
   int timeout;
   pid_t pid;
};
 
struct ArgusProbeStruct {
   struct ArgusQueueHeader qhdr;
   struct ArgusHashTableHdr *htblhdr;
   struct ArgusTransportStruct trans;
   unsigned int status;
   struct timeval start;
   struct ArgusAdjustStruct nadp;
   struct RaBinProcessStruct *rbp;
};

struct ArgusListStruct *ArgusScriptList = NULL;
struct ArgusScriptStruct *ArgusCurrentScript = NULL;


struct RaBinProcessStruct *ArgusBinProcess = NULL;
struct RaBinProcessStruct *RaBinProcess = NULL;

int ArgusRunScript (struct ArgusParserStruct *, struct ArgusWfileStruct *);
int ArgusSetCurrentWfile (struct ArgusParserStruct *, struct RaBinProcessStruct *, struct ArgusRecordStruct *);
struct ArgusWfileStruct *ArgusFindFile (struct ArgusHashTable *, struct ArgusHashStruct *);

struct ArgusAdjustStruct RaStreamDefaultNadp, *RaStreamNadp;
struct RaBinProcessStruct *RaNewBinProcess (struct ArgusParserStruct *, int);

int ArgusRmonMode = 0;


struct RaBinProcessStruct *
RaNewBinProcess (struct ArgusParserStruct *parser, int size)
{
   struct RaBinProcessStruct *retn = NULL;
   struct ArgusAdjustStruct *tnadp;
 
   parser->ArgusReverse = 0;

   if ((retn = (struct RaBinProcessStruct *)ArgusCalloc(1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "RaNewBinProcess: ArgusCalloc error %s", strerror(errno));

   tnadp = &retn->nadp;
   bcopy((char *)&RaStreamDefaultNadp, (char *)tnadp, sizeof(*tnadp));

   tnadp->mode    = -1;
   tnadp->modify  =  1;
   tnadp->slen    =  2;
   tnadp->count   = 1;
   tnadp->value   = 1;

   return (retn);
}


struct ArgusHashTable ArgusProbeTable;
struct ArgusQueueStruct *ArgusProbeQueue = NULL;

void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusAdjustStruct *nadp = &RaStreamDefaultNadp;
   struct ArgusModeStruct *mode = NULL;
   char outputfile[MAXSTRLEN];
   int i = 0, ind = 0, size = 1;

   parser->RaWriteOut = 0;
   *outputfile = '\0';

   if (!(parser->RaInitialized)) {

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);

      if (parser->Sflag)
         parser->ArgusReliableConnection++;

      if (parser->dflag) {
         int pid;

         ArgusLog(LOG_WARNING, "started");
         if (chdir ("/") < 0)
            ArgusLog (LOG_ERR, "Can't chdir to / %s", strerror(errno));

         if ((pid = fork ()) < 0) {
            ArgusLog (LOG_ERR, "Can't fork daemon %s", strerror(errno));
         } else {
            if (pid) {
               struct timespec ts = {0, 20000000};
               int status;
               nanosleep(&ts, NULL);   
               waitpid(pid, &status, WNOHANG);
               if (kill(pid, 0) < 0) {
                  exit (1);
               } else
                  exit (0);
            } else {
               parser->ArgusSessionId = setsid();
               fclose (stdin);
               fclose (stdout);
               fclose (stderr);
            }
         }
      }

      bzero ((char *)nadp, sizeof(*nadp));
      nadp->mode      = -1;
      nadp->modify    =  1;
      nadp->slen      =  2;

      if (parser->vflag)
         ArgusReverseSortDir++;

      parser->RaCumulativeMerge = 0;
 
      if ((ArgusSorter = ArgusNewSorter()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewSorter error %s", strerror(errno));

      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            if (isdigit((int) *mode->mode)) {
               ind = 0;
            } else {
               if (!(strncasecmp (mode->mode, "rtime", 5)) ||
                  (!(strncasecmp (mode->mode, "realtime", 8)))) {
                  char *ptr = NULL;
                  RaRealTime++;
                  if ((ptr = strchr(mode->mode, ':')) != NULL) {
                     double value = 0.0;
                     char *endptr = NULL;
                     ptr++;
                     value = strtod(ptr, &endptr);
                     if (ptr != endptr) {
                        RaUpdateRate = value;
                     }
                  }
               } else

               if (!(strncasecmp (mode->mode, "rmon", 4))) {
                  ArgusRmonMode++;
               } else {
                  for (i = 0, ind = -1; i < ARGUSSPLITMODENUM; i++) {
                     if (!(strncasecmp (mode->mode, RaSplitModes[i], 3))) {
                        ind = i;
                        switch (ind) {
                           case ARGUSSPLITTIME:
                           case ARGUSSPLITSIZE:
                           case ARGUSSPLITCOUNT:
                              if ((mode = mode->nxt) == NULL)
                                 usage();
                              break;
                        }
                     }
                  }
               }
            }

            if (ind < 0)
               usage();

            switch (ind) {
               case ARGUSSPLITTIME:
                  nadp->mode = ind;
                  if (isdigit((int)*mode->mode)) {
                     char *ptr = NULL;
                     nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                     if (ptr == mode->mode)
                        usage();
                     else {
                        switch (*ptr) {
                           case 'y':
                              nadp->qual = ARGUSSPLITYEAR;  
                              size = nadp->value * 31556926;
                              break;
                           case 'M':
                              nadp->qual = ARGUSSPLITMONTH; 
                              size = nadp->value * 2629744;
                              break;
                           case 'w':
                              nadp->qual = ARGUSSPLITWEEK;  
                              size = nadp->value * 604800;
                              break;
                           case 'd':
                              nadp->qual = ARGUSSPLITDAY;   
                              size = nadp->value * 86400;
                              break;
                           case 'h':
                              nadp->qual = ARGUSSPLITHOUR;  
                              size = nadp->value * 3600;
                              break;
                           case 'm':
                              nadp->qual = ARGUSSPLITMINUTE;
                              size = nadp->value * 60;
                              break;
                            default:
                              nadp->qual = ARGUSSPLITSECOND;
                              size = nadp->value;
                              break;
                        }
                     }
                  }
                  ArgusSorter->ArgusSortAlgorithms[0] = ArgusSortStartTime;
                  break;

               case ARGUSSPLITSIZE:
               case ARGUSSPLITCOUNT:
                  nadp->mode = ind;
                  nadp->count = 1;

                  if (isdigit((int)*mode->mode)) {
                     char *ptr = NULL;
                     nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                     if (ptr == mode->mode)
                        usage();
                     else {
                        switch (*ptr) {
                           case 'B':   
                           case 'b':  nadp->value *= 1000000000; break;
                            
                           case 'M':   
                           case 'm':  nadp->value *= 1000000; break;
                            
                           case 'K':   
                           case 'k':  nadp->value *= 1000; break;
                        }
                     }
                  }
                  ArgusSorter->ArgusSortAlgorithms[0] = NULL;
                  break;

               case ARGUSSPLITNOMODIFY:
                  nadp->modify = 0;
                  break;

               case ARGUSSPLITSOFT:
                  nadp->soft++;
                  break;

               case ARGUSSPLITZERO:
                  nadp->zero++;
                  break;
            }

            mode = mode->nxt;
         }
      }

      if (nadp->mode < 0) {
          nadp->modify = 0;
          nadp->count = 1;
          nadp->value = 1;
      }

      if ((ArgusBinProcess = RaNewBinProcess(parser, size)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusCalloc error %s", strerror(errno));
 
      ArgusBinProcess->size  = size;

      nadp->count = ((size > parser->Bflag) ? size : parser->Bflag);
      bcopy ((char *) nadp, (char *)&ArgusBinProcess->nadp, sizeof (*nadp));

      /* if content substitution, either time or any field, is used,
         size and count modes will not work properly.  If using
         the default count, set the value so that we generate only
         one filename.

         if no substitution, then we need to add "aa" suffix to the
         output file for count and size modes.
      */


      if (parser->ArgusWfileList != NULL) {
         struct ArgusWfileStruct *wfile = NULL;
         int count = parser->ArgusWfileList->count;

         if (count > 1)
            usage();

         if ((wfile = (struct ArgusWfileStruct *)ArgusPopFrontList(parser->ArgusWfileList, ARGUS_LOCK)) != NULL) {
            if (strcmp(wfile->filename, "-")) {
               strncpy (outputfile, wfile->filename, MAXSTRLEN);

               if (!(strstr(outputfile, "$srcid")))
                  ArgusLog(LOG_ERR, "output string requires $srcid");
         
               if ((strchr(outputfile, '%')) || (strchr(outputfile, '$'))) {
                  switch (nadp->mode) {
                     case ARGUSSPLITCOUNT:
                        nadp->count = -1;
                        break;

                     case ARGUSSPLITSIZE:
                        for (i = 0; i < nadp->slen; i++) 
                           strcat(outputfile, "a");
                        break;
                  }

               } else {
                  switch (nadp->mode) {
                     case ARGUSSPLITSIZE:
                     case ARGUSSPLITCOUNT:
                        for (i = 0; i < nadp->slen; i++) 
                           strcat(outputfile, "a");
                        break;
                  }
               }

               if (!(strchr(outputfile, '%'))) {
                  switch (nadp->mode) {
                     case ARGUSSPLITTIME:
                       break;
                  }
               }

               nadp->filename = strdup(outputfile);
               setArgusWfile (parser, outputfile, NULL);

            } else
               setArgusWfile (parser, "-", NULL);
         }
      }

      parser->RaClientTimeout.tv_sec  = 0;
      parser->RaClientTimeout.tv_usec = 330330;
      parser->RaInitialized++;

      if (ArgusParser->startime_t && ArgusParser->lasttime_t) {
         ArgusBinProcess->startpt.tv_sec  = ArgusParser->startime_t;
         ArgusBinProcess->startpt.tv_usec = 0;

         ArgusBinProcess->endpt.tv_sec    = ArgusParser->lasttime_t;
         ArgusBinProcess->endpt.tv_usec   = 0;

      } else {
         if (ArgusParser->Bflag > 0) {
            ArgusBinProcess->rtime.tv_sec = ArgusParser->ArgusRealTime.tv_sec;
         }
      }

      if (parser->Gflag) {
         parser->uflag++;
         parser->RaFieldDelimiter = ',';
      }

      for (i = 0; parser->RaPrintAlgorithmList[i] != NULL; i++) {
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintProto) {
            parser->RaPrintMode |= RA_PRINTPROTO;
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintSrcPort) {
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintDstPort) {
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintSourceID) {
            parser->RaPrintMode |= RA_PRINTSRCID;
            break;
         }
      }

      if ((parser->ArgusAggregator = ArgusNewAggregator(parser, NULL)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");

      if ((ArgusProbeQueue = ArgusNewQueue()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewQueue error");
 
      ArgusProbeTable.size  = 1024;
      if ((ArgusProbeTable.array = (struct ArgusHashTableHdr **)
                  ArgusCalloc (1024, sizeof (struct ArgusHashTableHdr))) == NULL)
         ArgusLog (LOG_ERR, "RaTimeInit: ArgusCalloc error %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientInit()\n");
#endif
}


void RaArgusInputComplete (struct ArgusInput *input) { return; }

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int i;
   
   if (sig >= 0)
      if ((sig == SIGINT) || (sig == SIGQUIT))
         exit(0);

   if (!(RaParseCompleting++)) {
      struct ArgusProbeStruct *probe = NULL;
 
      if ((probe = (struct ArgusProbeStruct *) ArgusProbeQueue->start) != NULL) {
         do {
            if (probe->start.tv_sec > 0) {
               RaBinProcess = probe->rbp;

               if (RaBinProcess != NULL) {
                  struct RaBinStruct *bin = NULL;
                  struct ArgusRecordStruct *ns = NULL;

                  if ((ArgusParser->ArgusWfileList == NULL) && (ArgusParser->Gflag)) {
                     if (ArgusParser->Hstr) {

                     } else {
                        char stimebuf[128], dtimebuf[128];

                        ArgusPrintTime(ArgusParser, stimebuf, &RaBinProcess->startpt);
                        ArgusPrintTime(ArgusParser, dtimebuf, &RaBinProcess->endpt);
                        stimebuf[strlen(stimebuf) - 1] = '\0';
                        dtimebuf[strlen(dtimebuf) - 1] = '\0';

                        printf ("StartTime=%s\n", stimebuf);
                        printf ("StopTime=%s\n", dtimebuf);
                        printf ("Seconds=%d\n", RaBinProcess->scalesecs);
                        printf ("BinSize=%1.*f\n", ArgusParser->pflag, RaBinProcess->size * 1.0);
                        printf ("Bins=%d\n", (RaBinProcess->scalesecs + (RaBinProcess->size - 1))/RaBinProcess->size);
                     }
                  }

                  for (i = 0; i < RaBinProcess->arraylen; i++) {
                     if ((bin = RaBinProcess->array[i]) != NULL) {
                        ArgusSortQueue(ArgusSorter, bin->queue);
                        while ((ns = (struct ArgusRecordStruct *) ArgusPopQueue(bin->queue, ARGUS_LOCK)) != NULL) {
                           if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList))))
                              ArgusSetCurrentWfile (ArgusParser, RaBinProcess, ns);
 
                           RaSendArgusRecord (ns);
                           ArgusDeleteRecordStruct(ArgusParser, ns);
                        }

                     } else {
                        if (RaBinProcess->nadp.zero && ((i >= RaBinProcess->index) && (((i - RaBinProcess->index) * RaBinProcess->size) < RaBinProcess->scalesecs))) {
                           ns = ArgusGenerateRecordStruct(NULL, NULL, NULL);
                           ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.start.tv_sec  = RaBinProcess->start + (RaBinProcess->size * (i - RaBinProcess->index));
                           ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.start.tv_usec = 0;
                           ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.end.tv_sec    = RaBinProcess->start + (RaBinProcess->size * ((i + 1) - RaBinProcess->index));
                           ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.end.tv_usec   = 0;

                           RaSendArgusRecord (ns);
                           ArgusDeleteRecordStruct(ArgusParser, ns);
                        }
                     }
                  }
               }
            }

            probe = (struct ArgusProbeStruct *) probe->qhdr.nxt;

         } while (probe != (struct ArgusProbeStruct *) ArgusProbeQueue->start);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParseComplete(%d)\n", sig);
#endif
}


void
ArgusClientTimeout ()
{
   struct ArgusRecordStruct *ns = NULL;
   struct RaBinStruct *bin = NULL;
 
   if (RaRealTime) {  /* establish value for time comparison */
      gettimeofday(&ArgusParser->ArgusRealTime, 0);
      ArgusAdjustGlobalTime(ArgusParser, &ArgusParser->ArgusRealTime);

      if (ArgusLastTime.tv_sec != 0) {
         if (ArgusLastRealTime.tv_sec > 0) {
            dRealTime = *RaDiffTime(&ArgusParser->ArgusRealTime, &ArgusLastRealTime);
            thisUsec = ((dRealTime.tv_sec * 1000000) + dRealTime.tv_usec) * RaUpdateRate;
            dRealTime.tv_sec  = thisUsec / 1000000;
            dRealTime.tv_usec = thisUsec % 1000000;
 
 
            ArgusLastTime.tv_sec  += dRealTime.tv_sec;
            ArgusLastTime.tv_usec += dRealTime.tv_usec;
 
            if (ArgusLastTime.tv_usec > 1000000) {
               ArgusLastTime.tv_sec++;
               ArgusLastTime.tv_usec -= 1000000;
            }
         }
 
         ArgusLastRealTime = ArgusParser->ArgusRealTime;
      }
   }

   if (ArgusScriptList) {
      struct ArgusScriptStruct *script = NULL;
      int retn = 0, status;

      if ((script = ArgusCurrentScript) != NULL) {
         if (script->pid > 0) {
            if ((retn = waitpid(script->pid, &status, WNOHANG)) != -1) {
#ifdef ARGUSDEBUG
               ArgusDebug (1, "ArgusClientTimeout(): waitpid(%d) returned %d", script->pid, retn);
#endif
               if (WIFEXITED(status)) {
#ifdef ARGUSDEBUG
                  ArgusDebug (1, "ArgusTask(%d): task %s completed", script->pid, script->cmd);
#endif
                  if (script->filename)
                     free(script->filename);
                  if (script->script)
                     free(script->script);
                  if (script->cmd)
                     free(script->cmd);
                  ArgusFree(script);
                  ArgusCurrentScript = NULL;

               }
            }
         }
      }

      if (ArgusCurrentScript == NULL) {
         if ((script = (struct ArgusScriptStruct *) ArgusFrontList(ArgusScriptList)) != NULL) {
            ArgusPopFrontList(ArgusScriptList, ARGUS_LOCK); 

            if ((script->pid = fork()) < 0)
               ArgusLog (LOG_ERR, "ArgusRunScript (%s) fork() error %s\n", script->cmd, strerror(errno));

            if (script->pid > 0) {
               ArgusCurrentScript = script;
            } else {
#ifdef ARGUSDEBUG
               ArgusDebug (1, "ArgusRunScript calling %s", script->cmd);
#endif
               exit(execv(script->script, script->args));
            }
         }
      }

/*
      for (i = 0, count = ArgusScriptList->count; i < count; i++) {
         int done = 0;

         if ((script = (struct ArgusScriptStruct *) ArgusFrontList(ArgusScriptList)) != NULL) {
            ArgusPopFrontList(ArgusScriptList, ARGUS_LOCK); 
            if (script->pid > 0) {
               if ((retn = kill(script->pid, 0)) < 0) {
                  if (errno != ESRCH)
                     ArgusLog (LOG_WARNING, "ArgusClientTimeout: script %d kill error %s", script->pid, strerror(errno));
                  done++;
               } else {
                  if ((waitpid(script->pid, &status, 0)) == script->pid)
                     done++;
               }
#ifdef ARGUSDEBUG
               ArgusDebug (1, "ArgusClientTimeout(): kill(%d, 0) returned %d", script->pid, retn);
#endif
            } else {
               ArgusLog (LOG_WARNING, "ArgusClientTimeout: script list contains pid %d", script->pid);
               done++;
            }

            if (done) {
               if (script->file) {
                  free(script->file->filename);
                  ArgusFree(script->file);
               } else
                  if (script->filename)
                     free(script->filename);
               if (script->script)
                  free(script->script);
               if (script->cmd)
                  free(script->cmd);
               ArgusFree(script);

            } else 
               ArgusPushBackList(ArgusScriptList, (struct ArgusListRecord *) script, ARGUS_LOCK);
         }
      }
*/
   }

   if (ArgusParser->Bflag > 0) {
      struct ArgusProbeStruct *probe = NULL;
      int i, count = 1;

      if ((probe = (struct ArgusProbeStruct *) ArgusProbeQueue->start) != NULL) {
         do {
            struct RaBinProcessStruct *rbps = NULL;

            rbps = probe->rbp;

            if (rbps->rtime.tv_sec) {
               if ((RaDiffTime(&ArgusParser->ArgusRealTime, &rbps->rtime))->tv_sec >= ArgusParser->Bflag) {
                  count = RaBinProcess->arraylen - RaBinProcess->index;

                  if (rbps->array != NULL) {

                     if ((bin = rbps->array[rbps->index]) != NULL) {
                        ArgusSortQueue(ArgusSorter, bin->queue);

                        while ((ns = (struct ArgusRecordStruct *) ArgusPopQueue(bin->queue, ARGUS_LOCK)) != NULL) {
                           if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList))))
                              ArgusSetCurrentWfile (ArgusParser, RaBinProcess, ns);

                           RaSendArgusRecord (ns);
                           ArgusDeleteRecordStruct(ArgusParser, ns);
                        }

                        RaDeleteBin(ArgusParser, bin);

                     } else {
                        if (rbps->nadp.zero && ((i >= rbps->index) && (((i - rbps->index) * rbps->size) < rbps->scalesecs))) {
                           ns = ArgusGenerateRecordStruct(NULL, NULL, NULL);
                           ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->start.tv_sec  = rbps->start + (rbps->size * (i - rbps->index));
                           ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->start.tv_usec = 0;
                           ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->end.tv_sec    = rbps->start + (rbps->size * ((i + 1) - rbps->index));
                           ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->end.tv_usec   = 0;

                           ns->dsrs[ARGUS_TRANSPORT_INDEX] = &ns->canon.trans.hdr;

                           RaSendArgusRecord (ns);
                           ArgusDeleteRecordStruct(ArgusParser, ns);
#ifdef ARGUSDEBUG
                           ArgusDebug (2, "ArgusClientTimeout() RaBinProcess: creating zero record\n");
#endif
                        }
                     }

                     for (i = 0; i < count; i++)
                        rbps->array[rbps->index + i] = rbps->array[rbps->index + (i + 1)];

                     rbps->start += rbps->size;
                     rbps->end += rbps->size;
                     rbps->array[rbps->index + count] = NULL;
                     rbps->startpt.tv_sec  += rbps->size;
                     rbps->endpt.tv_sec    += rbps->size;

                     if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
                        struct ArgusWfileStruct *wfile = NULL;
                        struct ArgusListObjectStruct *lobj = NULL;
                        int i, count = ArgusParser->ArgusWfileList->count;

                        if ((lobj = ArgusParser->ArgusWfileList->start) != NULL) {
                           for (i = 0; i < count; i++) {
                              if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
                                 if (wfile->fd != NULL)
                                    fflush(wfile->fd);
                              }
                              lobj = lobj->nxt;
                           }
                        }
                     }
#ifdef ARGUSDEBUG
                     ArgusDebug (2, "ArgusClientTimeout() RaBinProcess: probe %s shifting array rtime %d start %d end %d count %d index %d\n",
                        ArgusGetName (ArgusParser, (unsigned char *)&probe->trans.srcid.a_un.value), rbps->rtime.tv_sec,
                        rbps->startpt.tv_sec, rbps->endpt.tv_sec, rbps->count, rbps->index);
#endif
                  }

                  rbps->rtime.tv_sec += rbps->size;
               }

            } else
               rbps->rtime = ArgusParser->ArgusRealTime;

            probe = (struct ArgusProbeStruct *) probe->qhdr.nxt;

         } while (probe != (struct ArgusProbeStruct *) ArgusProbeQueue->start);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusClientTimeout() done");
#endif
}


void parse_arg (int argc, char**argv) {}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rastream Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -b                 dump packet-matching code.\n");
   fprintf (stderr, "         -C <[host]:port>   specify remote Cisco Netflow source.\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   fprintf (stderr, "         -E <file>          write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                 print help.\n");

   fprintf (stderr, "         -M <mode>          supported modes of operation:\n");
   fprintf (stderr, "            time n[smhdwmy] split output into time series bins\n");
   fprintf (stderr, "            nomodify        don't modify input records\n");
   fprintf (stderr, "            zero            generate zero records for gaps\n");

   fprintf (stderr, "         -m <mode>          supported aggregation objects:\n");
   fprintf (stderr, "           none             no flow key\n");
   fprintf (stderr, "           saddr            include the source address\n");
   fprintf (stderr, "           daddr            include the destination address\n");
   fprintf (stderr, "           proto            include the destination proto\n");
   fprintf (stderr, "           sport            include the source port\n");
   fprintf (stderr, "           dport            include the destination port\n");
   fprintf (stderr, "           srcid            include the source identifier\n");

   fprintf (stderr, "         -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]>   specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                   format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 [yyyy/]mm/dd\n");
   fprintf (stderr, "                                                 -%%d{yMdhms}\n");
   fprintf (stderr, "         -T <secs>          attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>          write output to <file>. '-' denotes stdout.\n");
   exit(1);
}

void RaProcessThisRecord (struct ArgusParserStruct *, struct ArgusRecordStruct *);



void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   struct ArgusRecordStruct *tns = NULL;
   struct ArgusProbeStruct *probe = NULL;
   struct ArgusHashTableHdr *htbl = NULL;
   unsigned short *sptr;
                                                                                                                    
   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      struct ArgusHashStruct ArgusHash;
      unsigned int key = ns->canon.trans.srcid.a_un.value;
      int i, len, s = sizeof(short);

      ArgusThisTime.tv_sec  = ns->canon.time.src.start.tv_sec;
      ArgusThisTime.tv_usec = ns->canon.time.src.start.tv_usec;

      if (RaRealTime) {
         if (ArgusLastTime.tv_sec == 0)
            ArgusLastTime = ArgusThisTime;

         if (!((ArgusLastTime.tv_sec  > ArgusThisTime.tv_sec) ||
            ((ArgusLastTime.tv_sec == ArgusThisTime.tv_sec) &&
             (ArgusLastTime.tv_usec > ArgusThisTime.tv_usec)))) {

            while ((ArgusThisTime.tv_sec  > ArgusLastTime.tv_sec) ||
                  ((ArgusThisTime.tv_sec == ArgusLastTime.tv_sec) &&
                   (ArgusThisTime.tv_usec > ArgusLastTime.tv_usec))) {
               struct timespec ts = {0, 0};
               int thisRate;

               dThisTime = *RaDiffTime(&ArgusThisTime, &ArgusLastTime);
               thisRate = ((dThisTime.tv_sec * 1000000) + dThisTime.tv_usec)/RaUpdateRate;
               thisRate = (thisRate > 100000) ? 100000 : thisRate;

               ts.tv_nsec = thisRate * 1000;
               nanosleep (&ts, NULL);

               ArgusClientTimeout ();

               gettimeofday(&parser->ArgusRealTime, 0);

               if (ArgusLastRealTime.tv_sec > 0) {
                  dRealTime = *RaDiffTime(&parser->ArgusRealTime, &ArgusLastRealTime);
                  thisUsec = ((dRealTime.tv_sec * 1000000) + dRealTime.tv_usec) * RaUpdateRate;
                  dRealTime.tv_sec  = thisUsec / 1000000;
                  dRealTime.tv_usec = thisUsec % 1000000;

                  ArgusLastTime.tv_sec  += dRealTime.tv_sec;
                  ArgusLastTime.tv_usec += dRealTime.tv_usec;
                  if (ArgusLastTime.tv_usec > 1000000) {
                     ArgusLastTime.tv_sec++;
                     ArgusLastTime.tv_usec -= 1000000;
                  }
               }
               ArgusLastRealTime = parser->ArgusRealTime;
            }
         }
      } else
         ArgusLastTime = parser->ArgusRealTime;
 
      bzero ((char *)&ArgusHash, sizeof(ArgusHash));
      ArgusHash.len = 4;
      ArgusHash.buf = &key;

      sptr = (unsigned short *) ArgusHash.buf;
      for (i = 0, len = ArgusHash.len / s; i < len; i++)
         ArgusHash.hash += *sptr++;

      if ((htbl = ArgusFindHashEntry(&ArgusProbeTable, &ArgusHash)) == NULL) {
         if ((probe = (struct ArgusProbeStruct *) ArgusCalloc (1, sizeof(*probe))) != NULL) {
            probe->htblhdr = ArgusAddHashEntry (&ArgusProbeTable, (void *)probe, &ArgusHash);
            ArgusAddToQueue(ArgusProbeQueue, &probe->qhdr, ARGUS_LOCK);

            if ((probe->rbp = RaNewBinProcess(parser, ArgusBinProcess->size)) == NULL)
               ArgusLog (LOG_ERR, "RaProcessRecord: RaNewBinProcess error %s\n", strerror(errno));

            bcopy ((char *)&ns->canon.trans, (char *)&probe->trans, sizeof(probe->trans)); 

            probe->rbp->size = 1;
            bcopy ((char *)&RaStreamDefaultNadp, (char *)&probe->nadp, sizeof(probe->nadp));
            bcopy ((char *)&RaStreamDefaultNadp, (char *)&probe->rbp->nadp, sizeof(probe->nadp));
 
#ifdef ARGUSDEBUG
            ArgusDebug (2, "RaProcessRecord(0x%x, 0x%x) New Probe %s", parser, ns,
                 ArgusGetName (parser, (unsigned char *)&key));
#endif

         } else
            ArgusLog (LOG_ERR, "RaProcessRecord: ArgusCalloc error %s\n", strerror(errno));
      } else
         probe = htbl->object;

      if (probe->start.tv_sec == 0)
         probe->start = ArgusParser->ArgusRealTime;

      RaBinProcess = probe->rbp;
      RaStreamNadp = &probe->nadp;

      if (ArgusRmonMode) {
         if ((tns = ArgusCopyRecordStruct(ns)) != NULL) {
            RaProcessThisRecord (parser, ns);
            ArgusReverseRecord(tns);
            RaProcessThisRecord (parser, tns);
            ArgusDeleteRecordStruct (parser, tns);
         }

      } else
         RaProcessThisRecord (parser, ns);
   }
}



void
RaProcessThisRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   extern struct RaBinProcessStruct *RaBinProcess;
   struct ArgusRecordStruct *tns = NULL;
   int retn;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      if (!parser->Sflag)
         ArgusClientTimeout ();

      RaBinProcess->nadp.tperiod = 0.0;
      RaStreamNadp->tperiod = 0.0;

      while ((tns = ArgusAlignRecord(parser, ns, RaStreamNadp)) != NULL) {
         if ((tns->canon.metric.src.pkts + tns->canon.metric.dst.pkts) > 0) {
            time_t secs;

            secs = tns->canon.time.src.start.tv_sec;
            localtime_r(&secs, &RaBinProcess->nadp.RaStartTmStruct);

            secs = tns->canon.time.src.end.tv_sec;
            localtime_r(&secs, &RaBinProcess->nadp.RaEndTmStruct);

            if ((retn = ArgusInsertRecord(parser, RaBinProcess, tns, parser->Bflag)) < 0)
               ArgusDeleteRecordStruct(parser, tns);

         } else 
            ArgusDeleteRecordStruct(parser, tns);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessThisRecord (0x%x) done\n", ns); 
#endif
}


int
RaSendArgusRecord(struct ArgusRecordStruct *argus)
{
   int retn = 1;
   char buf[MAXSTRLEN];

   if (!(retn = ArgusCheckTime (ArgusParser, argus)))
      return (retn);
 
   if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
      struct ArgusWfileStruct *wfile = NULL;
      struct ArgusListObjectStruct *lobj = NULL;
      int i, count = ArgusParser->ArgusWfileList->count;

      if ((lobj = ArgusParser->ArgusWfileList->start) != NULL) {
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
               int pass = 1;
               if (wfile->filterstr) {
                  struct nff_insn *wfcode = wfile->filter.bf_insns;
                  pass = ArgusFilterRecord (wfcode, argus);
               }

               if (pass != 0) {
                  if ((ArgusParser->exceptfile == NULL) || strcmp(wfile->filename, ArgusParser->exceptfile)) {
                     struct ArgusRecord *argusrec = NULL;
                     char sbuf[0x10000];
                     if ((argusrec = ArgusGenerateRecord (argus, 0L, sbuf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argusrec);
#endif
                        ArgusWriteNewLogfile (ArgusParser, argus->input, wfile, argusrec);
                     }
                  }
               }
            }
            lobj = lobj->nxt;
         }
      }

   } else {
      if (!ArgusParser->qflag) {
         if (ArgusParser->Lflag) {
            if (ArgusParser->RaLabel == NULL)
               ArgusParser->RaLabel = ArgusGenerateLabel(ArgusParser, argus);

            if (!(ArgusParser->RaLabelCounter++ % ArgusParser->Lflag)) {
               if (ArgusParser->Gflag)
                  printf ("Columns=%s\n", ArgusParser->RaLabel);
               else
                  printf ("%s", ArgusParser->RaLabel);
            }

            if (ArgusParser->Lflag < 0)
               ArgusParser->Lflag = 0;

            if (ArgusParser->Gflag) {
               switch (ArgusParser->RaPrintMode) {
                  case RA_PRINTSRCID:
                     printf ("Probes=\n");
                     break;

                  case RA_PRINTPROTO: {
                     printf ("Protos=\n");
                     break;
                  }
               }
            }
            printf ("\n");
         }

         *(int *)&buf = 0;
         ArgusPrintRecord(ArgusParser, buf, argus, MAXSTRLEN);
         fprintf (stdout, "%s\n", buf);
         fflush(stdout);
      }
   }

   return (retn);
}


void ArgusWindowClose(void);

void ArgusWindowClose(void) { 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusWindowClose () returning\n"); 
#endif
}


char *
RaSplitFilename (struct ArgusAdjustStruct *nadp)
{
   char *retn = NULL, tmpbuf[MAXSTRLEN];
   char *filename = nadp->filename;
   int len, i = 1, carry = 0;

   if (filename != NULL) {
      len = strlen(filename);

      for (i = 0; i < nadp->slen; i++)
         if (filename[len - (i + 1)] == 'z')
            carry++;

      if ((carry == (nadp->slen - 1)) && (filename[len - nadp->slen] == 'y')) {
         strncpy(tmpbuf, filename, MAXSTRLEN);
         tmpbuf[strlen(tmpbuf) - nadp->slen] = 'z';
         for (i = 0; i < nadp->slen; i++)
            strcat(tmpbuf, "a");
         nadp->slen++;

      } else {
         for (i = 0, carry = 0; i < nadp->slen; i++) {
            if (filename[len - (i + 1)] == 'z') {
               filename[len - (i + 1)] = 'a';
            } else {
               filename[len - (i + 1)]++;
               break;
            }
         }
         strncpy (tmpbuf, filename, MAXSTRLEN);
      }

      if (nadp->filename)
         free(nadp->filename);

      nadp->filename = strdup(tmpbuf);
      retn = nadp->filename;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaSplitFilename (0x%x) returning %s\n", nadp, retn); 
#endif

   return (retn);
}

int
RaProcessSplitOptions(struct ArgusParserStruct *parser, char *str, int len, struct ArgusRecordStruct *ns)
{
   char resultbuf[MAXSTRLEN], tmpbuf[MAXSTRLEN];
   char *ptr = NULL, *tptr = str;
   int retn = 0, i, x, slen = 0;

   bzero (resultbuf, len);

   while ((ptr = strchr (tptr, '$')) != NULL) {
      *ptr++ = '\0';
      slen = strlen(resultbuf);
      snprintf (&resultbuf[slen], MAXSTRLEN - slen, tptr);

      for (i = 0, x = 0; x < MAX_PRINT_ALG_TYPES; x++) {
         if (!strncmp (RaPrintAlgorithmTable[x].field, ptr, strlen(RaPrintAlgorithmTable[x].field))) {
            bzero (tmpbuf, MAXSTRLEN);
            RaPrintAlgorithmTable[x].print(parser, tmpbuf, ns, RaPrintAlgorithmTable[x].length);

            while (isspace((int)tmpbuf[strlen(tmpbuf) - 1]))
               tmpbuf[strlen(tmpbuf) - 1] = '\0';

            while (isspace((int)tmpbuf[i])) i++;
            slen = strlen(resultbuf);
            snprintf (&resultbuf[slen], MAXSTRLEN - slen, "%s", &tmpbuf[i]);

            ptr += strlen(RaPrintAlgorithmTable[x].field);
            while (*ptr && (*ptr != '$'))
               bcopy (ptr++, &resultbuf[strlen(resultbuf)], 1);
            break;
         }
      }

      tptr = ptr;
      retn++;
   }

   if (retn) {
      bzero (str, len);
      bcopy (resultbuf, str, strlen(resultbuf));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessSplitOptions(%s, %d, 0x%x): returns %d", str, len, ns, retn);
#endif

   return (retn);
}


int
ArgusRunScript (struct ArgusParserStruct *parser, struct ArgusWfileStruct *file)
{
   struct ArgusScriptStruct *script = NULL;
   int retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusRunScript(0x%x, %x) filename %s", parser, file, file->filename);
#endif

   if (file && parser->ArgusFlowModelFile) {
      char sbuf[1024];
      int i;

      if (ArgusScriptList == NULL)
         if ((ArgusScriptList = ArgusNewList()) == NULL)
            ArgusLog (LOG_ERR, "ArgusRunScript (%s) ArgusNewList() error %s\n", file->filename, strerror(errno));

      if ((script = (struct ArgusScriptStruct *) ArgusCalloc (1, sizeof(*script))) == NULL)
         ArgusLog (LOG_ERR, "ArgusRunScript (%s) ArgusCalloc() error %s\n", file->filename, strerror(errno));

      script->file = file;
      script->filename = strdup(file->filename);
      script->script = strdup(parser->ArgusFlowModelFile);
      script->startime = parser->ArgusRealTime;
      script->timeout = ARGUS_SCRIPT_TIMEOUT;
/*
      if ((script->pid = fork()) < 0)
         ArgusLog (LOG_ERR, "ArgusRunScript (%s) fork() error %s\n", filename, strerror(errno));

      if (script->pid > 0) {
         ArgusPushBackList(ArgusScriptList, (struct ArgusListRecord *) script, ARGUS_LOCK);
         retn++;

      } else {
         char *args[8], sbuf[1024];
         int i;

         bzero(args, sizeof(args));
         bzero(sbuf, sizeof(sbuf));

         args[0] = script->script;         
         args[1] = "-r";         
         args[2] = script->filename;

         for (i = 0; i < 4; i++) {
            if (args[i] != NULL) {
               int slen = strlen(sbuf);
               snprintf (&sbuf[slen], 1024 - slen, " %s", args[i]);
            }
         }

         script->cmd = strdup(sbuf);;

#ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusRunScript calling %s ", sbuf);
#endif
         exit(execv(script->script, args));
      }
*/


      bzero(script->args, sizeof(script->args));
      bzero(sbuf, sizeof(sbuf));

      script->args[0] = script->script;         
      script->args[1] = "-r";         
      script->args[2] = script->filename;

      for (i = 0; i < 4; i++) {
         if (script->args[i] != NULL) {
            int slen = strlen(sbuf);
            snprintf (&sbuf[slen], 1024 - slen, " %s", script->args[i]);
         }
      }

      script->cmd = strdup(sbuf);
#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusRunScript(0x%x, 0x%x) scheduling %s", parser, file, script->cmd);
#endif
      ArgusPushBackList(ArgusScriptList, (struct ArgusListRecord *) script, ARGUS_LOCK);

   } else
      retn = 1;

#ifdef ARGUSDEBUG
   if (script)
      ArgusDebug (1, "ArgusRunScript(0x%x, %x) returning %s", parser, file, script->cmd);
   else
      ArgusDebug (1, "ArgusRunScript(0x%x, %x) no script", parser, file);
#endif

   return (retn);
}


/*
   this routine takes the current state of the stream processor and decides
   if we need to create a new filename for the output.  The scenario we're
   designed to support is "processing a single radium composite stream,
   and outputing into a set of time dependant files using any number of output
   strategies that are specified using data directed labels ('$' in output
   filename template).

   So inorder to do this, we need to track outputfile streams. We do this
   by caching the outputfile stream using the '$' directives as keys.
   Do this as we always do, so use an ArgusAggregator that has the mask
   setup from the '$' directives in the filename, and lookup the output
   struct using this method.  If the cached filename has expired, then
   close it and open a new outputfile, and update the filename string.

*/

struct ArgusAggregatorStruct *RaStreamAggregator = NULL;
struct ArgusHashTable *RaStreamHashTable = NULL;

int
ArgusSetCurrentWfile (struct ArgusParserStruct *parser, struct RaBinProcessStruct *rbps, struct ArgusRecordStruct *ns)
{
   struct timeval startbuf, *start = &startbuf, endbuf, *end = &endbuf;
   struct ArgusAdjustStruct *nadp = RaStreamNadp;
   char tmpbuf[MAXSTRLEN], *filename = NULL;
   struct ArgusHashStruct *hstruct = NULL;

   struct ArgusWfileStruct *wfile = NULL;
   struct ArgusWfileStruct *tfile = NULL;

   char maskbuf[MAXSTRLEN], *masklist = maskbuf;
   int retn = 0, newfilename = 0;
   time_t secs;

   if (RaStreamAggregator == NULL) {
      char ptrbuf[MAXSTRLEN], *tptr = ptrbuf, *ptr;
      int i, x;

      bzero (maskbuf, MAXSTRLEN);
      snprintf (tptr, MAXSTRLEN, "%s", nadp->filename);

      while ((ptr = strchr (tptr, '$')) != NULL) {
         *ptr++ = '\0';
         for (i = 0, x = 0; x < MAX_PRINT_ALG_TYPES; x++) {
            if (!strncmp (RaPrintAlgorithmTable[x].field, ptr, strlen(RaPrintAlgorithmTable[x].field))) {
               int slen = strlen(maskbuf);
               snprintf (&maskbuf[slen], MAXSTRLEN - slen, "%s ", RaPrintAlgorithmTable[x].field);
               break;
            }
         }
         tptr = ptr;
      }

      if ((RaStreamAggregator = ArgusNewAggregator(parser, masklist)) == NULL)
         ArgusLog (LOG_ERR, "ArgusSetCurrentWfile: ArgusNewAggregator error");
   }

   if (RaStreamHashTable == NULL) {
      if ((RaStreamHashTable = (struct ArgusHashTable *) ArgusCalloc (1, sizeof(*RaStreamHashTable))) == NULL)
         ArgusLog (LOG_ERR, "ArgusCheckCurrentWfileStatus: ArgusCalloc error %s\n", strerror(errno));

      if ((RaStreamHashTable->array = (struct ArgusHashTableHdr **) ArgusCalloc (RABINS_HASHTABLESIZE,
                          sizeof (struct ArgusHashTableHdr *))) == NULL)
         ArgusLog (LOG_ERR, "ArgusCheckCurrentWfileStatus: ArgusCalloc error %s\n", strerror(errno));

      RaStreamHashTable->size = RABINS_HASHTABLESIZE;
   }

   if ((RaStreamAggregator->rap = RaFlowModelOverRides(RaStreamAggregator, ns)) == NULL)
      RaStreamAggregator->rap = RaStreamAggregator->drap;

   if ((hstruct = ArgusGenerateHashStruct(RaStreamAggregator, ns, NULL)) == NULL)
      ArgusLog (LOG_ERR, "RaProcessRecord: ArgusGenerateHashStruct error %s", strerror(errno));

   if ((tfile = ArgusFindFile(RaStreamHashTable, hstruct)) == NULL) {
#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusSetCurrentWfile (0x%x, 0x%x, 0x%x) need newfile", parser, rbps, ns);
#endif
      if ((tfile = (struct ArgusWfileStruct *)ArgusCalloc(1, sizeof(*tfile))) == NULL)
         ArgusLog (LOG_ERR, "ArgusCheckCurrentWfileStatus: ArgusCalloc error %s", strerror(errno));

      tfile->htblhdr = ArgusAddHashEntry (RaStreamHashTable, (void *)tfile, hstruct);

   } else {
#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusSetCurrentWfile (0x%x, 0x%x, 0x%x) found file %s", parser, rbps, ns, tfile->filename);
#endif
   }

   start->tv_sec  = ns->canon.time.src.start.tv_sec;
   start->tv_usec = ns->canon.time.src.start.tv_usec;
   end->tv_sec    = ns->canon.time.src.end.tv_sec;
   end->tv_usec   = ns->canon.time.src.end.tv_usec;

   secs = start->tv_sec;
   nadp->RaStartTmStruct = *localtime(&secs);

   switch (nadp->qual) {
      case ARGUSSPLITSECOND: {
         nadp->RaStartTmStruct.tm_sec = ((nadp->RaStartTmStruct.tm_sec/nadp->value)*nadp->value);
         secs = mktime(&nadp->RaStartTmStruct) + nadp->value;
         nadp->RaEndTmStruct = *localtime(&secs);
         break;
      }

      case ARGUSSPLITMINUTE: {
         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = ((nadp->RaStartTmStruct.tm_min/nadp->value)*nadp->value);
         secs = mktime(&nadp->RaStartTmStruct) + (nadp->value * 60);
         nadp->RaEndTmStruct = *localtime(&secs);
         break;
      }

      case ARGUSSPLITHOUR: {
         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = 0;
         nadp->RaStartTmStruct.tm_hour = ((nadp->RaStartTmStruct.tm_hour/nadp->value)*nadp->value);
         secs = mktime(&nadp->RaStartTmStruct) + (nadp->value * 3600);
         nadp->RaEndTmStruct = *localtime(&secs);
         break;
      }

      case ARGUSSPLITDAY: {
         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = 0;
         nadp->RaStartTmStruct.tm_hour = 0;
         secs = mktime(&nadp->RaStartTmStruct) + (nadp->value * (3600 * 24));
         nadp->RaEndTmStruct = *localtime(&secs);
         break;
      }

      case ARGUSSPLITWEEK: {
         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = 0;
         nadp->RaStartTmStruct.tm_hour = 0;
         nadp->RaStartTmStruct.tm_mday -= (nadp->RaStartTmStruct.tm_wday - 1);

         nadp->RaEndTmStruct = nadp->RaStartTmStruct;
         nadp->RaEndTmStruct.tm_mday += (7 * nadp->value);
         break;
      }

      case ARGUSSPLITMONTH: {
         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = 0;
         nadp->RaStartTmStruct.tm_hour = 0;
         nadp->RaStartTmStruct.tm_mday = 0;

         nadp->RaEndTmStruct = nadp->RaStartTmStruct;

         if (++nadp->RaEndTmStruct.tm_mon > 11) {
            nadp->RaEndTmStruct.tm_mon = 0;
            nadp->RaEndTmStruct.tm_year++;
         }
         break;
      }

      case ARGUSSPLITYEAR: {
         nadp->RaStartTmStruct.tm_sec  = 0;
         nadp->RaStartTmStruct.tm_min  = 0;
         nadp->RaStartTmStruct.tm_hour = 0;
         nadp->RaStartTmStruct.tm_mday = 0;
         nadp->RaStartTmStruct.tm_mon  = 0;

         nadp->RaEndTmStruct = nadp->RaStartTmStruct;
         nadp->RaEndTmStruct.tm_year++;

         break;
      }
   }

   switch (nadp->mode) {
      default:
      case ARGUSSPLITTIME: {
         time_t fileSecs;
         fileSecs = mktime(&nadp->RaStartTmStruct);

         if (strftime(tmpbuf, MAXSTRLEN, nadp->filename, localtime(&fileSecs)) <= 0)
            ArgusLog (LOG_ERR, "ArgusCheckCurrentWfilestatus () strftime %s", strerror(errno));

         RaProcessSplitOptions(parser, tmpbuf, MAXSTRLEN, ns);
           
         if ((tfile->filename == NULL) || strcmp(tfile->filename, tmpbuf)) {
            filename = tmpbuf;
            newfilename++;
         }
         break;
      }

      case ARGUSSPLITCOUNT:
         if ((nadp->value > 1) && (!(nadp->count % nadp->value)))
            newfilename++;

         if (nadp->count > 0)
            nadp->count++;
         break;

      case ARGUSSPLITSIZE:
         if ((nadp->value > 0) && (stat (tfile->filename, &tfile->statbuf) == 0))
            if ((tfile->statbuf.st_size + (ns->hdr.len * 4)) > nadp->value)
               newfilename++;
         break;
   }

   if (newfilename) {
      char *tptr = NULL, *pptr = NULL;

      if (tfile->filename != NULL) {
#ifdef ARGUSDEBUG
         ArgusDebug (2, "ArgusSetCurrentWfile (0x%x, 0x%x, 0x%x) file done %s", parser, rbps, ns, tfile->filename);
#endif
         if (tfile->htblhdr != NULL)
            ArgusRemoveHashEntry (tfile->htblhdr);

         if (tfile->fd != NULL) {
            fclose (tfile->fd);
            tfile->fd = NULL;
         }

         if (!(ArgusRunScript(parser, tfile))) {
            if (tfile->filename)
               free(tfile->filename);
            ArgusFree(tfile);
         }

         if ((tfile = (struct ArgusWfileStruct *)ArgusCalloc(1, sizeof(*tfile))) == NULL)
            ArgusLog (LOG_ERR, "ArgusCheckCurrentWfileStatus: ArgusCalloc error %s", strerror(errno));

         tfile->htblhdr = ArgusAddHashEntry (RaStreamHashTable, (void *)tfile, hstruct);
      }
 
      if (filename == NULL)
         if ((filename = RaSplitFilename(nadp)) == NULL)
            ArgusLog(LOG_ERR, "RaProcessRecord filename beyond space");

      tfile->filename = strdup(filename);

      /* got new filename, need to check the
         path to be sure that all the directories exist */

      strncpy (tmpbuf, tfile->filename, MAXSTRLEN);
      if ((tptr = strrchr(tmpbuf, (int) '/')) != NULL) {   /* if there is a path */
         *tptr = '\0';
         pptr = tptr;

         while ((pptr != NULL) && ((stat(tmpbuf, &tfile->statbuf)) < 0)) {
            switch (errno) {
               case ENOENT:
                  if ((pptr = strrchr(tmpbuf, (int) '/')) != NULL) {
                     if (pptr != tmpbuf) {
                        *pptr = '\0';
                     } else {
                        pptr = NULL;
                     }
                  }
                  break;

               default:
                  ArgusLog (LOG_ERR, "stat: %s %s\n", tmpbuf, strerror(errno));
            }
         }

         while (&tmpbuf[strlen(tmpbuf)] <= tptr) {
            if ((mkdir(tmpbuf, 0777)) < 0) {
               if (errno != EEXIST)
                  ArgusLog (LOG_ERR, "mkdir: %s %s\n", tmpbuf, strerror(errno));
            }
            tmpbuf[strlen(tmpbuf)] = '/';
         }
         *tptr = '/';
      }
   }

   if (tfile != (struct ArgusWfileStruct *)ArgusFrontList(parser->ArgusWfileList)) {
      if ((wfile = (struct ArgusWfileStruct *)ArgusPopFrontList(parser->ArgusWfileList, ARGUS_LOCK)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCheckCurrentWfileStatus: no output file in ArgusWfileList");

      ArgusPushFrontList(parser->ArgusWfileList, (struct ArgusListRecord *)tfile, ARGUS_LOCK);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusSetCurrentWfile (0x%x, 0x%x, 0x%x) return %d", parser, rbps, ns, retn);
#endif

   return (retn);
}



struct ArgusWfileStruct *
ArgusFindFile (struct ArgusHashTable *htable, struct ArgusHashStruct *hstruct)
{
   struct ArgusWfileStruct *retn = NULL;
   struct ArgusHashTableHdr *hashEntry = NULL, *target, *head;
   unsigned int ind = (hstruct->hash % htable->size), i, len;

#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&htable->lock);
#endif
   if ((target = htable->array[ind]) != NULL) {
      head = target;

      if (hstruct->len) {
         do {
            unsigned short *ptr1 = (unsigned short *) hstruct->buf;
            unsigned short *ptr2 = (unsigned short *) target->hstruct.buf;

            if (ptr1 && ptr2) {
               for (i = 0, len = hstruct->len/sizeof(unsigned short); i < len; i++)
                  if (*ptr1++ != *ptr2++)
                     break;
               if (i == len) {
                  hashEntry = target;
                  break;
               }

            } else
              if (!(ptr1 || ptr2)) {
                  hashEntry = target;
                  break;
              }

            target = target->nxt;
         } while (target != head);

      } else
         hashEntry = target;
 
      if (hashEntry != NULL) {
         if (hashEntry != head)
            htable->array[ind] = hashEntry;
         retn = hashEntry->object;
      }
   }
#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&htable->lock);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusFindFile () returning 0x%x\n", retn);
#endif
  
   return (retn);
}
