/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * ralabel - add descriptor labels to flows.
 *           this particular labeler adds descriptors based
 *           on addresses.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: //depot/gargoyle/gargoyle-4.0/clients/clients/ralabel.c#18 $
 * $DateTime: 2007/02/09 14:33:28 $
 * $Change: 1033 $
 */

#if defined(CYGWIN)
#define USE_IPV6
#endif

#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>

#if defined(HAVE_SOLARIS)
#include <strings.h>
#include <string.h>
#endif

#include <math.h>

#include <rabins.h>
#include <argus_util.h>
#include <argus_labeler.h>
#include <argus_client.h>
#include <argus_filter.h>
#include <argus_main.h>
#include <argus_cluster.h>


struct ArgusLabelerStruct *ArgusNewLabeler (struct ArgusParserStruct *);
struct ArgusLabelerStruct *ArgusLabeler = NULL;

int RaReadAddressConfig (struct ArgusParserStruct *, struct ArgusLabelerStruct *, char *);


void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusModeStruct *mode = NULL;
   parser->RaWriteOut = 0;

   if (!(parser->RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);

      if ((parser->ArgusLabeler = ArgusNewLabeler(parser)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewLabeler error");

      if ((parser->ArgusAggregator = ArgusNewAggregator(parser, NULL)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");

      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            mode = mode->nxt;
         }
      }

      parser->RaInitialized++;
   }
}

void RaArgusInputComplete (struct ArgusInput *input) { return; }


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;

      if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
         struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
         if ((wfile = (struct ArgusWfileStruct *) ArgusFrontList(ArgusParser->ArgusWfileList)) != NULL) {
            start = wfile;
            fflush(wfile->fd);
            ArgusPopFrontList(ArgusParser->ArgusWfileList, ARGUS_NOLOCK);
            ArgusPushBackList(ArgusParser->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_NOLOCK);
            wfile = (struct ArgusWfileStruct *) ArgusFrontList(ArgusParser->ArgusWfileList);
         } while (wfile != start);
      } 
   }

   fflush(stdout);
   exit(0);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete (%d) returning\n", sig);
#endif
}

void
ArgusClientTimeout ()
{

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusClientTimeout: returning\n");
#endif
}

void
parse_arg (int argc, char**argv)
{ 

#ifdef ARGUSDEBUG
   ArgusDebug (6, "parse_arg (%d, 0x%x) returning\n", argc, argv);
#endif
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Ratemplate Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -f <conffile>     read service signatures from <conffile>.\n");
   exit(1);
}

char *RaProcessAddress (struct ArgusParserStruct *, struct ArgusRecordStruct *, unsigned int *, int);
extern struct RaAddressStruct *RaFindAddress (struct ArgusParserStruct *, struct RaAddressStruct *, struct RaAddressStruct *, int);


char *
RaProcessAddress (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus, unsigned int *addr, int type)
{
   struct ArgusLabelerStruct *labeler = NULL;
   struct RaAddressStruct *raddr;
   char *retn = NULL;

   if ((labeler = parser->ArgusLabeler) == NULL)
      ArgusLog (LOG_ERR, "RaProcessAddress: No labeler\n");

   switch (type) {
      case ARGUS_TYPE_IPV4: {
         struct RaAddressStruct node;
         bzero ((char *)&node, sizeof(node));

         node.addr.type = AF_INET;
         node.addr.len = 4;
         node.addr.addr[0] = *addr;
         node.addr.masklen = 32;

         if ((raddr = RaFindAddress (parser, labeler->ArgusAddrTree[AF_INET], &node, ARGUS_EXACT_MATCH)) != NULL)
            retn = raddr->label;
         else {
            char *ptr, *sptr;
            if ((ptr = ArgusGetName(ArgusParser, (u_char *)addr)) != NULL) {
               if ((sptr = strrchr(ptr, '.')) != NULL) {
                  if (strlen(++sptr) == 2) {
                     if (!isdigit((int)*sptr)) {
                        char *tptr = sptr;
                        int i, ch;
                        for (i = 0; i < 2; i++, tptr++) {
                           ch = *tptr;
                           if (islower(ch)) {
                              ch = toupper(ch);
                              *tptr = ch;
                           }
                        }
                        retn = sptr;
                     }
                  }
               }
            }
         }

         break;
      }

      case ARGUS_TYPE_IPV6:
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessAddress (0x%x, 0x%x, 0x%x, %d) returning %s\n", parser, argus, addr, type, retn);
#endif

   return (retn);
}


void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusFlow *flow = (struct ArgusFlow *) argus->dsrs[ARGUS_FLOW_INDEX];
   char buf[0x10000];

   if (flow && (!(argus->hdr.type & ARGUS_MAR))) {
      switch (flow->hdr.subtype & 0x3F) {
         case ARGUS_FLOW_CLASSIC5TUPLE:
         case ARGUS_FLOW_LAYER_3_MATRIX: {
            char *sco = NULL, *dco = NULL;
 
            switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
               case ARGUS_TYPE_IPV4: {
                  sco = RaProcessAddress(parser, argus, &flow->ip_flow.ip_src, ARGUS_TYPE_IPV4);
                  dco = RaProcessAddress(parser, argus, &flow->ip_flow.ip_dst, ARGUS_TYPE_IPV4);
                  break;
               }
 
               case ARGUS_TYPE_IPV6:
                  sco = RaProcessAddress(parser, argus, (unsigned int *) &flow->ipv6_flow.ip_src, ARGUS_TYPE_IPV6);
                  dco = RaProcessAddress(parser, argus, (unsigned int *) &flow->ipv6_flow.ip_dst, ARGUS_TYPE_IPV6);
                  break;
            }
 
            if (sco || dco) {
               struct ArgusCountryCodeStruct *cocode = (void *) argus->dsrs[ARGUS_COCODE_INDEX];
               if (cocode == NULL) {
                  cocode = &argus->canon.cocode;
                  bzero((char *)cocode, sizeof(*cocode));
                  argus->dsrs[ARGUS_COCODE_INDEX] = &cocode->hdr;
                  argus->dsrindex |= (0x1 << ARGUS_COCODE_INDEX);
               }
               cocode->hdr.type              = ARGUS_COCODE_DSR;
               cocode->hdr.argus_dsrvl16.len = 2;
               if (sco) bcopy((char *)sco, (char *)&cocode->src, 2);
               if (dco) bcopy((char *)dco, (char *)&cocode->dst, 2);
            }
            break; 
         }
      }
   }

   if (parser->ArgusWfileList != NULL) {
      struct ArgusWfileStruct *wfile = NULL;
      struct ArgusListObjectStruct *lobj = NULL;
      int i, count = parser->ArgusWfileList->count;

      if ((lobj = parser->ArgusWfileList->start) != NULL) {
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
               if ((parser->exceptfile == NULL) || strcmp(wfile->filename, parser->exceptfile)) {
                  struct ArgusRecord *argusrec = NULL;
                  static char sbuf[0x10000];
                  if ((argusrec = ArgusGenerateRecord (argus, 0L, sbuf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                     ArgusHtoN(argusrec);
#endif
                     ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);
                  }
               }
            }

            lobj = lobj->nxt;
         }
      }

   } else {
      if (!parser->qflag) {
         if (parser->Lflag) {
            if (parser->RaLabel == NULL)
               parser->RaLabel = ArgusGenerateLabel(parser, argus);
 
            if (!(parser->RaLabelCounter++ % parser->Lflag))
               printf ("%s\n", parser->RaLabel);
 
            if (parser->Lflag < 0)
               parser->Lflag = 0;
         }

         *(int *)&buf = 0;
         ArgusPrintRecord(parser, buf, argus, MAXSTRLEN);
         fprintf (stdout, "%s ", buf);
      }
   }

   if (parser->ArgusWfileList == NULL)
      fprintf (stdout, "\n");

   fflush (stdout);
                 
#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessRecord (0x%x) returning\n", argus);
#endif
}


int
RaSendArgusRecord(struct ArgusRecordStruct *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaSendArgusRecord (0x%x) returning\n", argus);
#endif
   return 1;
}

void ArgusWindowClose(void) { } 

