/*
 * Dictionary functions
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <stdio.h>
#include <stdlib.h>

#include "stdlib.h"

/*
 * List of open dictionary handles
 */
handles *dicts;

/*
 * Flag for initialization of dictionary handles
 */
static int dicts_init = 0;

/*
 * Obtain new dictionary handle
 */
static int dict_add(symtab *s)
{
  if (!dicts_init) {
    dicts = handle_init();
    dicts_init = 1;
  }
  return handle_add(dicts, s);
}

/*
 * Get symtab pointer by dictionary handle
 */
static symtab *dict_get(int handle)
{
  if (!dicts_init) {
    return NULL;
  }
  return handle_get(dicts, handle);
}

/*
 * Release dictionary handle
 */
static void dict_del(int handle)
{
  if (!dicts_init) {
    return;
  }
  handle_del(dicts, handle);
}

/*
 * Tear down memory used by dictionaries
 */
void dict_teardown(void)
{
  if (dicts_init) {
    int i;
    
    for (i = 0; i < dicts->count; i++) {
      symtab_free(dicts->entries[i]);
      handle_del(dicts, i + 1);
    }
    handle_free(dicts);
  }
}

/*
 * Create a new dictionary
 *
 * Creates a new dictionary of the given order, where order n
 * means the dictionary will use 2**n hash buckets.
 */
value *dict_open(unsigned int argc, value **argv)
{
  int order = argv[0]->value_u.int_val;
  symtab *s;
  int handle;
  
  if (order < 1) {
    return value_make_void();
  }
  
  s = symtab_alloc(order);
  handle = dict_add(s);
  if (handle == -1) {
    symtab_free(s);
    return value_make_void();
  }
  return value_make_int(handle);
}

/*
 * Read dictionary entry
 *
 * Returns the value stored in the dictionary under the given
 * name, or void if the name is not contained in the dictionary.
 */
value *dict_read(unsigned int argc, value **argv)
{
  symtab *s = dict_get(argv[0]->value_u.int_val);
  const char *name = argv[1]->value_u.string_val.value;
  symtab_entry *entry;
  
  if (!s || !value_str_compat(argv[1])) {
    return value_make_void();
  }
  entry = symtab_lookup(s, name);
  if (!entry) {
    return value_make_void();
  }
  return value_copy(entry->entry_u.var);
}

/*
 * Write dictionary entry
 *
 * Stores a value in the dictionary under the given name. If the
 * name was used before, the old entry is removed first. Returns
 * true on success, false on failure.
 */
value *dict_write(unsigned int argc, value **argv)
{
  symtab *s = dict_get(argv[0]->value_u.int_val);
  const char *name = argv[1]->value_u.string_val.value;
  value *val = argv[2];
  
  if (!s || !value_str_compat(argv[1])) {
    return value_make_bool(0);
  }
  
  symtab_add_variable(s, name, val);
  return value_make_bool(1);
}

/*
 * Remove name from dictionary
 *
 * Returns true on success and false on failure.
 */
value *dict_remove(unsigned int argc, value **argv)
{
  symtab *s = dict_get(argv[0]->value_u.int_val);
  const char *name = argv[1]->value_u.string_val.value;
  
  if (!s || !value_str_compat(argv[1])) {
    return value_make_bool(0);
  }
  
  symtab_delete(s, name);
  return value_make_bool(1);
}

/*
 * Check dictionary entry
 *
 * Returns true if the given name exists in the dictionary.
 */
value *dict_exists(unsigned int argc, value **argv)
{
  symtab *s = dict_get(argv[0]->value_u.int_val);
  const char *name = argv[1]->value_u.string_val.value;
  symtab_entry *entry;
  
  if (!s || !value_str_compat(argv[1])) {
    return value_make_bool(0);
  }
  entry = symtab_lookup(s, name);
  return value_make_bool(entry != NULL);
}

/*
 * Close dictionary
 *
 * This function closes a dictionary, which means all entries
 * will be removed and the space used by the dictionary freed.
 */
value *dict_close(unsigned int argc, value **argv)
{
  symtab *s = dict_get(argv[0]->value_u.int_val);
  
  if (s) {
    symtab_free(s);
    dict_del(argv[0]->value_u.int_val);
  }
  return value_make_void();
}
