/*
 * Method call evaluation
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */
 
#include <stdio.h>
#include <stdlib.h>

#include "eval.h"

/*
 * Enter struct namespace
 */
static void enter_struct(value *val)
{
  symtab_stack_enter();
  symtab_stack_add_variable("this", val);
}

/*
 * Leave struct namespace
 */
static value *leave_struct(signature *sig, unsigned int argc, expr **argv)
{
  symtab_entry *entry;
  value *res = NULL;
  
  entry = symtab_stack_lookup("this");
  if (entry) {
    res = value_copy(entry->entry_u.var);
  }
  update_call_args(sig, argc, argv);
  return res;
}

/*
 * Constructor to call
 */
static char *new_cons     = NULL;
static signature *new_sig = NULL;

/*
 * Recursively construct instance
 */
static void getinstance(const char *name)
{
  symtab_entry *entry;
  
  entry = symtab_stack_lookup(name);
  if (!entry || entry->type != SYMTAB_ENTRY_TMPL) {
    fatal("use of undefined template `%s'", name);
  }
  if (entry->entry_u.def.parent) {
    getinstance(entry->entry_u.def.parent);
  }
  eval_stmt_list((stmt_list *) entry->entry_u.def.def, 0);
  
  entry = symtab_stack_lookup(name);
  if (entry && entry->type == SYMTAB_ENTRY_FUNCTION) {
    new_cons = entry->symbol;
    new_sig  = entry->entry_u.sig;
  }
}

/*
 * Evaluate constructor expression
 */
value *eval_new(expr *ex)
{
  value *res, *tname;

  sanity(ex && ex->name);
  
  symtab_stack_enter();
  new_cons = NULL;
  new_sig  = NULL;
  getinstance(ex->name);

  res = value_make_struct();
  symtab_free(res->value_u.struct_val);
  res->value_u.struct_val = symtab_stack_pop();
  
  tname = value_make_string(ex->name);
  value_set_struct(res, "__template", tname);
  value_free(tname);
  
  if (new_cons) {
    value **argv;
    value *temp, *cons;
    
    eval_call_args(ex->argc, ex->argv, &argv);
    enter_struct(res);
    
    cons = call_function(new_cons, new_sig, ex->argc, argv);
    value_free(cons);
    
    temp = leave_struct(new_sig, ex->argc, ex->argv);
    if (!temp) {
      fatal("no `this' at constructor `%s' exit", new_cons);
    }
    free_call_args(ex->argc, &argv);
    
    value_free(res);
    res = temp;
  }

  return res;
}

/*
 * Evaluate static method call
 */
value *eval_static(expr *ex)
{
  symtab_entry *entry;
  signature *sig;
  value **argv, *res;
  
  sanity(ex && ex->tname && ex->name);

  symtab_stack_enter();
  getinstance(ex->tname);
  
  entry = symtab_stack_lookup(ex->name);
  if (!entry || entry->type != SYMTAB_ENTRY_FUNCTION ||
      !symtab_stack_local(ex->name)) {
    fatal("call to undefined method `%s::%s'", ex->tname, ex->name);
  }
  sig = call_sig_copy(entry->entry_u.sig);
  symtab_stack_leave();
  
  eval_call_args(ex->argc, ex->argv, &argv);
  symtab_stack_enter();

  res = call_function(ex->name, sig, ex->argc, argv);
  update_call_args(sig, ex->argc, ex->argv);

  symtab_stack_leave();
  free_call_args(ex->argc, &argv);
  
  call_sig_free(sig);
  return res;
}

/*
 * Evaluate static reference
 */
value *eval_static_ref(expr *ex)
{
  symtab *sym;
  symtab_entry *entry;
  value *res;
  
  sanity(ex && ex->tname && ex->name);
  
  symtab_stack_enter();
  getinstance(ex->tname);
  sym = symtab_stack_pop();
  
  entry = symtab_lookup(sym, ex->name);
  if (!entry) {
    fatal("use of undefined template member `%s::%s'",
      ex->tname, ex->name);
  }
  if (entry->type == SYMTAB_ENTRY_VAR) {
    res = value_copy(entry->entry_u.var);
  } else {
    res = value_make_fn(entry->entry_u.sig);
  }
  symtab_free(sym);
  return res;
}

/*
 * Evaluate method call
 */
value *eval_method(expr *ex)
{
  symtab_entry *entry;
  value *val, *res, *temp, **argv;

  sanity(ex && ex->inner && ex->name);
  
  val = eval_expr(ex->inner);
  if (val->type != VALUE_TYPE_STRUCT) {
    fatal("method call on non-struct value");
  }
  
  entry = symtab_lookup(val->value_u.struct_val, ex->name);
  if (!entry || entry->type != SYMTAB_ENTRY_FUNCTION) {
    fatal("call to undefined method `%s'", ex->name);
  }

  eval_call_args(ex->argc, ex->argv, &argv);
  enter_struct(val);

  res = call_function(ex->name, entry->entry_u.sig, ex->argc, argv);

  temp = leave_struct(entry->entry_u.sig, ex->argc, ex->argv);
  free_call_args(ex->argc, &argv);

  value_free(val);
  
  if (temp) {
    if (ex->inner->type == EXPR_REF) {
      symtab_stack_add_variable(ex->inner->name, temp);
    }
    if (ex->inner->type == EXPR_REF_ARRAY) {
      eval_assign_array_direct(ex->inner->name, ex->inner->argc,
        ex->inner->argv, temp);
    }
    value_free(temp);
  }
  
  return res;
}
