/*
 * Evaluation library
 *
 * This library provides functions needed to evaluate statements
 * and expressions in accordance with the defined semantics of
 * the language.
 *
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */
#ifndef LIBEXEC_H
#define LIBEXEC_H

#include "../libmisc/misc.h"
#include "../libruntime/runtime.h"
#include "../libparser/parser.h"

/*
 * Exception handling
 */
extern int except_flag;

/*
 * Helper function
 */
void eval_call_args(unsigned int argc, expr **args, value ***argv);
void free_call_args(unsigned int argc, value ***argv);
void update_call_args(signature *sig, unsigned int argc, expr **args);

/*
 * Simple expressions
 */
value *eval_const_void(void);
value *eval_const_bool(expr *ex);
value *eval_const_int(expr *ex);
value *eval_const_float(expr *ex);
value *eval_const_string(expr *ex);
value *eval_cast(expr *ex);
value *eval_assign(expr *ex);
value *eval_assign_array(expr *ex);
value *eval_ref(expr *ex);
value *eval_ref_array(expr *ex);
value *eval_ref_index(expr *ex);
value *eval_call(expr *ex);
value *eval_new(expr *ex);
value *eval_static(expr *ex);
value *eval_method(expr *ex);
value *eval_static_ref(expr *ex);
value *eval_prefix(expr *ex);
value *eval_postfix(expr *ex);
value *eval_infix(expr *ex);
value *eval_lambda(expr *ex);

void eval_assign_array_direct(const char *name, int argc, expr **index,
  value *val);

/*
 * Math evaluation
 */
value *eval_math_plus(value *a, value *b);
value *eval_math_minus(value *a, value *b);
value *eval_math_mul(value *a, value *b);
value *eval_math_div(value *a, value *b);
value *eval_math_mod(value *a, value *b);
value *eval_math_pow(value *a, value *b);

/*
 * Boolean math evaluation
 */
value *eval_bool_and(expr *a, expr *b);
value *eval_bool_or(expr *a, expr *b);
value *eval_expr_if(expr *ex);

/*
 * Bitwise math evaluation
 */
value *eval_bit_and(value *a, value *b);
value *eval_bit_or(value *a, value *b);
value *eval_bit_xor(value *a, value *b);
value *eval_bit_lshift(value *a, value *b);
value *eval_bit_rshift(value *a, value *b);

/*
 * Order evaluation
 */
value *eval_order_equal(value *a, value *b);
value *eval_order_not_equal(value *a, value *b);
value *eval_order_seq(value *a, value *b);
value *eval_order_leq(value *a, value *b);
value *eval_order_smaller(value *a, value *b);
value *eval_order_larger(value *a, value *b);

/*
 * Top-level expression evaluation
 */
value *eval_expr(expr *ex);

/*
 * Statement evaluation
 */
void eval_stmt(stmt *st, int cookie);
void eval_stmt_switch(stmt *st, int cookie);
void eval_stmt_list(stmt_list *list, int cookie);

#endif
