/*
 * word2text.c
 * Copyright (C) 1998-2002 A.J. van Os; Released under GPL
 *
 * Description:
 * MS Word to text functions
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if defined(__riscos)
#include "visdelay.h"
#endif /* __riscos */
#include "antiword.h"


#define INITIAL_SIZE		40
#define EXTENTION_SIZE		20


/* Macros to make sure all such statements will be identical */
#define OUTPUT_LINE()		\
	do {\
		vAlign2Window(pDiag, pAnchor, lWidthMax, ucAlignment);\
		pAnchor = pStartNewOutput(pAnchor, NULL);\
		pOutput = pAnchor;\
	} while(0)

#define RESET_LINE()		\
	do {\
		pAnchor = pStartNewOutput(pAnchor, NULL);\
		pOutput = pAnchor;\
	} while(0)

#if defined(__riscos)
/* Length of the document in characters */
static long	lDocumentLength;
/* Number of characters processed so far */
static long	lCharCounter;
static int	iCurrPct, iPrevPct;
#endif /* __riscos */
/* The document is in the format belonging to this version of Word */
static int	iWordVersion = -1;
/* Special treatment for files from Word 6 on an Apple Macintosh */
static BOOL	bWord6MacFile = FALSE;
/* Section Information */
static const section_block_type	*pSection = NULL;
static const section_block_type	*pSectionNext = NULL;
/* All the (command line) options */
static options_type	tOptions;
/* Needed for reading a complete table row */
static const row_block_type	*pRowInfo = NULL;
static BOOL	bStartRow = FALSE;
static BOOL	bEndRowNorm = FALSE;
static BOOL	bEndRowFast = FALSE;
static BOOL	bIsTableRow = FALSE;
/* Index of the next style and font information */
static USHORT	usIstdNext = ISTD_NORMAL;
/* Needed for finding the start of a style */
static const style_block_type	*pStyleInfo = NULL;
static style_block_type		tStyleNext;
static BOOL	bStartStyle = FALSE;
static BOOL	bStartStyleNext = FALSE;
/* Needed for finding the start of a font */
static const font_block_type	*pFontInfo = NULL;
static font_block_type		tFontNext;
static BOOL	bStartFont = FALSE;
static BOOL	bStartFontNext = FALSE;
/* Needed for finding an image */
static ULONG	ulFileOffsetImage = FC_INVALID;


/*
 * vUpdateCounters - Update the counters for the hourglass
 */
static void
vUpdateCounters(void)
{
#if defined(__riscos)
	lCharCounter++;
	iCurrPct = (int)((lCharCounter * 100) / lDocumentLength);
	if (iCurrPct != iPrevPct) {
		visdelay_percent(iCurrPct);
		iPrevPct = iCurrPct;
	}
#endif /* __riscos */
} /* end of vUpdateCounters */

/*
 * bOutputContainsText - see if the output contains more than white space
 */
static BOOL
bOutputContainsText(output_type *pAnchor)
{
	output_type	*pCurr;
	int	iIndex;

	fail(pAnchor == NULL);

	for (pCurr = pAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		fail(pCurr->lStringWidth < 0);
		for (iIndex = 0; iIndex < pCurr->iNextFree; iIndex++) {
			if (isspace((int)pCurr->szStorage[iIndex])) {
				continue;
			}
#if defined(DEBUG)
			if (pCurr->szStorage[iIndex] == FILLER_CHAR) {
				continue;
			}
#endif /* DEBUG */
			return TRUE;
		}
	}
	return FALSE;
} /* end of bOutputContainsText */

/*
 * lTotalStringWidth - compute the total width of the output string
 */
static long
lTotalStringWidth(output_type *pAnchor)
{
	output_type	*pCurr;
	long		lTotal;

	lTotal = 0;
	for (pCurr = pAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_DEC_C(pCurr->lStringWidth < 0, pCurr->lStringWidth);
		fail(pCurr->lStringWidth < 0);
		lTotal += pCurr->lStringWidth;
	}
	return lTotal;
} /* end of lTotalStringWidth */

/*
 * vStoreByte - store one byte
 */
static void
vStoreByte(UCHAR ucChar, output_type *pOutput)
{
	fail(pOutput == NULL);

	if (ucChar == 0) {
		pOutput->szStorage[pOutput->iNextFree] = '\0';
		return;
	}

	while (pOutput->iNextFree + 2 > (int)pOutput->tStorageSize) {
		pOutput->tStorageSize += EXTENTION_SIZE;
		pOutput->szStorage = xrealloc(pOutput->szStorage,
					pOutput->tStorageSize);
	}
	pOutput->szStorage[pOutput->iNextFree] = (char)ucChar;
	pOutput->szStorage[pOutput->iNextFree + 1] = '\0';
	pOutput->iNextFree++;
} /* end of vStoreByte */

/*
 * vStoreCharacter - store one character
 */
static void
vStoreCharacter(ULONG ulChar, output_type *pOutput)
{
	int	iLen;

	fail(pOutput == NULL);

	if (tOptions.eEncoding == encoding_utf8) {
		DBG_HEX_C(ulChar > 0xffff, ulChar);
		fail(ulChar > 0xffff);
		if (ulChar < 0x80) {
			vStoreByte((UCHAR)ulChar, pOutput);
			iLen = 1;
		} else if (ulChar < 0x800) {
			vStoreByte((UCHAR)(0xc0 | ulChar >> 6),
								pOutput);
			vStoreByte((UCHAR)(0x80 | (ulChar & 0x3f)),
								pOutput);
			iLen = 2;
		} else {
			vStoreByte((UCHAR)(0xe0 | ulChar >> 12),
								pOutput);
			vStoreByte((UCHAR)(0x80 | (ulChar >> 6 & 0x3f)),
								pOutput);
			vStoreByte((UCHAR)(0x80 | (ulChar & 0x3f)),
								pOutput);
			iLen = 3;
		}
	} else {
		DBG_HEX_C(ulChar > 0xff, ulChar);
		fail(ulChar > 0xff);
		vStoreByte((UCHAR)ulChar, pOutput);
		iLen = 1;
	}
	pOutput->lStringWidth += lComputeStringWidth(
				pOutput->szStorage + pOutput->iNextFree - iLen,
				iLen,
				pOutput->tFontRef,
				pOutput->sFontsize);
} /* end of vStoreCharacter */

/*
 * vStoreString - store a string
 */
static void
vStoreString(const char *szString, int iStringLength, output_type *pOutput)
{
	int	iIndex;

	fail(szString == NULL || iStringLength < 0 || pOutput == NULL);

	for (iIndex = 0; iIndex < iStringLength; iIndex++) {
		vStoreCharacter(szString[iIndex], pOutput);
	}
} /* end of vStoreString */

/*
 * vStoreNumberAsDecimal - store a number as a decimal number
 */
static void
vStoreNumberAsDecimal(UINT uiNumber, output_type *pOutput)
{
	int	iLen;
	char	szString[15];

	fail(uiNumber == 0);
	fail(pOutput == NULL);

	iLen = sprintf(szString, "%u", uiNumber);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreNumberAsDecimal */

/*
 * vStoreNumberAsRoman - store a number as a roman numerical
 */
static void
vStoreNumberAsRoman(UINT uiNumber, output_type *pOutput)
{
	int	iLen;
	char	szString[15];

	fail(uiNumber == 0);
	fail(pOutput == NULL);

	iLen = iNumber2Roman(uiNumber, FALSE, szString);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreNumberAsRoman */

/*
 * vStoreStyle - store a style
 */
static void
vStoreStyle(output_type *pOutput, const style_block_type *pStyle)
{
	int	iLen;
	char	szString[120];

	fail(pOutput == NULL);
	fail(pStyle == NULL);

	iLen = iStyle2Window(szString, pStyle, pSection);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreStyle */

/*
 * Create an empty line by adding a extra "newline"
 */
static void
vEmptyLine2Diagram(diagram_type *pDiag, draw_fontref tFontRef, short sFontsize)
{
	fail(pDiag == NULL);
	fail(sFontsize < MIN_FONT_SIZE || sFontsize > MAX_FONT_SIZE);

	if (pDiag->lXleft > 0) {
		/* To the start of the line */
		vMove2NextLine(pDiag, tFontRef, sFontsize);
	}
	/* Empty line */
	vMove2NextLine(pDiag, tFontRef, sFontsize);
} /* end of vEmptyLine2Diagram */

/*
 * vPutIndentation - output the given amount of indentation
 */
static void
vPutIndentation(diagram_type *pDiag, output_type *pOutput, BOOL bNumPause,
	UINT uiListNumber, UCHAR ucNFC, char cListCharacter,
	long lLeftIndentation)
{
	long	lWidth;
	int	iNextFree;
	char	szLine[30];

	fail(pDiag == NULL || pOutput == NULL);
	fail(lLeftIndentation < 0);

	if (lLeftIndentation <= 0) {
		return;
	}
	if (bNumPause) {
		vSetLeftIndentation(pDiag, lLeftIndentation);
		return;
	}
	fail(iscntrl((int)cListCharacter));

	switch (ucNFC) {
	case LIST_ARABIC_NUM:
		iNextFree = sprintf(szLine, "%u", uiListNumber);
		break;
	case LIST_ROMAN_NUM_UPPER:
	case LIST_ROMAN_NUM_LOWER:
		iNextFree = iNumber2Roman(uiListNumber,
				ucNFC == LIST_ROMAN_NUM_UPPER, szLine);
		break;
	case LIST_UPPER_ALPHA:
	case LIST_LOWER_ALPHA:
		iNextFree = iNumber2Alpha(uiListNumber,
				ucNFC == LIST_UPPER_ALPHA, szLine);
		break;
	case LIST_ORDINAL_NUM:
		if (uiListNumber % 10 == 1 && uiListNumber != 11) {
			iNextFree = sprintf(szLine, "%ust", uiListNumber);
		} else if (uiListNumber % 10 == 2 && uiListNumber != 12) {
			iNextFree = sprintf(szLine, "%und", uiListNumber);
		} else if (uiListNumber % 10 == 3 && uiListNumber != 13) {
			iNextFree = sprintf(szLine, "%urd", uiListNumber);
		} else {
			iNextFree = sprintf(szLine, "%uth", uiListNumber);
		}
		break;
	case LIST_BULLETS:
		iNextFree = 0;
		break;
	default:
		DBG_DEC(ucNFC);
		DBG_FIXME();
		iNextFree = sprintf(szLine, "%u", uiListNumber);
		break;
	}
	szLine[iNextFree++] = cListCharacter;
	szLine[iNextFree++] = ' ';
	szLine[iNextFree] = '\0';
	lWidth = lComputeStringWidth(szLine, iNextFree,
				pOutput->tFontRef, pOutput->sFontsize);
	lLeftIndentation -= lWidth;
	if (lLeftIndentation > 0) {
		vSetLeftIndentation(pDiag, lLeftIndentation);
	}
	vStoreString(szLine, iNextFree, pOutput);
} /* end of vPutIndentation */

/*
 * vPutNoteSeparator - output a note separator
 *
 * A note separator is a horizontal line two inches long.
 * Two inches equals 144000 millipoints.
 */
static void
vPutNoteSeparator(output_type *pOutput)
{
	long	lCharWidth;
	int	iCounter, iChars;
	char	szOne[2];

	fail(pOutput == NULL);

	szOne[0] = OUR_EM_DASH;
	szOne[1] = '\0';
	lCharWidth = lComputeStringWidth(szOne, 1,
				pOutput->tFontRef, pOutput->sFontsize);
	DBG_DEC(lCharWidth);
	iChars = (int)((144000 + lCharWidth / 2) / lCharWidth);
	DBG_DEC(iChars);
	for (iCounter = 0; iCounter < iChars; iCounter++) {
		vStoreCharacter(OUR_EM_DASH, pOutput);
	}
} /* end of vPutNoteSeparator */

/*
 *
 */
static output_type *
pStartNextOutput(output_type *pCurrent)
{
	output_type	*pNew;

	if (pCurrent->iNextFree == 0) {
		/* The current record is empty, re-use */
		fail(pCurrent->szStorage[0] != '\0');
		fail(pCurrent->lStringWidth != 0);
		return pCurrent;
	}
	/* The current record is in use, make a new one */
	pNew = xmalloc(sizeof(*pNew));
	pCurrent->pNext = pNew;
	pNew->tStorageSize = INITIAL_SIZE;
	pNew->szStorage = xmalloc(pNew->tStorageSize);
	pNew->szStorage[0] = '\0';
	pNew->iNextFree = 0;
	pNew->lStringWidth = 0;
	pNew->iColor = FONT_COLOR_DEFAULT;
	pNew->ucFontstyle = FONT_REGULAR;
	pNew->tFontRef = (draw_fontref)0;
	pNew->sFontsize = DEFAULT_FONT_SIZE;
	pNew->pPrev = pCurrent;
	pNew->pNext = NULL;
	return pNew;
} /* end of pStartNextOutput */

/*
 * pStartNewOutput
 */
static output_type *
pStartNewOutput(output_type *pAnchor, output_type *pLeftOver)
{
	output_type	*pCurr, *pNext;
	int		iColor;
	short		sFontsize;
	draw_fontref	tFontRef;
	UCHAR	ucFontstyle;

	iColor = FONT_COLOR_DEFAULT;
	ucFontstyle = FONT_REGULAR;
	tFontRef = (draw_fontref)0;
	sFontsize = DEFAULT_FONT_SIZE;
	/* Free the old output space */
	pCurr = pAnchor;
	while (pCurr != NULL) {
		pNext = pCurr->pNext;
		pCurr->szStorage = xfree(pCurr->szStorage);
		if (pCurr->pNext == NULL) {
			iColor = pCurr->iColor;
			ucFontstyle = pCurr->ucFontstyle;
			tFontRef = pCurr->tFontRef;
			sFontsize = pCurr->sFontsize;
		}
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}
	if (pLeftOver == NULL) {
		/* Create new output space */
		pLeftOver = xmalloc(sizeof(*pLeftOver));
		pLeftOver->tStorageSize = INITIAL_SIZE;
		pLeftOver->szStorage = xmalloc(pLeftOver->tStorageSize);
		pLeftOver->szStorage[0] = '\0';
		pLeftOver->iNextFree = 0;
		pLeftOver->lStringWidth = 0;
		pLeftOver->iColor = iColor;
		pLeftOver->ucFontstyle = ucFontstyle;
		pLeftOver->tFontRef = tFontRef;
		pLeftOver->sFontsize = sFontsize;
		pLeftOver->pPrev = NULL;
		pLeftOver->pNext = NULL;
	}
	fail(!bCheckDoubleLinkedList(pLeftOver));
	return pLeftOver;
} /* end of pStartNewOutput */

/*
 * ulGetChar - get the next character from the given list
 *
 * returns the next character of EOF
 */
static ULONG
ulGetChar(FILE *pFile, list_id_enum eListID)
{
	const font_block_type	*pCurr;
	ULONG		ulChar, ulFileOffset, ulTextOffset;
	row_info_enum	eRowInfo;
	USHORT		usChar, usPropMod;
	BOOL		bSkip;

	fail(pFile == NULL);

	pCurr = pFontInfo;
	bSkip = FALSE;
	for (;;) {
		usChar = usNextChar(pFile, eListID,
				&ulFileOffset, &ulTextOffset, &usPropMod);
		if (usChar == (USHORT)EOF) {
			return (ULONG)EOF;
		}

		vUpdateCounters();

		eRowInfo = ePropMod2RowInfo(usPropMod, iWordVersion);
		if (!bStartRow) {
#if 0
			bStartRow = eRowInfo == found_a_cell ||
				(pRowInfo != NULL &&
				 ulFileOffset == pRowInfo->ulFileOffsetStart &&
				 eRowInfo != found_not_a_cell);
#else
			bStartRow = pRowInfo != NULL &&
				ulFileOffset == pRowInfo->ulFileOffsetStart;
#endif
			NO_DBG_HEX_C(bStartRow, pRowInfo->ulFileOffsetStart);
		}
		if (!bEndRowNorm) {
#if 0
			bEndRow = eRowInfo == found_end_of_row ||
				(pRowInfo != NULL &&
				 ulFileOffset == pRowInfo->ulFileOffsetEnd &&
				 eRowInfo != found_not_end_of_row);
#else
			bEndRowNorm = pRowInfo != NULL &&
				ulFileOffset == pRowInfo->ulFileOffsetEnd;
#endif
			NO_DBG_HEX_C(bEndRowNorm, pRowInfo->ulFileOffsetEnd);
		}
		if (!bEndRowFast) {
			bEndRowFast = eRowInfo == found_end_of_row;
			NO_DBG_HEX_C(bEndRowFast, pRowInfo->ulFileOffsetEnd);
		}

		if (!bStartStyle) {
			bStartStyle = pStyleInfo != NULL &&
				ulFileOffset == pStyleInfo->ulFileOffset;
			NO_DBG_HEX_C(bStartStyle, ulFileOffset);
		}
		if (pCurr != NULL && ulFileOffset == pCurr->ulFileOffset) {
			bStartFont = TRUE;
			NO_DBG_HEX(ulFileOffset);
			pFontInfo = pCurr;
			pCurr = pGetNextFontInfoListItem(pCurr);
		}

		/* Skip embedded characters */
		if (usChar == START_EMBEDDED) {
			bSkip = TRUE;
			continue;
		}
		if (usChar == END_IGNORE || usChar == END_EMBEDDED) {
			bSkip = FALSE;
			continue;
		}
		if (bSkip) {
			continue;
		}
		ulChar = ulTranslateCharacters(usChar,
					ulFileOffset,
					iWordVersion,
					tOptions.eEncoding,
					bWord6MacFile);
		if (ulChar == IGNORE_CHARACTER) {
			continue;
		}
		if (ulChar == PICTURE) {
			ulFileOffsetImage = ulGetPictInfoListItem(ulFileOffset);
		} else {
			ulFileOffsetImage = FC_INVALID;
		}
		if (ulChar == PAR_END) {
			/* End of paragraph seen, prepare for the next */
			vFillStyleFromStylesheet(usIstdNext, &tStyleNext);
			vCorrectStyleValues(&tStyleNext);
			bStartStyleNext = TRUE;
			vFillFontFromStylesheet(usIstdNext, &tFontNext);
			vCorrectFontValues(&tFontNext);
			bStartFontNext = TRUE;
		}
		if (ulChar == PAGE_BREAK) {
			/* Might be the start of a new section */
			pSectionNext = pGetSectionInfo(pSection, ulTextOffset);
		}
		return ulChar;
	}
} /* end of ulGetChar */

/*
 * bWord2Text - translate Word to text or Postcript
 *
 * returns TRUE when succesful, otherwise FALSE
 */
BOOL
bWord2Text(FILE *pFile, long lFilesize, diagram_type *pDiag)
{
	imagedata_type	tImage;
	const style_block_type	*pStyleTmp;
	const font_block_type	*pFontTmp;
	output_type	*pAnchor, *pOutput, *pLeftOver;
	ULONG	ulChar;
	long	lBeforeIndentation, lAfterIndentation;
	long	lLeftIndentation, lRightIndentation;
	long	lWidthCurr, lWidthMax, lDefaultTabWidth, lTmp;
	list_id_enum 	eListID;
	image_info_enum	eRes;
	UINT	uiListNumber, uiFootnoteNumber, uiEndnoteNumber;
	BOOL	bWasTableRow, bTableFontClosed, bWasInList, bWasEndOfParagraph;
	BOOL	bNumPause, bAllCapitals, bHiddenText, bSuccess;
	short	sFontsize;
	UCHAR	ucFontnumber, ucFontcolor, ucTmp;
	UCHAR	ucFontstyle, ucFontstyleMinimal;
	UCHAR	ucNFC, ucAlignment;
	char	cListChar;

	fail(pFile == NULL || lFilesize <= 0 || pDiag == NULL);

	DBG_MSG("bWord2Text");

	iWordVersion = iInitDocument(pFile, lFilesize);
	if (iWordVersion < 0) {
		DBG_DEC(iWordVersion);
		return FALSE;
	}
	vAddFonts2Diagram(pDiag);

	/* Initialisation */
#if defined(__riscos)
	lCharCounter = 0;
	iCurrPct = 0;
	iPrevPct = -1;
	lDocumentLength = (long)ulGetDocumentLength();
#endif /* __riscos */
	bWord6MacFile = bIsWord6MacFile();
	pSection = pGetSectionInfo(NULL, 0);
	pSectionNext = pSection;
	lDefaultTabWidth = lGetDefaultTabWidth();
	DBG_DEC_C(lDefaultTabWidth != 36000, lDefaultTabWidth);
	pRowInfo = pGetNextRowInfoListItem();
	DBG_HEX_C(pRowInfo != NULL, pRowInfo->ulFileOffsetStart);
	DBG_HEX_C(pRowInfo != NULL, pRowInfo->ulFileOffsetEnd);
	DBG_MSG_C(pRowInfo == NULL, "No rows at all");
	bStartRow = FALSE;
	bEndRowNorm = FALSE;
	bEndRowFast = FALSE;
	bIsTableRow = FALSE;
	bWasTableRow = FALSE;
	vResetStyles();
	pStyleInfo = pGetNextStyleInfoListItem(NULL);
	bStartStyle = FALSE;
	bWasInList = FALSE;
	usIstdNext = ISTD_NORMAL;
	pAnchor = NULL;
	pFontInfo = pGetNextFontInfoListItem(NULL);
	DBG_HEX_C(pFontInfo != NULL, pFontInfo->ulFileOffset);
	DBG_MSG_C(pFontInfo == NULL, "No fonts at all");
	bStartFont = FALSE;
	ucFontnumber = 0;
	ucFontstyleMinimal = FONT_REGULAR;
	ucFontstyle = FONT_REGULAR;
	sFontsize = DEFAULT_FONT_SIZE;
	ucFontcolor = FONT_COLOR_DEFAULT;
	pAnchor = pStartNewOutput(pAnchor, NULL);
	pOutput = pAnchor;
	pOutput->iColor = (int)ucFontcolor;
	pOutput->ucFontstyle = ucFontstyle;
	pOutput->tFontRef = tOpenFont(ucFontnumber, ucFontstyle, sFontsize);
	pOutput->sFontsize = sFontsize;
	bTableFontClosed = TRUE;
	lBeforeIndentation = 0;
	lAfterIndentation = 0;
	lLeftIndentation = 0;
	lRightIndentation = 0;
	bWasEndOfParagraph = TRUE;
	bNumPause = TRUE;
	ucNFC = LIST_BULLETS;
	cListChar = OUR_BULLET;
	uiListNumber = 0;
	ucAlignment = ALIGNMENT_LEFT;
	bAllCapitals = FALSE;
	bHiddenText = FALSE;
	vGetOptions(&tOptions);
	fail(tOptions.iParagraphBreak < 0);
	if (tOptions.iParagraphBreak == 0) {
		lWidthMax = LONG_MAX;
	} else if (tOptions.iParagraphBreak < MIN_SCREEN_WIDTH) {
		lWidthMax = lChar2MilliPoints(MIN_SCREEN_WIDTH);
	} else if (tOptions.iParagraphBreak > MAX_SCREEN_WIDTH) {
		lWidthMax = lChar2MilliPoints(MAX_SCREEN_WIDTH);
	} else {
		lWidthMax = lChar2MilliPoints(tOptions.iParagraphBreak);
	}
	NO_DBG_DEC(lWidthMax);

	visdelay_begin();

	uiFootnoteNumber = 0;
	uiEndnoteNumber = 0;
	eListID = text_list;
	for(;;) {
		ulChar = ulGetChar(pFile, eListID);
		if (ulChar == (ULONG)EOF) {
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			switch (eListID) {
			case text_list:
				eListID = footnote_list;
				if (uiFootnoteNumber != 0) {
					vPutNoteSeparator(pAnchor);
					OUTPUT_LINE();
					uiFootnoteNumber = 0;
				}
				break;
			case footnote_list:
				eListID = endnote_list;
				if (uiEndnoteNumber != 0) {
					vPutNoteSeparator(pAnchor);
					OUTPUT_LINE();
					uiEndnoteNumber = 0;
				}
				break;
			case endnote_list:
			default:
				eListID = end_of_lists;
				break;
			}
			if (eListID == end_of_lists) {
				break;
			}
			continue;
		}

		if (ulChar == UNKNOWN_NOTE_CHAR) {
			switch (eListID) {
			case footnote_list:
				ulChar = FOOTNOTE_CHAR;
				break;
			case endnote_list:
				ulChar = ENDNOTE_CHAR;
				break;
			default:
				break;
			}
		}

		if (bStartRow) {
			/* Begin of a tablerow found */
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			fail(pAnchor != pOutput);
			if (bTableFontClosed) {
				/* Start special table font */
				vCloseFont();
				/*
				 * Compensate for the fact that Word uses
				 * proportional fonts for its tables and we
				 * only one fixed-width font
				 */
				pOutput->sFontsize =
					(sFontsize <= DEFAULT_FONT_SIZE ?
					 (DEFAULT_FONT_SIZE * 5 + 3) / 6 :
					 (sFontsize * 5 + 3) / 6);
				pOutput->tFontRef =
					tOpenTableFont(pOutput->sFontsize);
				pOutput->ucFontstyle = FONT_REGULAR;
				pOutput->iColor = FONT_COLOR_BLACK;
				bTableFontClosed = FALSE;
			}
			bIsTableRow = TRUE;
			bStartRow = FALSE;
		}

		if (bWasTableRow &&
		    !bIsTableRow &&
		    ulChar != PAR_END &&
		    ulChar != HARD_RETURN &&
		    ulChar != PAGE_BREAK &&
		    ulChar != COLUMN_FEED) {
			/*
			 * The end of a table should be followed by an
			 * empty line, like the end of a paragraph
			 */
			OUTPUT_LINE();
			vEndOfParagraph2Diagram(pDiag,
					pOutput->tFontRef,
					pOutput->sFontsize,
					(long)pOutput->sFontsize * 600);
		}

		switch (ulChar) {
		case PAGE_BREAK:
		case COLUMN_FEED:
			if (bIsTableRow) {
				vStoreCharacter((ULONG)'\n', pOutput);
				break;
			}
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			if (ulChar == PAGE_BREAK) {
				vEndOfPage2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->sFontsize,
						lAfterIndentation);
			} else {
				vEndOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->sFontsize,
						lAfterIndentation);
			}
			break;
		default:
			break;
		}

		if (bStartFont || (bStartFontNext && ulChar != PAR_END)) {
			/* Begin of a font found */
			if (bStartFont) {
				/* bStartFont takes priority */
				fail(pFontInfo == NULL);
				pFontTmp = pFontInfo;
			} else {
				pFontTmp = &tFontNext;
			}
			bAllCapitals = bIsCapitals(pFontTmp->ucFontstyle);
			bHiddenText = bIsHidden(pFontTmp->ucFontstyle);
			ucTmp = pFontTmp->ucFontstyle &
				(FONT_BOLD|FONT_ITALIC|FONT_UNDERLINE|
				 FONT_STRIKE|FONT_MARKDEL);
			if (!bIsTableRow &&
			    (sFontsize != pFontTmp->sFontsize ||
			    ucFontnumber != pFontTmp->ucFontnumber ||
			    ucFontstyleMinimal != ucTmp ||
			    ucFontcolor != pFontTmp->ucFontcolor)) {
				pOutput = pStartNextOutput(pOutput);
				vCloseFont();
				pOutput->iColor = (int)pFontTmp->ucFontcolor;
				pOutput->ucFontstyle = pFontTmp->ucFontstyle;
				pOutput->sFontsize = pFontTmp->sFontsize;
				pOutput->tFontRef = tOpenFont(
						pFontTmp->ucFontnumber,
						pFontTmp->ucFontstyle,
						pFontTmp->sFontsize);
				fail(!bCheckDoubleLinkedList(pAnchor));
			}
			ucFontnumber = pFontTmp->ucFontnumber;
			sFontsize = pFontTmp->sFontsize;
			ucFontcolor = pFontTmp->ucFontcolor;
			ucFontstyle = pFontTmp->ucFontstyle;
			ucFontstyleMinimal = ucTmp;
			if (bStartFont) {
				/* Get the next font info */
				pFontInfo = pGetNextFontInfoListItem(pFontInfo);
				NO_DBG_HEX_C(pFontInfo != NULL,
						pFontInfo->ulFileOffset);
				DBG_MSG_C(pFontInfo == NULL, "No more fonts");
			}
			bStartFont = FALSE;
			bStartFontNext = FALSE;
		}

		if (bStartStyle || (bStartStyleNext && ulChar != PAR_END)) {
			/* Begin of a style found */
			if (bStartStyle) {
				/* bStartStyle takes priority */
				fail(pStyleInfo == NULL);
				pStyleTmp = pStyleInfo;
			} else {
				pStyleTmp = &tStyleNext;
			}
			if (!bIsTableRow) {
				vStoreStyle(pOutput, pStyleTmp);
			}
			usIstdNext = pStyleTmp->usIstdNext;
			lBeforeIndentation =
				lTwips2MilliPoints(pStyleTmp->usBeforeIndent);
			lAfterIndentation =
				lTwips2MilliPoints(pStyleTmp->usAfterIndent);
			lLeftIndentation =
				lTwips2MilliPoints(pStyleTmp->sLeftIndent);
			lRightIndentation =
				lTwips2MilliPoints(pStyleTmp->sRightIndent);
			bNumPause = !pStyleTmp->bInList ||
						pStyleTmp->bNumPause;
			ucNFC = pStyleTmp->ucNFC;
			cListChar = (char)pStyleTmp->ucListCharacter;
			ucAlignment = pStyleTmp->ucAlignment;
			if (pStyleTmp->bInList) {
				if (bWasInList) {
					if (!pStyleTmp->bNumPause) {
						uiListNumber++;
					}
				} else {
					uiListNumber =
						(UINT)pStyleTmp->usStartAt;
				}
			} else {
				uiListNumber = 0;
			}
			bWasInList = pStyleTmp->bInList;
			if (bStartStyle) {
				pStyleInfo =
					pGetNextStyleInfoListItem(pStyleInfo);
				NO_DBG_HEX_C(pStyleInfo != NULL,
						pStyleInfo->ulFileOffset);
				NO_DBG_MSG_C(pStyleInfo == NULL,
						"No more styles");
			}
			bStartStyle = FALSE;
			bStartStyleNext = FALSE;
		}

		if (bWasEndOfParagraph) {
			vStartOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->sFontsize,
						lBeforeIndentation);
			bWasEndOfParagraph = FALSE;
		}

		if (!bIsTableRow &&
		    lTotalStringWidth(pAnchor) == 0) {
			vPutIndentation(pDiag, pAnchor, bNumPause,
					uiListNumber, ucNFC, cListChar,
					lLeftIndentation);
			/* One number or mark per paragraph will do */
			bNumPause = TRUE;
		}

		switch (ulChar) {
		case PICTURE:
			(void)memset(&tImage, 0, sizeof(tImage));
			eRes = eExamineImage(pFile, ulFileOffsetImage, &tImage);
			switch (eRes) {
			case image_no_information:
				bSuccess = FALSE;
				break;
			case image_minimal_information:
			case image_full_information:
#if 0
				if (bOutputContainsText(pAnchor)) {
					OUTPUT_LINE();
				} else {
					RESET_LINE();
				}
#endif
				bSuccess = bTranslateImage(pDiag, pFile,
					eRes == image_minimal_information,
					ulFileOffsetImage, &tImage);
				break;
			default:
				DBG_DEC(eRes);
				bSuccess = FALSE;
				break;
			}
			if (!bSuccess) {
				vStoreString("[pic]", 5, pOutput);
			}
			break;
		case FOOTNOTE_CHAR:
			uiFootnoteNumber++;
			vStoreCharacter((ULONG)'[', pOutput);
			vStoreNumberAsDecimal(uiFootnoteNumber, pOutput);
			vStoreCharacter((ULONG)']', pOutput);
			break;
		case ENDNOTE_CHAR:
			uiEndnoteNumber++;
			vStoreCharacter((ULONG)'[', pOutput);
			vStoreNumberAsRoman(uiEndnoteNumber, pOutput);
			vStoreCharacter((ULONG)']', pOutput);
			break;
		case UNKNOWN_NOTE_CHAR:
			vStoreString("[?]", 3, pOutput);
			break;
		case PAR_END:
			if (bIsTableRow) {
				vStoreCharacter((ULONG)'\n', pOutput);
				break;
			}
			OUTPUT_LINE();
			vEndOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->sFontsize,
						lAfterIndentation);
			bWasEndOfParagraph = TRUE;
			break;
		case HARD_RETURN:
			if (bIsTableRow) {
				vStoreCharacter((ULONG)'\n', pOutput);
				break;
			}
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			}
			vEmptyLine2Diagram(pDiag,
					pOutput->tFontRef,
					pOutput->sFontsize);
			break;
		case PAGE_BREAK:
		case COLUMN_FEED:
			pSection = pSectionNext;
			break;
		case TABLE_SEPARATOR:
			if (bIsTableRow) {
				vStoreCharacter(ulChar, pOutput);
				break;
			}
			vStoreCharacter((ULONG)' ', pOutput);
			vStoreCharacter(TABLE_SEPARATOR_CHAR, pOutput);
			break;
		case TAB:
			if (bIsTableRow) {
				vStoreCharacter((ULONG)' ', pOutput);
				break;
			}
			if (tOptions.iParagraphBreak == 0 &&
			    !tOptions.bUseOutlineFonts) {
				/* No logical lines, so no tab expansion */
				vStoreCharacter(TAB, pOutput);
				break;
			}
			lTmp = lTotalStringWidth(pAnchor);
			lTmp += lDrawUnits2MilliPoints(pDiag->lXleft);
			lTmp /= lDefaultTabWidth;
			do {
				vStoreCharacter(FILLER_CHAR, pOutput);
				lWidthCurr = lTotalStringWidth(pAnchor);
				lWidthCurr +=
					lDrawUnits2MilliPoints(pDiag->lXleft);
			} while (lTmp == lWidthCurr / lDefaultTabWidth &&
				 lWidthCurr < lWidthMax + lRightIndentation);
			break;
		default:
			if (bHiddenText && tOptions.bHideHiddenText) {
				continue;
			}
			if (bAllCapitals) {
				ulChar = ulToUpper(ulChar);
			}
			vStoreCharacter(ulChar, pOutput);
			break;
		}

		if (bWasTableRow && !bIsTableRow) {
			/* End of a table, resume normal font */
			NO_DBG_MSG("End of table font");
			vCloseFont();
			bTableFontClosed = TRUE;
			pOutput->iColor = ucFontcolor;
			pOutput->ucFontstyle = ucFontstyle;
			pOutput->sFontsize = sFontsize;
			pOutput->tFontRef =
				tOpenFont(ucFontnumber, ucFontstyle, sFontsize);
		}
		bWasTableRow = bIsTableRow;

		if (bIsTableRow) {
			fail(pAnchor != pOutput);
			if (!bEndRowNorm && !bEndRowFast) {
				continue;
			}
			/* End of a table row */
			if (bEndRowNorm) {
				fail(pRowInfo == NULL);
				vTableRow2Window(pDiag, pAnchor, pRowInfo);
			} else {
				fail(!bEndRowFast);
			}
			/* Reset */
			pAnchor = pStartNewOutput(pAnchor, NULL);
			pOutput = pAnchor;
			if (bEndRowNorm) {
				pRowInfo = pGetNextRowInfoListItem();
			}
			bIsTableRow = FALSE;
			bEndRowNorm = FALSE;
			bEndRowFast = FALSE;
			NO_DBG_HEX_C(pRowInfo != NULL,
						pRowInfo->ulFileOffsetStart);
			NO_DBG_HEX_C(pRowInfo != NULL,
						pRowInfo->ulFileOffsetEnd);
			continue;
		}
		lWidthCurr = lTotalStringWidth(pAnchor);
		lWidthCurr += lDrawUnits2MilliPoints(pDiag->lXleft);
		if (lWidthCurr < lWidthMax + lRightIndentation) {
			continue;
		}
		pLeftOver = pSplitList(pAnchor);
		vJustify2Window(pDiag, pAnchor,
				lWidthMax, lRightIndentation, ucAlignment);
		pAnchor = pStartNewOutput(pAnchor, pLeftOver);
		for (pOutput = pAnchor;
		     pOutput->pNext != NULL;
		     pOutput = pOutput->pNext)
			;	/* EMPTY */
		fail(pOutput == NULL);
		if (lTotalStringWidth(pAnchor) > 0) {
			vSetLeftIndentation(pDiag, lLeftIndentation);
		}
	}

	pAnchor = pStartNewOutput(pAnchor, NULL);
	pAnchor->szStorage = xfree(pAnchor->szStorage);
	pAnchor = xfree(pAnchor);
	vCloseFont();
	vFreeDocument();
	visdelay_end();
	return TRUE;
} /* end of bWord2Text */
