/*
 * stylelist.c
 * Copyright (C) 1998-2002 A.J. van Os; Released under GPL
 *
 * Description:
 * Build, read and destroy a list of Word style information
 */

#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include "antiword.h"


/*
 * Private structure to hide the way the information
 * is stored from the rest of the program
 */
typedef struct style_mem_tag {
	style_block_type	tInfo;
	ULONG			ulSequenceNumber;
	struct style_mem_tag	*pNext;
} style_mem_type;

/* Variables needed to write the Style Information List */
static style_mem_type	*pAnchor = NULL;
static style_mem_type	*pStyleLast = NULL;
/* Values for efficiency reasons */
static const style_mem_type	*pMidPtr = NULL;
static BOOL		bMoveMidPtr = FALSE;
static BOOL		bInSequence = TRUE;


/*
 * vDestroyStyleInfoList - destroy the Style Information List
 */
void
vDestroyStyleInfoList(void)
{
	style_mem_type	*pCurr, *pNext;

	DBG_MSG("vDestroyStyleInfoList");

	/* Free the Style Information List */
	pCurr = pAnchor;
	while (pCurr != NULL) {
		pNext = pCurr->pNext;
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}
	pAnchor = NULL;
	/* Reset all control variables */
	pStyleLast = NULL;
	pMidPtr = NULL;
	bMoveMidPtr = FALSE;
	bInSequence = TRUE;
} /* end of vDestroyStyleInfoList */

/*
 * ucChooseListCharacter - choose our list character
 */
static UCHAR
ucChooseListCharacter(UCHAR ucNFC, UCHAR ucListCharacter)
{
	if (isprint(ucListCharacter)) {
		return ucListCharacter;
	}
	if (ucNFC == LIST_BULLETS) {
		switch (ucListCharacter) {
		case 0x00:
		case 0xb7:
			return (UCHAR)OUR_BULLET;
		case 0xa8:
			return (UCHAR)OUR_DIAMOND;
		case 0xde:
			return (UCHAR)'=';
		case 0xe0:
			return (UCHAR)'o';
		case 0xe1:
			return (UCHAR)'(';
		case 0xfe:
			return (UCHAR)' ';
		default:
			DBG_HEX(ucListCharacter);
			return (UCHAR)OUR_BULLET;
		}
	}
	return (UCHAR)'.';
} /* end of ucChooseListCharacter */

/*
 * eGetNumType - get level type from level number
 */
level_type_enum
eGetNumType(UCHAR ucLevelNumber)
{
	switch (ucLevelNumber) {
	case 10:
		return level_type_numbering;
	case 11:
		return level_type_sequence;
	case 12:
		return level_type_pause;
	default:
		if ((int)ucLevelNumber >= 1 && (int)ucLevelNumber <= 9) {
			return level_type_outline;
		}
		return level_type_none;
	}
} /* end of eGetNumType */

/*
 * vCorrectStyleValues - correct style values that Antiword can't use
 */
void
vCorrectStyleValues(style_block_type *pStyleBlock)
{
	if (pStyleBlock->usBeforeIndent > 0x7fff) {
		pStyleBlock->usBeforeIndent = 0;
	} else if (pStyleBlock->usBeforeIndent > 2160) {
		/* 2160 twips = 1.5 inches or 38.1 mm */
		DBG_DEC(pStyleBlock->usBeforeIndent);
		pStyleBlock->usBeforeIndent = 2160;
	}
	if (pStyleBlock->usIstd >= 1 &&
	    pStyleBlock->usIstd <= 9 &&
	    pStyleBlock->usBeforeIndent < HEADING_GAP) {
		NO_DBG_DEC(pStyleBlock->usBeforeIndent);
		pStyleBlock->usBeforeIndent = HEADING_GAP;
	}
	if (pStyleBlock->usAfterIndent > 0x7fff) {
		pStyleBlock->usAfterIndent = 0;
	} else if (pStyleBlock->usAfterIndent > 2160) {
		/* 2160 twips = 1.5 inches or 38.1 mm */
		DBG_DEC(pStyleBlock->usAfterIndent);
		pStyleBlock->usAfterIndent = 2160;
	}
	if (pStyleBlock->usIstd >= 1 &&
	    pStyleBlock->usIstd <= 9 &&
	    pStyleBlock->usAfterIndent < HEADING_GAP) {
		NO_DBG_DEC(pStyleBlock->usAfterIndent);
		pStyleBlock->usAfterIndent = HEADING_GAP;
	}
	if (pStyleBlock->sLeftIndent < 0) {
		pStyleBlock->sLeftIndent = 0;
	}
	if (pStyleBlock->sRightIndent > 0) {
		pStyleBlock->sRightIndent = 0;
	}
	pStyleBlock->ucListCharacter =
			ucChooseListCharacter(
				pStyleBlock->ucNFC,
				pStyleBlock->ucListCharacter);
} /* end of vCorrectStyleValues */

/*
 * vAdd2StyleInfoList - Add an element to the Style Information List
 */
void
vAdd2StyleInfoList(const style_block_type *pStyleBlock)
{
	style_mem_type	*pListMember;

	fail(pStyleBlock == NULL);

	NO_DBG_MSG("bAdd2StyleInfoList");

	if (pStyleBlock->ulFileOffset == FC_INVALID) {
		NO_DBG_DEC(pStyleBlock->usIstd);
		return;
	}

	NO_DBG_HEX(pStyleBlock->ulFileOffset);
	NO_DBG_DEC_C(pStyleBlock->sLeftIndent != 0,
					pStyleBlock->sLeftIndent);
	NO_DBG_DEC_C(pStyleBlock->sRightIndent != 0,
					pStyleBlock->sRightIndent);
	NO_DBG_DEC_C(pStyleBlock->bInList, pStyleBlock->bInList);
	NO_DBG_DEC_C(pStyleBlock->bNumPause, pStyleBlock->bNumPause);
	NO_DBG_DEC_C(pStyleBlock->usIstd != 0, pStyleBlock->usIstd);
	NO_DBG_DEC_C(pStyleBlock->usStartAt != 0, pStyleBlock->usStartAt);
	NO_DBG_DEC_C(pStyleBlock->usAfterIndent != 0,
					pStyleBlock->usAfterIndent);
	NO_DBG_DEC_C(pStyleBlock->ucAlignment != 0, pStyleBlock->ucAlignment);
	NO_DBG_DEC_C(pStyleBlock->bInList, pStyleBlock->ucNFC);
	NO_DBG_HEX_C(pStyleBlock->bInList, pStyleBlock->ucListCharacter);
	NO_DBG_DEC_C(pStyleBlock->ucNFC != LIST_BULLETS,
					pStyleBlock->usStartAt);

	if (pStyleLast != NULL &&
	    pStyleLast->tInfo.ulFileOffset == pStyleBlock->ulFileOffset) {
		/*
		 * If two consecutive styles share the same
		 * offset, remember only the last style
		 */
		fail(pStyleLast->pNext != NULL);
		pStyleLast->tInfo = *pStyleBlock;
		return;
	}

	/* Create list member */
	pListMember = xmalloc(sizeof(style_mem_type));
	/* Fill the list member */
	pListMember->tInfo = *pStyleBlock;
	pListMember->pNext = NULL;
	/* Add the sequence number */
	pListMember->ulSequenceNumber =
			ulGetSeqNumber(pListMember->tInfo.ulFileOffset);
	/* Correct the values where needed */
	vCorrectStyleValues(&pListMember->tInfo);
	/* Add the new member to the list */
	if (pAnchor == NULL) {
		pAnchor = pListMember;
		/* For efficiency */
		pMidPtr = pAnchor;
		bMoveMidPtr = FALSE;
		bInSequence = TRUE;
	} else {
		fail(pStyleLast == NULL);
		pStyleLast->pNext = pListMember;
		/* For efficiency */
		if (bMoveMidPtr) {
			pMidPtr = pMidPtr->pNext;
			bMoveMidPtr = FALSE;
		} else {
			bMoveMidPtr = TRUE;
		}
		if (bInSequence) {
			bInSequence = pListMember->ulSequenceNumber >
					pStyleLast->ulSequenceNumber;
		}
	}
	pStyleLast = pListMember;
} /* end of vAdd2StyleInfoList */

/*
 * Get the record that follows the given recored in the Style Information List
 */
const style_block_type *
pGetNextStyleInfoListItem(const style_block_type *pCurr)
{
	const style_mem_type	*pRecord;
	size_t	tOffset;

	if (pCurr == NULL) {
		if (pAnchor == NULL) {
			/* There are no records */
			return NULL;
		}
		/* The first record is the only one without a predecessor */
		return &pAnchor->tInfo;
	}
	tOffset = offsetof(style_mem_type, tInfo);
	/* Many casts to prevent alignment warnings */
	pRecord = (style_mem_type *)(void *)((char *)pCurr - tOffset);
	fail(pCurr != &pRecord->tInfo);
	if (pRecord->pNext == NULL) {
		/* The last record has no successor */
		return NULL;
	}
	return &pRecord->pNext->tInfo;
} /* end of pGetNextStyleInfoListItem */

/*
 * usGetIstd - get the istd that belongs to the given file offset
 */
USHORT
usGetIstd(ULONG ulFileOffset)
{
	const style_mem_type	*pCurr, *pBest, *pStart;
	ULONG	ulSeq, ulBest;

	ulSeq = ulGetSeqNumber(ulFileOffset);
	if (ulSeq == FC_INVALID) {
		return ISTD_NORMAL;
	}
	NO_DBG_HEX(ulFileOffset);
	NO_DBG_DEC(ulSeq);

	if (bInSequence &&
	    pMidPtr != NULL &&
	    ulSeq > pMidPtr->ulSequenceNumber) {
		/* The istd is in the second half of the chained list */
		pStart = pMidPtr;
	} else {
		pStart = pAnchor;
	}

	pBest = NULL;
	ulBest = 0;
	for (pCurr = pStart; pCurr != NULL; pCurr = pCurr->pNext) {
		if (pCurr->ulSequenceNumber != FC_INVALID &&
		    (pBest == NULL || pCurr->ulSequenceNumber > ulBest) &&
		    pCurr->ulSequenceNumber <= ulSeq) {
			pBest = pCurr;
			ulBest = pCurr->ulSequenceNumber;
		}
		if (bInSequence && pCurr->ulSequenceNumber > ulSeq) {
			break;
		}
	}
	NO_DBG_DEC(ulBest);

	if (pBest == NULL) {
		return ISTD_NORMAL;
	}

	NO_DBG_DEC(pBest->tInfo.usIstd);
	return pBest->tInfo.usIstd;
} /* end of usGetIstd */
