/*
 * wordlib.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Deal with the internals of a MS Word file
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "antiword.h"

typedef struct pps_entry_tag {
	char	szName[32];
	int	iType;
	int	iNext;
	int	iPrev;
	int	iDir;
	int	iSb;
	int	iSize;
	int	iLevel;
} pps_entry_type;

static BOOL	bMacFile = FALSE;

/* Macro to make sure all such statements will be identical */
#define FREE_ALL()		\
	do {\
		vDestroySmallBlockList();\
		aiRootList = xfree(aiRootList);\
		aiSbdList = xfree(aiSbdList);\
		aiBbdList = xfree(aiBbdList);\
		aiSBD = xfree(aiSBD);\
		aiBBD = xfree(aiBBD);\
	} while(0)


/*
 * ulReadLong - read four bytes from the given file and offset
 */
static unsigned long
ulReadLong(FILE *pFile, long lOffset)
{
	unsigned char	aucBytes[4];

	fail(pFile == NULL || lOffset < 0);

	if (!bReadBytes(aucBytes, 4, lOffset, pFile)) {
		werr(1, "Read long %ld not possible", lOffset);
	}
	return ulGetLong(0, aucBytes);
} /* end of ulReadLong */

/*
 * vName2String - turn the name into a proper string.
 */
static void
vName2String(char *szName, const unsigned char *aucBytes, int iNameSize)
{
	char	*pcChar;
	int	iIndex;

	fail(aucBytes == NULL || szName == NULL);

	if (iNameSize <= 0) {
		szName[0] = '\0';
		return;
	}
	for (iIndex = 0, pcChar = szName;
	     iIndex < 2 * iNameSize;
	     iIndex += 2, pcChar++) {
		*pcChar = (char)aucBytes[iIndex];
	}
	szName[iNameSize - 1] = '\0';
} /* end of vName2String */

/*
 * iReadBlockIndices - read the Big/Small Block Depot indices
 *
 * returns the number of indices read
 */
static int
iReadBlockIndices(FILE *pFile, int *aiBlockDepot, int iMaxRec, int iOffset)
{
	int	iIndex, iDone;
	unsigned char	aucBytes[BIG_BLOCK_SIZE];

	fail(pFile == NULL || aiBlockDepot == NULL);
	fail(iMaxRec < 0 || iOffset < 0);

	/* Read a big block with BBD or SBD indices */
	if (!bReadBytes(aucBytes, BIG_BLOCK_SIZE, iOffset, pFile)) {
		werr(0, "Reading big block from %d is not possible", iOffset);
		return -1;
	}
	/* Split the big block into indices, an index is four bytes */
	iDone = min(iMaxRec, BIG_BLOCK_SIZE / 4);
	for (iIndex = 0; iIndex < iDone; iIndex++) {
		aiBlockDepot[iIndex] = (int)ulGetLong(4 * iIndex, aucBytes);
		NO_DBG_DEC(aiBlockDepot[iIndex]);
	}
	return iDone;
} /* end of iReadBlockIndices */

/*
 * bGetBBD - get the Big Block Depot indices from the index-blocks
 */
static BOOL
bGetBBD(FILE *pFile, const int *aiDepot, int iDepotLen,
			int *aiBBD, int iBBDLen)
{
	int	iIndex, iBegin, iToGo, iDone;

	fail(pFile == NULL || aiDepot == NULL || aiBBD == NULL);
	fail(iDepotLen < 0 || iBBDLen < 0);

	DBG_MSG("bGetBBD");

	iToGo = iBBDLen;
	for (iIndex = 0; iIndex < iDepotLen && iToGo > 0; iIndex++) {
		iBegin = (aiDepot[iIndex] + 1) * BIG_BLOCK_SIZE;
		NO_DBG_HEX(iBegin);
		iDone = iReadBlockIndices(pFile, aiBBD, iToGo, iBegin);
		if (iDone < 0) {
			return FALSE;
		}
		aiBBD += iDone;
		iToGo -= iDone;
	}
	return iToGo <= 0;
} /* end of bGetBBD */

/*
 * bGetSBD - get the Small Block Depot indices from the index-blocks
 */
static BOOL
bGetSBD(FILE *pFile, const int *aiDepot, int iDepotLen,
			int *aiSBD, int iSBDLen)
{
	int	iIndex, iBegin, iToGo, iDone;

	fail(pFile == NULL || aiDepot == NULL || aiSBD == NULL);
	fail(iDepotLen < 0 || iSBDLen < 0);

	DBG_MSG("bGetSBD");

	iToGo = iSBDLen;
	for (iIndex = 0; iIndex < iDepotLen && iToGo > 0; iIndex++) {
		iBegin = (aiDepot[iIndex] + 1) * BIG_BLOCK_SIZE;
		NO_DBG_HEX(iBegin);
		iDone = iReadBlockIndices(pFile, aiSBD, iToGo, iBegin);
		if (iDone < 0) {
			return FALSE;
		}
		aiSBD += iDone;
		iToGo -= iDone;
	}
	return iToGo <= 0;
} /* end of bGetSBD */

/*
 * vComputePPSlevels - compute the levels of the Property Set Storage entries
 */
static void
vComputePPSlevels(pps_entry_type *atPPSlist, pps_entry_type *ptNode,
			int iLevel, int iRecursionLevel)
{
	fail(atPPSlist == NULL || ptNode == NULL);
	fail(iLevel < 0 || iRecursionLevel < 0);

	if (iRecursionLevel > 50) {
		/* Remove the possibility of an infinite recursion */
		DBG_DEC(iRecursionLevel);
		return;
	}

	ptNode->iLevel = iLevel;

	if (ptNode->iDir != -1) {
		vComputePPSlevels(atPPSlist,
				&atPPSlist[ptNode->iDir],
				iLevel + 1,
				iRecursionLevel + 1);
	}
	if (ptNode->iNext != -1) {
		vComputePPSlevels(atPPSlist,
				&atPPSlist[ptNode->iNext],
				iLevel,
				iRecursionLevel + 1);
	}
	if (ptNode->iPrev != -1) {
		vComputePPSlevels(atPPSlist,
				&atPPSlist[ptNode->iPrev],
				iLevel,
				iRecursionLevel + 1);
	}
} /* end of vComputePPSlevels */

/*
 * bGetPPS - search the Property Set Storage for three sets
 *
 * Return TRUE if the WordDocument PPS is found
 */
static BOOL
bGetPPS(FILE *pFile,
		const int *aiRootList, int iRootListLen,
		pps_info_type *pPPS)
{
	pps_entry_type	*atPPSlist;
	int	iTmp, iIndex, iStartBlock, iOffset, iBegin;
	int	iNameSize, iRootIndex, iNbrOfPPS;
	BOOL	bWord, bExcel;
	unsigned char	aucBytes[PROPERTY_SET_STORAGE_SIZE];

	fail(pFile == NULL || pPPS == NULL);
	fail(aiRootList == NULL || iRootListLen < 0);

	DBG_MSG("bGetPPS");
	NO_DBG_DEC(iRootListLen);

	bWord = FALSE;
	bExcel = FALSE;
	(void)memset(pPPS, 0, sizeof(*pPPS));

	/* Read and store all the Property Set Storage entries */
	iNbrOfPPS = iRootListLen * BIG_BLOCK_SIZE / PROPERTY_SET_STORAGE_SIZE;
	atPPSlist = xmalloc(iNbrOfPPS * sizeof(pps_entry_type));
	iRootIndex = 0;
	for (iIndex = 0; iIndex < iNbrOfPPS; iIndex++) {
		iTmp = iIndex * PROPERTY_SET_STORAGE_SIZE;
		iStartBlock = iTmp / BIG_BLOCK_SIZE;
		iOffset = iTmp % BIG_BLOCK_SIZE;
		iBegin = (aiRootList[iStartBlock] + 1) * BIG_BLOCK_SIZE +
			iOffset;
		NO_DBG_HEX(iBegin);
		if (!bReadBytes(aucBytes, PROPERTY_SET_STORAGE_SIZE,
							iBegin, pFile)) {
			werr(0, "Reading PPS %d is not possible", iIndex);
			atPPSlist = xfree(atPPSlist);
			return FALSE;
		}
		iNameSize = (int)usGetWord(0x40, aucBytes);
		iNameSize = (iNameSize + 1) / 2;
		vName2String(atPPSlist[iIndex].szName, aucBytes, iNameSize);
		atPPSlist[iIndex].iType = (int)ucGetByte(0x42, aucBytes);
		if (atPPSlist[iIndex].iType == 5) {
			iRootIndex = iIndex;
		}
		atPPSlist[iIndex].iPrev = (int)ulGetLong(0x44, aucBytes);
		atPPSlist[iIndex].iNext = (int)ulGetLong(0x48, aucBytes);
		atPPSlist[iIndex].iDir = (int)ulGetLong(0x4c, aucBytes);
		atPPSlist[iIndex].iSb = (int)ulGetLong(0x74, aucBytes);
		atPPSlist[iIndex].iSize = (int)ulGetLong(0x78, aucBytes);
		atPPSlist[iIndex].iLevel = -1;
		if (atPPSlist[iIndex].iPrev < -1 ||
		    atPPSlist[iIndex].iPrev >= iNbrOfPPS ||
		    atPPSlist[iIndex].iNext < -1 ||
		    atPPSlist[iIndex].iNext >= iNbrOfPPS ||
		    atPPSlist[iIndex].iDir < -1 ||
		    atPPSlist[iIndex].iDir >= iNbrOfPPS) {
			werr(0, "The Property Set Storage is damaged");
			atPPSlist = xfree(atPPSlist);
			return FALSE;
		}
	}

#if 0 /* defined(DEBUG) */
	DBG_MSG("Before");
	for (iIndex = 0; iIndex < iNbrOfPPS; iIndex++) {
		DBG_MSG(atPPSlist[iIndex].szName);
		DBG_HEX(atPPSlist[iIndex].iDir);
		DBG_HEX(atPPSlist[iIndex].iPrev);
		DBG_HEX(atPPSlist[iIndex].iNext);
		DBG_DEC(atPPSlist[iIndex].iSb);
		DBG_HEX(atPPSlist[iIndex].iSize);
		DBG_DEC(atPPSlist[iIndex].iLevel);
	}
#endif /* DEBUG */

	/* Add level information to each entry */
	vComputePPSlevels(atPPSlist, &atPPSlist[iRootIndex], 0, 0);

	/* Check the entries on level 1 for the required information */
	NO_DBG_MSG("After");
	for (iIndex = 0; iIndex < iNbrOfPPS; iIndex++) {
#if 0 /* defined(DEBUG) */
		DBG_MSG(atPPSlist[iIndex].szName);
		DBG_HEX(atPPSlist[iIndex].iDir);
		DBG_HEX(atPPSlist[iIndex].iPrev);
		DBG_HEX(atPPSlist[iIndex].iNext);
		DBG_DEC(atPPSlist[iIndex].iSb);
		DBG_HEX(atPPSlist[iIndex].iSize);
		DBG_DEC(atPPSlist[iIndex].iLevel);
#endif /* DEBUG */
		if (atPPSlist[iIndex].iLevel != 1 ||
		    atPPSlist[iIndex].iType != 2 ||
		    atPPSlist[iIndex].szName[0] == '\0' ||
		    atPPSlist[iIndex].iSize <= 0) {
			continue;
		}
		if (pPPS->tWordDocument.iSize <= 0 &&
		    StrEq(atPPSlist[iIndex].szName, "WordDocument")) {
			pPPS->tWordDocument.iSb = atPPSlist[iIndex].iSb;
			pPPS->tWordDocument.iSize = atPPSlist[iIndex].iSize;
			bWord = TRUE;
		} else if (pPPS->t0Table.iSize <= 0 &&
			   StrEq(atPPSlist[iIndex].szName, "0Table")) {
			pPPS->t0Table.iSb = atPPSlist[iIndex].iSb;
			pPPS->t0Table.iSize = atPPSlist[iIndex].iSize;
		} else if (pPPS->t1Table.iSize <= 0 &&
			   StrEq(atPPSlist[iIndex].szName, "1Table")) {
			pPPS->t1Table.iSb = atPPSlist[iIndex].iSb;
			pPPS->t1Table.iSize = atPPSlist[iIndex].iSize;
		} else if (StrEq(atPPSlist[iIndex].szName, "Book") ||
			   StrEq(atPPSlist[iIndex].szName, "Workbook")) {
			bExcel = TRUE;
		}
	}

	/* Free the space for the Property Set Storage entries */
	atPPSlist = xfree(atPPSlist);

	/* Draw your conclusions */
	if (bWord) {
		return TRUE;
	}
	if (bExcel) {
		werr(0, "Sorry, but this is an Excel spreadsheet");
	} else {
		werr(0, "This OLE file does not contain a Word document");
	}
	return FALSE;
} /* end of bGetPPS */

/*
 * bGetDocumentText - make a list of the text blocks of a Word document
 */
static BOOL
bGetDocumentText(FILE *pFile, const pps_info_type *pPPS,
		const int *aiBBD, int iBBDLen,
		const int *aiSBD, int iSBDLen,
		const unsigned char *aucHeader, int iWordVersion)
{
	int	iBeginOfText;
	int	iTextLen, iFootnoteLen, iEndnoteLen;
	int	iHeaderLen, iMacroLen, iAnnotationLen;
	int	iTextBoxLen, iHdrTextBoxLen;
	BOOL	bFarEastWord, bFastSaved, bEncrypted, bSuccess;
	text_info_enum	eResult;
	unsigned short	usDocStatus, usIdent;

	fail(pFile == NULL || pPPS == NULL);
	fail(aiBBD == NULL || iBBDLen < 0);

	DBG_MSG("bGetDocumentText");

	/* Get the "magic number" from the header */
	usIdent = usGetWord(0x00, aucHeader);
	DBG_HEX(usIdent);
	bFarEastWord = usIdent == 0x8098 || usIdent == 0x8099 ||
			usIdent == 0xa697 || usIdent == 0xa699;
	/* Get the status flags from the header */
	usDocStatus = usGetWord(0x0a, aucHeader);
	DBG_HEX(usDocStatus);
	bFastSaved = (usDocStatus & BIT(2)) != 0;
	DBG_MSG_C(bFastSaved, "This document is Fast Saved");
	DBG_DEC_C(bFastSaved, (usDocStatus & 0xf0) >> 4);
	bEncrypted = (usDocStatus & BIT(8)) != 0;
	if (bEncrypted) {
		werr(0, "Encrypted documents are not supported");
		return FALSE;
	}

	/* Get length information */
	iBeginOfText = (int)ulGetLong(0x18, aucHeader);
	DBG_HEX(iBeginOfText);
	if (iWordVersion == 6 || iWordVersion == 7) {
		iTextLen = (int)ulGetLong(0x34, aucHeader);
		iFootnoteLen = (int)ulGetLong(0x38, aucHeader);
		iHeaderLen = (int)ulGetLong(0x3c, aucHeader);
		iMacroLen = (int)ulGetLong(0x40, aucHeader);
		iAnnotationLen = (int)ulGetLong(0x44, aucHeader);
		iEndnoteLen = (int)ulGetLong(0x48, aucHeader);
		iTextBoxLen = (int)ulGetLong(0x4c, aucHeader);
		iHdrTextBoxLen = (int)ulGetLong(0x50, aucHeader);
	} else {
		iTextLen = (int)ulGetLong(0x4c, aucHeader);
		iFootnoteLen = (int)ulGetLong(0x50, aucHeader);
		iHeaderLen = (int)ulGetLong(0x54, aucHeader);
		iMacroLen = (int)ulGetLong(0x58, aucHeader);
		iAnnotationLen = (int)ulGetLong(0x5c, aucHeader);
		iEndnoteLen = (int)ulGetLong(0x60, aucHeader);
		iTextBoxLen = (int)ulGetLong(0x64, aucHeader);
		iHdrTextBoxLen = (int)ulGetLong(0x68, aucHeader);
	}
	DBG_DEC(iTextLen);
	DBG_DEC(iFootnoteLen);
	DBG_DEC(iHeaderLen);
	DBG_DEC(iMacroLen);
	DBG_DEC(iAnnotationLen);
	DBG_DEC(iEndnoteLen);
	DBG_DEC(iTextBoxLen);
	DBG_DEC(iHdrTextBoxLen);

	/* Make a list of the text blocks */
	switch (iWordVersion) {
	case 6:
	case 7:
		if (bFastSaved) {
			eResult = eGet6DocumentText(pFile,
					bFarEastWord,
					pPPS->tWordDocument.iSb,
					aiBBD, iBBDLen,
					aucHeader);
			bSuccess = eResult == text_success;
		} else {
		  	bSuccess = bAddBlocks(iBeginOfText,
				iTextLen +
				iFootnoteLen +
				iHeaderLen + iMacroLen + iAnnotationLen +
				iEndnoteLen +
				iTextBoxLen + iHdrTextBoxLen,
				bFarEastWord,
				pPPS->tWordDocument.iSb,
				aiBBD, iBBDLen);
		}
		break;
	case 8:
		eResult = eGet8DocumentText(pFile,
				pPPS,
				aiBBD, iBBDLen, aiSBD, iSBDLen,
				aucHeader);
		if (eResult == text_no_information && !bFastSaved) {
			/* Assume there is only one block */
			bSuccess = bAddBlocks(iBeginOfText,
				iTextLen +
				iFootnoteLen +
				iHeaderLen + iMacroLen + iAnnotationLen +
				iEndnoteLen +
				iTextBoxLen + iHdrTextBoxLen,
				FALSE,
				pPPS->tWordDocument.iSb,
				aiBBD, iBBDLen);
		} else {
			bSuccess = eResult == text_success;
		}
		break;
	default:
		werr(0, "This version of Word is not supported");
		bSuccess = FALSE;
		break;
	}

	if (bSuccess) {
		vSplitBlockList(iTextLen,
				iFootnoteLen,
				iHeaderLen + iMacroLen + iAnnotationLen,
				iEndnoteLen,
				iTextBoxLen + iHdrTextBoxLen,
				!bFastSaved && iWordVersion == 8);
	} else {
		vDestroyTextBlockList();
		werr(0, "I can't find the text of this document");
	}
	return bSuccess;
} /* end of bGetDocumentText */

/*
 * pInitFile - open and check the file
 */
static FILE *
pInitFile(const char *szFilename)
{
	FILE	*pFile;

	fail(szFilename == NULL || szFilename[0] == '\0');

	pFile = fopen(szFilename, "rb");
	if (pFile == NULL) {
		werr(0, "I can't open '%s' for reading", szFilename);
		return NULL;
	}
	return pFile;
} /* end of pInitFile */

static BOOL
bInitDoc(FILE *pFile, off_t tFilesize)
{
	pps_info_type	PPS_info;
	int	*aiBBD, *aiRootList, *aiBbdList;
	int	*aiSBD, *aiSbdList;
	size_t	tSize;
	int	iRootListLen, iRootStartblock;
	int	iSbdStartblock, iSBLstartblock;
	int	iBBDLen, iSBDLen;
	int	iWordVersion, iIndex, iTmp;
	int	iMaxBlock, iNumBbdBlocks, iMaxSmallBlock;
	BOOL	bResult;
	unsigned short	usIdent, usFib, usChse;
	unsigned char	aucHeader[HEADER_SIZE];

	fail(pFile == NULL);

	iMaxBlock = (int)(tFilesize / BIG_BLOCK_SIZE) - 2;
	DBG_DEC(iMaxBlock);
	if (iMaxBlock < 1) {
		return FALSE;
	}
	iBBDLen = iMaxBlock + 1;
	iNumBbdBlocks = (int)ulReadLong(pFile, 0x2c);
	DBG_DEC(iNumBbdBlocks);
	if (iNumBbdBlocks > 109) {
		DBG_DEC(tFilesize);
		if (tFilesize > 109 * (BIG_BLOCK_SIZE/4) * BIG_BLOCK_SIZE) {
			werr(0, "I'm afraid this file is too big to handle.");
		} else {
			werr(0, "This file is probably damaged. Sorry.");
		}
		return FALSE;
	}
	iRootStartblock = (int)ulReadLong(pFile, 0x30);
	DBG_DEC(iRootStartblock);
	iSbdStartblock = (int)ulReadLong(pFile, 0x3c);
	DBG_DEC(iSbdStartblock);
	iSBLstartblock = (int)ulReadLong(pFile,
		((long)iRootStartblock + 1) * BIG_BLOCK_SIZE + 0x74);
	DBG_DEC(iSBLstartblock);
	iMaxSmallBlock = (int)(ulReadLong(pFile,
		((long)iRootStartblock + 1) *
		BIG_BLOCK_SIZE + 0x78) / SMALL_BLOCK_SIZE) - 1;
	DBG_DEC(iMaxSmallBlock);
	iSBDLen = iMaxSmallBlock + 1;
	/* All to be xmalloc-ed pointers to NULL */
	aiRootList = NULL;
	aiSbdList = NULL;
	aiBbdList = NULL;
	aiSBD = NULL;
	aiBBD = NULL;
/* Big Block Depot */
	tSize = iNumBbdBlocks * sizeof(int);
	aiBbdList = xmalloc(tSize);
	tSize = iBBDLen * sizeof(int);
	aiBBD = xmalloc(tSize);
	for (iIndex = 0; iIndex < iNumBbdBlocks; iIndex++) {
		aiBbdList[iIndex] =
			(int)ulReadLong(pFile, 0x4c + 4 * (long)iIndex);
		NO_DBG_HEX(aiBbdList[iIndex]);
	}
	if (!bGetBBD(pFile, aiBbdList, iNumBbdBlocks, aiBBD, iBBDLen)) {
		FREE_ALL();
		return FALSE;
	}
	aiBbdList = xfree(aiBbdList);
/* Small Block Depot */
	tSize = iBBDLen * sizeof(int);
	aiSbdList = xmalloc(tSize);
	tSize = iSBDLen * sizeof(int);
	aiSBD = xmalloc(tSize);
	for (iIndex = 0, iTmp = iSbdStartblock;
	     iIndex < iBBDLen && iTmp != END_OF_CHAIN;
	     iIndex++, iTmp = aiBBD[iTmp]) {
		if (iTmp < 0 || iTmp >= iBBDLen) {
			werr(1, "The Big Block Depot is damaged");
		}
		aiSbdList[iIndex] = iTmp;
		NO_DBG_HEX(aiSbdList[iIndex]);
	}
	if (!bGetSBD(pFile, aiSbdList, iBBDLen, aiSBD, iSBDLen)) {
		FREE_ALL();
		return FALSE;
	}
	aiSbdList = xfree(aiSbdList);
/* Root list */
	for (iRootListLen = 0, iTmp = iRootStartblock;
	     iRootListLen < iBBDLen && iTmp != END_OF_CHAIN;
	     iRootListLen++, iTmp = aiBBD[iTmp]) {
		if (iTmp < 0 || iTmp >= iBBDLen) {
			werr(1, "The Big Block Depot is damaged");
		}
	}
	if (iRootListLen <= 0) {
		werr(0, "No Rootlist found");
		FREE_ALL();
		return FALSE;
	}
	tSize = iRootListLen * sizeof(int);
	aiRootList = xmalloc(tSize);
	for (iIndex = 0, iTmp = iRootStartblock;
	     iIndex < iBBDLen && iTmp != END_OF_CHAIN;
	     iIndex++, iTmp = aiBBD[iTmp]) {
		if (iTmp < 0 || iTmp >= iBBDLen) {
			werr(1, "The Big Block Depot is damaged");
		}
		aiRootList[iIndex] = iTmp;
		NO_DBG_DEC(aiRootList[iIndex]);
	}
	fail(iRootListLen != iIndex);
	bResult = bGetPPS(pFile, aiRootList, iRootListLen, &PPS_info);
	aiRootList = xfree(aiRootList);
	if (!bResult) {
		FREE_ALL();
		return FALSE;
	}
/* Small block list */
	if (!bCreateSmallBlockList(iSBLstartblock, aiBBD, iBBDLen)) {
		FREE_ALL();
		return FALSE;
	}

	if (PPS_info.tWordDocument.iSize < MIN_SIZE_FOR_BBD_USE) {
		DBG_DEC(PPS_info.tWordDocument.iSize);
		FREE_ALL();
		werr(0, "I'm afraid the text stream of this file "
			"is too small to handle.");
		return FALSE;
	}
	/* Read the headerblock */
	if (!bReadBuffer(pFile, PPS_info.tWordDocument.iSb,
			aiBBD, iBBDLen, BIG_BLOCK_SIZE,
			aucHeader, 0, HEADER_SIZE)) {
		FREE_ALL();
		return FALSE;
	}
	usIdent = usGetWord(0x00, aucHeader);
	DBG_HEX(usIdent);
	fail(usIdent != 0x8098 &&	/* Word 7 for oriental languages */
	     usIdent != 0x8099 &&	/* Word 7 for oriental languages */
	     usIdent != 0xa5dc &&	/* Word 6 & 7 */
	     usIdent != 0xa5ec &&	/* Word 7 & 97 & 98 */
	     usIdent != 0xa697 &&	/* Word 7 for oriental languages */
	     usIdent != 0xa699);	/* Word 7 for oriental languages */
	usFib = usGetWord(0x02, aucHeader);
	DBG_DEC(usFib);
	if (usFib < 101) {
		FREE_ALL();
		werr(0, "This file is from a version of Word before Word 6.");
		return FALSE;
	}
	usChse = usGetWord(0x14, aucHeader);
	DBG_DEC(usChse);
	bMacFile = FALSE;
	/* Get the version of Word */
	switch (usFib) {
	case 101:
	case 102:
		iWordVersion = 6;
		DBG_MSG("Word 6 for Windows");
		break;
	case 103:
	case 104:
		switch (usChse) {
		case 0:
			iWordVersion = 7;
			DBG_MSG("Word 7 for Win95");
			break;
		case 256:
			iWordVersion = 6;
			DBG_MSG("Word 6 for Macintosh");
			bMacFile = TRUE;
			break;
		default:
			if (ucGetByte(0x05, aucHeader) == 0xe0) {
				iWordVersion = 7;
				DBG_MSG("Word 7 for Win95");
			} else {
				iWordVersion = 6;
				DBG_MSG("Word 6 for Macintosh");
				bMacFile = TRUE;
			}
		}
		break;
	default:
		iWordVersion = 8;
		DBG_MSG_C(usChse != 256, "Word97 for Win95/98/NT");
		DBG_MSG_C(usChse == 256, "Word98 for Macintosh");
		/* NOTE: No Macintosh character set !! */
		break;
	}
	bResult = bGetDocumentText(pFile, &PPS_info,
			aiBBD, iBBDLen, aiSBD, iSBDLen,
			aucHeader, iWordVersion);
	if (bResult) {
		vSetDefaultTabWidth(pFile, &PPS_info,
			aiBBD, iBBDLen, aiSBD, iSBDLen,
			aucHeader, iWordVersion);
		vGetParagraphInfo(pFile, &PPS_info,
			aiBBD, iBBDLen, aiSBD, iSBDLen,
			aucHeader, iWordVersion);
		vGetNotesInfo(pFile, &PPS_info,
			aiBBD, iBBDLen, aiSBD, iSBDLen,
			aucHeader, iWordVersion);
	}
	FREE_ALL();
	return bResult;
} /* end of bInitDoc */

/*
 * pOpenDocument - open and initialize the document
 */
FILE *
pOpenDocument(const char *szFilename)
{
	FILE	*pFile;
	off_t	tFilesize;

	tFilesize = tGetFilesize(szFilename);
	if (tFilesize == 0 || tFilesize == (off_t)-1) {
		return NULL;
	}
	pFile = pInitFile(szFilename);
	if (pFile == NULL) {
		return NULL;
	}
	if (bInitDoc(pFile, tFilesize)) {
		return pFile;
	}
	(void)fclose(pFile);
	return NULL;
} /* end of pOpenDocument*/

/*
 * vCloseDocument
 */
void
vCloseDocument(FILE *pFile)
{
	DBG_MSG("vCloseDocument");

	/* Free the memory */
	vDestroyTextBlockList();
	vDestroyRowInfoList();
	vDestroyStyleInfoList();
	vDestroyFontInfoList();
	vDestroyNotesInfoLists();
#if defined(__riscos)
	vDestroyFontTable();
#endif /* __riscos */
	/* Close the file */
	(void)fclose(pFile);
} /* end of vCloseDocument */

/*
 * Common part of the file checking functions
 */
static BOOL
bCheckBytes(const char *szFilename, const unsigned char *aucBytes, int iBytes)
{
	FILE	*pFile;
	int	iIndex, iChar;
	BOOL	bResult;

	fail(aucBytes == NULL || iBytes <= 0);

	if (szFilename == NULL || szFilename[0] == '\0') {
		return FALSE;
	}
	pFile = fopen(szFilename, "rb");
	if (pFile == NULL) {
		return FALSE;
	}
	bResult = TRUE;
	for (iIndex = 0; iIndex < iBytes; iIndex++) {
		iChar = getc(pFile);
		if (iChar == EOF || iChar != (int)aucBytes[iIndex]) {
			bResult = FALSE;
			break;
		}
	}
	(void)fclose(pFile);
	return bResult;
} /* end of bCheckBytes */

/*
 * This function checks whether the given file is or is not a Word6 (or later)
 * document
 */
BOOL
bIsSupportedWordFile(const char *szFilename)
{
	static unsigned char	aucBytes[] =
		{ 0xd0, 0xcf, 0x11, 0xe0, 0xa1, 0xb1, 0x1a, 0xe1 };
	off_t	tSize;

	if (szFilename == NULL || szFilename[0] == '\0') {
		return FALSE;
	}
	tSize = tGetFilesize(szFilename);
	if (tSize < 3 * BIG_BLOCK_SIZE || tSize % BIG_BLOCK_SIZE != 0) {
		return FALSE;
	}
	return bCheckBytes(szFilename, aucBytes, elementsof(aucBytes));
} /* end of bIsSupportedWordFile */

/*
 * This function checks whether the given file is or is not a "Word2, 4, 5"
 * document
 */
BOOL
bIsWord245File(const char *szFilename)
{
	static unsigned char	aucBytes[6][8] = {
		{ 0x31, 0xbe, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00 },
		{ 0xdb, 0xa5, 0x2d, 0x00, 0x00, 0x00, 0x09, 0x04 },
		{ 0xdb, 0xa5, 0x2d, 0x00, 0x31, 0x40, 0x09, 0x08 },
		{ 0xdb, 0xa5, 0x2d, 0x00, 0x31, 0x40, 0x09, 0x0c },
		{ 0xfe, 0x37, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00 },
		{ 0xfe, 0x37, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00 },
	};
	int	iIndex;

	for (iIndex = 0; iIndex < elementsof(aucBytes); iIndex++) {
		if (bCheckBytes(szFilename,
				aucBytes[iIndex],
				elementsof(aucBytes[iIndex]))) {
			return TRUE;
		}
	}
	return FALSE;
} /* end of bIsWord245File */

/*
 * This function checks whether the given file is or is not a RTF document
 */
BOOL
bIsRtfFile(const char *szFilename)
{
	static unsigned char	aucBytes[] =
		{ '{', '\\', 'r', 't', 'f', '1' };

	return bCheckBytes(szFilename, aucBytes, elementsof(aucBytes));
} /* end of bIsRtfFile */

/*
 * TRUE if the current file was made on an Apple Macintosh, otherwise FALSE.
 * This function hides the methode of how to find out from the rest of the
 * program.
 *
 * note: this should grow into a real function
 */
BOOL
bFileFromTheMac(void)
{
	return bMacFile;
} /* end of bOriginatesOnMac */
