/*
 * postscript.c
 * Copyright (C) 1999 A.J. van Os
 *
 * Description:
 * Functions to deal with the PostScript format
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <time.h>
#include "version.h"
#include "antiword.h"

/* The output must be in PostScript */
static BOOL	bUsePostScript = FALSE;
/* The height of an PostScript page */
static int	iPageHeight = INT_MAX;
/* PostScript output must use colour or grayscale */
static BOOL	bUseColour = TRUE;
/* Current font information */
static draw_fontref	tFontRefCurr = (draw_fontref)-1;
static int		iFontsizeCurr = -1;
static int		iColourCurr = -1;
/* Current vertical position information */
static int		iYtopCurr = -1;
/* PostScript page counter */
static int	iPageNumber = 0;
static int	iPageCount = 0;

static char iso_8859_1_data[] = { "\
/newcodes	% foreign character encodings\n\
[\n\
160/space 161/exclamdown 162/cent 163/sterling 164/currency\n\
165/yen 166/brokenbar 167/section 168/dieresis 169/copyright\n\
170/ordfeminine 171/guillemotleft 172/logicalnot 173/hyphen 174/registered\n\
175/macron 176/degree 177/plusminus 178/twosuperior 179/threesuperior\n\
180/acute 181/mu 182/paragraph 183/periodcentered 184/cedilla\n\
185/onesuperior 186/ordmasculine 187/guillemotright 188/onequarter\n\
189/onehalf 190/threequarters 191/questiondown 192/Agrave 193/Aacute\n\
194/Acircumflex 195/Atilde 196/Adieresis 197/Aring 198/AE 199/Ccedilla\n\
200/Egrave 201/Eacute 202/Ecircumflex 203/Edieresis 204/Igrave 205/Iacute\n\
206/Icircumflex 207/Idieresis 208/Eth 209/Ntilde 210/Ograve 211/Oacute\n\
212/Ocircumflex 213/Otilde 214/Odieresis 215/multiply 216/Oslash\n\
217/Ugrave 218/Uacute 219/Ucircumflex 220/Udieresis 221/Yacute 222/Thorn\n\
223/germandbls 224/agrave 225/aacute 226/acircumflex 227/atilde\n\
228/adieresis 229/aring 230/ae 231/ccedilla 232/egrave 233/eacute\n\
234/ecircumflex 235/edieresis 236/igrave 237/iacute 238/icircumflex\n\
239/idieresis 240/eth 241/ntilde 242/ograve 243/oacute 244/ocircumflex\n\
245/otilde 246/odieresis 247/divide 248/oslash 249/ugrave 250/uacute\n\
251/ucircumflex 252/udieresis 253/yacute 254/thorn 255/ydieresis\n\
] def\n\
\n\
/reencdict 12 dict def\n\
\n\
" };

static char iso_8859_1_func[] = { "\
% change fonts using ISO-8859-1 characters\n\
/ChgFnt		% size psname natname => font\n\
{\n\
	dup FontDirectory exch known		% is re-encoded name known?\n\
	{ exch pop }				% yes, get rid of long name\n\
	{ dup 3 1 roll ReEncode } ifelse	% no, re-encode it\n\
	findfont exch scalefont setfont\n\
} def\n\
\n\
/ReEncode	%\n\
{\n\
reencdict begin\n\
	/newname exch def\n\
	/basename exch def\n\
	/basedict basename findfont def\n\
	/newfont basedict maxlength dict def\n\
	basedict\n\
	{ exch dup /FID ne\n\
		{ dup /Encoding eq\n\
			{ exch dup length array copy newfont 3 1 roll put }\n\
			{ exch newfont 3 1 roll put } ifelse\n\
		}\n\
		{ pop pop } ifelse\n\
	} forall\n\
	newfont /FontName newname put\n\
	newcodes aload pop newcodes length 2 idiv\n\
	{ newfont /Encoding get 3 1 roll put } repeat\n\
	newname newfont definefont pop\n\
end\n\
} def\n\
\n\
% draw a line and show the string\n\
/LineShow	% string linewidth movement\n\
{\n\
	gsave\n\
		0 exch rmoveto\n\
		setlinewidth\n\
		dup\n\
		stringwidth pop\n\
		0 rlineto stroke\n\
	grestore\n\
	show\n\
} bind def\n\
\n\
" };


/*
 * vPrologue - perform the PostScript initialisation
 */
static void
vPrologue(FILE *pFile, const char *szTask, const char *szFilename)
{
	options_type	tOptions;
	const char	*szTmp;
	time_t	tTime;

	fail(pFile == NULL);
	fail(szTask == NULL || szTask[0] == '\0');

	vGetOptions(&tOptions);
	iPageHeight = tOptions.iPageHeight;
	bUsePostScript = tOptions.bUseOutlineFonts;
	bUseColour = tOptions.bUseColour;
	tFontRefCurr = (draw_fontref)-1;
	iFontsizeCurr = -1;
	iColourCurr = -1;
	iYtopCurr = -1;

	if (!bUsePostScript) {
		return;
	}
	fprintf(pFile, "%%!PS-Adobe-1.0\n");
	szTmp = strrchr(szFilename, '/');
	if (szTmp == NULL) {
		szTmp = szFilename;
	} else {
		szTmp++;
	}
	fprintf(pFile, "%%%%Title: %s\n", szTmp);

	fprintf(pFile, "%%%%Creator: %s %s\n", szTask, VERSIONSTRING);
	szTmp = getenv("LOGNAME");
	if (szTmp == NULL || szTmp[0] == '\0') {
		szTmp = getenv("USER");
		if (szTmp == NULL || szTmp[0] == '\0') {
			szTmp = "unknown";
		}
	}
	fprintf(pFile, "%%%%For: %s\n", szTmp);
	errno = 0;
	tTime = time(NULL);
	if (tTime == (time_t)-1 && errno != 0) {
		szTmp = NULL;
	} else {
		szTmp = ctime(&tTime);
	}
	if (szTmp == NULL || szTmp[0] == '\0') {
		szTmp = "unknown\n";
	}
	fprintf(pFile, "%%%%CreationDate: %s", szTmp);
} /* end of vPrologue */

/*
 * vEpilogue - clean up after PostScript is done
 */
static void
vEpilogue(FILE *pFile)
{
	if (!bUsePostScript) {
		return;
	}
	fprintf(pFile, "%%%%Trailer\n");
	fprintf(pFile, "%%%%Pages: %d\n", iPageCount);
} /* end of vEpilogue */

/*
 * pCreateDiagram - create and initialize a diagram
 *
 * remark: does not return if the diagram can't be created
 */
diagram_type *
pCreateDiagram(char *szTask, const char *szFilename)
{
	diagram_type	*pDiag;

	fail(szTask == NULL || szTask[0] == '\0');
	DBG_MSG("pCreateDiagram");

	/* Get the necessary memory */
	pDiag = xmalloc(sizeof(diagram_type));
	/* Initialisation */
	pDiag->pFile = stdout;
	vPrologue(pDiag->pFile, szTask, szFilename);
	iPageNumber = 0;
	pDiag->iXleft = 0;
	if (bUsePostScript) {
		pDiag->iYtop = iPageHeight - PS_TOP_MARGIN;
	} else {
		pDiag->iYtop = 0;
	}
	/* Return success */
	return pDiag;
} /* end of pCreateDiagram */

/*
 * vDestroyDiagram - remove a diagram by freeing the memory it uses
 */
void
vDestroyDiagram(diagram_type *pDiag, BOOL bLastFile)
{
	DBG_MSG("vDestroyDiagram");

	fail(pDiag == NULL);

	if (pDiag == NULL) {
		return;
	}
	if (bUsePostScript && pDiag->iYtop < iPageHeight - PS_TOP_MARGIN) {
		fprintf(pDiag->pFile, "showpage\n");
	}
	vEpilogue(pDiag->pFile);
	pDiag = xfree(pDiag);
} /* end of vDestroyDiagram */

/*
 * vAddFonts2Diagram - add the list of fonts and complete the prologue
 */
void
vAddFonts2Diagram(diagram_type *pDiag)
{
	FILE	*pFile;
	font_table_type *pTable, *pTmp, *pTmp2;
	int	iRecords;
	BOOL	bFound;

	fail(pDiag == NULL);
	fail(pDiag->pFile == NULL);

	if (!bUsePostScript) {
		return;
	}

	pFile = pDiag->pFile;
	fprintf(pFile, "%%%%DocumentFonts: ");
	vGetFontTranslationTable(&pTable, &iRecords);
	if (pTable == NULL || iRecords <= 0) {
		fprintf(pFile, "Courier");
	} else {
		for (pTmp = pTable; pTmp < pTable + iRecords; pTmp++) {
			bFound = FALSE;
			for (pTmp2 = pTable; pTmp2 < pTmp; pTmp2++) {
				bFound = StrEq(pTmp2->szOurFontname,
						pTmp->szOurFontname);
				if (bFound) {
					break;
				}
			}
			if (!bFound) {
				fprintf(pFile, "%s ", pTmp->szOurFontname);
			}
		}
	}
	fprintf(pFile, "\n");
	iPageCount = 0;
	fprintf(pFile, "%%%%Pages: (atend)\n");
	fprintf(pFile, "%%%%EndComments\n");
	/* Allow ISO-8859-1 character set */
	fprintf(pFile, "%s\n%s", iso_8859_1_data, iso_8859_1_func);
	fprintf(pFile, "%%%%EndProlog\n");
	fprintf(pDiag->pFile, "%%%%Page %d %d\n", ++iPageNumber, ++iPageCount);
} /* end of vAddFonts2Diagram */

/*
 * vPrintPS - print a PostScript string
 */
static void
vPrintPS(FILE *pFile, const char *szString, int iStringLength,
		unsigned char ucFontstyle)
{
	int iCount;
	const unsigned char *ucBytes;

	fail(szString == NULL || iStringLength < 0);

	if (szString == NULL || szString[0] == '\0' || iStringLength <= 0) {
		return;
	}

	ucBytes = (unsigned char *)szString;
	putc('(', pFile);
	for (iCount = 0; iCount < iStringLength ; iCount++) {
		switch (ucBytes[iCount]) {
		case '(':
		case ')':
		case '\\':
			putc('\\', pFile);
			putc(szString[iCount], pFile);
			break;
		default:
			if (ucBytes[iCount] < 0x20 ||
			    (ucBytes[iCount] >= 0x7f &&
			     ucBytes[iCount] < 0xa0)) {
				DBG_HEX(ucBytes[iCount]);
				putc(' ', pFile);
			} else if (ucBytes[iCount] >= 0x80) {
				fprintf(pFile, "\\%03o", ucBytes[iCount]);
			} else {
				putc(szString[iCount], pFile);
			}
			break;
		}
	}
	fprintf(pFile, ") ");
	if (bIsStrike(ucFontstyle) && iFontsizeCurr > 0) {
		fprintf(pFile, "%.2f %.2f LineShow\n",
			iFontsizeCurr * 0.02, iFontsizeCurr * 0.12);
	} else if (bIsUnderline(ucFontstyle) && iFontsizeCurr > 0) {
		fprintf(pFile, "%.2f %.2f LineShow\n",
			iFontsizeCurr * 0.02, iFontsizeCurr * -0.06);
	} else {
		fprintf(pFile, "show\n");
	}
} /* end of vPrintPS */

/*
 * vSetColour - move to the given colour
 *
 */
static void
vSetColour(FILE *pFile, int iColour)
{
	unsigned int	uiTmp;
	int	iRed, iGreen, iBlue;

	uiTmp = iColour2Colour(iColour);
	iRed   = (uiTmp & 0x0000ff00) >> 8;
	iGreen = (uiTmp & 0x00ff0000) >> 16;
	iBlue  = (uiTmp & 0xff000000) >> 24;
	if (bUseColour) {
		fprintf(pFile, "%.3f %.3f %.3f setrgbcolor\n",
				iRed / 255.0, iGreen / 255.0, iBlue / 255.0);
	} else {
		fprintf(pFile, "%.3f setgray\n",
		(iRed * 0.299 + iGreen * 0.487 + iBlue * 0.114) / 255.0);
	}
} /* end of vSetColour */

/*
 * vMove2NextPage - move to the start of the next page
 */
static void
vMove2NextPage(diagram_type *pDiag)
{
	fail(pDiag == NULL);
	fail(!bUsePostScript);

	fprintf(pDiag->pFile, "showpage\n");
	fprintf(pDiag->pFile, "%%%%Page %d %d\n", ++iPageNumber, ++iPageCount);
	pDiag->iXleft = 0;
	pDiag->iYtop = iPageHeight - PS_TOP_MARGIN;
	iYtopCurr = -1;
} /* end of vMove2NextPage */

/*
 * vMoveTo - move to the given X,Y coordinates
 *
 * Move the current position of the given diagram to its X,Y coordianates,
 * start on a new page if needed
 */
static void
vMoveTo(diagram_type *pDiag)
{
	if (pDiag->iYtop < PS_BOTTOM_MARGIN) {
		vMove2NextPage(pDiag);
	}
	fail(pDiag->iYtop < PS_BOTTOM_MARGIN);

	if (pDiag->iYtop != iYtopCurr) {
		fprintf(pDiag->pFile, "%.2f %.2f moveto\n",
				(pDiag->iXleft + PS_LEFT_MARGIN) / 640.0,
				pDiag->iYtop / 640.0);
		iYtopCurr = pDiag->iYtop;
	}
} /* end of vMoveTo */

/*
 * vMove2NextLine - move to the starting point of the next line
 */
void
vMove2NextLine(diagram_type *pDiag, draw_fontref tFontRef, int iFontsize)
{
	fail(pDiag == NULL);
	fail(pDiag->pFile == NULL);
	fail(iFontsize < MIN_FONT_SIZE || iFontsize > MAX_FONT_SIZE);

	pDiag->iXleft = 0;
	if (bUsePostScript) {
		pDiag->iYtop -= iWord2DrawUnits20(iFontsize);
	} else {
		(void)fprintf(pDiag->pFile, "\n");
	}
} /* end of vMove2NextLine */

/*
 * vSubstring2Diagram - put a substring into a diagram
 */
void
vSubstring2Diagram(diagram_type *pDiag,
		char *szString, int iStringLength, int iStringWidth,
		int iColour, unsigned char ucFontstyle, draw_fontref tFontRef,
		int iFontsize, int iMaxFontsize)
{
	const char	*szOurFontname;

	fail(pDiag == NULL || szString == NULL);
	fail(pDiag->pFile == NULL);
	fail(pDiag->iXleft < 0);
	fail(iStringLength != strlen(szString));
	fail(iFontsize < MIN_FONT_SIZE || iFontsize > MAX_FONT_SIZE);
	fail(iMaxFontsize < MIN_FONT_SIZE || iMaxFontsize > MAX_FONT_SIZE);
	fail(iFontsize > iMaxFontsize);

	if (szString[0] == '\0' || iStringLength <= 0) {
		return;
	}

	if (bUsePostScript) {
		if (tFontRef != tFontRefCurr || iFontsize != iFontsizeCurr) {
			szOurFontname = szGetFontname(tFontRef);
			fail(szOurFontname == NULL);
			fprintf(pDiag->pFile,
				"%.1f /%s /%s-ISO-8859-1 ChgFnt\n",
				(double)iFontsize / 2.0,
				szOurFontname, szOurFontname);
			tFontRefCurr = tFontRef;
			iFontsizeCurr = iFontsize;
		}
		if (iColour != iColourCurr) {
			vSetColour(pDiag->pFile, iColour);
			iColourCurr = iColour;
		}
		vMoveTo(pDiag);
		vPrintPS(pDiag->pFile, szString, iStringLength, ucFontstyle);
	} else {
		fprintf(pDiag->pFile, "%.*s", iStringLength, szString);
	}
	pDiag->iXleft += iStringWidth;
} /* end of vSubstring2Diagram */

/*
 * vSetLeftIndentation - set the left indentation of the given diagram
 */
void
vSetLeftIndentation(diagram_type *pDiag, int iLeftIndentation)
{
	int	iX, iCount, iNbr;

	fail(pDiag == NULL || iLeftIndentation < 0);
	fail(pDiag->pFile == NULL);

	iX = iMilliPoints2DrawUnits(iLeftIndentation);
	if (iX > pDiag->iXleft) {
		iNbr = iDrawUnits2Char(iX - pDiag->iXleft);
		pDiag->iXleft = iX;
		if (!bUsePostScript) {
			for (iCount = 0; iCount < iNbr; iCount++) {
				fprintf(pDiag->pFile, "%c", FILLER_CHAR);
			}
		}
	}
} /* end of vSetLeftIndentation */

/*
 * Create an end of paragraph by moving the y-high mark 1/3 line down
 */
void
vEndOfParagraph2Diagram(diagram_type *pDiag,
			draw_fontref tFontRef, int iFontsize)
{
	fail(pDiag == NULL);
	fail(pDiag->pFile == NULL);
	fail(iFontsize < MIN_FONT_SIZE || iFontsize > MAX_FONT_SIZE);

	if (pDiag->iXleft > 0) {
		/* To the start of the line */
		vMove2NextLine(pDiag, tFontRef, iFontsize);
	}
	/* Empty line */
	if (bUsePostScript) {
		pDiag->iXleft = 0;
		pDiag->iYtop -= iWord2DrawUnits20(iFontsize) / 3;
	} else {
		vMove2NextLine(pDiag, tFontRef, iFontsize);
	}
} /* end of vEndOfParagraph2Diagram */

/*
 * Create an empty line by moving the y-high mark one line down
 */
void
vEmptyLine2Diagram(diagram_type *pDiag,
			draw_fontref tFontRef, int iFontsize)
{
	fail(pDiag == NULL);
	fail(pDiag->pFile == NULL);
	fail(iFontsize < MIN_FONT_SIZE || iFontsize > MAX_FONT_SIZE);

	if (pDiag->iXleft > 0) {
		/* To the start of the line */
		vMove2NextLine(pDiag, tFontRef, iFontsize);
	}
	/* Empty line */
	vMove2NextLine(pDiag, tFontRef, iFontsize);
} /* end of vEmptyLine2Diagram */

/*
 * Create an end of page
 */
void
vEndOfPage2Diagram(diagram_type *pDiag,
			draw_fontref tFontRef, int iFontsize)
{
	if (bUsePostScript) {
		vMove2NextPage(pDiag);
	} else {
		vEndOfParagraph2Diagram(pDiag, tFontRef, iFontsize);
	}
} /* end of vEndOfPage2Diagram */
