/*
 * out2window.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Output to a text window
 */

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <limits.h>
#include "antiword.h"


/* Used for numbering the chapters */
static int	aiHdrCounter[9];


/*
 * vString2Diagram - put a string into a diagram
 */
static void
vString2Diagram(diagram_type *pDiag, output_type *pAnchor)
{
	output_type	*pOutput;
	int		iWidth;
	unsigned char	ucMaxFontsize;

	fail(pDiag == NULL);
	fail(pAnchor == NULL);

	/* Compute the maximum fontsize in this string */
	ucMaxFontsize = MIN_FONT_SIZE;
	for (pOutput = pAnchor; pOutput != NULL; pOutput = pOutput->pNext) {
		if (pOutput->ucFontsize > ucMaxFontsize) {
			ucMaxFontsize = pOutput->ucFontsize;
		}
	}
	/* Output all substrings */
	for (pOutput = pAnchor; pOutput != NULL; pOutput = pOutput->pNext) {
		iWidth = iMilliPoints2DrawUnits(pOutput->iStringWidth);
		vSubstring2Diagram(pDiag, pOutput->szStorage,
			pOutput->iNextFree, iWidth, pOutput->iColour,
			pOutput->ucFontstyle, pOutput->tFontRef,
			pOutput->ucFontsize, ucMaxFontsize);
	}
	/* Goto the start of the next line */
	vMove2NextLine(pDiag, pAnchor->tFontRef, ucMaxFontsize);
} /* end of vString2Diagram */

/*
 * iComputeNetWidth - compute the net string width
 */
static int
iComputeNetWidth(output_type *pAnchor)
{
	output_type	*pTmp;
	int		iNetWidth;

	fail(pAnchor == NULL);

	/* Step 1: Count all but the last sub-string */
	iNetWidth = 0;
	for (pTmp = pAnchor; pTmp->pNext != NULL; pTmp = pTmp->pNext) {
		fail(pTmp->iStringWidth < 0);
		iNetWidth += pTmp->iStringWidth;
	}
	fail(pTmp == NULL);
	fail(pTmp->pNext != NULL);

	/* Step 2: remove the white-space from the end of the string */
	while (pTmp->iNextFree > 0 &&
	       isspace(pTmp->szStorage[pTmp->iNextFree - 1])) {
		pTmp->szStorage[pTmp->iNextFree - 1] = '\0';
		pTmp->iNextFree--;
		NO_DBG_DEC(pTmp->iStringWidth);
		pTmp->iStringWidth = iComputeStringWidth(
						pTmp->szStorage,
						pTmp->iNextFree,
						pTmp->tFontRef,
						pTmp->ucFontsize);
		NO_DBG_DEC(pTmp->iStringWidth);
	}

	/* Step 3: Count the last sub-string */
	iNetWidth += pTmp->iStringWidth;
	return iNetWidth;
} /* end of iComputeNetWidth */

/*
 * iComputeHoles - compute number of holes
 * (A hole is a number of whitespace characters followed by a
 *  non-whitespace character)
 */
static int
iComputeHoles(output_type *pAnchor)
{
	output_type	*pTmp;
	int	iIndex, iCounter;
	BOOL	bWasSpace, bIsSpace;

	fail(pAnchor == NULL);

	iCounter = 0;
	bIsSpace = FALSE;
	/* Count the holes */
	for (pTmp = pAnchor; pTmp != NULL; pTmp = pTmp->pNext) {
		fail(pTmp->iNextFree != strlen(pTmp->szStorage));
		for (iIndex = 0; iIndex <= pTmp->iNextFree; iIndex++) {
			bWasSpace = bIsSpace;
			bIsSpace = isspace(pTmp->szStorage[iIndex]);
			if (bWasSpace && !bIsSpace) {
				iCounter++;
			}
		}
	}
	return iCounter;
} /* end of iComputeHoles */

/*
 * Align a string and insert it into the text
 */
void
vAlign2Window(diagram_type *pDiag, output_type *pAnchor,
		int iScreenWidth, unsigned char ucAlignment)
{
	int	iNetWidth;

	fail(pDiag == NULL || pAnchor == NULL);
	fail(iScreenWidth < iChar2MilliPoints(MIN_SCREEN_WIDTH));

	NO_DBG_MSG("vAlign2Window");

	iNetWidth = iComputeNetWidth(pAnchor);

	if (iScreenWidth > iChar2MilliPoints(MAX_SCREEN_WIDTH) ||
	    iNetWidth <= 0) {
		/*
		 * Screenwidth is "infinite", so no alignment is possible
		 * Don't bother to align an empty line
		 */
		vString2Diagram(pDiag, pAnchor);
		return;
	}

	switch (ucAlignment) {
	case ALIGNMENT_CENTER:
		vSetLeftIndentation(pDiag, (iScreenWidth - iNetWidth) / 2);
		break;
	case ALIGNMENT_RIGHT:
		vSetLeftIndentation(pDiag, iScreenWidth - iNetWidth);
		break;
	case ALIGNMENT_JUSTIFY:
	case ALIGNMENT_LEFT:
	default:
		break;
	}
	vString2Diagram(pDiag, pAnchor);
} /* end of vAlign2Window */

/*
 * vJustify2Window
 */
void
vJustify2Window(diagram_type *pDiag, output_type *pAnchor,
		int iScreenWidth, unsigned char ucAlignment)
{
	output_type	*pTmp;
	char	*pcNew, *pcOld, *szStorage;
	int	iNetWidth, iSpaceWidth, iFillerLen, iToAdd, iHoles;

	fail(pDiag == NULL || pAnchor == NULL);
	fail(iScreenWidth < MIN_SCREEN_WIDTH);

	NO_DBG_MSG("vJustify2Window");

	if (ucAlignment != ALIGNMENT_JUSTIFY) {
		vAlign2Window(pDiag, pAnchor, iScreenWidth, ucAlignment);
		return;
	}

	iNetWidth = iComputeNetWidth(pAnchor);

	if (iScreenWidth > iChar2MilliPoints(MAX_SCREEN_WIDTH) ||
	    iNetWidth <= 0) {
		/*
		 * Screenwidth is "infinite", so justify is not possible
		 * Don't bother to align an empty line
		 */
		vString2Diagram(pDiag, pAnchor);
		return;
	}

	/* Justify */
	fail(ucAlignment != ALIGNMENT_JUSTIFY);
	iSpaceWidth = iComputeStringWidth(" ", 1,
				pAnchor->tFontRef, pAnchor->ucFontsize);
	DBG_DEC(iSpaceWidth);
	iToAdd = iScreenWidth -
			iNetWidth -
			iDrawUnits2MilliPoints(pDiag->iXleft);
	DBG_DEC_C(iToAdd < 0, iToAdd);
	DBG_DEC_C(iToAdd < 0, iScreenWidth);
	DBG_DEC_C(iToAdd < 0, iNetWidth);
	DBG_DEC_C(iToAdd < 0, iDrawUnits2MilliPoints(pDiag->iXleft));
	DBG_DEC_C(iToAdd < 0, pDiag->iXleft);
	iToAdd /= iSpaceWidth;
	DBG_DEC(iToAdd);
	if (iToAdd <= 0) {
		vString2Diagram(pDiag, pAnchor);
		return;
	}

	iHoles = iComputeHoles(pAnchor);
	/* Justify by adding spaces */
	for (pTmp = pAnchor; pTmp != NULL; pTmp = pTmp->pNext) {
		fail(pTmp->iNextFree != strlen(pTmp->szStorage));
		fail(iToAdd < 0);
		szStorage = xmalloc(pTmp->iNextFree + iToAdd + 1);
		pcNew = szStorage;
		for (pcOld = pTmp->szStorage; *pcOld != '\0'; pcOld++) {
			*pcNew++ = *pcOld;
			if (*pcOld == ' ' &&
			    *(pcOld + 1) != ' ' &&
			    iHoles > 0) {
				iFillerLen = iToAdd / iHoles;
				iToAdd -= iFillerLen;
				iHoles--;
				for (; iFillerLen > 0; iFillerLen--) {
					*pcNew++ = ' ';
				}
			}
		}
		*pcNew = '\0';
		pTmp->szStorage = xfree(pTmp->szStorage);
		pTmp->szStorage = szStorage;
		pTmp->iStorageSize = pTmp->iNextFree + iToAdd + 1;
		pTmp->iStringWidth +=
			(pcNew - szStorage - pTmp->iNextFree) * iSpaceWidth;
		pTmp->iNextFree = pcNew - szStorage;
		fail(pTmp->iNextFree != strlen(pTmp->szStorage));
	}
	DBG_DEC_C(iToAdd != 0, iToAdd);
	vString2Diagram(pDiag, pAnchor);
} /* end of vJustify2Window */

/*
 * vResetStyles - reset the style information variables
 */
void
vResetStyles(void)
{
	memset(aiHdrCounter, 0, sizeof(aiHdrCounter));
} /* end of vResetStyles */

/*
 * Add the style characters to the line
 */
int
iStyle2Window(char *szLine, const style_block_type *pStyleInfo)
{
	char	*pcTxt;
	int	iIndex, iStyleIndex;

	fail(szLine == NULL || pStyleInfo == NULL);

	pcTxt = szLine;
	if ((int)pStyleInfo->ucStyle >= 1 && (int)pStyleInfo->ucStyle <= 9) {
		iStyleIndex = (int)pStyleInfo->ucStyle - 1;
		for (iIndex = 0; iIndex < 9; iIndex++) {
			if (iIndex == iStyleIndex) {
				aiHdrCounter[iIndex]++;
			} else if (iIndex > iStyleIndex) {
				aiHdrCounter[iIndex] = 0;
			} else if (aiHdrCounter[iIndex] < 1) {
				aiHdrCounter[iIndex] = 1;
			}
			if (iIndex <= iStyleIndex) {
				pcTxt += sprintf(pcTxt, "%d",
						aiHdrCounter[iIndex]);
				if (iIndex < iStyleIndex) {
					*pcTxt++ = '.';
				}
			}
		}
		*pcTxt++ = ' ';
	}
	*pcTxt = '\0';
	DBG_MSG_C(szLine[0] != '\0', szLine);
	return pcTxt - szLine;
} /* end of iStyle2Window */

/*
 * vRemoveRowEnd - remove the end of table row indicator
 *
 * Remove the double TABLE_SEPARATOR characters from the end of the string.
 */
static void
vRemoveRowEnd(char *szRowTxt)
{
	char	*pcTmp;

	fail(szRowTxt == NULL || szRowTxt[0] == '\0');
	fail(szRowTxt[strlen(szRowTxt) - 1] != TABLE_SEPARATOR);
	fail(szRowTxt[strlen(szRowTxt) - 2] != TABLE_SEPARATOR);

	pcTmp = strrchr(szRowTxt, TABLE_SEPARATOR);
	if (pcTmp == NULL || pcTmp <= szRowTxt) {
		DBG_MSG(szRowTxt);
		return;
	}
	*pcTmp = '\0';		/* Remove the last TABLE_SEPARATOR */
	pcTmp--;
	if (*pcTmp == TABLE_SEPARATOR) {
		*pcTmp = '\0';	/* And the second last */
		return;
	}
	DBG_MSG(szRowTxt);
} /* end of vRemoveRowEnd */

/*
 * vTableRow2Window - put a table row into a diagram
 */
void
vTableRow2Window(diagram_type *pDiag, output_type *pOutput,
		const row_block_type *pRowInfo)
{
	output_type	tRow;
	char	*aszColTxt[TABLE_COLUMN_MAX];
	char	*szLine, *pcTmp, *pcTxt;
	size_t	tSize;
	int	iIndex, iNbrOfColumns, iColumnWidth, iTmp;
	int	iCharWidthLarge, iCharWidthSmall;
	int	iLen1, iLen2, iLen;
	BOOL	bNotReady;

	fail(pDiag == NULL || pOutput == NULL || pRowInfo == NULL);
	fail(pOutput->szStorage == NULL);
	fail(pOutput->pNext != NULL);

	/* Character sizes */
	iCharWidthLarge = iComputeStringWidth("W", 1,
				pOutput->tFontRef, pOutput->ucFontsize);
	NO_DBG_DEC(iCharWidthLarge);
	iCharWidthSmall = iComputeStringWidth("i", 1,
				pOutput->tFontRef, pOutput->ucFontsize);
	NO_DBG_DEC(iCharWidthSmall);
	/* For the time being: use a fixed width font */
	fail(iCharWidthLarge != iCharWidthSmall);

	/* Make room for the row */
	tSize = iTwips2MilliPoints(pRowInfo->iColumnWidthSum) /
			iCharWidthSmall +
			(int)pRowInfo->ucNumberOfColumns + 3;
	szLine = xmalloc(tSize);

	vRemoveRowEnd(pOutput->szStorage);

	/* Split the row text into column texts */
	aszColTxt[0] = pOutput->szStorage;
	for (iNbrOfColumns = 1;
	     iNbrOfColumns < TABLE_COLUMN_MAX;
	     iNbrOfColumns++) {
		aszColTxt[iNbrOfColumns] =
				strchr(aszColTxt[iNbrOfColumns - 1],
					TABLE_SEPARATOR);
		if (aszColTxt[iNbrOfColumns] == NULL) {
			break;
		}
		*aszColTxt[iNbrOfColumns] = '\0';
		aszColTxt[iNbrOfColumns]++;
		NO_DBG_DEC(iNbrOfColumns);
		NO_DBG_MSG(aszColTxt[iNbrOfColumns]);
	}

	DBG_DEC_C(iNbrOfColumns != pRowInfo->ucNumberOfColumns,
		iNbrOfColumns);
	DBG_DEC_C(iNbrOfColumns != pRowInfo->ucNumberOfColumns,
		pRowInfo->ucNumberOfColumns);
	if (iNbrOfColumns != pRowInfo->ucNumberOfColumns) {
		werr(0, "Skipping an unmatched table row");
		/* Clean up before you leave */
		szLine = xfree(szLine);
		return;
	}

	do {
		/* Print a table row line */
		bNotReady = FALSE;
		pcTxt = szLine;
		*pcTxt++ = TABLE_SEPARATOR_CHAR;
		for (iIndex = 0; iIndex < iNbrOfColumns; iIndex++) {
			iColumnWidth =
			iTwips2MilliPoints(pRowInfo->asColumnWidth[iIndex]) /
				iCharWidthLarge;
			fail(iColumnWidth < 0);
			if (iColumnWidth < 1) {
				/* Minimum column width */
				iColumnWidth = 1;
			} else if (iColumnWidth > 1) {
				/* Room for the TABLE_SEPARATOR_CHAR */
				iColumnWidth--;
			}
			NO_DBG_DEC(iColumnWidth);
			/* Compute the length of the text for a column */
			if (aszColTxt[iIndex] == NULL) {
				iLen = 0;
			} else {
				pcTmp = strchr(aszColTxt[iIndex], '\n');
				if (pcTmp == NULL) {
					iLen1 = INT_MAX;
				} else {
					iLen1 =
					pcTmp - aszColTxt[iIndex] + 1;
				}
				iLen2 = strlen(aszColTxt[iIndex]);
				if (iLen2 > iColumnWidth) {
					iLen2 = iColumnWidth;
				}
				iLen = min(iLen1, iLen2);
			}
			NO_DBG_DEC(iLen);
			fail(iLen < 0 || iLen > iColumnWidth);
			if (iLen >= 1 &&
			    aszColTxt[iIndex][iLen - 1] == '\n') {
				aszColTxt[iIndex][iLen - 1] = ' ';
			}
			if (iLen == iColumnWidth &&
			    !isspace(aszColTxt[iIndex][iLen])) {
				/* Search for a breaking point */
				for (iTmp = iLen - 1; iTmp >= 0; iTmp--) {
					if (isspace(aszColTxt[iIndex][iTmp])) {
						/* Found a breaking point */
						iLen = iTmp + 1;
						NO_DBG_DEC(iLen);
						break;
					}
				}
			}
			/* Print the text */
			if (iLen <= 0) {
				aszColTxt[iIndex] = NULL;
			} else {
				pcTxt += sprintf(pcTxt,
					"%.*s", iLen, aszColTxt[iIndex]);
				aszColTxt[iIndex] += iLen;
				while (*aszColTxt[iIndex] == ' ') {
					aszColTxt[iIndex]++;
				}
				if (*aszColTxt[iIndex] != '\0') {
					/* The row takes more lines */
					bNotReady = TRUE;
				}
			}
			/* Print the filler */
			for (iTmp = 0; iTmp < iColumnWidth - iLen; iTmp++) {
				*pcTxt++ = (char)FILLER_CHAR;
			}
			*pcTxt++ = TABLE_SEPARATOR_CHAR;
			*pcTxt = '\0';
		}
		/* Output the table row line */
		*pcTxt = '\0';
		tRow = *pOutput;
		tRow.szStorage = szLine;
		tRow.iNextFree = pcTxt - szLine;
		tRow.iStringWidth = iComputeStringWidth(
					tRow.szStorage,
					tRow.iNextFree,
					tRow.tFontRef,
					tRow.ucFontsize);
		vString2Diagram(pDiag, &tRow);
	} while (bNotReady);
	/* Clean up before you leave */
	szLine = xfree(szLine);
} /* end of vTableRow2Window */
