/*
 * notes.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Functions to tell the difference between footnotes and endnotes
 */

#include "antiword.h"

/* Variables needed to write the Footnote and Endnote Lists */
static int	*aiFootnoteList = NULL;
static int	iFootnoteListLength = 0;
static int	*aiEndnoteList = NULL;
static int	iEndnoteListLength = 0;


/*
 * Destroy the lists with footnote and endnote information
 */
void
vDestroyNotesInfoLists(void)
{
	DBG_MSG("vDestroyNotesInfoLists");

	/* Free the lists and reset all control variables */
	aiEndnoteList = xfree(aiEndnoteList);
	aiFootnoteList = xfree(aiFootnoteList);
	iEndnoteListLength = 0;
	iFootnoteListLength = 0;
} /* end of vDestroyNotesInfoLists */

/*
 * Build the list with footnote information for Word 6/7 files
 */
static void
vGet6FootnotesInfo(FILE *pFile, int iStartBlock,
	const int *aiBBD, int iBBDLen,
	const unsigned char *aucHeader)
{
	unsigned char	*aucBuffer;
	int	iBeginOfText, iBeginFootnoteInfo, iFootnoteInfoLen;
	int	iIndex, iOffset, iFileOffset;

	fail(pFile == NULL || aucHeader == NULL);
	fail(iStartBlock < 0);
	fail(aiBBD == NULL || iBBDLen < 0);

	iBeginOfText = (int)ulGetLong(0x18, aucHeader);
	DBG_HEX(iBeginOfText);
	iBeginFootnoteInfo = (int)ulGetLong(0x68, aucHeader);
	DBG_HEX(iBeginFootnoteInfo);
	iFootnoteInfoLen = (int)ulGetLong(0x6c, aucHeader);
	DBG_DEC(iFootnoteInfoLen);

	if (iFootnoteInfoLen < 10) {
		DBG_MSG("No Footnotes in this document");
		return;
	}

	aucBuffer = xmalloc(iFootnoteInfoLen);
	if (!bReadBuffer(pFile, iStartBlock,
			aiBBD, iBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, iBeginFootnoteInfo, iFootnoteInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, iFootnoteInfoLen);

	fail(iFootnoteListLength != 0);
	iFootnoteListLength = (iFootnoteInfoLen - 4) / 6;
	fail(iFootnoteListLength <= 0);

	fail(aiFootnoteList != NULL);
	aiFootnoteList = xmalloc(sizeof(int) * iFootnoteListLength);

	for (iIndex = 0; iIndex < iFootnoteListLength; iIndex++) {
		iOffset = (int)ulGetLong(iIndex * 4, aucBuffer);
		DBG_HEX(iOffset);
		iFileOffset = iTextOffset2FileOffset(iOffset + iBeginOfText);
		DBG_HEX(iFileOffset);
		aiFootnoteList[iIndex] = iFileOffset;
	}
	aucBuffer = xfree(aucBuffer);
} /* end of vGet6FootnotesInfo */

/*
 * Build the list with endnote information for Word 6/7 files
 */
static void
vGet6EndnotesInfo(FILE *pFile, int iStartBlock,
	const int *aiBBD, int iBBDLen,
	const unsigned char *aucHeader)
{
	unsigned char	*aucBuffer;
	int	iBeginOfText, iBeginEndnoteInfo, iEndnoteInfoLen;
	int	iIndex, iOffset, iFileOffset;

	fail(pFile == NULL || aucHeader == NULL);
	fail(iStartBlock < 0);
	fail(aiBBD == NULL || iBBDLen < 0);

	iBeginOfText = (int)ulGetLong(0x18, aucHeader);
	DBG_HEX(iBeginOfText);
	iBeginEndnoteInfo = (int)ulGetLong(0x1d2, aucHeader);
	DBG_HEX(iBeginEndnoteInfo);
	iEndnoteInfoLen = (int)ulGetLong(0x1d6, aucHeader);
	DBG_DEC(iEndnoteInfoLen);

	if (iEndnoteInfoLen < 10) {
		DBG_MSG("No Endnotes in this document");
		return;
	}

	aucBuffer = xmalloc(iEndnoteInfoLen);
	if (!bReadBuffer(pFile, iStartBlock,
			aiBBD, iBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, iBeginEndnoteInfo, iEndnoteInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, iEndnoteInfoLen);

	fail(iEndnoteListLength != 0);
	iEndnoteListLength = (iEndnoteInfoLen - 4) / 6;
	fail(iEndnoteListLength <= 0);

	fail(aiEndnoteList != NULL);
	aiEndnoteList = xmalloc(sizeof(int) * iEndnoteListLength);

	for (iIndex = 0; iIndex < iEndnoteListLength; iIndex++) {
		iOffset = (int)ulGetLong(iIndex * 4, aucBuffer);
		DBG_HEX(iOffset);
		iFileOffset = iTextOffset2FileOffset(iOffset + iBeginOfText);
		DBG_HEX(iFileOffset);
		aiEndnoteList[iIndex] = iFileOffset;
	}
	aucBuffer = xfree(aucBuffer);
} /* end of vGet6EndnotesInfo */

/*
 * Build the lists note information for Word 6/7 files
 */
static void
vGet6NotesInfo(FILE *pFile, int iStartBlock,
	const int *aiBBD, int iBBDLen,
	const unsigned char *aucHeader)
{
	vGet6FootnotesInfo(pFile, iStartBlock,
			aiBBD, iBBDLen, aucHeader);
	vGet6EndnotesInfo(pFile, iStartBlock,
			aiBBD, iBBDLen, aucHeader);
} /* end of vGet6NotesInfo */

/*
 * Build the list with footnote information for Word 8/97 files
 */
static void
vGet8FootnotesInfo(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader)
{
	const int	*aiBlockDepot;
	unsigned char	*aucBuffer;
	int	iTableStartBlock, iTableSize, iBlockDepotLen, iBlockSize;
	int	iBeginOfText, iBeginFootnoteInfo, iFootnoteInfoLen;
	int	iIndex, iOffset, iFileOffset;
	unsigned short	usDocStatus;

	iBeginOfText = (int)ulGetLong(0x18, aucHeader);
	DBG_HEX(iBeginOfText);
	iBeginFootnoteInfo = (int)ulGetLong(0xaa, aucHeader);
	DBG_HEX(iBeginFootnoteInfo);
	iFootnoteInfoLen = (int)ulGetLong(0xae, aucHeader);
	DBG_DEC(iFootnoteInfoLen);

	if (iFootnoteInfoLen < 10) {
		DBG_MSG("No Footnotes in this document");
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		iTableStartBlock = pPPS->t1Table.iSb;
		iTableSize = pPPS->t1Table.iSize;
	} else {
		iTableStartBlock = pPPS->t0Table.iSb;
		iTableSize = pPPS->t0Table.iSize;
	}
	DBG_DEC(iTableStartBlock);
	if (iTableStartBlock < 0) {
		DBG_DEC(iTableStartBlock);
		DBG_MSG("No notes information");
		return;
	}
	DBG_HEX(iTableSize);
	if (iTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aiBlockDepot = aiSBD;
		iBlockDepotLen = iSBDLen;
		iBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aiBlockDepot = aiBBD;
		iBlockDepotLen = iBBDLen;
		iBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(iFootnoteInfoLen);
	if (!bReadBuffer(pFile, iTableStartBlock,
			aiBlockDepot, iBlockDepotLen, iBlockSize,
			aucBuffer, iBeginFootnoteInfo, iFootnoteInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, iFootnoteInfoLen);

	fail(iFootnoteListLength != 0);
	iFootnoteListLength = (iFootnoteInfoLen - 4) / 6;
	fail(iFootnoteListLength <= 0);

	fail(aiFootnoteList != NULL);
	aiFootnoteList = xmalloc(sizeof(int) * iFootnoteListLength);

	for (iIndex = 0; iIndex < iFootnoteListLength; iIndex++) {
		iOffset = (int)ulGetLong(iIndex * 4, aucBuffer);
		DBG_HEX(iOffset);
		iFileOffset = iTextOffset2FileOffset(iOffset + iBeginOfText);
		DBG_HEX(iFileOffset);
		aiFootnoteList[iIndex] = iFileOffset;
	}
	aucBuffer = xfree(aucBuffer);
} /* end of vGet8FootnotesInfo */

/*
 * Build the list with endnote information for Word 8/97 files
 */
static void
vGet8EndnotesInfo(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader)
{
	const int	*aiBlockDepot;
	unsigned char	*aucBuffer;
	int	iTableStartBlock, iTableSize, iBlockDepotLen, iBlockSize;
	int	iBeginOfText, iBeginEndnoteInfo, iEndnoteInfoLen;
	int	iIndex, iOffset, iFileOffset;
	unsigned short	usDocStatus;

	iBeginOfText = (int)ulGetLong(0x18, aucHeader);
	DBG_HEX(iBeginOfText);
	iBeginEndnoteInfo = (int)ulGetLong(0x20a, aucHeader);
	DBG_HEX(iBeginEndnoteInfo);
	iEndnoteInfoLen = (int)ulGetLong(0x20e, aucHeader);
	DBG_DEC(iEndnoteInfoLen);

	if (iEndnoteInfoLen < 10) {
		DBG_MSG("No Endnotes in this document");
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		iTableStartBlock = pPPS->t1Table.iSb;
		iTableSize = pPPS->t1Table.iSize;
	} else {
		iTableStartBlock = pPPS->t0Table.iSb;
		iTableSize = pPPS->t0Table.iSize;
	}
	DBG_DEC(iTableStartBlock);
	if (iTableStartBlock < 0) {
		DBG_DEC(iTableStartBlock);
		DBG_MSG("No notes information");
		return;
	}
	DBG_HEX(iTableSize);
	if (iTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aiBlockDepot = aiSBD;
		iBlockDepotLen = iSBDLen;
		iBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aiBlockDepot = aiBBD;
		iBlockDepotLen = iBBDLen;
		iBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(iEndnoteInfoLen);
	if (!bReadBuffer(pFile, iTableStartBlock,
			aiBlockDepot, iBlockDepotLen, iBlockSize,
			aucBuffer, iBeginEndnoteInfo, iEndnoteInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	DBG_PRINT_BLOCK(aucBuffer, iEndnoteInfoLen);

	fail(iEndnoteListLength != 0);
	iEndnoteListLength = (iEndnoteInfoLen - 4) / 6;
	fail(iEndnoteListLength <= 0);

	fail(aiEndnoteList != NULL);
	aiEndnoteList = xmalloc(sizeof(int) * iEndnoteListLength);

	for (iIndex = 0; iIndex < iEndnoteListLength; iIndex++) {
		iOffset = (int)ulGetLong(iIndex * 4, aucBuffer);
		DBG_HEX(iOffset);
		iFileOffset = iTextOffset2FileOffset(iOffset + iBeginOfText);
		DBG_HEX(iFileOffset);
		aiEndnoteList[iIndex] = iFileOffset;
	}
	aucBuffer = xfree(aucBuffer);
} /* end of vGet8EndnotesInfo */

/*
 * Build the lists with footnote and endnote information for Word 8/97 files
 */
static void
vGet8NotesInfo(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader)
{
	vGet8FootnotesInfo(pFile, pPPS,
			aiBBD, iBBDLen, aiSBD, iSBDLen, aucHeader);
	vGet8EndnotesInfo(pFile, pPPS,
			aiBBD, iBBDLen, aiSBD, iSBDLen, aucHeader);
} /* end of vGet8NotesInfo */

/*
 * Build the lists with footnote and endnote information
 */
void
vGetNotesInfo(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader, int iWordVersion)
{
	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(iWordVersion < 6 || iWordVersion > 8);
	fail(aiBBD == NULL || iBBDLen < 0);
	fail(aiSBD == NULL || iSBDLen < 0);

	switch (iWordVersion) {
	case 6:
	case 7:
		vGet6NotesInfo(pFile, pPPS->tWordDocument.iSb,
			aiBBD, iBBDLen, aucHeader);
		break;
	case 8:
		vGet8NotesInfo(pFile, pPPS,
			aiBBD, iBBDLen, aiSBD, iSBDLen, aucHeader);
		break;
	default:
		werr(0, "Sorry, no notes information");
		break;
	}
} /* end of vGetNotesInfo */

/*
 * Get the notetype of the note at the given fileoffset
 */
notetype_enum
eGetNotetype(int iFileOffset)
{
	int	iIndex;

	fail(iFootnoteListLength < 0);
	fail(iEndnoteListLength < 0);
	fail(aiFootnoteList == NULL && iFootnoteListLength > 0);
	fail(aiEndnoteList == NULL && iEndnoteListLength > 0);

	/* Go for the easy answer first */
	if (iFootnoteListLength <= 0 && iEndnoteListLength <= 0) {
		return notetype_is_unknown;
	}
	if (iEndnoteListLength <= 0) {
		return notetype_is_footnote;
	}
	if (iFootnoteListLength <= 0) {
		return notetype_is_endnote;
	}
	/* No easy answer, so we search */
	for (iIndex = 0; iIndex < iFootnoteListLength; iIndex++) {
		if (aiFootnoteList[iIndex] == iFileOffset) {
			return notetype_is_footnote;
		}
	}
	for (iIndex = 0; iIndex < iEndnoteListLength; iIndex++) {
		if (aiEndnoteList[iIndex] == iFileOffset) {
			return notetype_is_endnote;
		}
	}
	/* Not found */
	return notetype_is_unknown;
} /* end of eGetNotetype */
