/*
 * fonts_u.c
 * Copyright (C) 1999 A.J. van Os
 *
 * Description:
 * Functions to deal with fonts (Unix version)
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "antiword.h"
#include "fontinfo.h"

/* Don't use fonts, just plain text */
static BOOL	bUsePlainText = TRUE;

/* Default contents of the fontnames translation table */
static char *szFonttable[] = {
"# Default fontnames translation table\n",
"# uses only Times, Helvetica and Courier\n",
"#\n",
"# MS-Word fontname,	Italic,	Bold,	PostScript fontname,	Special\n",
"Arial,			0,	0,	Helvetica,		0\n",
"Arial,			0,	1,	Helvetica-Bold,		0\n",
"Arial,			1,	0,	Helvetica-Oblique,	0\n",
"Arial,			1,	1,	Helvetica-BoldOblique,	0\n",
"Courier,		0,	0,	Courier,		0\n",
"Courier,		0,	1,	Courier-Bold,		0\n",
"Courier,		1,	0,	Courier-Oblique,	0\n",
"Courier,		1,	1,	Courier-BoldOblique,	0\n",
"Courier New,		0,	0,	Courier,		0\n",
"Courier New,		0,	1,	Courier-Bold,		0\n",
"Courier New,		1,	0,	Courier-Oblique,	0\n",
"Courier New,		1,	1,	Courier-BoldOblique,	0\n",
"Fixedsys,		0,	0,	Courier,		0\n",
"Fixedsys,		0,	1,	Courier-Bold,		0\n",
"Fixedsys,		1,	0,	Courier-Oblique,	0\n",
"Fixedsys,		1,	1,	Courier-BoldOblique,	0\n",
"Helvetica,		0,	0,	Helvetica,		0\n",
"Helvetica,		0,	1,	Helvetica-Bold,		0\n",
"Helvetica,		1,	0,	Helvetica-Oblique,	0\n",
"Helvetica,		1,	1,	Helvetica-BoldOblique,	0\n",
"Lucida Console,	0,	0,	Courier,		0\n",
"Lucida Console,	0,	1,	Courier-Bold,		0\n",
"Lucida Console,	1,	0,	Courier-Oblique,	0\n",
"Lucida Console,	1,	1,	Courier-BoldOblique,	0\n",
"Swiss,			0,	0,	Helvetica,		0\n",
"Swiss,			0,	1,	Helvetica-Bold,		0\n",
"Swiss,			1,	0,	Helvetica-Oblique,	0\n",
"Swiss,			1,	1,	Helvetica-BoldOblique,	0\n",
"Univers,		0,	0,	Helvetica,		0\n",
"Univers,		0,	1,	Helvetica-Bold,		0\n",
"Univers,		1,	0,	Helvetica-Oblique,	0\n",
"Univers,		1,	1,	Helvetica-BoldOblique,	0\n",
"# All the other fonts\n",
"*,			0,	0,	Times-Roman,		0\n",
"*,			0,	1,	Times-Bold,		0\n",
"*,			1,	0,	Times-Italic,		0\n",
"*,			1,	1,	Times-BoldItalic,	0\n",
};


/*
 * pOpenFontTableFile - open the Font translation file
 * Copy the file to the proper place if necessary.
 *
 * Returns the file pointer or NULL
 */
FILE *
pOpenFontTableFile(void)
{
	FILE	*pFile;
	char	*szHome;
	int	iIndex;
	BOOL	bFailed;
	char	szFontNamesFile[PATH_MAX+1];
	
	szHome = getenv("HOME");
	if (szHome == NULL || szHome[0] == '\0') {
		werr(0, "I can't find the name of your HOME directory");
		return NULL;
	}
	if (strlen(szHome) + strlen(FONTNAMES_FILE) >=
						sizeof(szFontNamesFile)) {
		werr(0, "The name of your HOME directory is too long");
		return NULL;
	}

	sprintf(szFontNamesFile, "%s/%s", szHome, FONTNAMES_FILE);
	DBG_MSG(szFontNamesFile);

	pFile = fopen(szFontNamesFile, "r");
	if (pFile != NULL) {
		/* The font table is already in the right directory */
		return pFile;
	}

	if (!bMakeDirectory(szFontNamesFile)) {
		werr(0,
		"I can't make a directory for the fontnames file");
		return NULL;
	}
	/* Here the proper directory is known to exist */

	pFile = fopen(szFontNamesFile, "w");
	if (pFile == NULL) {
		werr(0, "I can't create a default fontnames file");
		return NULL;
	}
	/* Here the proper directory is known to be writeable */

	bFailed = FALSE;
	/* Copy the default fontnames file */
	for (iIndex = 0; iIndex < elementsof(szFonttable); iIndex++) {
		if (fputs(szFonttable[iIndex], pFile) < 0) {
			DBG_MSG("Write error");
			bFailed = TRUE;
			break;
		}
	}
	(void)fclose(pFile);
	if (bFailed) {
		DBG_MSG("Copying the fontnames file failed");
		(void)remove(szFontNamesFile);
		return NULL;
	}
	return fopen(szFontNamesFile, "r");
} /* end of pOpenFontTableFile */

/*
 * vCloseFont - close the current font, if any
 */
void
vCloseFont(void)
{
	DBG_MSG("vCloseFont");
	/* For safety: to be overwritten at the next call of tOpenfont() */
	bUsePlainText = TRUE;
} /* end of vCloseFont */

/*
 * tOpenFont - make the given font the current font
 *
 * Returns the font reference number
 */
draw_fontref
tOpenFont(diagram_type *pDiag, unsigned char ucWordFontNumber,
		unsigned char ucFontstyle, int iWordFontsize)
{
	options_type	tOptions;
	const char	*szOurFontname;
	int	iIndex, iFontnumber;

	DBG_MSG("tOpenFont");
	DBG_DEC(ucWordFontNumber);
	DBG_HEX(ucFontstyle);
	DBG_DEC(iWordFontsize);

	/* Keep the relevant bits */
	ucFontstyle &= FONT_BOLD|FONT_ITALIC;
	DBG_HEX(ucFontstyle);

	vGetOptions(&tOptions);
	bUsePlainText = !tOptions.bUseOutlineFonts;
	if (bUsePlainText) {
		/* No outline fonts, no postscript just plain text */
		return 0;
	}

	iFontnumber = iGetFontByNumber(ucWordFontNumber, ucFontstyle);
	if (iFontnumber < 0 || iFontnumber >= UCHAR_MAX) {
		DBG_DEC(iFontnumber);
		return 0;
	}
	szOurFontname = szGetOurFontname(iFontnumber);
	if (szOurFontname == NULL || szOurFontname[0] == '\0') {
		return 0;
	}
	DBG_MSG(szOurFontname);

	for (iIndex = 0; iIndex < elementsof(szFontnames); iIndex++) {
		if (StrEq(szFontnames[iIndex], szOurFontname)) {
			DBG_DEC(iIndex);
			return iIndex;
		}
	}
	return 0;
} /* end of tOpenFont */

/*
 * tOpenTableFont - make the table font the current font
 *
 * Returns the font reference number
 */
draw_fontref
tOpenTableFont(diagram_type *pDiag, int iWordFontsize)
{
	options_type	tOptions;
	int	iWordFontnumber;

	DBG_MSG("tOpenTableFont");

	vGetOptions(&tOptions);
	bUsePlainText = !tOptions.bUseOutlineFonts;
	if (bUsePlainText) {
		/* No outline fonts, no postscript just plain text */
		return 0;
	}     

	iWordFontnumber = iFontname2Fontnumber(TABLE_FONT, FONT_REGULAR);
	if (iWordFontnumber < 0 || iWordFontnumber > UCHAR_MAX) {
		DBG_DEC(iWordFontnumber);
		return 0;
	}

	return tOpenFont(pDiag, iWordFontnumber, FONT_REGULAR, iWordFontsize);
} /* end of tOpenTableFont */

/*
 * szGetFontname - get the fontname
 */
const char *
szGetFontname(draw_fontref tFontRef)
{
	fail((int)tFontRef >= elementsof(szFontnames));
	return szFontnames[(unsigned char)tFontRef];
} /* end of szGetFontname */

/*
 * iComputeStringWidth - compute the string width
 *
 * Returns string width in millipoints
 */
int
iComputeStringWidth(char *szString, int iStringLength,
		draw_fontref tFontRef, int iFontsize)
{
	unsigned char	*pucChar;
	int	iIndex, iWidth, iFontRef;

	fail(szString == NULL || iStringLength < 0);
	fail(iFontsize < MIN_FONT_SIZE || iFontsize > MAX_FONT_SIZE);

	if (szString[0] == '\0' || iStringLength <= 0) {
		/* Empty string */
		return 0;
	}

	if (bUsePlainText) {
		/* No current font, use systemfont */
		return iChar2MilliPoints(iStringLength);
	}

	iFontRef = (int)(unsigned char)tFontRef;
	iWidth = 0;
	for (iIndex = 0, pucChar = (unsigned char *)szString;
	     iIndex < iStringLength;
	     iIndex++, pucChar++) {
		iWidth += aiCharacterWidths[iFontRef][(int)*pucChar];
	}
	return iWidth * iFontsize / 2;
} /* end of iComputeStringWidth */
