/*
 * fonts.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Functions to deal with fonts (generic)
 */

#include <ctype.h>
#include <string.h>
#include "antiword.h"

/* Font Translation Table */
static int		iFontTableRecords = 0;
static font_table_type	*pFontTable = NULL;

/*
 * Find the given font in the font table
 *
 * returns the index into the FontTable, -1 if not found
 */
int
iGetFontByNumber(unsigned char ucWordFontnumber, unsigned char ucFontstyle)
{
	int	iIndex;

	for (iIndex = 0; iIndex < iFontTableRecords; iIndex++) {
		if (pFontTable[iIndex].ucWordFontnumber ==
							ucWordFontnumber &&
		    pFontTable[iIndex].ucFontstyle == ucFontstyle &&
		    pFontTable[iIndex].szOurFontname[0] != '\0') {
			return iIndex;
		}
	}
	return -1;
} /* end of iGetFontByNumber */

/*
 * szGetOurFontName - Get our font name
 *
 * return our font name from the given index, NULL if not found
 */
const char *
szGetOurFontname(int iIndex)
{
	if (iIndex < 0 || iIndex >= iFontTableRecords) {
		return NULL;
	}
	return pFontTable[iIndex].szOurFontname;
} /* end of szGetOurFontname */

/*
 * Find the given font in the font table
 *
 * returns the Word font number, -1 if not found
 */
int
iFontname2Fontnumber(const char *szOurFontname, unsigned char ucFontstyle)
{
	int	iIndex;

	for (iIndex = 0; iIndex < iFontTableRecords; iIndex++) {
		if (pFontTable[iIndex].ucFontstyle == ucFontstyle &&
		    StrEq(pFontTable[iIndex].szOurFontname, szOurFontname)) {
			return pFontTable[iIndex].ucWordFontnumber;
		}
	}
	return -1;
} /* end of iGetFontByName */

/*
 * See if the fontname from the Word file matches the fontname from the
 * font translation file.
 * If iBytesPerChar is one than szWord is in ISO-8859-1 (Word 6/7),
 * if iBytesPerChar is two than szWord is in Unicode (Word 8/97).
 */
static BOOL
bFontEqual(const char *szWord, const char *szTable, int iBytesPerChar)
{
	const char	*pcTmp1, *pcTmp2;

	fail(szWord == NULL || szTable == NULL);
	fail(iBytesPerChar != 1 && iBytesPerChar != 2);

	for (pcTmp1 = szWord, pcTmp2 = szTable;
	     *pcTmp1 != '\0';
	     pcTmp1 += iBytesPerChar, pcTmp2++) {
		if (iToUpper(*pcTmp1) != iToUpper(*pcTmp2)) {
			return FALSE;
		}
	}
	return *pcTmp2 == '\0';
} /* end of bFontEqual */

/*
 * vCreateFontTable - Create and initialize the internal font table
 */
static void
vCreateFontTable(void)
{
	font_table_type	*pTmp;
	int	iSize, iNbr;

	if (iFontTableRecords <= 0) {
		pFontTable = NULL;
		iFontTableRecords = 0;
		return;
	}

	/* Create the font table */
	iSize = iFontTableRecords * sizeof(*pFontTable);
	pFontTable = xmalloc(iSize);
	/* Initialize the font table */
	(void)memset(pFontTable, 0, iSize);
	for (iNbr = 0, pTmp = pFontTable;
	     pTmp < pFontTable + iFontTableRecords;
	     iNbr++, pTmp++) {
		pTmp->ucWordFontnumber = iNbr / 4;
		switch (iNbr % 4) {
		case 0:
			pTmp->ucFontstyle = FONT_REGULAR;
			break;
		case 1:
			pTmp->ucFontstyle = FONT_BOLD;
			break;
		case 2:
			pTmp->ucFontstyle = FONT_ITALIC;
			break;
		case 3:
			pTmp->ucFontstyle = FONT_BOLD|FONT_ITALIC;
			break;
		default:
			DBG_DEC(iNbr);
			break;
		}
	}
} /* end of vCreateFontTable */

/*
 * vMinimizeFontTable - make the font table as small as possible
 */
static void
vMinimizeFontTable(void)
{
	const font_block_type	*pFont;
	font_table_type		*pTmp;
	int	iUnUsed;
	BOOL	bMustAddTableFont;

	NO_DBG_MSG("vMinimizeFontTable");

	if (pFontTable == NULL || iFontTableRecords <= 0) {
		return;
	}

#if 0
	DBG_MSG("Before");
	DBG_DEC(iFontTableRecords);
	for (pTmp = pFontTable;
	     pTmp < pFontTable + iFontTableRecords;
	     pTmp++) {
		DBG_DEC(pTmp->ucWordFontnumber);
		DBG_HEX(pTmp->ucFontstyle);
		DBG_MSG(pTmp->szWordFontname);
		DBG_MSG(pTmp->szOurFontname);
	}
#endif /* DEBUG */

	/* Default font/style is by definition in use */
	pFontTable[0].ucInUse = TRUE;

	/* See which fonts/styles are really being used */
	bMustAddTableFont = TRUE;
	pFont = NULL;
	while((pFont = pGetNextFontInfoListItem(pFont)) != NULL) {
		pTmp = pFontTable + 4 * (int)pFont->ucFontnumber;
		if (bIsBold(pFont->ucFontstyle)) {
			pTmp++;
		}
		if (bIsItalic(pFont->ucFontstyle)) {
			pTmp += 2;
		}
		if (pTmp >= pFontTable + iFontTableRecords) {
			continue;
		}
		if (StrEq(pTmp->szOurFontname, TABLE_FONT)) {
			/* The table font is already present */
			bMustAddTableFont = FALSE;
		}
		pTmp->ucInUse = TRUE;
	}

	/* Remove the unused font entries from the font table */
	iUnUsed = 0;
	for (pTmp = pFontTable;
	     pTmp < pFontTable + iFontTableRecords;
	     pTmp++) {
		if (!pTmp->ucInUse) {
			iUnUsed++;
			continue;
		}
		if (iUnUsed > 0) {
			fail(pTmp - iUnUsed <= pFontTable);
			*(pTmp - iUnUsed) = *pTmp;
		}
	}
	iFontTableRecords -= iUnUsed;
	if (bMustAddTableFont) {
		pTmp = pFontTable + iFontTableRecords;
	  	pTmp->ucWordFontnumber = (pTmp - 1)->ucWordFontnumber + 1;
		pTmp->ucFontstyle = FONT_REGULAR;
		pTmp->ucInUse = TRUE;
		strcpy(pTmp->szWordFontname, "Extra Table Font");
		strcpy(pTmp->szOurFontname, TABLE_FONT);
		iFontTableRecords++;
		iUnUsed--;
	}
	if (iUnUsed > 0) {
		/* Resize the font table */
		pFontTable = xrealloc(pFontTable,
				iFontTableRecords * sizeof(*pFontTable));
	}
#if 0
	/* Report the used but untranslated fontnames */
	for (pTmp = pFontTable;
	     pTmp < pFontTable + iFontTableRecords;
	     pTmp++) {
		if (pTmp->szOurFontname[0] == '\0') {
			DBG_MSG(pTmp->szWordFontname);
			werr(0, "%s is not in the FontNames file",
				pTmp->szWordFontname);
		}
	}
#endif /* 0 */
#if defined(DEBUG)
	DBG_MSG("After");
	DBG_DEC(iFontTableRecords);
	for (pTmp = pFontTable;
	     pTmp < pFontTable + iFontTableRecords;
	     pTmp++) {
		DBG_DEC(pTmp->ucWordFontnumber);
		DBG_HEX(pTmp->ucFontstyle);
		DBG_MSG(pTmp->szWordFontname);
		DBG_MSG(pTmp->szOurFontname);
	}
#endif /* DEBUG */
} /* end of vMinimizeFontTable */

/*
 * bReadFontFile - read and check a line from the font translation file
 *
 * returns TRUE when a correct line has been read, otherwise FALSE
 */
static BOOL
bReadFontFile(FILE *pFontTableFile, char *szWordFont,
	int *piItalic, int *piBold, char *szOurFont, int *piSpecial)
{
	char	*pcTmp;
	int	iFields;
	char	szLine[81];

	fail(szWordFont == NULL || szOurFont == NULL);
	fail(piItalic == NULL || piBold == NULL || piSpecial == NULL);

	while (fgets(szLine, sizeof(szLine), pFontTableFile) != NULL) {
		if (szLine[0] == OPTIONS_COMMENT_CHAR ||
		    szLine[0] == '\n' ||
		    szLine[0] == '\r') {
			continue;
		}
		iFields = sscanf(szLine, "%[^,],%d,%d,%1s%[^,],%d",
		    	szWordFont, piItalic, piBold,
		    	&szOurFont[0], &szOurFont[1], piSpecial);
		if (iFields != 6) {
			pcTmp = strchr(szLine, '\r');
			if (pcTmp != NULL) {
				*pcTmp = '\0';
			}
			pcTmp = strchr(szLine, '\n');
			if (pcTmp != NULL) {
				*pcTmp = '\0';
			}
			DBG_DEC(iFields);
			werr(0, "Syntax error in: '%s'", szLine);
			continue;
		}
		if (strlen(szWordFont) >=
		    		sizeof(pFontTable[0].szWordFontname) ||
		    strlen(szOurFont) >=
		    		sizeof(pFontTable[0].szOurFontname)) {
			continue;
		}
		/* The current line passed all the tests */
		return TRUE;
	}
	return FALSE;
} /* end of bReadFontFile */

/*
 * vCreate6FontTable - create a font table for Word 6/7
 */
void
vCreate6FontTable(FILE *pFile, int iStartBlock,
	const int *aiBBD, int iBBDLen,
	const unsigned char *aucHeader)
{
	FILE	*pFontTableFile;
	font_table_type	*pTmp;
	char	*szFont, *szAltFont;
	unsigned char	*aucBuffer;
	int	iNbr, iPos, iRecLen, iOffsetAltName;
	int	iBold, iItalic, iSpecial, iBeginFontInfo, iFontInfoLen;
	char	szWordFont[81], szOurFont[81];

	fail(pFile == NULL || aucHeader == NULL);
	fail(iStartBlock < 0);
	fail(aiBBD == NULL || iBBDLen < 0);

	pFontTableFile = pOpenFontTableFile();
	if (pFontTableFile == NULL) {
		/* No translation table file, no translation table */
		return;
	}

	iBeginFontInfo = (int)ulGetLong(0xd0, aucHeader);
	DBG_HEX(iBeginFontInfo);
	iFontInfoLen = (int)ulGetLong(0xd4, aucHeader);
	DBG_DEC(iFontInfoLen);
	fail(iFontInfoLen < 9);

	aucBuffer = xmalloc(iFontInfoLen);
	if (!bReadBuffer(pFile, iStartBlock,
			aiBBD, iBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, iBeginFontInfo, iFontInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		(void)fclose(pFontTableFile);
		return;
	}
	DBG_DEC(usGetWord(0, aucBuffer));

	/* Compute the maximum number of entries in the font table */
	iFontTableRecords = 0;
	iPos = 2;
	while (iPos + 6 < iFontInfoLen) {
		iRecLen = (int)ucGetByte(iPos, aucBuffer);
		NO_DBG_DEC(iRecLen);
		iOffsetAltName = (int)ucGetByte(iPos + 5, aucBuffer);
		NO_DBG_MSG(aucBuffer + iPos + 6);
		NO_DBG_MSG_C(iOffsetAltName > 0,
				aucBuffer + iPos + 6 + iOffsetAltName);
		iPos += iRecLen + 1;
		iFontTableRecords++;
	}
	iFontTableRecords *= 4;	/* Regular, Bold, Italic and Bold/italic */
	iFontTableRecords++;	/* One extra for the table-font */
	vCreateFontTable();

	/* Read the font translation file */
	while (bReadFontFile(pFontTableFile, szWordFont,
			&iItalic, &iBold, szOurFont, &iSpecial)) {
		pTmp = pFontTable;
		if (iBold != 0) pTmp++;
		if (iItalic != 0) pTmp += 2;
		iNbr = 0;
		iPos = 2;
		while (iPos + 6 < iFontInfoLen) {
			iRecLen = (int)ucGetByte(iPos, aucBuffer);
			iOffsetAltName = (int)ucGetByte(iPos + 5, aucBuffer);
			szFont = (char *)aucBuffer + iPos + 6;
			szAltFont = szFont + iOffsetAltName;
			if (bFontEqual(szFont, szWordFont, 1) ||
			    (iOffsetAltName > 0 &&
			     bFontEqual(szAltFont, szWordFont, 1)) ||
			    (pTmp->szWordFontname[0] == '\0' &&
			     szWordFont[0] == '*' &&
			     szWordFont[1] == '\0')) {
			  	strcpy(pTmp->szWordFontname, szFont);
				strcpy(pTmp->szOurFontname, szOurFont);
			}
			pTmp += 4;
			iNbr++;
			iPos += iRecLen + 1;
		}
	}
	vMinimizeFontTable();
	aucBuffer = xfree(aucBuffer);
	(void)fclose(pFontTableFile);
} /* end of vCreate6FontTable */

/*
 * vCreate6FontTable - create a font table for Word 8/97
 */
void
vCreate8FontTable(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader)
{
	FILE	*pFontTableFile;
	font_table_type	*pTmp;
	const int	*aiBlockDepot;
	char		*szFont, *szAltFont;
	unsigned char	*aucBuffer;
	int	iTableStartBlock, iTableSize, iBlockDepotLen, iBlockSize;
	int	iNbr, iPos, iRecLen, iOffsetAltName;
	int	iBold, iItalic, iSpecial, iBeginFontInfo, iFontInfoLen;
	unsigned short	usDocStatus;
	char	szWordFont[81], szOurFont[81];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(aiBBD == NULL || iBBDLen < 0);

	pFontTableFile = pOpenFontTableFile();
	if (pFontTableFile == NULL) {
		/* No translation table file, no translation table */
		return;
	}

	iBeginFontInfo = (int)ulGetLong(0x112, aucHeader);
	DBG_HEX(iBeginFontInfo);
	iFontInfoLen = (int)ulGetLong(0x116, aucHeader);
	DBG_DEC(iFontInfoLen);
	fail(iFontInfoLen < 46);

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		iTableStartBlock = pPPS->t1Table.iSb;
		iTableSize = pPPS->t1Table.iSize;
	} else {
		iTableStartBlock = pPPS->t0Table.iSb;
		iTableSize = pPPS->t0Table.iSize;
	}
	DBG_DEC(iTableStartBlock);
	if (iTableStartBlock < 0) {
		DBG_DEC(iTableStartBlock);
		DBG_MSG("No fontname table");
		return;
	}
	DBG_HEX(iTableSize);
	if (iTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aiBlockDepot = aiSBD;
		iBlockDepotLen = iSBDLen;
		iBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aiBlockDepot = aiBBD;
		iBlockDepotLen = iBBDLen;
		iBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(iFontInfoLen);
	if (!bReadBuffer(pFile, iTableStartBlock,
			aiBlockDepot, iBlockDepotLen, iBlockSize,
			aucBuffer, iBeginFontInfo, iFontInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		(void)fclose(pFontTableFile);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, iFontInfoLen);

	/* Get the maximum number of entries in the font table */
	iFontTableRecords = (int)usGetWord(0, aucBuffer);
	iFontTableRecords *= 4;	/* Regular, Bold, Italic and Bold/italic */
	iFontTableRecords++;	/* One extra for the table-font */
	vCreateFontTable();

	/* Read the font translation file */
	while (bReadFontFile(pFontTableFile, szWordFont,
			&iItalic, &iBold, szOurFont, &iSpecial)) {
		pTmp = pFontTable;
		if (iBold != 0) pTmp++;
		if (iItalic != 0) pTmp += 2;
		iNbr = 0;
		iPos = 4;
		while (iPos + 40 < iFontInfoLen) {
			iRecLen = (int)ucGetByte(iPos, aucBuffer);
#if 0
			iOffsetAltName = (int)ucGetByte(iPos + 5, aucBuffer);
#else
			iOffsetAltName = 0;
#endif /* 0 */
			szFont = (char *)aucBuffer + iPos + 40;
			szAltFont = szFont + iOffsetAltName;
			if (bFontEqual(szFont, szWordFont, 2) ||
			    (iOffsetAltName > 0 &&
			     bFontEqual(szAltFont, szWordFont, 2)) ||
			    (pTmp->szWordFontname[0] == '\0' &&
			     szWordFont[0] == '*' &&
			     szWordFont[1] == '\0')) {
				unicpy(pTmp->szWordFontname, szFont);
				strcpy(pTmp->szOurFontname, szOurFont);
			}
			pTmp += 4;
			iNbr++;
			iPos += iRecLen + 1;
		}
	}
	vMinimizeFontTable();
	aucBuffer = xfree(aucBuffer);
	(void)fclose(pFontTableFile);
} /* end of vCreate8FontTable */

void
vDestroyFontTable(void)
{
	DBG_MSG("vDestroyFontTable");

	pFontTable = xfree(pFontTable);
	iFontTableRecords = 0;
} /* end of vDestroyFontTable */

/*
 * vGetFontTranslationTable - make the table known to the outside world
 */
void
vGetFontTranslationTable(font_table_type **ppTable, int *piRecords)
{
	if (ppTable != NULL) {
		*ppTable = pFontTable;
	}
	if (piRecords != NULL) {
		*piRecords = iFontTableRecords;
	}
} /* end of vGetFontTranslationTable */
