/*
 * blocklist.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Build, read and destroy a list of Word text blocks
 */

#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "antiword.h"


/*
 * Privat structure to hide the way the information
 * is stored from the rest of the program
 */
typedef struct list_mem_tag {
	text_block_type		tInfo;
	struct list_mem_tag	*pNext;
} list_mem_type;

/* Variables to describe the start of the five block lists */
static list_mem_type	*pTextAnchor = NULL;
static list_mem_type	*pFootAnchor = NULL;
static list_mem_type	*pUnused1Anchor = NULL;
static list_mem_type	*pEndAnchor = NULL;
static list_mem_type	*pUnused2Anchor = NULL;
/* Variable needed to read the block list */
static list_mem_type	*pBlockLast = NULL;
/* Variable needed to read the text block list */
static list_mem_type	*pTextBlockCurrent = NULL;
/* Variable needed to read the footnote block list */
static list_mem_type	*pFootBlockCurrent = NULL;
/* Variable needed to read the endnote block list */
static list_mem_type	*pEndBlockCurrent = NULL;
/* Last block read */
static unsigned char	aucBlock[BIG_BLOCK_SIZE];


/*
 * vDestroyTextBlockList - destroy the text block list
 */
void
vDestroyTextBlockList(void)
{
	list_mem_type	*apAnchor[5];
	list_mem_type	*pCurr, *pNext;
	int	iIndex;

	DBG_MSG("vDestroyTextBlockList");

	apAnchor[0] = pTextAnchor;
	apAnchor[1] = pFootAnchor;
	apAnchor[2] = pUnused1Anchor;
	apAnchor[3] = pEndAnchor;
	apAnchor[4] = pUnused2Anchor;

	for (iIndex = 0; iIndex < 5; iIndex++) {
		pCurr = apAnchor[iIndex];
		while (pCurr != NULL) {
			pNext = pCurr->pNext;
			pCurr = xfree(pCurr);
			pCurr = pNext;
		}
	}
	/* Show that there are no lists any more */
	pTextAnchor = NULL;
	pFootAnchor = NULL;
	pUnused1Anchor = NULL;
	pEndAnchor = NULL;
	pUnused2Anchor = NULL;
	/* Reset all the controle variables */
	pBlockLast = NULL;
	pTextBlockCurrent = NULL;
	pFootBlockCurrent = NULL;
} /* end of vDestroyTextBlockList */

/*
 * bAdd2TextBlockList - add an element to the text block list
 *
 * returns: TRUE when successful, otherwise FALSE
 */
BOOL
bAdd2TextBlockList(text_block_type *pTextBlock)
{
	list_mem_type	*pListMember;

	fail(pTextBlock == NULL);
	fail(pTextBlock->iFileOffset < 0);
	fail(pTextBlock->iTextOffset < 0);
	fail(pTextBlock->iLength <= 0);
	fail(pTextBlock->bUsesUnicode != !!pTextBlock->bUsesUnicode);
	fail(pTextBlock->bUsesUnicode && odd(pTextBlock->iLength));

	NO_DBG_MSG("bAdd2TextBlockList");
	NO_DBG_HEX(pTextBlock->iFileOffset);
	NO_DBG_HEX(pTextBlock->iTextOffset);
	NO_DBG_HEX(pTextBlock->iLength);
	NO_DBG_DEC(pTextBlock->bUsesUnicode);

	if (pTextBlock->iFileOffset < 0 ||
	    pTextBlock->iTextOffset < 0 ||
	    pTextBlock->iLength <= 0) {
		werr(0, "Software (textblock) error");
		return FALSE;
	}
	/* Check for continuous blocks of the same character size */
	if (pBlockLast != NULL &&
	    pBlockLast->tInfo.iFileOffset +
	     pBlockLast->tInfo.iLength == pTextBlock->iFileOffset &&
	    pBlockLast->tInfo.iTextOffset +
	     pBlockLast->tInfo.iLength == pTextBlock->iTextOffset &&
	    pBlockLast->tInfo.bUsesUnicode == pTextBlock->bUsesUnicode) {
		/* These are continous blocks */
		pBlockLast->tInfo.iLength += pTextBlock->iLength;
		return TRUE;
	}
	/* Make a new block */
	pListMember = xmalloc(sizeof(list_mem_type));
	/* Add the block to the list */
	pListMember->tInfo = *pTextBlock;
	pListMember->pNext = NULL;
	if (pTextAnchor == NULL) {
		pTextAnchor = pListMember;
	} else {
		fail(pBlockLast == NULL);
		pBlockLast->pNext = pListMember;
	}
	pBlockLast = pListMember;
	return TRUE;
} /* end of bAdd2TextBlockList */


/*
 * vSplitBlockList - split the block list in three parts
 *
 * Split the blocklist in a Text block list, a Footnote block list, a
 * Endnote block list and two Unused lists.
 *
 * NOTE:
 * The various i*Len input parameters are given in characters, but the
 * length of the blocks are in bytes.
 */
void
vSplitBlockList(int iTextLen, int iFootnoteLen,
		int iUnused1Len, int iEndnoteLen, int iUnused2Len,
		BOOL bMustExtend)
{
	list_mem_type	*apAnchors[5];
	list_mem_type	*pGarbageAnchor, *pCurr, *pNext;
	int		iIndex, iCharsToGo, iBytesTooFar;
#if defined(DEBUG)
	int		iTotal;
#endif /* DEBUG */

	DBG_MSG("vSplitBlockList");

/* Text block list */
	if (iTextLen > 0) {
		DBG_MSG("Text block list");
		DBG_DEC(iTextLen);
		iCharsToGo = iTextLen;
		iBytesTooFar = -1;
		for (pCurr = pTextAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.iLength);
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iCharsToGo -= pCurr->tInfo.iLength / 2;
				if (iCharsToGo < 0) {
					iBytesTooFar = -2 * iCharsToGo;
				}
			} else {
				iCharsToGo -= pCurr->tInfo.iLength;
				if (iCharsToGo < 0) {
					iBytesTooFar = -iCharsToGo;
				}
			}
			if (iCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (iTextLen <= 0) {
		/* Empty text blocks list */
		pFootAnchor = pTextAnchor;
		pTextAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No footnote blocks */
		pFootAnchor = NULL;
	} else if (iCharsToGo == 0) {
		/* Move the integral number of footnote blocks */
		pFootAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-text block, part-footnote block */
		DBG_DEC(iBytesTooFar);
		fail(iBytesTooFar <= 0);
		pFootAnchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.iFileOffset);
		pFootAnchor->tInfo.iFileOffset =
				pCurr->tInfo.iFileOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pFootAnchor->tInfo.iFileOffset);
		DBG_HEX(pCurr->tInfo.iTextOffset);
		pFootAnchor->tInfo.iTextOffset =
				pCurr->tInfo.iTextOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pFootAnchor->tInfo.iTextOffset);
		pFootAnchor->tInfo.iLength = iBytesTooFar;
		pCurr->tInfo.iLength -= iBytesTooFar;
		pFootAnchor->tInfo.bUsesUnicode = pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of footnote blocks */
		pFootAnchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Footnote block list */
	if (iFootnoteLen > 0) {
		DBG_MSG("Footnote block list");
		DBG_DEC(iFootnoteLen);
		iCharsToGo = iFootnoteLen;
		iBytesTooFar = -1;
		for (pCurr = pFootAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.iLength);
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iCharsToGo -= pCurr->tInfo.iLength / 2;
				if (iCharsToGo < 0) {
					iBytesTooFar = -2 * iCharsToGo;
				}
			} else {
				iCharsToGo -= pCurr->tInfo.iLength;
				if (iCharsToGo < 0) {
					iBytesTooFar = -iCharsToGo;
				}
			}
			if (iCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (iFootnoteLen <= 0) {
		/* Empty footnote list */
		pUnused1Anchor = pFootAnchor;
		pFootAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No unused1 blocks */
		pUnused1Anchor = NULL;
	} else if (iCharsToGo == 0) {
		/* Move the integral number of unused1-list blocks */
		pUnused1Anchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
	  	/* Split the part-footnote block, part-unused1 block */
		DBG_DEC(iBytesTooFar);
		fail(iBytesTooFar <= 0);
		pUnused1Anchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.iFileOffset);
		pUnused1Anchor->tInfo.iFileOffset =
				pCurr->tInfo.iFileOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pUnused1Anchor->tInfo.iFileOffset);
		DBG_HEX(pCurr->tInfo.iTextOffset);
		pUnused1Anchor->tInfo.iTextOffset =
				pCurr->tInfo.iTextOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pUnused1Anchor->tInfo.iTextOffset);
		pUnused1Anchor->tInfo.iLength = iBytesTooFar;
		pCurr->tInfo.iLength -= iBytesTooFar;
		pUnused1Anchor->tInfo.bUsesUnicode =
				pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of unused1 blocks */
		pUnused1Anchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Unused1 block list */
	if (iUnused1Len > 0) {
		DBG_MSG("Unused1 block list");
		DBG_DEC(iUnused1Len);
		iCharsToGo = iUnused1Len;
		iBytesTooFar = -1;
		for (pCurr = pUnused1Anchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.iLength);
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iCharsToGo -= pCurr->tInfo.iLength / 2;
				if (iCharsToGo < 0) {
					iBytesTooFar = -2 * iCharsToGo;
				}
			} else {
				iCharsToGo -= pCurr->tInfo.iLength;
				if (iCharsToGo < 0) {
					iBytesTooFar = -iCharsToGo;
				}
			}
			if (iCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (iUnused1Len <= 0) {
		/* Empty unused1 list */
		pEndAnchor = pUnused1Anchor;
		pUnused1Anchor = NULL;
	} else if (pCurr == NULL) {
		/* No endnote blocks */
		pEndAnchor = NULL;
	} else if (iCharsToGo == 0) {
		/* Move the intergral number of endnote blocks */
		pEndAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-unused1-list block, part-endnote block */
		DBG_DEC(iBytesTooFar);
		fail(iBytesTooFar <= 0);
		pEndAnchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.iFileOffset);
		pEndAnchor->tInfo.iFileOffset =
				pCurr->tInfo.iFileOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pEndAnchor->tInfo.iFileOffset);
		DBG_HEX(pCurr->tInfo.iTextOffset);
		pEndAnchor->tInfo.iTextOffset =
				pCurr->tInfo.iTextOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pEndAnchor->tInfo.iTextOffset);
		pEndAnchor->tInfo.iLength = iBytesTooFar;
		pCurr->tInfo.iLength -= iBytesTooFar;
		pEndAnchor->tInfo.bUsesUnicode = pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of endnote blocks */
		pEndAnchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Endnote block list */
	if (iEndnoteLen > 0) {
		DBG_MSG("Endnote block list");
		DBG_DEC(iEndnoteLen);
		iCharsToGo = iEndnoteLen;
		iBytesTooFar = -1;
		for (pCurr = pEndAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.iLength);
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iCharsToGo -= pCurr->tInfo.iLength / 2;
				if (iCharsToGo <= 0) {
					iBytesTooFar = -2 * iCharsToGo;
				}
			} else {
				iCharsToGo -= pCurr->tInfo.iLength;
				if (iCharsToGo <= 0) {
					iBytesTooFar = -iCharsToGo;
				}
			}
			if (iCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (iEndnoteLen <= 0) {
		/* Empty endnote list */
		pUnused2Anchor = pEndAnchor;
		pEndAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No unused2 blocks */
		pUnused2Anchor = NULL;
	} else if (iCharsToGo == 0) {
		/* Move the intergral number of unused2 blocks */
		pUnused2Anchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-endnote block, part-unused2 block */
		DBG_DEC(iBytesTooFar);
		fail(iBytesTooFar <= 0);
		pUnused2Anchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.iFileOffset);
		pUnused2Anchor->tInfo.iFileOffset =
				pCurr->tInfo.iFileOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pUnused2Anchor->tInfo.iFileOffset);
		DBG_HEX(pCurr->tInfo.iTextOffset);
		pUnused2Anchor->tInfo.iTextOffset =
				pCurr->tInfo.iTextOffset +
				pCurr->tInfo.iLength -
				iBytesTooFar;
		DBG_HEX(pUnused2Anchor->tInfo.iTextOffset);
		pUnused2Anchor->tInfo.iLength = iBytesTooFar;
		pCurr->tInfo.iLength -= iBytesTooFar;
		pUnused2Anchor->tInfo.bUsesUnicode =
				pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of unused2 blocks */
		pUnused2Anchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Unused2 block list */
	if (iUnused2Len > 0) {
		DBG_MSG("Unused2 block list");
		DBG_DEC(iUnused2Len);
		iCharsToGo = iUnused2Len;
		iBytesTooFar = -1;
		for (pCurr = pUnused2Anchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.iLength);
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iCharsToGo -= pCurr->tInfo.iLength / 2;
				if (iCharsToGo < 0) {
					iBytesTooFar = -2 * iCharsToGo;
				}
			} else {
				iCharsToGo -= pCurr->tInfo.iLength;
				if (iCharsToGo < 0) {
					iBytesTooFar = -iCharsToGo;
				}
			}
			if (iCharsToGo <= 0) {
				break;
			}
		}
	}
	if (iUnused2Len <= 0) {
		/* Empty unused2 list */
		pGarbageAnchor = pUnused2Anchor;
		pUnused2Anchor = NULL;
	} else if (pCurr == NULL) {
		/* No garbage block list */
		pGarbageAnchor = NULL;
	} else if (iCharsToGo == 0) {
		/* Move the intergral number of garbage blocks */
		pGarbageAnchor = pCurr->pNext;
	} else {
		/* Reduce the part-endnote block */
		DBG_DEC(iBytesTooFar);
		fail(iBytesTooFar <= 0);
		pCurr->tInfo.iLength -= iBytesTooFar;
		/* Move the integral number of garbage blocks */
		pGarbageAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Free the garbage block list, this should never be needed */
	pCurr = pGarbageAnchor;
	while (pCurr != NULL) {
		DBG_HEX(pCurr->tInfo.iFileOffset);
		DBG_DEC(pCurr->tInfo.iLength);
		pNext = pCurr->pNext;
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}

#if defined(DEBUG)
	/* Check the number of bytes in the block lists */
	iTotal = 0;
	for (pCurr = pTextAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		NO_DBG_HEX(pCurr->tInfo.iFileOffset);
		NO_DBG_HEX(pCurr->tInfo.iTextOffset);
		NO_DBG_DEC(pCurr->tInfo.iLength);
		fail(pCurr->tInfo.iLength <= 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.iLength));
			iTotal += pCurr->tInfo.iLength / 2;
		} else {
			iTotal += pCurr->tInfo.iLength;
		}
	}
	DBG_DEC(iTotal);
	if (iTotal != iTextLen) {
		DBG_DEC(iTextLen);
		werr(1, "Software error (Text)");
	}
	iTotal = 0;
	for (pCurr = pFootAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.iFileOffset);
		NO_DBG_HEX(pCurr->tInfo.iTextOffset);
		DBG_DEC(pCurr->tInfo.iLength);
		fail(pCurr->tInfo.iLength <= 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.iLength));
			iTotal += pCurr->tInfo.iLength / 2;
		} else {
			iTotal += pCurr->tInfo.iLength;
		}
	}
	DBG_DEC(iTotal);
	if (iTotal != iFootnoteLen) {
		DBG_DEC(iFootnoteLen);
		werr(1, "Software error (Footnotes)");
	}
	iTotal = 0;
	for (pCurr = pUnused1Anchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.iFileOffset);
		NO_DBG_HEX(pCurr->tInfo.iTextOffset);
		DBG_DEC(pCurr->tInfo.iLength);
		fail(pCurr->tInfo.iLength <= 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.iLength));
			iTotal += pCurr->tInfo.iLength / 2;
		} else {
			iTotal += pCurr->tInfo.iLength;
		}
	}
	DBG_DEC(iTotal);
	if (iTotal != iUnused1Len) {
		DBG_DEC(iUnused1Len);
		werr(1, "Software error (Unused1-list)");
	}
	iTotal = 0;
	for (pCurr = pEndAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.iFileOffset);
		NO_DBG_HEX(pCurr->tInfo.iTextOffset);
		DBG_DEC(pCurr->tInfo.iLength);
		fail(pCurr->tInfo.iLength <= 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.iLength));
			iTotal += pCurr->tInfo.iLength / 2;
		} else {
			iTotal += pCurr->tInfo.iLength;
		}
	}
	DBG_DEC(iTotal);
	if (iTotal != iEndnoteLen) {
		DBG_DEC(iEndnoteLen);
		werr(1, "Software error (Endnotes)");
	}
	iTotal = 0;
	for (pCurr = pUnused2Anchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.iFileOffset);
		NO_DBG_HEX(pCurr->tInfo.iTextOffset);
		DBG_DEC(pCurr->tInfo.iLength);
		fail(pCurr->tInfo.iLength <= 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.iLength));
			iTotal += pCurr->tInfo.iLength / 2;
		} else {
			iTotal += pCurr->tInfo.iLength;
		}
	}
	DBG_DEC(iTotal);
	if (iTotal != iUnused2Len) {
		DBG_DEC(iUnused2Len);
		werr(1, "Software error (Unused2-list)");
	}
#endif /* DEBUG */

	if (!bMustExtend) {
		return;
	}
	/*
	 * All blocks (except the last one) must have a length that
	 * is a multiple of the Big Block Size
	 */

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pUnused1Anchor;
	apAnchors[3] = pEndAnchor;
	apAnchors[4] = pUnused2Anchor;

	for (iIndex = 0; iIndex < 5; iIndex++) {
		for (pCurr = apAnchors[iIndex];
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			if (pCurr->pNext != NULL &&
			    pCurr->tInfo.iLength % BIG_BLOCK_SIZE != 0) {
				DBG_HEX(pCurr->tInfo.iFileOffset);
				DBG_HEX(pCurr->tInfo.iTextOffset);
				DBG_DEC(pCurr->tInfo.iLength);
				pCurr->tInfo.iLength /= BIG_BLOCK_SIZE;
				pCurr->tInfo.iLength++;
				pCurr->tInfo.iLength *= BIG_BLOCK_SIZE;
				DBG_DEC(pCurr->tInfo.iLength);
			}
		}
	}
} /* end of vSplitBlockList */

/*
 * uiGetDocumentLength - get the combined character length of the three lists
 *
 * returns: The total number of characters
 */
unsigned int
uiGetDocumentLength(void)
{
	list_mem_type	*apAnchors[3];
	list_mem_type	*pCurr;
	int	iTotal, iIndex;

	DBG_MSG("uiGetDocumentLength");

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pEndAnchor;

	iTotal = 0;
	for (iIndex = 0; iIndex < 3; iIndex++) {
		for (pCurr = apAnchors[iIndex];
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			fail(pCurr->tInfo.iLength <= 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.iLength));
				iTotal += pCurr->tInfo.iLength / 2;
			} else {
				iTotal += pCurr->tInfo.iLength;
			}
		}
	}
	DBG_DEC(iTotal);
	return iTotal;
} /* end of uiGetDocumentLength */

/*
 * iNextTextByte - get the next byte from the text block list
 */
static int
iNextTextByte(FILE *pFile, int *piOffset)
{
	static int	iByteNext = 0;
	static int	iBlockOffset = 0;
	int	iReadLen, iReadOff;

	if (pTextBlockCurrent == NULL ||
	    iByteNext >= sizeof(aucBlock) ||
	    iBlockOffset + iByteNext >= pTextBlockCurrent->tInfo.iLength) {
		if (pTextBlockCurrent == NULL) {
			/* First block, first part */
			pTextBlockCurrent = pTextAnchor;
			iBlockOffset = 0;
		} else if (iBlockOffset + sizeof(aucBlock) <
					pTextBlockCurrent->tInfo.iLength) {
			/* Same block, next part */
			iBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pTextBlockCurrent = pTextBlockCurrent->pNext;
			iBlockOffset = 0;
		}
		if (pTextBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return EOF;
		}
		iReadLen = pTextBlockCurrent->tInfo.iLength - iBlockOffset;
		if (iReadLen > sizeof(aucBlock)) {
			iReadLen = sizeof(aucBlock);
		}
		iReadOff = pTextBlockCurrent->tInfo.iFileOffset +
				iBlockOffset;
		if (!bReadBytes(aucBlock, iReadLen, iReadOff, pFile)) {
			return EOF;
		}
		iByteNext = 0;
	}
	if (piOffset != NULL) {
		*piOffset = pTextBlockCurrent->tInfo.iFileOffset +
			iBlockOffset +
			iByteNext;
	}
	return aucBlock[iByteNext++];
} /* end of iNextTextByte */

/*
 * iNextFootByte - get the next byte from the footnote block list
 */
static int
iNextFootByte(FILE *pFile, int *piOffset)
{
	static int	iByteNext = 0;
	static int	iBlockOffset = 0;
	int	iReadLen, iReadOff;

	if (pFootBlockCurrent == NULL ||
	    iByteNext >= sizeof(aucBlock) ||
	    iBlockOffset + iByteNext >= pFootBlockCurrent->tInfo.iLength) {
		if (pFootBlockCurrent == NULL) {
			/* First block, first part */
			pFootBlockCurrent = pFootAnchor;
			iBlockOffset = 0;
		} else if (iBlockOffset + sizeof(aucBlock) <
					pFootBlockCurrent->tInfo.iLength) {
			/* Same block, next part */
			iBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pFootBlockCurrent = pFootBlockCurrent->pNext;
			iBlockOffset = 0;
		}
		if (pFootBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return EOF;
		}
		iReadLen = pFootBlockCurrent->tInfo.iLength - iBlockOffset;
		if (iReadLen > sizeof(aucBlock)) {
			iReadLen = sizeof(aucBlock);
		}
		iReadOff = pFootBlockCurrent->tInfo.iFileOffset +
				iBlockOffset;
		if (!bReadBytes(aucBlock, iReadLen, iReadOff, pFile)) {
			return EOF;
		}
		iByteNext = 0;
	}
	if (piOffset != NULL) {
		*piOffset = pFootBlockCurrent->tInfo.iFileOffset +
			iBlockOffset +
			iByteNext;
	}
	return aucBlock[iByteNext++];
} /* end of iNextFootByte */

/*
 * iNextEndByte - get the next byte from the endnote block list
 */
static int
iNextEndByte(FILE *pFile, int *piOffset)
{
	static int	iByteNext = 0;
	static int	iBlockOffset = 0;
	int	iReadLen, iReadOff;

	if (pEndBlockCurrent == NULL ||
	    iByteNext >= sizeof(aucBlock) ||
	    iBlockOffset + iByteNext >= pEndBlockCurrent->tInfo.iLength) {
		if (pEndBlockCurrent == NULL) {
			/* First block, first part */
			pEndBlockCurrent = pEndAnchor;
			iBlockOffset = 0;
		} else if (iBlockOffset + sizeof(aucBlock) <
					pEndBlockCurrent->tInfo.iLength) {
			/* Same block, next part */
			iBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pEndBlockCurrent = pEndBlockCurrent->pNext;
			iBlockOffset = 0;
		}
		if (pEndBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return EOF;
		}
		iReadLen = pEndBlockCurrent->tInfo.iLength - iBlockOffset;
		if (iReadLen > sizeof(aucBlock)) {
			iReadLen = sizeof(aucBlock);
		}
		iReadOff = pEndBlockCurrent->tInfo.iFileOffset +
				iBlockOffset;
		if (!bReadBytes(aucBlock, iReadLen, iReadOff, pFile)) {
			return EOF;
		}
		iByteNext = 0;
	}
	if (piOffset != NULL) {
		*piOffset = pEndBlockCurrent->tInfo.iFileOffset +
			iBlockOffset +
			iByteNext;
	}
	return aucBlock[iByteNext++];
} /* end of iNextEndByte */

/*
 * iNextTextChar - get the next character from the text block list
 */
static int
iNextTextChar(FILE *pFile, int *piOffset)
{
	int	iChar, iMSB;

	iChar = iNextTextByte(pFile, piOffset);
	if (iChar == EOF) {
		return EOF;
	}
	if (pTextBlockCurrent->tInfo.bUsesUnicode) {
		iMSB = iNextTextByte(pFile, NULL);
	} else {
		iMSB = 0x00;
	}
	if (iMSB == EOF) {
		DBG_MSG("iNextTextChar: Unexpected EOF");
		DBG_HEX_C(piOffset != NULL, *piOffset);
		return EOF;
	}
	iChar |= iMSB << 8;
	return iChar;
} /* end of iNextTextChar */

/*
 * iNextFootChar - get the next character from the footnote block list
 */
static int
iNextFootChar(FILE *pFile, int *piOffset)
{
	int	iChar, iMSB;

	iChar = iNextFootByte(pFile, piOffset);
	if (iChar == EOF) {
		return EOF;
	}
	if (pFootBlockCurrent->tInfo.bUsesUnicode) {
		iMSB = iNextFootByte(pFile, NULL);
	} else {
		iMSB = 0x00;
	}
	if (iMSB == EOF) {
		DBG_MSG("iNextFootChar: Unexpected EOF");
		DBG_HEX_C(piOffset != NULL, *piOffset);
		return EOF;
	}
	iChar |= iMSB << 8;
	return iChar;
} /* end of iNextFootChar */

/*
 * iNextEndChar - get the next character from the endnote block list
 */
static int
iNextEndChar(FILE *pFile, int *piOffset)
{
	int	iChar, iMSB;

	iChar = iNextEndByte(pFile, piOffset);
	if (iChar == EOF) {
		return EOF;
	}
	if (pEndBlockCurrent->tInfo.bUsesUnicode) {
		iMSB = iNextEndByte(pFile, NULL);
	} else {
		iMSB = 0x00;
	}
	if (iMSB == EOF) {
		DBG_MSG("iNextEndChar: Unexpected EOF");
		DBG_HEX_C(piOffset != NULL, *piOffset);
		return EOF;
	}
	iChar |= iMSB << 8;
	return iChar;
} /* end of iNextEndChar */

/*
 * iNextChar - get the next character from the given block list
 */
int
iNextChar(FILE *pFile, list_id_enum eListID, int *piOffset)
{
	fail(pFile == NULL);

	switch (eListID) {
	case text_list:
		return iNextTextChar(pFile, piOffset);
	case footnote_list:
		return iNextFootChar(pFile, piOffset);
	case endnote_list:
		return iNextEndChar(pFile, piOffset);
	default:
		if (piOffset != NULL) {
			*piOffset = -1;
		}
		return EOF;
	}
} /* end of iNextChar */

/*
 * Translate the start from the begin of the text to an offset in the file.
 *
 * Returns:	 -1: in case of error
 *		>=0: the computed file offset
 */
int
iTextOffset2FileOffset(int iTextOffset)
{
	list_mem_type	*apAnchors[5];
	list_mem_type	*pCurr;
	int		iIndex;
	BOOL		bStartOfList;

	fail(iTextOffset < 0);

	if (iTextOffset < 0) {
		return -1;
	}

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pUnused1Anchor;
	apAnchors[3] = pEndAnchor;
	apAnchors[4] = pUnused2Anchor;

	bStartOfList = FALSE;

	for (iIndex = 0; iIndex < 5; iIndex++) {
		pCurr = apAnchors[iIndex];
		while (pCurr != NULL) {
			if (bStartOfList) {
				NO_DBG_DEC(iIndex);
				NO_DBG_HEX(pCurr->tInfo.iFileOffset);
				if (iIndex >= 4) {
					/* Past the last used byte */
					return -1;
				}
				return pCurr->tInfo.iFileOffset;
			}
			if (iTextOffset < pCurr->tInfo.iTextOffset ||
			    iTextOffset >= pCurr->tInfo.iTextOffset +
			     pCurr->tInfo.iLength) {
				pCurr = pCurr->pNext;
				continue;
			}
			switch (iIndex) {
			case 0:
			case 1:
			case 3:
			/* The textoffset is in the current block */
				return pCurr->tInfo.iFileOffset +
					iTextOffset -
					pCurr->tInfo.iTextOffset;
			case 2:
			/* Use the start of the next non-empty list */
				bStartOfList = TRUE;
				break;
			case 4:
			/* In Unused2, means after the last used byte */
				return -1;
			default:
			/* This should not happen */
				return -1;
			}
			if (bStartOfList) {
				/* To the start of the next list */
				break;
			}
		}
	}
	/* Passed beyond the end of the last list */
	DBG_HEX(iTextOffset);
	return -1;
} /* end of iTextOffset2FileOffset */
