#include <iostream>
#include "../include/animorph/Mesh.h"

#ifdef DEBUG
  #include <StopClock/StopClock.h>
#endif //DEBUG

#include <assert.h>

using namespace std;
using namespace Animorph;

// Control the Target loading policy

/* Set this to get additional notification messages when the targets are loaded.
   This may be helpful for debugging purposes */
// #define DEBUG_LAZY_LOADING

/* This define turns lazy loading ON! 
 * you may restore the old loading policy (eager loading when the program starts) 
 * If you unset this! */
#define USE_LAZY_LOAD

Mesh::Mesh()
: facevector             (),
  hotspotmap             (),
  vertexvector_morph     (),  // container for modified mesh
  vertexvector_morph_copy(),  // copy container for morphed mesh  
  vertexvector_orginal   (),  // container for orginal mesh
  vgroup                 (),
  bodyset                (),
  targetmap              (),
  materialvector         (),
  centeroid              (),
  texture_vector         (),
  poses                  (),  // container for applied poses
  posemap                (),  // container for all poses  
  charactersmap          ()   // container for all characters
{
}

Mesh::~Mesh()
{
    clearTargetmap();
    clearPosemap();
}

bool Mesh::makeMuscles (const string& muscle_group, const Euler& e)
{
  BodySettings sample_bs;

  for (int r1 = -90; r1 <= 90; r1 += 30)
  {
    for (int r2 = -90; r2 <= 90; r2 += 30)
    {
      for (int r3 = -90; r3 <= 90; r3 += 30)
      {
        string sample_name = muscle_group + "/" +
                             toString (r1) + "_"  +
                             toString (r2) + "_" +
                             toString (r3) + ".target";

        TargetMap::iterator target_it = targetmap.find (sample_name);

        if (target_it != targetmap.end ())
        {

          Euler sample_rot (static_cast <float> (r1),
                            static_cast <float> (r2),
                            static_cast <float> (r3), Euler::XYZ);
          float dist = vectorDistance (e, sample_rot);

          if (dist < 30)
          {
            float weight = 1 - (dist / 30.0);
            sample_bs[sample_name] = weight;

            cout << "Using sample: " << sample_name << endl;

          }
        }
      }
    }
  }

  doMorph (sample_bs, true);

  if (sample_bs.size () == 0)
    return false;
  else
    return true;
}

void Mesh::clearTargetmap()
{
    TargetMap::iterator i;
    for (i=targetmap.begin(); i != targetmap.end(); ++i)
    {
        delete i->second;
    }
    targetmap.clear();
}

void Mesh::clearPosemap()
{
    PoseMap::iterator i;
    for (i=posemap.begin(); i != posemap.end(); ++i)
    {
        delete i->second;
    }
    posemap.clear();
}


void Mesh::rotateLimb (const string& joint, const Matrix& m)
{
  VGroupData &vgroup_data = vgroup[joint];
  Vector3f &center = centeroid[joint+"Joint"];

  for (VGroupData::iterator vg_data_it = vgroup_data.begin ();
       vg_data_it != vgroup_data.end ();
       vg_data_it++)
  {
    int vn = (*vg_data_it).first;

    Vertex &vertex = vertexvector_morph[vn];

    Vector3f& rot(vertex.co);

    rot -= center;
    rot  = rot * m;
    rot += center;
  }
}

Vector3f Mesh::calcCenteroid(const vector<int>& vertexNumbers)
{
  Vector3f center;
  center.zero();
  
  for (vector<int>::const_iterator v_it = vertexNumbers.begin ();
       v_it != vertexNumbers.end ();
       v_it++)
  {
    int vn = *v_it;

    const Vertex &vertex(vertexvector_morph_copy[vn]);
    center += vertex.co;
  }

  if (vertexNumbers.size ())
  {
    center /= vertexNumbers.size ();    
  }
  
  return center;
}

void Mesh::calcCenteroids ()
{
  for (VertexGroup::iterator vg_it = vgroup.begin ();
       vg_it != vgroup.end ();
       vg_it++)
  {
    string centeroid_name = (*vg_it).first;
    VGroupData &vgroup_data = (*vg_it).second;
    Vector3f center;
    center.zero ();

    for (VGroupData::iterator vg_data_it = vgroup_data.begin ();
         vg_data_it != vgroup_data.end ();
         vg_data_it++)
    {
      int vn = (*vg_data_it).first;

      const Vertex &vertex(vertexvector_morph[vn]);

      center += vertex.co;
    }

    if (vgroup_data.size ())
    {
      center /= vgroup_data.size ();
      centeroid[centeroid_name] = center;
      cout << "Centeroid: " << centeroid_name << ": " << center << endl;
    }
  }
}

void Mesh::calcFormFactor ()
{
  for (VertexGroup::iterator vg_it = vgroup.begin ();
       vg_it != vgroup.end ();
       vg_it++)
  {
    string vgroup_name = (*vg_it).first;
    VGroupData &vgroup_data = (*vg_it).second;
    Vector3f form_factor;
    form_factor.zero ();
    int n = 0;

    for (VGroupData::iterator vg_data_it = vgroup_data.begin ();
         vg_data_it != vgroup_data.end ();
         vg_data_it++)
    {
      int vn = (*vg_data_it).first;

      const Vertex &vertex(vertexvector_morph[vn]);

      for (VGroupData::const_iterator vg_data_it2 = vgroup_data.begin ();
           vg_data_it2 != vgroup_data.end ();
           vg_data_it2++)
      {
        int vn2 = (*vg_data_it2).first;

        const Vertex &vertex2(vertexvector_morph[vn2]);

        // do this better?
        form_factor.x += abs <float> (vertex.co.x - vertex2.co.x);
        form_factor.y += abs <float> (vertex.co.y - vertex2.co.y);
        form_factor.z += abs <float> (vertex.co.z - vertex2.co.z);

        n++;
      }
    }

    if (n > 0)
      form_factor /= n;

    cout << vgroup_name << form_factor << endl;
    // TODO: write names and vectors into FormFactor
  }
}

// save the information into each vertex to which face it belong
void Mesh::calcSharedVertices ()
{
  for (unsigned int i = 0; i < facevector.size (); i++)
  {
    int vertex_number;

    const Face &face(facevector[i]);

    for (unsigned int j = 0; j < face.getSize (); j++)
    {
      vertex_number = face.getVertexAtIndex(j);

      Vertex &vertex = vertexvector_morph[vertex_number];

      vertex.addSharedFace (i);
    }
  }
}

void Mesh::calcVertexNormals ()
{
  for (unsigned int i = 0; i < vertexvector_morph.size (); i++)
  {
    Vertex &vertex(vertexvector_morph[i]);

    vector <int> & faces(vertex.getSharedFaces ());
    for (unsigned int j = 0; j < faces.size (); j++)
    {
      try
      {
        const Vector3f &face_normal(facevector.at (faces[j]).no);
        vertex.no += face_normal;
      }
      catch (const exception &e)
      {
        return;
      }
    }

    vertex.no.normalize ();
  }
}

void Mesh::calcFaceNormals ()
{
  for (unsigned int i = 0; i < facevector.size (); i++)
  {
    Face &face = facevector[i];
    if (face.getSize () >= 3)
    {
      const Vertex &vertex1(vertexvector_morph[face.getVertexAtIndex(0)]);
      const Vertex &vertex2(vertexvector_morph[face.getVertexAtIndex(1)]);
      const Vertex &vertex3(vertexvector_morph[face.getVertexAtIndex(2)]);

      const Vector3f v1_tmp(vertex2.co - vertex1.co);
      const Vector3f v2_tmp(vertex3.co - vertex1.co);

      face.no = crossProduct (v1_tmp, v2_tmp);
      face.no.normalize ();
    }
    else
    {
      cerr << "Error: a face need at least 3 vertices!" << endl;
      return;
    }
  }
}

void Mesh::calcNormals ()
{
  calcFaceNormals ();
  calcVertexNormals ();
}
/*
void Mesh::calcMuscles ()
{
  BodySettings tmp = poses;
  doPose(tmp, true);
}
*/
bool Mesh::loadMeshFactory (const string& mesh_filename, const string& faces_filename)
{
  bool vload = vertexvector_morph.load (mesh_filename);
  bool fload = facevector.loadGeometry (faces_filename);

  if (!vload || !fload)
    return false;

  // calculate shared vertices
  calcSharedVertices ();

  calcNormals ();

  // create initial copy for orginal mesh
  vertexvector_orginal.clear ();
  for (unsigned int i = 0; i < vertexvector_morph.size (); i++)
  {
    const Vertex &vertex_morph(vertexvector_morph[i]);
    vertexvector_orginal.push_back (vertex_morph.co);
  }

  return true;
}

/* ========================================================================== */
/** Get a Ptr to a Pose Target Object which belongs to a given name
 * If the pose target does not exists return NULL.
 *
 * @param inTargetname The name of the Pose Target to get.
 * @return a pointer to the pose target to get or NULL if this target does not 
 *         exists
 */
/* ========================================================================== */
PoseTarget* Mesh::getPoseTargetForName(const string& inTargetname) const
{
  // Check if a pose entry for the given name exists
  PoseMap::const_iterator i = posemap.find(inTargetname);
  if (i == posemap.end())
    return NULL; // There is no PoseTarget for this named entry!

  PoseEntry *poseEntry = i->second;
  if (poseEntry == NULL)
    return NULL; // There is no PoseEntry is not valid!

  // get the actual poseTarget for this entry (eventually by lazy loading!)
  PoseTarget *poseTarget = poseEntry->getTarget();
  if (poseTarget == NULL)
    return NULL; // There is no poseTarget is not valid!

  return poseTarget;
}

bool Mesh::loadMaterialFactory (const string& material_filename, const string& face_colors_filename)
{
  bool mload = materialvector.loadMaterials (material_filename);
  bool fcload = facevector.loadColors (face_colors_filename);

  if (!mload || !fcload)
    return false;

  return true;
}

void Mesh::loadAuxiliaryPoseTargetsFactory (const string& target_root_path, int recursive_level)
{
  const string negative_type("-.rot");
  const string positive_type(".rot");
  
  DirectoryList dir_list;

  dir_list.setRootPath (target_root_path);
  dir_list.setRecursive (recursive_level);
  dir_list.setFileFilter (positive_type);

  const StringList &str_list(dir_list.getDirectoryList ());
     
  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string filename (file);

    string target_name (filename);
    target_name.erase (0, target_root_path.length()+1);
    
    bool is_negative = (target_name.find(negative_type) != string::npos);
    
    if(is_negative)
    {
      target_name = target_name.substr(0, target_name.length() - negative_type.length() - 3) + ".target";
    }
    else
    {
      target_name = target_name.substr(0, target_name.length() - positive_type.length() - 3) + ".target";
    }
    
    PoseMap::const_iterator i = posemap.find(target_name);
    if (i != posemap.end()) 
    {
      PoseEntry *tmpPoseEntry = i->second;
      if (tmpPoseEntry != NULL)
      {
        tmpPoseEntry->addAuxRotation(filename, is_negative);
      }
    }
  }    
}

void Mesh::loadPoseTargetsFactory (const string& target_root_path, int recursive_level)
{
  const string negative_type("-.target");
        
  clearPosemap(); // wipe out all old posmeap entries first

  // reading all the targets recursive
  DirectoryList dir_list;

  dir_list.setRootPath (target_root_path);
  dir_list.setRecursive (recursive_level);
  dir_list.setFileFilter (".target");

  const StringList &str_list(dir_list.getDirectoryList ());

#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadPoseTargetsFactory () : Reading %lu entries...\n", str_list.size());
#endif

  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string target_name (file);

    target_name.erase (0, target_root_path.length()+1);
    
    bool is_negative = (target_name.find(negative_type) != string::npos);
    
    if(is_negative)
    {
      target_name = target_name.substr(0, target_name.length() - negative_type.length()) + ".target";
      const string filename = file.substr(0, file.length() - negative_type.length()) + ".target";
      
#ifdef USE_LAZY_LOAD
      posemap[target_name] = new PoseEntry(filename, false, true);      
#else
      posemap[target_name] = new PoseEntry(filename, true, true); // true -> signal "eager load"; true -> mHasNegative;
#endif
    }
    else 
    {
      if (!posemap.count (target_name))
      {
#ifdef USE_LAZY_LOAD
        posemap[target_name] = new PoseEntry(file, false, false);
#else
        posemap[target_name] = new PoseEntry(file, true, false); // true -> signal "eager load"
#endif            
      }
    }
  }     

  loadAuxiliaryPoseTargetsFactory(target_root_path, recursive_level);
#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadPoseTargetsFactory () : Ready!\n");
#endif
}

void Mesh::loadCharactersFactory (const string& characters_root_path, int recursive_level)
{
  // reading all the targets recursive
  DirectoryList dir_list;

  dir_list.setRootPath (characters_root_path);
  dir_list.setRecursive (recursive_level);
  dir_list.setFileFilter (".bs");

  const StringList &str_list(dir_list.getDirectoryList ());

#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadCharactersFactory () : Reading %lu entries...\n", str_list.size());
#endif

  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string character_name (file);
    
    character_name.erase (0, characters_root_path.length()+1);

    BodySettings character;

    character.load (file);

    charactersmap[character_name] = character;
  }  

#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadCharactersFactory () : Ready!\n");
#endif
}

void Mesh::loadTargetsFactory (const string& target_root_path, int recursive_level)
{
  clearTargetmap(); // wipe out all old targetmap entries first

  // reading all the targets recursive
  DirectoryList dir_list;

  dir_list.setRootPath (target_root_path);
  dir_list.setRecursive (recursive_level);
  dir_list.setFileFilter (".target");

  const StringList &str_list(dir_list.getDirectoryList ());

#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadTargetsFactory () : Reading %lu entries...\n", str_list.size());
#endif

  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string target_name (file);
    target_name.erase (0, target_root_path.length()+1);

#ifdef USE_LAZY_LOAD
    TargetEntry *targetEntry = new TargetEntry(file);
#else
    TargetEntry *targetEntry = new TargetEntry(file, true); // true -> signal "eager load"
#endif

    targetmap[target_name] = targetEntry;
  }

#ifdef DEBUG_LAZY_LOADING
  fprintf(stderr, "Mesh::loadTargetsFactory () : Ready loading\n");
#endif

}

Mesh Mesh::getTriangleMesh ()
{
  Mesh triangle_mesh;
  FaceVector   &tri_facevector(triangle_mesh.getFaceVectorRef ());
  VertexVector &tri_vertexvector(triangle_mesh.getVertexVectorRef ());

  // same vertices for trimesh object!
  tri_vertexvector = vertexvector_morph;

  // calculate new faces
  for (unsigned int i = 0; i < facevector.size(); i++)
  {
    const Face &face = facevector[i];

    if (face.getSize () == 3)
    {
      // yet a trangle
      tri_facevector.push_back (face);
    }
    else if (face.getSize () == 4) // split quat into two triangles
    {
      int a = face.getVertexAtIndex(0);
      int b = face.getVertexAtIndex(1);
      int c = face.getVertexAtIndex(2);
      int d = face.getVertexAtIndex(3);

      Face tri_face1 (a, b, c);
      Face tri_face2 (c, d, a);

      tri_face1.setMaterialIndex (face.getMaterialIndex ());
      tri_face2.setMaterialIndex (face.getMaterialIndex ());

      tri_facevector.push_back (tri_face1);
      tri_facevector.push_back (tri_face2);
    }
    else
    {
      cerr << "Sorry only quads and triangles are supported!" << endl;
    }
  }

  return triangle_mesh;
}

Mesh Mesh::getVGroupMesh (const string& vgroup_ident)
{
  Mesh vgroup_mesh;
  VertexVector &vertexvector_vg = vgroup_mesh.getVertexVectorRef ();
  FaceVector &facevector_vg = vgroup_mesh.getFaceVectorRef ();

  if (!vgroup.count (vgroup_ident))
  {
    cerr << "Vertex Group with name '" << vgroup_ident
    << "' doesn't exist!" << endl;
    // TODO: throw an exeption here instead of returning an empty Mesh!
    return vgroup_mesh;
  }

  VGroupData &vgroup_data = vgroup[vgroup_ident];
  vector <int> faces_list;
  map <int, int> vt_map; // vertex translation map
  //unsigned int new_vn = 0;

  // save translation from old to new vertex numbers...
  for (VGroupData::iterator vg_it = vgroup_data.begin ();
       vg_it != vgroup_data.end ();
       vg_it++)
  {
    int old_vn = (*vg_it).first;

    vt_map[old_vn] = vertexvector_vg.size (); // ...through size ()
    vertexvector_vg.push_back (vertexvector_morph[old_vn]);
    //Vector3f &v = vertexvector_morph[vertex_number].co;
    //cout << "v " << v.x << " " << v.y << " " << v.z << endl;
  }

  // create copy of all Materials
  // better is to copy only needed materials, but could be fixed later...
  vgroup_mesh.materialvector = materialvector;

  for (unsigned int i = 0; i < facevector.size (); i++)
  {
    const Face &face(facevector[i]);

    bool all = true;
    for (unsigned int n = 0; n < face.getSize (); n++)
    {
      int vnum = face.getVertexAtIndex(n);

      if (!vgroup_data.count (vnum))
      {
        all = false;
        break;
      }
    }

    if (all)
    {
      faces_list.push_back (i);
    }
  }
  // now faces_list contains all faces of the new mesh

  for (unsigned int i = 0; i < faces_list.size ();  i++)
  {
    Face &face = facevector[faces_list[i]];

    // create link to material index
    Material mat = materialvector[face.getMaterialIndex ()];
    Face *face_fg = NULL;

    if (face.getSize() == 3)
    {
        int v0 = face.getVertexAtIndex(0);
        int v1 = face.getVertexAtIndex(1);
        int v2 = face.getVertexAtIndex(2);

        face_fg = new Face(vt_map[v0], vt_map[v1], vt_map[v2]);
    }
    else if (face.getSize() == 4)
    {
        int v0 = face.getVertexAtIndex(0);
        int v1 = face.getVertexAtIndex(1);
        int v2 = face.getVertexAtIndex(2);
        int v3 = face.getVertexAtIndex(3);

        face_fg = new Face(vt_map[v0], vt_map[v1], vt_map[v2], vt_map[v3]);

    }
    else
    {
        cerr << "Face with illegal size of " << face.getSize() << endl;
        assert(false);
    }

    face_fg->setMaterialIndex (face.getMaterialIndex ());
    facevector_vg.push_back(*face_fg);
    delete face_fg;
}

  vgroup_mesh.calcSharedVertices ();

  vgroup_mesh.calcNormals ();

  return vgroup_mesh;
}


const Target* Mesh::getTargetForName(const string& inTargetname)
{
  TargetMap::iterator i = targetmap.find(inTargetname);
  if (i == targetmap.end())
    return NULL;

  return (i->second)->getTarget();
}

bool Mesh::doMorph (const string& target_name, float morph_value)
{
#ifdef DEBUG
  StopClock sc;
  float elapsed_time;
  sc.resetClock ();
#endif //DEBUG

  // return if target doesn't exist
  if (!targetmap.count (target_name))
  {
    cerr << "a target with name \"" <<  target_name
    << "\" wasn't found in targetmap" << endl;
    return false;
  }

  float real_morph_value;
  float bs_morph_value = bodyset[target_name];

  // TODO: round morph value??

  // get correct additive morph value
  if (morph_value == 0.0)
  {
    real_morph_value = bs_morph_value * -1;
  }
  else
  {
    real_morph_value = (bs_morph_value * -1) + morph_value;
  }

  const Target *target = getTargetForName(target_name);

  for (Target::const_iterator target_it = target->begin ();
       target_it != target->end ();
       target_it++)
  {
    const TargetData &td(*target_it);

    vertexvector_morph[td.vertex_number].co += 
      (td.morph_vector - vertexvector_orginal[td.vertex_number]) * real_morph_value;
  }

  if(morph_value == 0.0)
  {
    bodyset.erase(target_name);
  }
  else 
  {
    bodyset[target_name] = morph_value;     
  }

#ifdef DEBUG
  elapsed_time = sc.getElapsedTime (StopClock::TIME_UNIT_SECONDS);
  printf ("morph time: %f\n", elapsed_time);
#endif //DEBUG

  return true;
}

void Mesh::doMorph (const BodySettings &bs, bool clear)
{
#ifdef DEBUG
  StopClock sc;
  float elapsed_time;
  sc.resetClock ();
#endif //DEBUG

  if (clear)
  {
    bodyset.clear ();

    vertexvector_morph.setCoordinates (vertexvector_orginal);
  }

  for (BodySettings::const_iterator bs_it = bs.begin ();
       bs_it != bs.end ();
       bs_it++)
  {
    const string &target_name((*bs_it).first);
    float morph_value = (*bs_it).second;

    doMorph (target_name, morph_value);
  }


#ifdef DEBUG
  elapsed_time = sc.getElapsedTime (StopClock::TIME_UNIT_SECONDS);
  printf ("bodyset morph time: %f\n", elapsed_time);
#endif //DEBUG
}

void Mesh::doMorph (const BodySettings &bs, float value, bool clear)
{
  if (clear)
  {
    bodyset.clear ();

    vertexvector_morph.setCoordinates (vertexvector_orginal);
  }

  for (BodySettings::const_iterator bs_it = bs.begin ();
       bs_it != bs.end ();
       bs_it++)
  {
    const string &target_name((*bs_it).first);
    float morph_value = (*bs_it).second;

    doMorph (target_name, morph_value * value);
  }
}

void Mesh::poseMode()
{
  vertexvector_morph_copy = vertexvector_morph;

  for (PoseMap::iterator target_it = posemap.begin ();
       target_it != posemap.end ();
       target_it++)
  {
    PoseEntry* poseEntry = target_it->second;
    assert(poseEntry);
    PoseTarget *tmp = poseEntry->getTarget();

    if(tmp->hasNegative())
    {
      tmp->getNegativeTarget().setCenter(calcCenteroid(tmp->getNegativeTarget().getCenterVertexNumbers()));
      
      // calcCenteroid of the auxiliary rotations
      for (vector<PoseSemiTarget>::iterator v_it = tmp->getAuxNegativeTargets().begin ();
           v_it != tmp->getAuxNegativeTargets().end ();
           v_it++)
      {
        (*v_it).setCenter(calcCenteroid((*v_it).getCenterVertexNumbers()));
      }
    } 
    if(tmp->hasPositive())
    {
      tmp->getPositiveTarget().setCenter(calcCenteroid(tmp->getPositiveTarget().getCenterVertexNumbers()));
      
      // calcCenteroid of the auxiliary rotations
      for (vector<PoseSemiTarget>::iterator v_it = tmp->getAuxPositiveTargets().begin ();
           v_it != tmp->getAuxPositiveTargets().end ();
           v_it++)
      {
        (*v_it).setCenter(calcCenteroid((*v_it).getCenterVertexNumbers()));
      }
    }     
  }  

  BodySettings tmp = poses;
  
  doPose(tmp, true);
}

void Mesh::bodyDetailsMode()
{
  vertexvector_morph = vertexvector_morph_copy;
}

void Mesh::resetMorph ()
{
  BodySettings bodyset_empty;

  // call doMorph with empty bodyset does effectiv reset the mesh
  doMorph (bodyset_empty, true);
}

void Mesh::resetPose (const PoseSemiTarget &target)
{ 
  for (PoseSemiTarget::const_iterator target_it = target.begin ();
       target_it != target.end ();
       target_it++)
  {
    const PoseTargetData &td = (*target_it);
    
    vertexvector_morph[td.vertex_number].co = vertexvector_morph_copy[td.vertex_number].co;
  }
}

bool Mesh::setPose (const std::string& target_name, float morph_value)
{  
  // return if target doesn't exist
  if (!posemap.count (target_name))
  {
    cerr << "a target with name \"" <<  target_name
    << "\" wasn't found in posemap" << endl;
    return false;
  }
  
  if(morph_value == 0.0)
  {
    poses.erase(target_name);
  }
  else 
  {       
    poses[target_name] = morph_value;     
  }
  
  PoseTarget *poseTarget = getPoseTargetForName(target_name);
  assert(poseTarget);

  PoseSemiTarget &target = (morph_value < 0) ? poseTarget->getNegativeTarget() : 
                                               poseTarget->getPositiveTarget();  
  
  UsedVertex &modVertex(target.getModVertex());
  
  // reset of the auxiliary rots
  vector<PoseSemiTarget> &auxTargets = (morph_value < 0) ? poseTarget->getAuxNegativeTargets() :
                                                           poseTarget->getAuxPositiveTargets();

  for (vector<PoseSemiTarget>::iterator v_it = auxTargets.begin ();
       v_it != auxTargets.end ();
       v_it++)
  {
    PoseSemiTarget &auxTarget = (*v_it);
    UsedVertex &usedVertex = auxTarget.getModVertex();
    
    resetPose(auxTarget);
    modVertex.insert(usedVertex.begin(), usedVertex.end());
  }
  
  resetPose(target);

  for (BodySettings::iterator poses_it = poses.begin ();
       poses_it != poses.end ();
       poses_it++)
  {
    string target_name = (*poses_it).first;
    float morph_value = (*poses_it).second;    

    doPose (target_name, morph_value, modVertex);
  }
  return true;
}

void Mesh::doPose (const string& target_name, float morph_value, const UsedVertex& modVertex)
{
  float theta;
  
  PoseTarget *poseTarget = getPoseTargetForName(target_name);
  assert(poseTarget);

  const PoseSemiTarget &target = (morph_value < 0) ? poseTarget->getNegativeTarget() : 
                                                     poseTarget->getPositiveTarget();  
  RotateAxis axis = target.getAxis();
  
  // rotation matrix
  Matrix rotMatrix;

  // application of the auxiliary rots
  const vector<PoseSemiTarget> &auxTargets = (morph_value < 0) ? poseTarget->getAuxNegativeTargets() :
                                                                 poseTarget->getAuxPositiveTargets();

  for (vector<PoseSemiTarget>::const_iterator v_it = auxTargets.begin ();
       v_it != auxTargets.end ();
       v_it++)
  {
    const PoseSemiTarget &auxTarget = (*v_it);

    for (PoseSemiTarget::const_iterator target_it = auxTarget.begin ();
         target_it != auxTarget.end ();
         target_it++)
    {
      const PoseTargetData &td = (*target_it);

      // continue if the Pose Target datas Vertex is not part of the modVertex
      if(modVertex.find(td.vertex_number) == modVertex.end())
      {
        continue;
      }

      theta = morph_value * td.rotation;

      rotMatrix.setRotation(theta, auxTarget.getAxis());
      vertexvector_morph[td.vertex_number].co = ((vertexvector_morph[td.vertex_number].co - auxTarget.getCenter()) * rotMatrix) + auxTarget.getCenter();
    }
  }

  for (PoseSemiTarget::const_iterator target_it = target.begin ();
       target_it != target.end ();
       target_it++)
  {        
    const PoseTargetData &td(*target_it);

    // continue if the Pose Target datas Vertex is not part of the modVertex
    if(modVertex.find(td.vertex_number) == modVertex.end()) 
    {
      continue;                                     
    }    
    
    theta = morph_value * td.rotation;
    rotMatrix.setRotation(theta, axis); 
    
    vertexvector_morph[td.vertex_number].co = ((vertexvector_morph[td.vertex_number].co - target.getCenter()) * rotMatrix) + target.getCenter();
  }
  poses[target_name] = morph_value;    
}

bool Mesh::doPose (const string& target_name, float morph_value)
{
#ifdef DEBUG
  StopClock sc;
  float elapsed_time;
  sc.resetClock ();
#endif //DEBUG

  // return if target doesn't exist
  if (!posemap.count (target_name))
  {
    cerr << "a target with name \"" <<  target_name
    << "\" wasn't found in posemap" << endl;
    return false;
  }

  PoseTarget *poseTarget = getPoseTargetForName(target_name);
  assert(poseTarget);

  // rotation matrix
  Matrix rotMatrix;

  // application of the auxiliary rots
  const vector<PoseSemiTarget> &auxTargets = (morph_value < 0) ? poseTarget->getAuxNegativeTargets() :
                                                                 poseTarget->getAuxPositiveTargets();

  for (vector<PoseSemiTarget>::const_iterator v_it = auxTargets.begin ();
       v_it != auxTargets.end ();
       v_it++)
  {
    const PoseSemiTarget &auxTarget = (*v_it);

    for (PoseSemiTarget::const_iterator target_it = auxTarget.begin ();
         target_it != auxTarget.end ();
         target_it++)
    {
      const PoseTargetData &td = (*target_it);

      float theta = morph_value * td.rotation;

      rotMatrix.setRotation(theta, auxTarget.getAxis());
      vertexvector_morph[td.vertex_number].co = ((vertexvector_morph[td.vertex_number].co - auxTarget.getCenter()) * rotMatrix) + auxTarget.getCenter();
    }
  }

  const PoseSemiTarget &target = (morph_value < 0) ? poseTarget->getNegativeTarget() :
                                                     poseTarget->getPositiveTarget();
  
  for (PoseSemiTarget::const_iterator target_it = target.begin ();
       target_it != target.end ();
       target_it++)
  {
    const PoseTargetData &td = (*target_it);
    
    float theta = morph_value * td.rotation;
    
    rotMatrix.setRotation(theta, target.getAxis());    
    vertexvector_morph[td.vertex_number].co = ((vertexvector_morph[td.vertex_number].co - target.getCenter()) * rotMatrix) + target.getCenter();
  }

  poses[target_name] = morph_value;

#ifdef DEBUG
  elapsed_time = sc.getElapsedTime (StopClock::TIME_UNIT_SECONDS);
  printf ("morph time: %f\n", elapsed_time);
#endif //DEBUG

  return true;
}

void Mesh::doPose (const BodySettings &bs, bool clear)
{
#ifdef DEBUG
  StopClock sc;
  float elapsed_time;
  sc.resetClock ();
#endif //DEBUG

  if (clear)
  {
    poses.clear ();
    vertexvector_morph = vertexvector_morph_copy;
  }

  for (BodySettings::const_iterator bs_it = bs.begin ();
       bs_it != bs.end ();
       bs_it++)
  {
    string target_name = (*bs_it).first;
    float morph_value = (*bs_it).second;    
    doPose (target_name, morph_value);
  }

#ifdef DEBUG
  elapsed_time = sc.getElapsedTime (StopClock::TIME_UNIT_SECONDS);
  printf ("bodyset morph time: %f\n", elapsed_time);
#endif //DEBUG
}

Target Mesh::createTarget ()
{
  Vector3f v_zero;
  v_zero.zero ();
  const float tolerance = 0.000001;
  Target t;

  for (unsigned int i = 0; i < vertexvector_morph.size (); i++)
  {
    Vertex   &vertex_morph = vertexvector_morph[i];
    Vector3f &vertex_orginal = vertexvector_orginal[i];

    Vector3f v_diff = vertex_orginal - vertex_morph.co;

    // don't insert if smaller as tolerance
    if (v_diff.getMagnitude () > tolerance)
    {
      TargetData td;
      td.vertex_number = i;
      td.morph_vector = vertex_morph.co;
    
      t.push_back (td);
    }
  }

  return t;
}



/* ========================================================================== *
 * This Class encapsulates a Target in order to load a target lazily when 
 * it is actually used. 
 *
 * When this object is created just the filename of the target will be stored.
 * This ensures to load he target when they are actually used at run time!
 * ========================================================================== */

/* ========================================================================== *
 * TargetEntry - Public
 * ========================================================================== */

/* ========================================================================== */
/** The ctor just remembers the name of the filename which is associated to 
 *  the target with
 *
 * @param inFilename The name of the Target to load to.
 * @param inPreload set this to true if you want to preload the target hence
 *                  skipping the lazy load. You may omit this parameter. 
 *                  In this case it defaults to false ( = lazy loading)
 */
/* ========================================================================== */
TargetEntry::TargetEntry(const string &inFilename, bool inPreload)
: mFilename(new string(inFilename)), 
  mTarget(NULL),           // not initialized yet
  mTargetLoadTry(false)    // not yet tried
{
    if (inPreload)
        loadFromFile();
}

/* ========================================================================== */
/** The dtor destroys all encapsulated members and the Target too!
 */
/* ========================================================================== */
TargetEntry::~TargetEntry() 
{
    delete mFilename; 
    delete mTarget;
}

/* ========================================================================== */
/** Get the target. If the Target has not been loaded then load it now (lazyliy)
 *  If the target does not exists, then this call returns NULL;
 *
 * @return The Target or null if a target with the given filename (in the ctor)
 *         does not exists.
 */
/* ========================================================================== */
Target* TargetEntry::getTarget()
{
#ifdef DEBUG_LAZY_LOADING
    printf("TargetEntry::getTarget() : Requesting target %s... ", mFilename->c_str()); 
#endif    
    // check if the data has already been loaded!
    if (mTargetLoadTry == false)
    {
        // no - the try to load it...
        bool rc = loadFromFile();
#ifdef DEBUG_LAZY_LOADING
      printf("[got from File! (rc = %d)]\n", rc);
#endif
    }
#ifdef DEBUG_LAZY_LOADING
    else
    {
        printf("[got from Cache!]\n");
    }
#endif
    return mTarget;

}

// Privates

/* ========================================================================== *
 * TargetEntry - Private
 * ========================================================================== */


/* ========================================================================== */
/** Try to load the given target. The file is just attempted to load once.
 *
 *  @return return true when the file could be loaded or false otherwise.
 */
/* ========================================================================== */
bool TargetEntry::loadFromFile()
{
    // if the target has not tried to be loaded then try it NOW
    if (mTargetLoadTry == false)
    {
        mTargetLoadTry = true; // marks as 'tried to load' for further attempts

        mTarget = new(std::nothrow) Target();
        assert(mTarget);
        bool rc = mTarget->load(*mFilename);

        // When the attempt to load the target failed, then release the target
        if (rc == false)
        {
            delete mTarget;
            mTarget = NULL;
        }
    }
    return (mTarget != NULL); // when a target object exists, then return true
}

/* ========================================================================== *
 * This Class encapsulates a Target in order to load a target lazily when 
 * it is actually used. 
 *
 * When this object is created just the filename of the target will be stored.
 * This ensures to load he target when they are actually used at run time!
 * ========================================================================== */

/* ========================================================================== *
 * PoseEntry - Public
 * ========================================================================== */

/* ========================================================================== */
/** The ctor just remembers the name of the filename which is associated to 
 *  the target with.
 *
 * @param inFilename The name of the Pose Target to load to.
 * @param inPreload set this to true if you want to preload the target hence
 *                  skipping the lazy load. You may omit this parameter. 
 *                  In this case it defaults to false ( = lazy loading)
 */
/* ========================================================================== */
PoseEntry::PoseEntry(const string &inFilename, bool inPreload, bool inNegative)
: mFilename(new string(inFilename)), 
  mTarget(NULL),           // not initialized yet
  mTargetLoadTry(false),   // not yet tried
  mHasNegative(inNegative)
{
    if (inPreload)
        loadFromFile();
}

/* ========================================================================== */
/** The dtor destroys all encapsulated members and the Target too!
 */
/* ========================================================================== */
PoseEntry::~PoseEntry() 
{
    delete mFilename; 
    delete mTarget;
}

void PoseEntry::addAuxRotation(const string& filename, const bool negative)
{
  if(negative)
  {
    negativeAuxRotations.push_back(filename);            
  }
  else 
  {
    positiveAuxRotations.push_back(filename);       
  }
}

/* ========================================================================== */
/** Get the target. If the Target has not been loaded then load it now (lazyliy)
 *  If the target does not exists, then this call returns NULL;
 *
 * @return The Target or null if a target with the given filename (in the ctor)
 *         does not exists.
 */
/* ========================================================================== */
PoseTarget* PoseEntry::getTarget()
{
#ifdef DEBUG_LAZY_LOADING
    printf("PoseEntry::getTarget() : Requesting target %s... ", mFilename->c_str()); 
#endif

    // check if the data has already been loaded!
    if (mTargetLoadTry == false)
    {
        // no - the try to load it...
        bool rc = loadFromFile();

#ifdef DEBUG_LAZY_LOADING
      printf("[got from File! (rc = %d)]\n", rc);
#endif

    }
#ifdef DEBUG_LAZY_LOADING
    else
    {
      printf("[Reusing data!]\n");
    }
#endif

    return mTarget;
}

/* ========================================================================== *
 * PoseEntry - Private
 * ========================================================================== */


/* ========================================================================== */
/** Try to load the given target. The file is just attempted to load once.
 *
 *  @return return true when the file could be loaded or false otherwise.
 */
/* ========================================================================== */
bool PoseEntry::loadFromFile()
{
    // if the target has not tried to be loaded then try it NOW
    if (mTargetLoadTry == false)
    {
        const string rot_extension(".target");
            
        mTargetLoadTry = true; // marks as 'tried to load' for further attempts

        mTarget = new(std::nothrow) PoseTarget();
        assert(mTarget);                       

        bool rc_1 = true;
        bool rc_2 = true;
        
        if(mHasNegative)
        {
          rc_1 = mTarget->load((*mFilename).substr(0, (*mFilename).length() - rot_extension.length()) + "-.target", false); // negative SemiTarget
          rc_2 = mTarget->load(*mFilename, true);  // positive SemiTarget
        }
        else
        {
          rc_1 = mTarget->load(*mFilename, false); // only positive SemiTarget    
        }
        
        // When the attempt to load the target failed, then release the target
        if (rc_1 == false || rc_2 == false)
        {
            delete mTarget;
            mTarget = NULL;
        }
        else
        {
          // load auxiliary rotations
          for (vector<string>::const_iterator v_it = negativeAuxRotations.begin ();
               v_it != negativeAuxRotations.end ();
               v_it++)     
          {           
            mTarget->loadAuxiliaryRotation((*v_it), true);
          }
          
          for (vector<string>::const_iterator v_it = positiveAuxRotations.begin ();
               v_it != positiveAuxRotations.end ();
               v_it++)     
          {    
            mTarget->loadAuxiliaryRotation((*v_it), false);
          }          
        }
    }
    return (mTarget != NULL); // when a target object exists, then return true
}

