/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: window.c,v 1.1.1.1 2003/08/14 03:28:52 cosine Exp $
 */

#include <assert.h>
#include <string.h>

#include "alddefs.h"
#include "command.h"
#include "defs.h"
#include "misc.h"
#include "version.h"
#include "window.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

#ifdef USE_CURSES

#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#endif
#ifdef HAVE_PANEL_H
#include <panel.h>
#endif

static int MakeWindows();
static struct Frame *CreateFrame(int color, int rows, int columns,
                                 int tly, int tlx);
static void RedrawFrames();
static void DrawBorder(struct Frame *frame);
static void MakeBar(WINDOW *win, char *str);
static void wCenterString(WINDOW *win, int y, const char *str);

/*
 * Local: window where version information is displayed
 */
static struct Frame             *TitleFrame = 0;

/*
 * Global: frame where debugging instructions are displayed
 */
struct Frame                    *DebugFrame = 0;

/*
 * Global: frame where disassembled instructions go
 */
struct Frame                    *DisassemblyFrame = 0;

/*
 * Global: frame where command prompt is located
 */
struct Frame                    *CommandPromptFrame = 0;

/*
 * Global: frame where user commands are entered
 */
struct Frame                    *CommandInputFrame = 0;

/*
 * Global: frame where output from commands go
 */
struct Frame                    *CommandOutputFrame = 0;

/*
 * Global: frame where register values are displayed
 */
struct Frame                    *RegistersFrame = 0;

/*
 * Global: frame where process' output is displayed
 */
struct Frame                    *ProcessOutputFrame = 0;

/*
 * Global: current active frame
 */
struct Frame                    *CurrentFrame = 0;

/*
 * This array contains the sizes and positions of each frame/panel
 */
struct FrameInfo Frames[] = {
  { &DebugFrame, "Debugging Window" },
  { &CommandOutputFrame, "Command Output Window" },
  { &DisassemblyFrame, "Disassembly Window" },
  { &CommandInputFrame, "Command Input Window" },
  { &RegistersFrame, "Register Window" },
  { &ProcessOutputFrame, "Process Output Window" },

  { 0, 0 }
};

/*
InitWindows()
 Initialize window panels

Return: 1 if successful
        0 if unsuccessful
*/

int
InitWindows()

{
  if (!initscr())
  {
    fprintf(stderr, "initscr() error\n");
    return (0);
  }

  /*
   * Start up the command window
   */
  if (!MakeWindows())
  {
    fprintf(stderr, "InitWindows(): Error creating windows\n");
    return (0);
  }

  cbreak();
  noecho();

  return (1);
} /* InitWindows() */

/*
MakeWindows()
  Initialize panels and windows

Return: 1 if successful
        0 if not
*/

static int
MakeWindows()

{
  int plen,
      rlen;

  TitleFrame = CreateFrame(COLOR_BLACK, 1, COLS, 0, 0);

  DebugFrame = CreateFrame(COLOR_BLACK, LINES / 2, COLS, 1, 0);

  CommandOutputFrame = CreateFrame(COLOR_BLACK,
                                   (LINES / 2) - 1,
                                   COLS,
                                   (LINES / 2) + 1,
                                   0);

  DisassemblyFrame = CreateFrame(COLOR_BLACK, LINES / 2, COLS, 1, 0);

  plen = strlen(mainWorkspace_p->commandWorkspace_p->CmdPrompt) + 4;
  CommandPromptFrame = CreateFrame(COLOR_BLACK, 1, plen, LINES - 1, 0);
  CommandInputFrame = CreateFrame(COLOR_BLACK,
                                  1,
                                  COLS - plen - 1,
                                  LINES - 1,
                                  plen);

  rlen = 22; /* number of columns in register window */
  RegistersFrame = CreateFrame(COLOR_BLACK,
                               LINES - 2,
                               rlen,
                               1,
                               COLS - rlen);

  ProcessOutputFrame = CreateFrame(COLOR_BLACK,
                                   (LINES / 2) - 1,
                                   COLS,
                                   (LINES / 2) + 1,
                                   0);

  if (!TitleFrame || !DebugFrame || !CommandOutputFrame ||
      !DisassemblyFrame || !CommandInputFrame || !RegistersFrame ||
      !CommandPromptFrame || !ProcessOutputFrame)
    return (0);

  DebugFrame->number = 1;
  DebugFrame->flags = W_HIGHLIGHT | W_RAISED | W_AUTORAISE;

  CommandOutputFrame->number = 2;
  CommandOutputFrame->flags = W_SCROLL | W_BLOCKMODE | W_RAISED;

  DisassemblyFrame->number = 3;
  DisassemblyFrame->flags = W_HIGHLIGHT | W_AUTORAISE;

  CommandInputFrame->number = 4;

  RegistersFrame->number = 5;
  RegistersFrame->flags = W_SCROLL | W_STAYONTOP;

  ProcessOutputFrame->number = 6;
  ProcessOutputFrame->flags = W_SCROLL | W_BLOCKMODE;

  hide_panel(DisassemblyFrame->panel);
  hide_panel(RegistersFrame->panel);
  hide_panel(ProcessOutputFrame->panel);

  /*
   * Do not change this without removing W_RAISED from above
   * flags
   */
  CurrentFrame = DebugFrame;

  RedrawFrames();

  RefreshFrames();

  return (1);
} /* MakeWindows() */

/*
CreateFrame()
  Allocate space for a new Frame structure and return a pointer
to it.

Inputs: rows    - number of rows in panel
        columns - number of colums in panel
        tly     - top left Y position
        tlx     - top left X position
*/

static struct Frame *
CreateFrame(int color, int rows, int columns, int tly, int tlx)

{
  struct Frame *ptr;
  WINDOW *win;
  WINDOW *sub;
  PANEL *panel;

  if ((win = newwin(rows, columns, tly, tlx)) == 0)
    return (0);

  if (rows > 1)
  {
    /*
     * Special case: CommandInputWindow - do not create a subwin
     */
    sub = subwin(win, rows - 2, columns - 2, tly + 1, tlx + 1);
    if (!sub)
    {
      delwin(win);
      return (0);
    }
  }
  else
    sub = 0;

  if ((panel = new_panel(win)) == 0)
  {
    delwin(win);
    delwin(sub);
    return (0);
  }

  ptr = (struct Frame *) MyMalloc(sizeof(struct Frame));
  memset(ptr, 0, sizeof(struct Frame));

  ptr->mainwin = win;

  if (sub)
    ptr->window = sub;
  else
    ptr->window = win;

  ptr->panel = panel;
  ptr->bufptr = ptr->buffer;

  /*
   * Subtract 2 to account for the border
   */
  if (rows > 1)
  {
    ptr->lines = rows - 2;
    ptr->cols = columns - 2;
  }
  else
  {
    ptr->lines = rows;
    ptr->cols = columns;
  }

  keypad(ptr->window, TRUE);

#if 0
  if (has_colors())
  {
    int fg,      /* foreground color */
        bg;      /* background color */

    fg = COLOR_WHITE;
    bg = color;

    init_pair(color, fg, bg);
    wbkgdset(ptr->window, COLOR_PAIR(color) | ' ');
  }
  else
    wbkgdset(ptr->window, A_BOLD | ' ');
#endif

  return (ptr);
} /* CreateFrame() */

void
DeleteFrames()

{
#if 0 /* bingo */
#endif /* 0 */
} /* DeleteFrames() */

/*
RedrawFrames()
 Redraw all windows/frames, but do not refresh them
*/

static void
RedrawFrames()

{
  char buffer[MAXLINE];

  assert(DebugFrame != 0);
  assert(CommandInputFrame != 0);
  assert(CommandOutputFrame != 0);

  clear();

  wSetAttr(TitleFrame->window, aReverse, 1);

  Sprintf(buffer,
    "Assembly Language Debugger %s (C) 2000 Patrick Alken",
    aVersion);
  MakeBar(TitleFrame->window, buffer);

  DrawBorder(DebugFrame);
  DrawBorder(CommandOutputFrame);
  DrawBorder(DisassemblyFrame);
  DrawBorder(RegistersFrame);
  DrawBorder(ProcessOutputFrame);
} /* RedrawFrames() */

/*
RefreshFrames()
 Refresh all windows/frames
*/

void
RefreshFrames()

{
  assert(CommandInputFrame != 0);

  update_panels();

  /*
   * apparantly subwindows aren't refreshed by update_panels() -
   * not sure why
   */
  if (CommandOutputFrame->flags & W_RAISED)
    wnoutrefresh(CommandOutputFrame->window);
  if (DisassemblyFrame->flags & W_RAISED)
    wnoutrefresh(DisassemblyFrame->window);
  if (DebugFrame->flags & W_RAISED)
    wnoutrefresh(DebugFrame->window);
  if (RegistersFrame->flags & W_RAISED)
    wnoutrefresh(RegistersFrame->window);

  /*
  touchwin(CommandInputFrame->window);
  wnoutrefresh(CommandInputFrame->window);
  */

  /*
   * Do actual refresh
   */
  doupdate();

/*
  touchwin(CommandInputFrame->window);
  wrefresh(CommandInputFrame->window);
*/
} /* RefreshFrames() */

/*
DrawBorder()
  Draw a border around the given frame
*/

static void
DrawBorder(struct Frame *frame)

{
  assert(frame != 0);

  /*
   * Draw a box around the frame
   */
  box(frame->mainwin, 0, 0);

  /*
   * Replace some of the characters on the top of the border
   * with the frame's number
   */
  wmove(frame->mainwin, 0, 2);
  wprintw(frame->mainwin, "[%d]", frame->number);
} /* DrawBorder() */

/*
DisplayCursesPrompt()
  Display command prompt
*/

void
DisplayCursesPrompt()

{
  werase(CommandInputFrame->window);

  wmove(CommandPromptFrame->window, 0, 0);

  wSetAttr(CommandPromptFrame->window, aBold, 0);

  wprintw(CommandPromptFrame->window, "%s", mainWorkspace_p->commandWorkspace_p->CmdPrompt);

  /*
   * If CommandInputWindow is the current active window,
   * set it's number identifier in bold
   */
  if (CurrentFrame == CommandInputFrame)
    wSetAttr(CommandPromptFrame->window, aBold, 1);

  wprintw(CommandPromptFrame->window, "[%d] ",
    CommandInputFrame->number);

  wSetAttr(CommandPromptFrame->window, aBold, 0);

  RefreshFrames();
} /* DisplayCursesPrompt() */

/*
wSetAttr()
 Set window attributes

Inputs: win  - window in which to set the attribute
        attr - attribute to set
        on   - if 1, set attribute on, otherwise off

Return: none
*/

void
wSetAttr(WINDOW *win, unsigned int attr, int on)

{
  assert(win != 0);

#ifdef A_BOLD

  if (attr & aBold)
  {
    if (on)
      wattron(win, A_BOLD);
    else
      wattroff(win, A_BOLD);
  }

#endif /* A_BOLD */

#ifdef A_REVERSE

  if (attr & aReverse)
  {
    if (on)
      wattron(win, A_REVERSE);
    else
      wattroff(win, A_REVERSE);
  }

#else

  if (attr & aReverse)
  {
    if (on)
      wstandout(win);
    else
      wstandend(win);
  }

#endif /* !A_REVERSE */

#ifdef A_UNDERLINE

  if (attr & aUnderline)
  {
    if (on)
      wattron(win, A_UNDERLINE);
    else
      wattroff(win, A_UNDERLINE);
  }

#else

  if (attr & aUnderline)
  {
    if (on)
      wstandout(win);
    else
      wstandend(win);
  }

#endif /* !A_UNDERLINE */
} /* wSetAttr() */

/*
MakeBar()
 Draw a bar in window with given text in the middle

Inputs: win - window to write to
        str - string to write
*/

static void
MakeBar(WINDOW *win, char *str)

{
  char buffer[MAXLINE];

  assert(win && str);

  /*
   * Draw spaces all across the window to color in the reverse
   * mode if set. For example, DividerWin1 will look like a white
   * bar going across the screen. Use TotalMaxX - 1 here, because
   * IdWin takes up the last column.
   */
  memset(buffer, ' ', COLS - 1);
  buffer[COLS] = '\0';

  wmove(win, 0, 0);
  waddstr(win, buffer);

  /*
   * Now center the given string in the middle
   */
  wCenterString(win, 0, str);
} /* MakeBar() */

/*
wCenterString()
 Output the string in the center of the line

Inputs: win - window to write to
        y   - line number to write string to
        str - string to write
*/

static void
wCenterString(WINDOW *win, int y, const char *str)

{
  int x;
  int maxx, maxy;

  assert(win != 0);

  getmaxyx(win, maxy, maxx);
  x = (maxx - strlen(str)) / 2;
  if (x < 0)
    x = 0;

  wmove(win, y, x);
  waddstr(win, str);
} /* wCenterString() */

/*
RaiseFrame()
  Raise frame to the top of the screen
*/

void
RaiseFrame(struct Frame *frame)

{
  assert(frame != 0);

  bottom_panel(CurrentFrame->panel);
  top_panel(frame->panel);

  if (frame == DebugFrame)
  {
    DisassemblyFrame->flags &= ~W_RAISED;
    RegistersFrame->flags &= ~W_RAISED;
  }
  else if (frame == DisassemblyFrame)
  {
    DebugFrame->flags &= ~W_RAISED;
    RegistersFrame->flags &= ~W_RAISED;
  }
  else if (frame == CommandOutputFrame)
  {
    RegistersFrame->flags &= ~W_RAISED;
    ProcessOutputFrame->flags &= ~W_RAISED;
  }
  else if (frame == RegistersFrame)
  {
    DisassemblyFrame->flags &= ~W_RAISED;
    DebugFrame->flags &= ~W_RAISED;
    CommandOutputFrame->flags &= ~W_RAISED;
    ProcessOutputFrame->flags &= ~W_RAISED;
  }
  else if (frame == ProcessOutputFrame)
  {
    CommandOutputFrame->flags &= ~W_RAISED;
    RegistersFrame->flags &= ~W_RAISED;
  }

  CommandInputFrame->flags &= ~W_RAISED;

  frame->flags |= W_RAISED;
  CurrentFrame = frame;

  RefreshFrames();
} /* RaiseFrame() */

/*
ReRaiseFrame()
  Re-raise the current frame by first raising 'other' and then
raising CurrentFrame
*/

void
ReRaiseFrame(struct Frame *other)

{
  struct Frame *cur;

  cur = CurrentFrame;
  RaiseFrame(other);
  RaiseFrame(cur);
} /* ReRaiseFrame() */

/*
FindFrameByNumber()
  Attempt to locate a frame by it's number

Inputs: number - frame number

Return: pointer to frame
*/

struct Frame *
FindFrameByNumber(int number)

{
  struct FrameInfo *ptr;

  for (ptr = Frames; ptr->frame; ++ptr)
  {
    if ((*(ptr->frame))->number == number)
      return (*(ptr->frame));
  }

  return (0);
} /* FindFrameByNumber() */

/*
SwitchCurrentFrame()
  Switch to frame given by 'winnum'
*/

WINDOW *
SwitchCurrentFrame(int winnum)

{
  struct Frame *fptr;

  if (!(fptr = FindFrameByNumber(winnum)))
    return (CurrentFrame->window);

  /*
   * Raise the new frame
   */
  RaiseFrame(fptr);

  return (CurrentFrame->window);
} /* SwitchCurrentFrame() */

/*
IsHidden()
  Determine if frame is completely hidden (ie: no part of it
is exposed)

Inputs: frame - frame to check

Return: 1 if it is completely hidden
        0 if not
*/

int
IsHidden(struct Frame *frame)

{
  assert(frame != 0);

  if (frame == RegistersFrame)
  {
    if (((DebugFrame->flags & W_RAISED) ||
         (DisassemblyFrame->flags & W_RAISED)) &&
        (CommandOutputFrame->flags & W_RAISED))
      return (1);
  }
  else if (frame == DebugFrame)
  {
    if (DisassemblyFrame->flags & W_RAISED)
      return (1);
  }
  else if (frame == DisassemblyFrame)
  {
    if (DebugFrame->flags & W_RAISED)
      return (1);
  }
  else if (frame == CommandOutputFrame)
  {
    if (ProcessOutputFrame->flags & W_RAISED)
      return (1);
  }
  else if (frame == ProcessOutputFrame)
  {
    if (CommandOutputFrame->flags & W_RAISED)
      return (1);
  }

  return (0);
} /* IsHidden() */

#endif /* USE_CURSES */
