/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: window.h,v 1.1.1.1 2003/08/14 03:28:50 cosine Exp $
 */

#ifndef INCLUDED_window_h
#define INCLUDED_window_h

#ifndef INCLUDED_stdio_h
#include <stdio.h>         /* FILE * */
#define INCLUDED_stdio_h
#endif

#ifndef INCLUDED_alddefs_h
#include "alddefs.h"       /* MAXLINE */
#define INCLUDED_alddefs_h
#endif

#ifndef INCLUDED_defs_h
#include "defs.h"          /* USE_CURSES */
#define INCLUDED_defs_h
#endif

#ifdef USE_CURSES

#ifndef INCLUDED_ncurses_h
#include <ncurses.h>       /* WINDOW */
#define INCLUDED_ncurses_h
#endif

#ifndef INCLUDED_panel_h
#  ifdef HAVE_PANEL_H
#    include <panel.h>
#  endif
#define INCLUDED_panel_h
#endif

/*
 * This struct is used for windows which will manipulate text
 */
struct Frame
{
  PANEL *panel;              /* panel containing window */
  WINDOW *mainwin;           /* window inside panel */
  WINDOW *window;            /* window to output text to */
  unsigned int number;       /* window number */
  unsigned int lines;        /* number of lines in window */
  unsigned int cols;         /* number of columns in window */

  /*
   *  These arrays hold the X and Y coordinates of the top left
   * and bottom right corner of the window. The coordinates are
   * with respect to the very top of the screen (0, 0).
   *  The first array index is the X coord and the second is
   * the Y coord.
   */
  int TopLeft[2];            /* top left corner */
  int BottomRight[2];        /* bottom right corner */

  char **data;               /* first piece of window data */
  int DataElements;          /* number of indices in 'data' */

  /*
   * This is used for DebugFrame only - it's indices
   * correspond to those of 'data' and it stores the addresses
   * of each instruction
   */
  unsigned int *addresses;

  /*
   * Index of 'data' that is appearing in the first line of
   * the window (for scrolling purposes)
   */
  int TopElement;

  /*
   * Index of 'data' that the window cursor is on
   */
  int CurrentElement;

  /*
   * These are used when we want to RawPrint() without printing a
   * \n character. The partial line is stored in buffer[] until
   * either Print() is called or until RawPrint() is called with
   * a terminating \n. At this point, the contents of buffer[] will
   * be added to the 'data' array it will be output to the
   * appropriate window.
   */
  char buffer[MAXLINE];
  char *bufptr;

  unsigned int flags; /* W_xxx bitmasks below */
};

struct FrameInfo
{
  struct Frame **frame; /* frame pointer */
  char *desc;           /* description */
};

/*
 * These bitmasks correspond to 'flags' in the Frame
 * structure
 */
#define W_HIGHLIGHT   (1 << 0) /* window highlights elements */
#define W_BLOCKMODE   (1 << 1) /* window is in block mode */
#define W_SCROLL      (1 << 2) /* scrolls when new text added (scrollok() call) */
#define W_NOBORDER    (1 << 3) /* no border around window */
#define W_RAISED      (1 << 4) /* window is currently raised */
#define W_AUTORAISE   (1 << 5) /* window is auto-raised when written to */
#define W_STAYONTOP   (1 << 6) /* window is immune to auto-raise */

/*
 * These bitmasks are for wSetAttr(). We can't use the actual
 * A_xxx bitmasks from ncurses.h because some curses libraries
 * may not have them.
 */
#define aBold         (1 << 0) /* bold */
#define aReverse      (1 << 1) /* reverse black and white */
#define aUnderline    (1 << 2) /* underline */

/*
 * Prototypes
 */

int InitWindows();
void DeleteFrames();
void wSetAttr(WINDOW *win, unsigned int attr, int on);
void RaiseFrame(struct Frame *frame);
void ReRaiseFrame(struct Frame *other);
struct Frame *FindFrameByNumber(int number);
WINDOW *SwitchCurrentFrame(int winnum);
int IsHidden(struct Frame *frame);

/*
 * External declarations
 */
extern struct Frame         *DebugFrame;
extern struct Frame         *CommandInputFrame;
extern struct Frame         *DisassemblyFrame;
extern struct Frame         *CommandOutputFrame;
extern struct Frame         *RegistersFrame;
extern struct Frame         *ProcessOutputFrame;
extern struct Frame         *CurrentFrame;
extern struct FrameInfo     Frames[];

#endif /* USE_CURSES */

/*
 * Non-curses specific stuff
 */

void DisplayCursesPrompt();
void RefreshFrames();

#endif /* INCLUDED_window_h */
