/* $Id: xtree.c,v 1.2 2003/03/03 11:38:36 kjc Exp $ */
/*
 * Copyright (C) 2001-2003 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifdef XTREE

#include <sys/types.h>
#include <sys/queue.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "aguri_tree.h"
#include "xtree.h"

struct wininfo {
	LIST_ENTRY(wininfo) next;
	struct	tree *tree;
	struct	tree_node *cur_top;
	Window	window;
	GC	gc;
	u_int  winW, winH;
	u_int  xoff, yoff;
	u_int  xsep, ysep;
};

#define BORDER		2
#define WINWIDTH	576	/* default window width */
#define WINHEIGHT	512	/* default window height */

int xtree = 0;
char *xtree_geometry = NULL;

static LIST_HEAD(winlist, wininfo) winlist = LIST_HEAD_INITIALIZER(&winlist);

static Display *dpy = NULL;
static Window root;
static int screen;
static Colormap cmap;
static unsigned long fgcol, bgcol;
static unsigned long pixels[10];
static unsigned int nodeW = 10;
static unsigned int nodeH = 10;
static int pause_flag = 0;

static struct wininfo *tree2wininfo(struct tree *tp);
static struct wininfo *window2wininfo(Window w);
static void draw_all(void);
static void draw_window(struct tree *tp);
struct tree_node *pos2node(struct wininfo *winfo, u_int cx, u_int cy);
static int get_position(struct tree_node *np, struct wininfo *winfo,
			u_int *x, u_int *y, u_int *px, u_int *py, int *depth);
static void drawtree(struct tree_node *np, struct wininfo *winfo, int erase);
static void do_drawtree(struct tree_node *np, struct wininfo *winfo,
		u_int x, u_int y, u_int px, u_int py, int depth, int erase);
static int makecindex(u_int64_t cnt, u_int64_t denom);
static void draw_node(struct tree_node *np, struct wininfo *winfo,
		      u_int x, u_int y, u_int px, u_int py);
static void erase_node(struct tree_node *np, struct wininfo *winfo,
		       u_int x, u_int y, u_int px, u_int py);

static struct wininfo *
tree2wininfo(struct tree *tp)
{
	struct wininfo *winfo;

	LIST_FOREACH(winfo, &winlist, next) {
		if (winfo->tree == tp)
			return (winfo);
	}
	return (NULL);
}

static struct wininfo *
window2wininfo(Window w)
{
	struct wininfo *winfo;

	LIST_FOREACH(winfo, &winlist, next) {
		if (winfo->window == w)
			return (winfo);
	}
	return (NULL);
}

int
xtree_init(struct tree *tp, const char *title, const char *geometry)
{
	struct wininfo *winfo;
	int x, y, width, height, i;
	XColor xcolor;

	if ((winfo = malloc(sizeof(*winfo))) == NULL)
		return (-1);
	winfo->tree = tp;
	winfo->cur_top = tp->tr_top;
	winfo->winW = winfo->winH = 0;
	winfo->xoff = winfo->yoff = winfo->xsep = winfo->ysep = 0;

	if (dpy == NULL) {
		dpy = XOpenDisplay(NULL);
		root = DefaultRootWindow(dpy);
		screen = DefaultScreen(dpy);
	}

	fgcol = BlackPixel(dpy, screen);
	bgcol = WhitePixel(dpy, screen);
	x = y = width = height = 0;
	if (geometry != NULL)
		XParseGeometry(geometry, &x, &y, &width, &height);
	if (width == 0)
		width = WINWIDTH;
	if (height == 0)
		height = WINHEIGHT;

	winfo->window = XCreateSimpleWindow(dpy, root, x, y, width, height,
					    BORDER, fgcol, bgcol);
	if (title != NULL)
		XStoreName(dpy, winfo->window, title);

	winfo->gc = XCreateGC(dpy, winfo->window, 0, NULL);
	XSetBackground(dpy, winfo->gc, bgcol);
	XSetForeground(dpy, winfo->gc, fgcol);

	XSelectInput(dpy, winfo->window,
	  ButtonPressMask | KeyPressMask | ExposureMask | StructureNotifyMask);

	XMapWindow(dpy, winfo->window);
	XFlush(dpy);

	/* if this is the first window, allocate colors */
	if (LIST_EMPTY(&winlist)) {
		/*
		 * create 10 colors
		 *   pixels[0]: white
		 *         ...
		 *   pixels[5]: yellow
		 *         ...
		 *   pixels[9]: red
		 */
		cmap = DefaultColormap(dpy, screen);
		for (i = 0; i < 10; i++) {
			xcolor.red = 255 << 8;
			if (i < 5)
				xcolor.green = 255 << 8;
			else
				xcolor.green = 255 * (10 - i) / 5 << 8;
			if (i < 5)
				xcolor.blue = 255 * (5 - i) / 5 << 8;
			else
				xcolor.blue = 0;
			xcolor.flags = DoRed | DoGreen | DoBlue;
			if (!XAllocColor(dpy, cmap, &xcolor)) {
				warnx("XAlocColor");
			}
			pixels[i] = xcolor.pixel;
		}
	}

	LIST_INSERT_HEAD(&winlist, winfo, next);
	
	return (0);
}

int
xtree_destroy(struct tree *tp)
{
	struct wininfo *winfo;

	if ((winfo = tree2wininfo(tp)) == NULL)
		return (-1);

	XFreeGC(dpy, winfo->gc);
	XDestroyWindow(dpy, winfo->window);

	LIST_REMOVE(winfo, next);
	free(winfo);

	if (LIST_EMPTY(&winlist)) {
		XCloseDisplay(dpy);
		dpy = NULL;
		return (1);
	}
	return (0);
}

int
xtree_doevent(int wait)
{
	struct wininfo *winfo;
	struct tree_node *np;
	XEvent event;

	if (dpy == NULL || (!wait && !XPending(dpy)))
		return (0);

	XNextEvent(dpy, &event);
	switch(event.type) {
	case Expose:
		if (event.xexpose.count == 0 &&
			(winfo = window2wininfo(event.xexpose.window)) != NULL) {
			draw_window(winfo->tree);
		}
		return (0);
		break;
	case ButtonPress:
		if ((winfo = window2wininfo(event.xexpose.window)) == NULL)
			break;
		switch (event.xbutton.button) {
		case Button1:
			/* toggle pause/resume */
			if (pause_flag == 0)
				draw_all();
			pause_flag = pause_flag ? 0 : 1;
			while (pause_flag && dpy != NULL)
				xtree_doevent(1);
			break;
		case Button2:
			return (xtree_destroy(winfo->tree));
			break;
		case Button3:
			/* zoom in/out the tree */
			xtree_erase(winfo->tree->tr_top);
			if ((np = pos2node(winfo, event.xbutton.x,
					   event.xbutton.y)) != NULL)
				/* new top node for the drawing area */
				winfo->cur_top = np;
			else if (winfo->cur_top->tn_parent != NULL)
				/* go up the top node */
				winfo->cur_top = winfo->cur_top->tn_parent;
			xtree_draw(winfo->tree->tr_top);
			break;
		}
		break;
	case KeyPress:
		{
			int n;
			char buf[64];

			if ((winfo = window2wininfo(event.xkey.window)) == NULL)
				break;
			n = XLookupString((XKeyEvent *)&event,
					  buf, sizeof(*buf), NULL, NULL);
			if (n > 0 && buf[0] == 'q')
				xtree_destroy(winfo->tree);
		}
		break;
	}
		
	return (0);
}

void
xtree_pause(void)
{
	if (dpy == NULL)
		return;

	draw_all();

	fprintf(stderr,
	    "press button 1: resume/pause 2: close window 3: zoom in/out\n");

	pause_flag = 1;
	while (pause_flag && dpy != NULL)
		xtree_doevent(1);
}

static void
draw_all(void)
{
	struct wininfo *winfo;

	LIST_FOREACH(winfo, &winlist, next) {
		draw_window(winfo->tree);
	}
}

static void
draw_window(struct tree *tp)
{
	struct wininfo *winfo;
	Window rw;
	int x, y;
	u_int width, height, border, depth;

	if ((winfo = tree2wininfo(tp)) == NULL)
		return;

	xtree_erase(tp->tr_top);

	XGetGeometry(dpy, winfo->window, &rw, &x, &y,
		     &width, &height, &border, &depth);

	/* recompute the size parameters if both width and height changed */
	if (width != winfo->winW && height != winfo->winH) {
		winfo->winW = width;
		winfo->winH = height;
		winfo->xoff = width / 2;
		winfo->yoff = height / 16;
		winfo->xsep = width / 4;
		winfo->ysep = height / 16;
	}

	xtree_draw(tp->tr_top);
}

void
xtree_draw(struct tree_node *np)
{
	struct wininfo *winfo;

	if ((winfo = tree2wininfo(np->tn_tree)) == NULL)
		return;
	drawtree(np, winfo, 0);
	XFlush(dpy);
}

void
xtree_erase(struct tree_node *np)
{
	struct wininfo *winfo;

	if ((winfo = tree2wininfo(np->tn_tree)) == NULL)
		return;
	drawtree(np, winfo, 1);
}

struct tree_node *
pos2node(struct wininfo *winfo, u_int cx, u_int cy)
{
	struct tree_node *np;
	u_int x, y, depth;

	np = winfo->cur_top;
	x = winfo->xoff;
	y = winfo->yoff;
	depth = 0;

	while (np != NULL) {
		if (cx >= x - nodeW/2 && cx <= x + nodeW/2 &&
		    cy >= y - nodeH/2 && cy <= y + nodeH/2)
			return (np);

		if (cx < x) {
			np = np->tn_left;
			x = x - (winfo->xsep >> depth);
		} else {
			np = np->tn_right;
			x = x + (winfo->xsep >> depth);
		}
		y = y + winfo->ysep;
		depth++;
	}
	return (NULL);
}

/*
 * compute the position of a node (x, y) and that of its parent (px, py)
 */
static int
get_position(struct tree_node *np, struct wininfo *winfo,
	     u_int *x, u_int *y, u_int *px, u_int *py, int *depth)
{
	int rval;

	if (np == winfo->cur_top) {
		*x = winfo->xoff;
		*y = winfo->yoff;
		*px = 0;
		*py = 0;
		*depth = 0;
		return (1);
	}
	if (np->tn_parent == NULL)
		/* not in the drawing area */
		return (0);

	rval = get_position(np->tn_parent, winfo, x, y, px, py, depth);
	if (rval == 0)
		return (0);
	*px = *x;
	*py = *y;
	if (np == np->tn_parent->tn_left)
		*x = *px - (winfo->xsep >> *depth);
	else
		*x = *px + (winfo->xsep >> *depth);
	*y = *py + winfo->ysep;
	(*depth)++;
	return (1);
}

static void
drawtree(struct tree_node *np, struct wininfo *winfo, int erase)
{
	u_int x, y, px, py;
	int depth;

	if (get_position(np, winfo, &x, &y, &px, &py, &depth) == 0) {
		/*
		 * the position of the specified node is out of the
		 * drawing area.  just use the current top.
		 */
		np = winfo->cur_top;
		get_position(np, winfo, &x, &y, &px, &py, &depth);
	}

	do_drawtree(np, winfo, x, y, px, py, depth, erase);

	/* re-draw parent */
	if (np != winfo->cur_top && np->tn_parent != NULL)
		draw_node(np->tn_parent, winfo, px, py, 0, 0);
}

static void
do_drawtree(struct tree_node *np, struct wininfo *winfo, u_int x, u_int y,
	    u_int px, u_int py, int depth, int erase)
{
	if (np->tn_left != NULL)
		do_drawtree(np->tn_left, winfo,
			    x - (winfo->xsep >> depth), y + winfo->ysep,
			    x, y, depth + 1, erase);
	if (np->tn_right != NULL)
		do_drawtree(np->tn_right, winfo,
			    x + (winfo->xsep >> depth), y + winfo->ysep,
			    x, y, depth + 1, erase);
	if (erase)
		erase_node(np, winfo, x, y, px, py);
	else
		draw_node(np, winfo, x, y, px, py);
}

/*
 * map a counter value to one of 10 color indices
 * this function is a simplified version of
 *	cindex = (int)(sqrt((double)cnt / denom * 2) * 10);
 * the color becomes yellow at 12.5% and red at 40.5%
 */
static int
makecindex(u_int64_t cnt, u_int64_t denom)
{
	int idx, val;

	if (denom == 0)
		return (0);

	val = (int)(cnt * 1000 / denom);
	if (val < 5)
		idx = 0;
	else if (val < 20)
		idx = 1;
	else if (val < 45)
		idx = 2;
	else if (val < 80)
		idx = 3;
	else if (val < 125)
		idx = 4;
	else if (val < 180)
		idx = 5;
	else if (val < 245)
		idx = 6;
	else if (val < 320)
		idx = 7;
	else if (val < 405)
		idx = 8;
	else
		idx = 9;
	return (idx);
}

static void
draw_node(struct tree_node *np, struct wininfo *winfo,
	  u_int x, u_int y, u_int px, u_int py)
{
	char buf[128];
	int cindex;

	cindex = makecindex(np->tn_count, np->tn_tree->tr_count);
	if (px != 0)
		XDrawLine(dpy, winfo->window, winfo->gc, x, y, px, py);
	if (np->tn_count > 0) {
		XSetForeground(dpy, winfo->gc, pixels[cindex]);
		XFillArc(dpy, winfo->window, winfo->gc,
			 x - nodeW/2, y - nodeH/2,
			 nodeW, nodeH, 0, 360*64);
		XSetForeground(dpy, winfo->gc, fgcol);
		XDrawArc(dpy, winfo->window, winfo->gc,
			 x - nodeW/2, y - nodeH/2, nodeW, nodeH, 0, 360*64);

		snprintf(buf, sizeof(buf), "/%u", (u_int)np->tn_prefixlen);
		XDrawString(dpy, winfo->window, winfo->gc,
			    x + nodeW/2, y - nodeH/2, buf, strlen(buf));
	} else
		XSetForeground(dpy, winfo->gc, fgcol);
}

static void
erase_node(struct tree_node *np, struct wininfo *winfo,
	   u_int x, u_int y, u_int px, u_int py)
{
	char buf[128];

	XSetForeground(dpy, winfo->gc, bgcol);
	XFillArc(dpy, winfo->window, winfo->gc,
		 x - nodeW/2, y - nodeH/2, nodeW, nodeH, 0, 360*64);
	XDrawArc(dpy, winfo->window, winfo->gc,
		 x - nodeW/2, y - nodeH/2, nodeW, nodeH, 0, 360*64);

	snprintf(buf, sizeof(buf), "/%u", (u_int)np->tn_prefixlen);
	XDrawString(dpy, winfo->window, winfo->gc,
		    x + nodeW/2, y - nodeH/2, buf, strlen(buf));

	if (px != 0)
		XDrawLine(dpy, winfo->window, winfo->gc, x, y, px, py);
	XSetForeground(dpy, winfo->gc, fgcol);
}

#endif /* XTREE */
