/*  $Id: sdlmain.cpp,v 1.16 2009/02/27 03:08:33 sarrazip Exp $
    sdlmain.cpp - main() function for Afternoon Stalker

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001-2009 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "AfternoonStalkerEngine.h"

#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>

#ifdef HAVE_GETOPT_LONG
    #include <getopt.h>
#endif

#include <iostream>
#include <string>

using namespace std;


#ifdef HAVE_GETOPT_LONG
static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "ms-per-frame",  required_argument, NULL, 'm' },
    { "no-sound",      no_argument,       NULL, 'n' },
    { "full-screen",   no_argument,       NULL, 'f' },
    { "maze",          required_argument, NULL, 's' },
    { "rules",         required_argument, NULL, 'r' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo(ostream &out)
{
    out << PACKAGE << " " << VERSION << "\n";
}


static void
displayHelp(ostream &out)
{
    out << "\n";

    displayVersionNo(out);

    out <<
"\n"
"Copyright (C) 2001, 2002, 2003 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    ;

    out <<
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--ms-per-frame=N   N milliseconds per animation frame [default: 55]\n"
"                   Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--full-screen      Attempt to use full screen mode [window mode]\n"
"--maze=M           Select maze M: 1 = original, 2 = alternative\n"
"--rules=R          Rules to use (R must be 1 or 2, default is 1)\n"
"                   1 = original Afternoon Stalker rules\n"
"                   2 = 2008 updated Afternoon Stalker rules\n"
"\n"
    ;
}
#endif  /* HAVE_GETOPT_LONG */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    int millisecondsPerFrame = 55;
    bool fullScreen = false;
    string mazeFilename = "level1.set";
    AfternoonStalkerEngine::Rules rules = AfternoonStalkerEngine::AS_RULES_1;

    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
	c = getopt_long(argc, argv, "", knownOptions, NULL);

	switch (c)
	{
	    case EOF:
		break;  // nothing to do

	    case 'm':
		{
		    errno = 0;
		    long n = strtol(optarg, NULL, 10);
		    if (errno == ERANGE || n < 1 || n > 1000)
		    {
			cerr << "Invalid number of ms per frame.\n";
			displayHelp(cerr);
			return EXIT_FAILURE;
		    }

		    millisecondsPerFrame = int(n);
		}
		break;
	    
	    case 'f':
		fullScreen = true;
		break;

	    case 's':
		{
		    string m = optarg;
		    if (m == "1")
			mazeFilename = "level1.set";
		    else if (m == "2")
			mazeFilename = "level2.set";
		    else
		    {
			cerr << "Invalid '--maze' argument \"" << optarg << "\".\n";
			displayHelp(cerr);
			return EXIT_FAILURE;
		    }
		}
		break;

	    case 'r':
		{
		    string r = optarg;
		    if (r == "1")
			rules = AfternoonStalkerEngine::AS_RULES_1;
		    else if (r == "2")
			rules = AfternoonStalkerEngine::AS_RULES_2;
		    else
		    {
			cerr << "Invalid '--rules' argument \"" << optarg << "\".\n";
			displayHelp(cerr);
			return EXIT_FAILURE;
		    }
		}
		break;

	    case 'v':
		displayVersionNo(cout);
		return EXIT_SUCCESS;

	    case 'h':
		displayHelp(cout);
		return EXIT_SUCCESS;

	    default:
		displayHelp(cerr);
		return EXIT_FAILURE;
	}
    } while (c != EOF && c != '?');

    #endif  /* HAVE_GETOPT_LONG */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    cerr << "seed = " << seed << "\n";
    cerr << "ms-per-frame = " << millisecondsPerFrame << "\n";
    #endif
    srand(seed);


    try
    {
	AfternoonStalkerEngine theAfternoonStalkerEngine(
			"Afternoon Stalker",
			fullScreen,
			mazeFilename,
			rules);
	theAfternoonStalkerEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
	cerr << PACKAGE << ": engine init error: " << s << "\n";
	return EXIT_FAILURE;
    }
    catch (int e)
    {
	cerr << PACKAGE << ": init error # " << e << "\n";
	return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
