/*  $Id: sdlmain.cpp,v 1.11 2004/05/02 04:55:46 sarrazip Exp $
    sdlmain.cpp - main() function for Afternoon Stalker

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "AfternoonStalkerEngine.h"

#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>

#ifdef HAVE_GETOPT_LONG
    #include <getopt.h>
#endif

#include <iostream>
#include <string>

using namespace std;


#ifdef HAVE_GETOPT_LONG
static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "ms-per-frame",  required_argument, NULL, 'm' },
    { "no-sound",      no_argument,       NULL, 'n' },
    { "full-screen",   no_argument,       NULL, 'f' },
    { "pedantic",      no_argument,       NULL, 'p' },
    { "only-blue",     no_argument,       NULL, 'b' },
    { "set-file",      required_argument, NULL, 's' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo(ostream &out)
{
    out << PACKAGE << " " << VERSION << "\n";
}


static void
displayHelp(ostream &out)
{
    out << "\n";

    displayVersionNo(out);

    out <<
"\n"
"Copyright (C) 2001, 2002, 2003 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    ;

    out <<
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--ms-per-frame=N   N milliseconds per animation frame [default: 55]\n"
"                   Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--no-sound         Disable sound effects [enabled]\n"
"--full-screen      Attempt to use full screen mode [window mode]\n"
"--only-blue        Practice mode: only send blue robots\n"
"--set-file=F       Use file F as the set description file [level1.set]\n"
"                   F must be in " PKGDATADIR "\n"
"\n"
    ;
}
#endif  /* HAVE_GETOPT_LONG */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    int millisecondsPerFrame = 55;
    bool useSound = true;
    bool pedantic = false;
    bool fullScreen = false;
    bool onlyBlueRobots = false;
    string setFilename = "level1.set";

    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
	c = getopt_long(argc, argv, "", knownOptions, NULL);

	switch (c)
	{
	    case EOF:
		break;  // nothing to do

	    case 'm':
		{
		    errno = 0;
		    long n = strtol(optarg, NULL, 10);
		    if (errno == ERANGE || n < 1 || n > 1000)
		    {
			cerr << "Invalid number of ms per frame.\n";
			displayHelp(cerr);
			return EXIT_FAILURE;
		    }

		    millisecondsPerFrame = int(n);
		}
		break;
	    
	    case 'n':
		useSound = false;
		break;
	    
	    case 'f':
		fullScreen = true;
		break;

	    case 'p':
		pedantic = true;
		break;

	    case 'b':
		onlyBlueRobots = true;
		break;

	    case 's':
		setFilename = optarg;
		break;

	    case 'v':
		displayVersionNo(cout);
		return EXIT_SUCCESS;

	    case 'h':
		displayHelp(cout);
		return EXIT_SUCCESS;

	    default:
		displayHelp(cerr);
		return EXIT_FAILURE;
	}
    } while (c != EOF && c != '?');

    #endif  /* HAVE_GETOPT_LONG */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    cerr << "seed = " << seed << "\n";
    cerr << "ms-per-frame = " << millisecondsPerFrame << "\n";
    #endif
    srand(seed);


    try
    {
	AfternoonStalkerEngine theAfternoonStalkerEngine(
			"Afternoon Stalker",
			useSound, pedantic, fullScreen, onlyBlueRobots,
			setFilename);
	theAfternoonStalkerEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
	cerr << PACKAGE << ": engine init error: " << s << "\n";
	return EXIT_FAILURE;
    }
    catch (int e)
    {
	cerr << PACKAGE << ": init error # " << e << "\n";
	return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
