/*  $Id: AfternoonStalkerEngine.h,v 1.23 2004/05/02 04:55:46 sarrazip Exp $
    AfternoonStalkerEngine.h - A robot-killing video game engine.

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_AfternoonStalkerEngine
#define _H_AfternoonStalkerEngine

#define PAUL 12

#include <flatzebra/GameEngine.h>
#include <flatzebra/Sprite.h>
#include <flatzebra/SoundMixer.h>
#include <flatzebra/KeyState.h>

class RobotSprite;

#include <string>
#include <vector>

using namespace flatzebra;


class AfternoonStalkerEngine : public GameEngine
/*  A robot-killing video game.
*/
{
public:

    enum { RIGHT, UP, LEFT, DOWN };
    /*	Possible directions of movement, in trigonometric order.
	This enumeration is garanteed to start at zero.
	The order of the constants is guaranteed, so they are 0, 1, 2, 3.
    */

    AfternoonStalkerEngine(const std::string &windowManagerCaption,
			    bool useSound,
			    bool pedantic,
			    bool fullScreen,
			    bool onlyBlueRobots,
			    const std::string &setFilename)
				throw(int, std::string);
    /*  See base class.

	'windowManagerCaption' must contain the title to give to the window.
	'useSound' determines if sound effects are produced.
	'pedantic' determines if a special sound effect is played when
	a bullet shot by the player does not hit anything.
	'fullScreen' attempts to display the fame in full screen mode
	instead of using an ordinary window.
	'onlyBlueRobots' sets a practice mode where only blue robots
	are sent out (they are smarter than gray robots, but only
	require one bullet to be killed).

	Throws a string or an integer code if an error occurs.
    */

    virtual ~AfternoonStalkerEngine();
    /*  Nothing interesting.
    */

    const AfternoonStalkerEngine &getInstance() const;
    AfternoonStalkerEngine &getInstance();

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

private:

    ///////////////////////////////////////////////////////////////////////////
    //
    //  LOCAL TYPES, CLASSES AND CONSTANTS
    //
    //


    enum TileNo
    {
	// The first value must be zero
	WALL_TILE,

	// The CORNER* and WALLEND* names must be contiguous
	CORNER0_TILE,
	CORNER1_TILE,
	CORNER2_TILE,
	CORNER3_TILE,
	WALLEND0_TILE,
	WALLEND1_TILE,
	WALLEND2_TILE,
	WALLEND3_TILE,

	FLOOR_TILE,
	COBWEB_TILE,

	BUNKER_TILE,

	// The FLOOR[0-3]* names must be contiguous
	BUNKER0_TILE,
	BUNKER1_TILE,
	BUNKER2_TILE,
	BUNKER3_TILE,

	BUNKER_DOOR_TILE
    };

    typedef std::vector<TileNo> TileMatrixRow;
    typedef std::vector<TileMatrixRow> TileMatrix;


    enum Privilege
    // Used in positionIsAllowed()
    {
	PLAYER_PRIV,
	ENEMY_PRIV,
	BULLET_PRIV,
	PIERCING_BULLET_PRIV
    };


    class Bomb
    {
    public:
	Couple pos;
	int ticksBeforeExplosion;
	int explosionRadius;

	Bomb()
	  : pos(), ticksBeforeExplosion(0), explosionRadius(0) {}

	void init(Couple pos, int ticksBeforeExplosion, int explosionRadius)
	{
	    this->pos = pos;
	    this->ticksBeforeExplosion = ticksBeforeExplosion;
	    this->explosionRadius = explosionRadius;
	}

	bool tickDown()
	{
	    if (ticksBeforeExplosion == 0)
		return false;
	    ticksBeforeExplosion--;
	    return (ticksBeforeExplosion == 0);
	}

	void reset()
	{
	    ticksBeforeExplosion = 0;
	}

	bool isTicking() const
	{
	    return (ticksBeforeExplosion > 0);
	}

	Couple getPos() const
	{
	    return pos;
	}

	int getExplosionRadius() const
	{
	    return explosionRadius;
	}
    };


    ///////////////////////////////////////////////////////////////////////////
    //
    //  DATA MEMBERS
    //
    //


    bool paused;
    unsigned long tickCount;
    bool useSound;


    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;


    /*  SETTING:
    */
    PixmapArray tilePA;
    TileMatrix  tileMatrix;


    /*  SCORE BOARD:
    */
    long theScore;
    Couple scorePos;

    Couple playerLivesPos;


    /*  PLAYER:
    */
    PixmapArray playerPA;
    Sprite *playerSprite;
    size_t numPlayerLives;
    Couple initPlayerPos;
    size_t numPlayerBullets;
    int lastPlayerDir;
    int lastRequestedDir;
    long playerParalysisTime;
    bool playerHasBomb;

    PixmapArray playerBulletPA;
    SpriteList playerBulletSprites;

    PixmapArray playerExplosionPA;
    SpriteList playerExplosionSprites;
	    // when this list is not empty, the player is dying

    PixmapArray gunPA;
    SpriteList gunSprites;

    PixmapArray bombPA;
    SpriteList bombSprites;
    long timeUntilNextBomb;

    Bomb theBomb;
    int bombExplosionTicks;
    PixmapArray bombExplosionPA;
    SpriteList bombExplosionSprites;


    /*  ENEMIES:

	CONVENTION: all robot sprites are the same size as the human.
    */
    PixmapArray grayRobotPA;
    PixmapArray blueRobotPA;
    PixmapArray whiteRobotPA;
    PixmapArray blackRobotPA;
    PixmapArray invisibleRobotPA;
    PixmapArray blinkPA;
    SpriteList robotSprites;
    long timeUntilNextRobot;
    bool sendOnlyBlueRobots;
    Couple initRobotPos;
    size_t numCreatedRobots;
    size_t maxNumRobots;

    PixmapArray robotBulletPA;
    PixmapArray bigRobotBulletPA;
    SpriteList robotBulletSprites;

    PixmapArray grayRobotExplosionPA;
    PixmapArray blueRobotExplosionPA;
    PixmapArray whiteRobotExplosionPA;
    PixmapArray blackRobotExplosionPA;
    PixmapArray invisibleRobotExplosionPA;
    SpriteList robotExplosionSprites;

    PixmapArray spiderPA;
    PixmapArray batPA;
    SpriteList spiderSprites;
    SpriteList batSprites;
    Couple initSpiderPos;
    long timeUntilNextAnimal;

    PixmapArray spiderExplosionPA;
    SpriteList spiderExplosionSprites;
    PixmapArray batExplosionPA;
    SpriteList batExplosionSprites;

    /*  DIGITS:
    */
    PixmapArray digitPA;
    SpriteList scoreSprites;


    /*  KEYBOARD COMMANDS:
    */
    KeyState quitKS;
    KeyState leftKS;
    KeyState rightKS;
    KeyState upKS;
    KeyState downKS;
    KeyState startKS;
    KeyState shootKS;
    KeyState pauseKS;
    KeyState bombKS;
    KeyState shootLeftKS;
    KeyState shootRightKS;
    KeyState shootUpKS;
    KeyState shootDownKS;


    /*  SOUND EFFECTS:
    */
    SoundMixer *theSoundMixer;  // see method playSoundEffect()
    SoundMixer::Chunk gunPickupSound;
    SoundMixer::Chunk playerBulletSound;
    SoundMixer::Chunk shootingBlanksSound;
    SoundMixer::Chunk playerHitSound;
    SoundMixer::Chunk robotBulletSound;
    SoundMixer::Chunk robotHitSound;
    SoundMixer::Chunk batKilledSound;
    SoundMixer::Chunk spiderKilledSound;
    SoundMixer::Chunk newLifeSound;
    SoundMixer::Chunk bombExplodesSound;
    SoundMixer::Chunk missedSound;


    ///////////////////////////////////////////////////////////////////////////
    //
    //  IMPLEMENTATION FUNCTIONS
    //
    //
    bool animatePlayer();
    int getRobotScore(const Sprite &robot) const;
    void makePlayerShoot(int shotDir);
    void layPlayerBomb();
    Couple attemptMove(const Sprite &s,
			const bool attemptedDirections[4],
			int speedFactor) const;
    Couple getSpeedFromAttemptedAndAllowedDirections(
			const bool attemptedDirections[4],
			const bool allowedDirections[4],
			int speedFactor,
			Couple delta) const;
    Couple determineAllowedDirections(const Sprite &s,
					bool forPlayer,
					int speedFactor, int tolerance,
					bool allowedDirections[4]) const;
    Couple getDistanceToPerfectPos(const Sprite &s) const;
    bool positionIsAllowed(int direction, Privilege priv,
				Couple pos, Couple size) const;
    std::pair<TileNo, TileNo> getSideTilesAtPos(
				int direction, Couple pos, Couple size) const;
    TileNo getTileAtPos(Couple pos) const;
    void setTileAtPos(Couple pos, TileNo t);

    void animateTemporarySprites(SpriteList &slist) const;
    void animateAutomaticCharacters();
    void killSpritesNearPosition(
			SpriteList &slist, Couple pos, int minDistance);
    void moveExplosionSprites(SpriteList &slist);
    bool isOutOfSetting(const Sprite &s) const;
    size_t getNumToughRobots() const;
    void moveEnemyList(SpriteList &slist, int speedFactor);
    void moveBullets(SpriteList &slist, bool playMissedSound = false);
    int chooseDirectionTowardTarget(Couple startPos,
                                    Couple targetPos,
                                    int speedFactor,
                                    const bool allowedDirections[4]) const;
    Couple chooseRandomAllowedTile();
    void makeRobotsShoot();

    void detectCollisions();
    void makePlayerDie();
    void paralyzePlayer();
    void createRobotExplosion(const RobotSprite &r);
    void createAnimalExplosion(Couple pos,
    				const PixmapArray &pa, SpriteList &slist);
    void killEnemy(const Sprite &s);
    void createPlayerExplosion(const Sprite &s);
    void addToScore(long n);
    void createScoreSprites(long n, Couple center);
    void loadPixmaps() throw(PixmapLoadError);
    void restoreBackground();
    void restoreForeground();
    void putSprite(const Sprite &s);
    void putSpriteList(const SpriteList &slist);
    void drawSprites();
    void drawScoreBoard();
    void displayPauseMessage();
    void displayStartMessage();
    void displayMessage(int row, const char *msg);
    void addToNumPlayerLives(int n);

    void displayErrorMessage(const std::string &msg) throw();
    void initializeLevel();
    void initializePlayerLife();
    void setTimeUntilNextBomb();
    void setTimeUntilNextRobot();
    void setTimeUntilNextAnimal();
    void doOneTimeInitializations(
			bool useSound,
			bool pedantic,
			const std::string &setFilename)
				throw(std::string);
    void loadLevel(int levelNo,
    			const std::string &setFilename) throw(std::string);

    void playSoundEffect(SoundMixer::Chunk &wb);


    /*	Forbidden operations:
    */
    AfternoonStalkerEngine(const AfternoonStalkerEngine &);
    AfternoonStalkerEngine &operator = (const AfternoonStalkerEngine &);
};


#endif  /* _H_AfternoonStalkerEngine */
