/*  $Id: AfternoonStalkerEngine.h,v 1.9 2002/01/19 03:12:50 sarrazip Exp $
AfternoonStalkerEngine.h - A robot-killing video game engine.

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_AfternoonStalkerEngine
#define _H_AfternoonStalkerEngine

#define PAUL 12

#define __STDC_LIMIT_MACROS  /* to get SIZE_MAX from <stdint.h> */

#include <gengameng/GameEngine.h>
#include <gengameng/Sprite.h>

#include "SoundPlayer.h"

class RobotSprite;

#include <stdint.h>

#include <string>

using namespace std;


class AfternoonStalkerEngine : public GameEngine
/*  A robot-killing video game.
*/
{
public:

    typedef PixmapArray::Pixmap Pixmap;


    enum { RIGHT, UP, LEFT, DOWN };
    /*	Possible directions of movement, in trigonometric order.
	This enumeration is garanteed to start at zero.
	The order of the constants is guaranteed, so they are 0, 1, 2, 3.
    */

    AfternoonStalkerEngine(const string &windowManagerCaption,
			    bool useSound, bool fullScreen)
						throw(int, string);
    /*  See base class.

	'windowManagerCaption' must contain the title to give to the window.

	Throws a string or an integer code if an error occurs.
    */

    virtual ~AfternoonStalkerEngine();
    /*  Nothing interesting.
    */

    const AfternoonStalkerEngine &getInstance() const;
    AfternoonStalkerEngine &getInstance();

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

private:

    ///////////////////////////////////////////////////////////////////////////
    //
    //  LOCAL TYPES, CLASSES AND CONSTANTS
    //
    //


    enum TileNo
    {
	// The first value must be zero
	WALL_TILE,

	// The CORNER* and WALLEND* names must be contiguous
	CORNER0_TILE,
	CORNER1_TILE,
	CORNER2_TILE,
	CORNER3_TILE,
	WALLEND0_TILE,
	WALLEND1_TILE,
	WALLEND2_TILE,
	WALLEND3_TILE,

	FLOOR_TILE,
	COBWEB_TILE,

	BUNKER_TILE,

	// The FLOOR[0-3]* names must be contiguous
	BUNKER0_TILE,
	BUNKER1_TILE,
	BUNKER2_TILE,
	BUNKER3_TILE,

	BUNKER_DOOR_TILE
    };

    typedef vector<TileNo> TileMatrixRow;
    typedef vector<TileMatrixRow> TileMatrix;


    enum Privilege
    // Used in positionIsAllowed()
    {
	PLAYER_PRIV,
	ENEMY_PRIV,
	BULLET_PRIV,
	PIERCING_BULLET_PRIV
    };



    ///////////////////////////////////////////////////////////////////////////
    //
    //  DATA MEMBERS
    //
    //


    bool paused;
    unsigned long tickCount;


    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;


    /*  SETTING:
    */
    PixmapArray tilePA;
    TileMatrix  tileMatrix;


    /*  SCORE BOARD:
    */
    long theScore;
    Couple scorePos;

    Couple playerLivesPos;


    /*  PLAYER:
    */
    PixmapArray playerPA;
    Sprite *playerSprite;
    size_t numPlayerLives;
    Couple initPlayerPos;
    size_t numPlayerBullets;
    int lastPlayerDir;
    int lastRequestedDir;
    long playerParalysisTime;

    PixmapArray playerBulletPA;
    SpriteList playerBulletSprites;

    PixmapArray playerExplosionPA;
    SpriteList playerExplosionSprites;
	    // when this list is not empty, the player is dying

    PixmapArray gunPA;
    SpriteList gunSprites;


    /*  ENEMIES:

	CONVENTION: all robot sprites are the same size as the human.
    */
    PixmapArray grayRobotPA;
    PixmapArray blueRobotPA;
    PixmapArray whiteRobotPA;
    PixmapArray blackRobotPA;
    PixmapArray invisibleRobotPA;
    PixmapArray blinkPA;
    SpriteList robotSprites;
    long timeUntilNextRobot;
    Couple initRobotPos;
    size_t numCreatedRobots;
    size_t maxNumRobots;

    PixmapArray robotBulletPA;
    PixmapArray bigRobotBulletPA;
    SpriteList robotBulletSprites;

    PixmapArray grayRobotExplosionPA;
    PixmapArray blueRobotExplosionPA;
    PixmapArray whiteRobotExplosionPA;
    PixmapArray blackRobotExplosionPA;
    PixmapArray invisibleRobotExplosionPA;
    SpriteList robotExplosionSprites;

    PixmapArray spiderPA;
    PixmapArray batPA;
    SpriteList spiderSprites;
    SpriteList batSprites;
    Couple initSpiderPos;
    long timeUntilNextAnimal;

    PixmapArray spiderExplosionPA;
    SpriteList spiderExplosionSprites;
    PixmapArray batExplosionPA;
    SpriteList batExplosionSprites;

    /*  DIGITS:
    */
    PixmapArray digitPA;
    SpriteList scoreSprites;


    /*  KEYBOARD COMMANDS:
    */
    bool escapePressed;
    bool leftArrowPressed;
    bool rightArrowPressed;
    bool upArrowPressed;
    bool downArrowPressed;
    bool spacePressed, spacePressedBefore;
    bool ctrlPressed, ctrlPressedBefore;
    bool letterPPressed, letterPPressedBefore;


    /*  SOUND EFFECTS:
    */
    SoundPlayer *theSoundPlayer;  // see method playSoundEffect()
    SoundPlayer::WavBuffer gunPickupSound;
    SoundPlayer::WavBuffer playerBulletSound;
    SoundPlayer::WavBuffer shootingBlanksSound;
    SoundPlayer::WavBuffer playerHitSound;
    SoundPlayer::WavBuffer robotBulletSound;
    SoundPlayer::WavBuffer robotHitSound;
    SoundPlayer::WavBuffer batKilledSound;
    SoundPlayer::WavBuffer spiderKilledSound;
    SoundPlayer::WavBuffer newLifeSound;


    ///////////////////////////////////////////////////////////////////////////
    //
    //  IMPLEMENTATION FUNCTIONS
    //
    //
    bool animatePlayer();
    void makePlayerShoot();
    Couple attemptMove(const Sprite &s,
			const bool attemptedDirections[4],
			int speedFactor) const;
    Couple getSpeedFromAttemptedAndAllowedDirections(
			const bool attemptedDirections[4],
			const bool allowedDirections[4],
			int speedFactor,
			Couple delta) const;
    Couple determineAllowedDirections(const Sprite &s,
					bool forPlayer,
					int speedFactor, int tolerance,
					bool allowedDirections[4]) const;
    Couple getDistanceToPerfectPos(const Sprite &s) const;
    bool positionIsAllowed(int direction, Privilege priv,
				Couple pos, Couple size) const;
    pair<TileNo, TileNo> getSideTilesAtPos(
				int direction, Couple pos, Couple size) const;
    TileNo getTileAtPos(Couple pos) const;
    void setTileAtPos(Couple pos, TileNo t);

    void animateTemporarySprites(SpriteList &slist) const;
    void animateAutomaticCharacters();
    void moveExplosionSprites(SpriteList &slist);
    bool isOutOfSetting(const Sprite &s) const;
    size_t getNumToughRobots() const;
    void moveEnemyList(SpriteList &slist, int speedFactor);
    void moveBullets(SpriteList &slist);
    int chooseDirectionTowardTarget(Couple startPos,
                                    Couple targetPos,
                                    int speedFactor,
                                    const bool allowedDirections[4]) const;
    Couple chooseRandomAllowedTile();
    void makeRobotsShoot();

    void detectCollisions();
    void makePlayerDie();
    void paralyzePlayer();
    void createRobotExplosion(const RobotSprite &r);
    void createAnimalExplosion(Couple pos,
    				const PixmapArray &pa, SpriteList &slist);
    void createPlayerExplosion(const Sprite &s);
    void addToScore(long n);
    void createScoreSprites(long n, Couple center);
    void loadPixmaps() throw(PixmapLoadError);
    void restoreBackground();
    void restoreForeground();
    void putSprite(const Sprite &s);
    void putSpriteList(const SpriteList &slist);
    void drawSprites();
    void drawScoreBoard();
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);
    void displayMessage(const char *msg);
    void addToNumPlayerLives(int n);

    void displayErrorMessage(const string &msg) throw();
    void initializeLevel();
    void initializePlayerLife();
    void setTimeUntilNextRobot();
    void setTimeUntilNextAnimal();
    void doOneTimeInitializations(bool useSound) throw(string);
    void loadLevel(int levelNo, const string &setFilename) throw(string);

    void playSoundEffect(SoundPlayer::WavBuffer &wb);


    /*	Forbidden operations:
    */
    AfternoonStalkerEngine(const AfternoonStalkerEngine &);
    AfternoonStalkerEngine &operator = (const AfternoonStalkerEngine &);
};


#endif  /* _H_AfternoonStalkerEngine */
