/* aewm - a minimalistic X11 window manager. ------- vim:sw=4:et
 * Copyright (c) 1998-2001 Decklin Foster <decklin@red-bean.com>
 * Free software! Please see README for details and license.  */

#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include "lib/panel-misc.h"
#include "lib/switch-misc.h"
#include "lib/palette-misc.h"

#define SPACING 4
#define NAME_SIZE 50

void make_cmd_button(char *, char *);
void make_client_button(Window);
void update_client_button(Window);
void remove_client_button(Window);
void watch_window(Window, long);
GdkFilterReturn check_event(GdkXEvent *, GdkEvent *, gpointer);
void show_menu_cb(GtkWidget *, gpointer);
void raise_win_cb(GtkWidget *, Window);
void fork_exec_cb(GtkWidget *, char *);

GtkWidget *menu, *clients_box;
extern gint gdk_error_warnings;

int main(int argc, char **argv)
{
    int width, height;
    GtkWidget *toplevel, *bbox, *button;
    struct sigaction act;

    act.sa_handler = sig_handler;
    act.sa_flags = 0;
    sigaction(SIGCHLD, &act, NULL);

    gtk_init(&argc, &argv);
    gdk_error_warnings = 0; /* gag me with a spoon... */
    dpy = GDK_DISPLAY();
    root = GDK_ROOT_WINDOW();
    client_tab = XUniqueContext();
    wm_state = XInternAtom(dpy, "WM_STATE", False);
    watch_window(root, SubstructureNotifyMask);

    gdk_window_get_size(GDK_ROOT_PARENT(), &width, &height);

    toplevel = gtk_window_new(GTK_WINDOW_POPUP);
    gtk_widget_set_usize(toplevel, width, 0);
    gtk_container_set_border_width(GTK_CONTAINER(toplevel), SPACING);

    menu = gtk_menu_new();

    bbox = gtk_hbox_new(FALSE, SPACING);
    gtk_container_add(GTK_CONTAINER(toplevel), bbox);

    button = gtk_button_new_with_label("Menu");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(show_menu_cb), menu);
    gtk_box_pack_start(GTK_BOX(bbox), button, FALSE, FALSE, 0);

    button = gtk_button_new_with_label("Quit");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
    gtk_box_pack_start(GTK_BOX(bbox), button, FALSE, FALSE, 0);

    clients_box = gtk_hbox_new(TRUE, SPACING);
    gtk_container_add(GTK_CONTAINER(bbox), clients_box);
    gtk_container_set_resize_mode(GTK_CONTAINER(clients_box), GTK_RESIZE_QUEUE);

    add_cmd_buttons(make_cmd_button);
    update_clients(root, make_client_button, update_client_button);

    gtk_widget_show_all(toplevel);

    gtk_main();
    return 0;
}

void make_cmd_button(char *label, char *cmd)
{
    GtkWidget *menuitem = gtk_menu_item_new_with_label(label);
    gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
        GTK_SIGNAL_FUNC(fork_exec_cb), cmd);
    gtk_menu_append(GTK_MENU(menu), menuitem);
    gtk_widget_show(menuitem);
}

void make_client_button(Window w)
{
    GtkWidget *button;
    char buf[NAME_SIZE];

    get_wm_name(w, buf, sizeof buf);

    button = gtk_button_new_with_label(buf);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(raise_win_cb), (gpointer) w);
    gtk_box_pack_start(GTK_BOX(clients_box), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    XSaveContext(dpy, w, client_tab, (XPointer)button);
    watch_window(w, StructureNotifyMask|PropertyChangeMask);
}

void update_client_button(Window w)
{
    GtkWidget *button;
    char buf[NAME_SIZE];

    if (get_wm_state(w) == WithdrawnState) {
        remove_client_button(w);
    } else if (XFindContext(dpy, w, client_tab, (XPointer*)&button) == Success) {
        get_wm_name(w, buf, sizeof buf);
        gtk_label_set_text(GTK_LABEL(GTK_BIN(button)->child), buf);
    }
}

void remove_client_button(Window w)
{
    GtkWidget *button;

    if (XFindContext(dpy, w, client_tab, (XPointer*)&button) == Success) {
        gtk_container_remove(GTK_CONTAINER(clients_box), button);
        XDeleteContext(dpy, w, client_tab);
    }
}

void watch_window(Window w, long mask)
{
    GdkWindow *gdkwin = gdk_window_lookup(w);

    XSelectInput(dpy, w, mask);
    gdk_window_add_filter(gdkwin, check_event, NULL);
}

GdkFilterReturn check_event(GdkXEvent *gdk_xevent, GdkEvent *event,
    gpointer dummy)
{
    XEvent *e = (XEvent *)gdk_xevent;

    switch (e->type) {
        case MapNotify:
            //printf("map %#lx\n", e->xmap.window);
            update_clients(e->xmap.window, make_client_button, update_client_button);
            break;
        case PropertyNotify:
            //printf("prop %#lx\n", e->xproperty.window);
            update_clients(e->xproperty.window, make_client_button, update_client_button);
            break;
        case DestroyNotify:
            remove_client_button(e->xdestroywindow.window);
            break;
    }

    return GDK_FILTER_CONTINUE;
}

void show_menu_cb(GtkWidget *widget, gpointer menu)
{
    gtk_menu_popup(menu, NULL, NULL, NULL, NULL, 0, 0);
}

void raise_win_cb(GtkWidget *widget, Window w)
{
    raise_win(w);
}

void fork_exec_cb(GtkWidget *widget, char *data)
{
    fork_exec(data);
}
