
/*
 *
 *  conff.c by JH <jheinonen@bigfoot.com>
 *
 *  Copyright (C) Jaakko Heinonen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * conff.c version 0.2.11
 */

#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include "conff.h"

#ifdef _AIX
int strcasecmp (const char *, const char *);
int strncasecmp (const char *, const char *, size_t);
#endif

#define COMMENT_CHAR	'#'

#define MAX_LINESIZE	1024 /* this is used only while loading the file */

conffile
conff_new()
{
	conffile tmp;

	tmp.data = NULL;
	tmp.items = 0;

	return tmp;
}

void
conff_free(conffile *file)
{
	int i;

	for(i = 0; i < file -> items; i++) {
		free(file -> data[i].key);
		free(file -> data[i].value);
	}

	free(file -> data);
}

void
conff_free_sections(conffile *file)
{
	int i;

	for(i = 0; i < file -> items; i++) {
		if( file -> data[i].key == NULL )
			free(file -> data[i].value);
	}
}

void
conff_free2(conffile *file)
{
	conff_free_sections(file);
	free(file -> data);
}

int
conff_add_section(conffile *file, char *section)
{
	if( conff_find_section(*file, section) >= 0 )
		return -1;
	
	file -> data = realloc(file -> data,
			sizeof(confline) * ++(file -> items) );

	file -> data[file -> items - 1].key = NULL;

	file -> data[file -> items - 1].value = strdup(section);

	return 0;
}

int
conff_find_section(conffile file, char *section)
{
	int i;

	if(file.items < 1)
		return -2;

	for(i = 0; i < file.items; i++)
		if( file.data[i].key == NULL )
			if( !strcasecmp( file.data[i].value, section ) )
				return i;

	return -1;
}

int
conff_find_section_end(conffile file, char *section)
{
	int start, i;

	if( ( start = conff_find_section(file, section) ) < 0 )
		return --start;

	for( i = start + 1; i < file.items; i++ )
		if( file.data[i].key == NULL )
			break;
	
	return i - 1;
}

int
conff_add_key(conffile *file, char *section, char *key, char *value)
{
	int last, i, place;
	char *dummy;
	
	last = conff_find_section_end(*file, section);
	if(last < 0 )
		return -1;

	if( (place = conff_get_value(*file, section, key, &dummy)) < 0 ) {
		file -> data = realloc(file -> data,
			sizeof(confline) * ++(file -> items) );

		for( i = file -> items - 1; i > last; i-- )
			file -> data[i] = file -> data[i - 1];
		place = last + 1;
	} else {
		free(file -> data[place].key);
		free(file -> data[place].value);
		free(dummy);
	}
	file -> data[place].key = strdup(key);
	file -> data[place].value = strdup(value);
	return 0;
}


int
conff_add_key2(conffile *file, char *section, char *key, char *value)
{
	int last, i, place;
	
	last = conff_find_section_end(*file, section);
	if(last < 0 )
		return -1;

	file -> data = realloc(file -> data,
		sizeof(confline) * ++(file -> items) );

	for( i = file -> items - 1; i > last; i-- )
		file -> data[i] = file -> data[i - 1];
	place = last + 1;

	file -> data[place].key = key;
	file -> data[place].value = value;
	return 0;
}

int
conff_add_key_noreplace(conffile *file, char *section, char *key, char *value)
{
	char *dummy;

	if( conff_get_value(*file, section, key, &dummy) < 0 ) {
		return conff_add_key(file, section, key, value);
	} else {
		free(dummy);
		return -2;
	}
}

int
conff_remove_section(conffile *file, char *section)
{
	int first, last, i;

	first = conff_find_section(*file, section);
	last = conff_find_section_end(*file, section);

	if(first < 0 || last < 0)
		return -1;
	
	for(i = first; i < last + 1; i++) {
		free(file -> data[i].key);
		free(file -> data[i].value);
		file -> data[i].value = NULL;
		file -> data[i].key = NULL;
	}
		
	conff_remove_items(file);

	return 0;
}

void
conff_remove_items(conffile *file)
{
	int i, j;

	for( j = 0; j < file -> items; j++ ) {
		if( file->data[j].key == NULL && file->data[j].value == NULL ) {
			for( i = j; i < file -> items -1; i++)
				file -> data[i] = file -> data[i+1];
			j--;
			file -> items --;
		}
	}

        file -> data = realloc(file -> data,
			sizeof(confline) * (file -> items) );
}

int
conff_get_value(conffile file, char *section, char *key, char **value)
{
	int first, last, i;

	first = conff_find_section(file, section);
	last = conff_find_section_end(file, section);

	for( i = first ; i < last + 1 ; i++ ) {
		if( file.data[i].key != NULL && file.data[i].key[0]
				!= COMMENT_CHAR )
			if( !strcasecmp(file.data[i].key, key) ) {
				*value = strdup(file.data[i].value);
				return i;
			}
	}

	return -1;
}

int
conff_remove_key(conffile *file, char *section, char *key)
{
	char *dummy;
	int item;

	
	if( (item = conff_get_value(*file, section, key, &dummy)) < 0 )
		return -1;
	else {
		free(dummy);
		free(file -> data[item].key);
		free(file -> data[item].value);
		file -> data[item].key = NULL;
		file -> data[item].value = NULL;
	}
	conff_remove_items(file);
	return 0;
}


int
conff_save_file(conffile file, char *filename)
{
	FILE *out;
	int i;

	if (!(out = fopen(filename, "w")))
		return -1;

	for(i = 0; i < file.items; i++) {
		if(file.data[i].key == NULL)
			fprintf(out, "\n[%s]\n", file.data[i].value);
		else
			if(file.data[i].key[0] == COMMENT_CHAR ) {
				fprintf(out, "%s\n", file.data[i].key);
			        continue;
			} else
				fprintf(out, "%s=%s\n", file.data[i].key,
						file.data[i].value);
	}

	fprintf(out,"\n");
	fclose(out);

	return 0;
}


/*
 * this is not really useful function
 *   !!! use with caution and remember the '#' aka COMMENT_CHAR !!!
 */
int
conff_add_comment(conffile *file, char *comment)
{

	file -> data = realloc(file -> data,
			sizeof(confline) * ++(file -> items) );

	file -> data[file -> items - 1].key = strdup(comment);
	file -> data[file -> items - 1].value = NULL; 

	return 0;
}

int
conff_add_comment2(conffile *file, char *comment)
{

        file -> data = realloc(file -> data,
	sizeof(confline) * ++(file -> items) );

	file -> data[file -> items - 1].key = comment;
	file -> data[file -> items - 1].value = NULL;

	return 0;
}

/*
 * todo:
 * 	- space removement
 */
void inline
trim_line(char *str)
{
	char *tmp;

	if( (tmp = strchr(str, '\n')) )
		*tmp = 0;
}

int
conff_load_config(conffile *file, char *filename, const int flags)
{
	FILE *in;
	char line[MAX_LINESIZE];
	char *tmp;
	int i = 0;
	char *sec = NULL;

	if (!(in = fopen(filename, "r")))
		return -1;

	for(;;) {
		i++;
		fgets(line, MAX_LINESIZE, in);
		if( feof(in) )
			break;

		trim_line(line);

		if(line[0] == 0 || line[0] == '\n' )
			continue;
		else
		if(line[0] == COMMENT_CHAR) {
			if(flags & CONFF_LOAD_COMMENTS)
				conff_add_comment(file, line);
		} else
		if(line[0] == '[' ) {
			if ((tmp = strchr(line, ']'))) {
				*tmp = '\0';
				tmp = line + 1;
				conff_add_section(file, tmp);
				free(sec);
				sec = strdup(tmp);
			} else {
/*				fprintf(stderr, "parse error,line #%d\n",i);*/
				fclose(in);
				return i;
			}
		} else
		if ( (tmp = strchr(line, '=') ) && sec != NULL ) {
			*tmp++ = '\0';
			if(flags & CONFF_LOAD_NOREPLACE)
				conff_add_key_noreplace(file, sec , line, tmp);
			else
				conff_add_key(file, sec , line, tmp);
		}
		else {
/*			fprintf(stderr, "parse error2,line #%d\n",i);*/
			fclose(in);
			return i;
		}
	}

	free(sec);
	fclose(in);
	return 0;
}

