
/*
 * filter.c for abook
 * by JH <jheinonen@bigfoot.com>
 *
 * Copyright (C) Jaakko Heinonen
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ncurses.h>
#include <pwd.h>
#include <sys/types.h>
#include "filter.h"
#include "abook.h"
#include "database.h"
#include "edit.h"
#include "list.h"
#include "misc.h"

extern int items;
extern int curitem;
extern int first_list_item;
extern list_item *database;
extern struct abook_field abook_fields[];

/*
 * function declarations
 */

static int      ldif_parse_file(FILE *handle);
static int	pine_parse_file(FILE *in);

static int	html_export_database(FILE *out);
static int	pine_export_database(FILE *out);
static int	csv_export_database(FILE *out);
static int	gcrd_export_database(FILE *out);
static int	mutt_alias_export(FILE *out);
static int	elm_alias_export(FILE *out);
static int	text_export_database(FILE *out);

struct abook_filter i_filters[] = {
	{ "abook", "abook native format", parse_database },
	{ "ldif", "ldif / Netscape addressbook", ldif_parse_file },
	{ "pine", "pine addressbook", pine_parse_file },
	{ "\0", NULL, NULL }
};

struct abook_filter e_filters[] = {
	{ "abook", "abook native format", write_database },
	{ "mutt", "mutt alias", mutt_alias_export },
	{ "html", "html document", html_export_database },
	{ "pine", "pine addressbook", pine_export_database },
	{ "gcrd", "GnomeCard (VCard) addressbook", gcrd_export_database },
	{ "csv", "comma separated values", csv_export_database },
	{ "elm", "elm alias", elm_alias_export },
	{ "text", "plain text", text_export_database },
	{ "\0", NULL, NULL }
};

/*
 * common functions
 */

void
print_filters()
{
	int i;
	
	puts("input:");
	for(i=0; *i_filters[i].filtname ; i++)
		printf("\t%s\t%s\n", i_filters[i].filtname,
			i_filters[i].desc);

	putchar('\n');
	
	puts("output:");
	for(i=0; *e_filters[i].filtname ; i++)
		printf("\t%s\t%s\n", e_filters[i].filtname,
			e_filters[i].desc);

	putchar('\n');
}

static int
number_of_filters(struct abook_filter * filters)
{
	int i;

	for(i=0; *filters[i].filtname ; i++);

	return i;
}

static char *
get_real_name(char *username)
{
	struct passwd *pwent;
	int rtn;
	char *tmp;

	pwent = getpwnam(username);

	tmp = malloc(strlen(pwent->pw_gecos) +1);
	rtn = sscanf(pwent->pw_gecos, "%[^,]", tmp);
	if (rtn == EOF || rtn == 0) {
		free(tmp);
		return strdup(username);
	} else
		return tmp;
}

/*
 * import
 */
	
static int		i_read_file(char *filename, int (*func) (FILE *in));

static void
import_screen()
{
	int i;
	
	clear();

	refresh_statusline();
	headerline("import database");

	mvaddstr(3, 1, "please select a filter");
	

	for(i=0; *i_filters[i].filtname ; i++)
		mvprintw(5 + i, 6, "%c -\t%s\t%s\n", 'a' + i,
			i_filters[i].filtname,
			i_filters[i].desc);

	mvprintw(6 + i, 6, "x -\tcancel");
}

int
import_database()
{
	int filter;
	char *filename;
	int tmp = items;

	import_screen();
	
	filter = getch() - 'a';
	if(filter == 'x' - 'a' || filter >= number_of_filters(e_filters) ||
			filter < 0) {
		refresh_screen();
		return 1;
	}
	
	mvaddstr(5+filter, 2, "->");
	
	filename = ask_filename("Filename: ", 1);
	if( !filename ) {
		refresh_screen();
		return 2;
	}
		
	if(  i_read_file(filename, i_filters[filter].func ) )
		statusline_msg("Error occured while opening the file");
	else
	if( tmp == items )
		statusline_msg("Hmm.., file seems not to be a valid file");
	
	check_list();
	refresh_screen();
	free(filename);

	return 0;
}



static int
i_read_file(char *filename, int (*func) (FILE *in))
{
	FILE *in;
	int ret = 0;

	if( ( in = fopen( filename, "r" ) ) == NULL )
		return 1;

	ret = (*func) (in);

	fclose(in);

	return ret;	
}

int
import(char filtname[FILTNAME_LEN], char *filename)
{
	int i;
	int tmp = items;
	int ret = 0;

	for(i=0;; i++) {
		if( ! strncmp(i_filters[i].filtname, filtname, FILTNAME_LEN) )
			break;
		if( ! *i_filters[i].filtname ) {
			i = -1;
			break;
		}
	}

	if( i<0 )
		return -1;

	if( !strcmp(filename, "-") )
		ret = (*i_filters[i].func) (stdin);
	else
		ret =  i_read_file(filename, i_filters[i].func);
	
	if( tmp == items )
		ret = 1;
	
	check_list();

	return ret;
}

/*
 * export
 */

static int		e_write_file(char *filename, int (*func) (FILE *in));

static void
export_screen()
{
	int i;
	
	clear();


	refresh_statusline();
	headerline("export database");

	mvaddstr(3, 1, "please select a filter");
	

	for(i=0; *e_filters[i].filtname ; i++)
		mvprintw(5 + i, 6, "%c -\t%s\t%s\n", 'a' + i,
			e_filters[i].filtname,
			e_filters[i].desc);

	mvprintw(6 + i, 6, "x -\tcancel");
}

int
export_database()
{
	int filter;
	char *filename;

	export_screen();
	
	filter = getch() - 'a';
	if(filter == 'x' - 'a' || filter >= number_of_filters(e_filters) ||
			filter < 0) {
		refresh_screen();
		return 1;
	}
	
	mvaddstr(5+filter, 2, "->");
	
	filename = ask_filename("Filename: ", 0);
	if( !filename ) {
		refresh_screen();
		return 2;
	}
	
	if(  e_write_file(filename, e_filters[filter].func ) )
		statusline_msg("Error occured while exporting");
	
	refresh_screen();
	free(filename);

	return 0;
}

static int
e_write_file(char *filename, int (*func) (FILE *in))
{
	FILE *out;
	int ret = 0;

	if( (out = fopen(filename, "a")) == NULL )
		return 1;

	if( ftell(out) )
		return 1;

	ret = (*func) (out);
	
	fclose(out);
	
	return ret;
}

int
fexport(char filtname[FILTNAME_LEN], FILE *handle)
{
	int i;

	for(i=0;; i++) {
		if( ! strncmp(e_filters[i].filtname, filtname, FILTNAME_LEN) )
			break;
		if( ! *e_filters[i].filtname ) {
			i = -1;
			break;
		}
	}

	return (e_filters[i].func) (handle);
}

	

int
export(char filtname[FILTNAME_LEN], char *filename)
{
	int i;
	int ret = 0;
	
	for(i=0;; i++) {
		if( ! strncmp(e_filters[i].filtname, filtname, FILTNAME_LEN) )
			break;
		if( ! *e_filters[i].filtname ) {
			i = -1;
			break;
		}
	}

	if( i<0 )
		return -1;

	if( !strcmp(filename, "-") )
		ret = (e_filters[i].func) (stdout);
	else
		ret =  e_write_file(filename, e_filters[i].func);

	return ret;
}

/*
 * end of common functions
 */

/*
 * ldif import
 */

#include "ldif.h"

static void	ldif_fix_string(char *str);

#ifndef LINESIZE
#	define LINESIZE 1024
#endif

#define	LDIF_ITEM_FIELDS	14

typedef char*  ldif_item[LDIF_ITEM_FIELDS];

static ldif_item ldif_field_names = {
	"cn",	
	"mail",
	"streetaddress",
        "locality",
	"st",
	"postalcode",
	"countryname",
	"homephone",
	"description",
	"homeurl",
	"facsimiletelephonenumber",
	"cellphone",
	"xmozillaanyphone",
	"objectclass", /* this must be the last entry */
};

static int ldif_conv_table[LDIF_ITEM_FIELDS] = {
	NAME,		/* "cn" */
	EMAIL,		/* "mail" */
	ADDRESS,	/* "streetaddress" */
        CITY,		/* "locality" */
	STATE,		/* "st" */
	ZIP,		/* "postalcode" */
	COUNTRY,	/* "countryname" */
	PHONE,		/* "homephone" */
	NOTES,		/* "description" */
	URL,		/* "homeurl" */
	FAX,		/* "facsimiletelephonenumber" */
	MOBILEPHONE,	/* "cellphone" */
	WORKPHONE,	/* "xmozillaanyphone" */
	-1,             /* "objectclass" */ /* this must be the last entry */
};


static char * 
ldif_read_line(FILE *in)
{
	char line[LINESIZE];
	char *buf=NULL;
	char *ptr, *tmp;
	long pos;
	int i;

	for(i=1;;i++) {
		pos = ftell(in);
		fgets(line, LINESIZE, in);
		
		if( feof(in) )
			break;
		
		if(i == 1) {
			buf = strdup(line);
			continue;
		}
		
		if(*line != ' ') {
			fseek(in, pos, SEEK_SET);
			break;
		}

		ptr = (char *)&line;
		while( *ptr == ' ')
			ptr++;

		tmp = buf;
		buf = strconcat(buf, ptr, NULL);
		free(tmp);
	}

	if( *buf == '#' ) {
		free(buf);
		return NULL;
	}
		
	if(buf) {
		int i,j;
		for(i=0,j=0; j < strlen(buf); i++, j++)
			buf[i] = buf[j] == '\n' ? buf[++j] : buf[j];
	}

	return buf;
}

static void
ldif_add_item(ldif_item ldif_item)
{
	list_item abook_item;
	int i;

	memset(abook_item, 0, sizeof(abook_item));
	
	if( safe_strcmp(ldif_item[LDIF_ITEM_FIELDS -1], "person") &&
			ldif_item[LDIF_ITEM_FIELDS -1] )
		goto bail_out;
	

	for(i=0; i < LDIF_ITEM_FIELDS; i++) {
		if(ldif_conv_table[i] >= 0 && ldif_item[i] && *ldif_item[i] )
			abook_item[ldif_conv_table[i]] = strdup(ldif_item[i]);
	}

	add_item2database(abook_item);

bail_out:
	for(i=0; i < LDIF_ITEM_FIELDS; i++)
		my_free(ldif_item[i]);

}

static void
ldif_convert(ldif_item item, char *type, char *value)
{
	int i;

	if( !strcmp(type, "dn") ) {
		ldif_add_item(item);
		return;
	}

	for(i=0; i < LDIF_ITEM_FIELDS; i++) {
		if( !safe_strcmp(ldif_field_names[i], type) && *value ) {
			if(item[i])
				my_free(item[i]);
			item[i] = strdup(value);
		}
	}
}

static int
ldif_parse_file(FILE *handle)
{
	char *line = NULL;
	char *type, *value;
	int vlen;
	ldif_item item;

	memset(item, 0, sizeof(item));

	do {
		if( ! (line = ldif_read_line(handle)) )
			continue;

		if( -1 == ( str_parse_line(line, &type, &value, &vlen)) ) {
			my_free(line);
			continue; /* just skip the errors */
		}
				
		ldif_fix_string(value);

		ldif_convert(item, type, value);

		my_free(line);
	} while ( !feof(handle) );

	ldif_convert(item, "dn", "");

	return 0;
}

static void
ldif_fix_string(char *str)
{
	int i, j;

	for( i = 0, j = 0; j < strlen(str); i++, j++)
		str[i] = ( str[j] == (char)0xc3 ?
				(char) str[++j] + (char) 0x40 :
				str[j] );

	str[i] = 0;
}

/*
 * end of ldif import
 */

/*
 * html export filter
 */

static void            html_export_write_head(FILE *out);
static void            html_export_write_tail(FILE *out);

static int
html_export_database(FILE *out)
{
	char tmp[MAX_EMAILSTR_LEN];
	int i;

	if( items < 1 )
		return 2;
	
	html_export_write_head(out);
	for( i = 0; i < items; i++ ) {
		get_first_email(tmp, i);
		fprintf(out, "<tr>\n<td><a href=\"mailto:%s\">%s</a>\n",
				tmp,
				database[i][NAME] );
				
		fprintf(out, "<td>%s\n<td>%s\n",
				database[i][EMAIL],
				safe_str(database[i][PHONE]) );
		fprintf(out, "</tr>\n\n");
	}
	html_export_write_tail(out);

	return 0;
}


static void
html_export_write_head(FILE *out)
{
	char *realname;

	realname = get_real_name(getenv("USER"));

	fprintf(out, "<html>\n<head>\n	<title>%s's addressbook</title>",
			realname );
	fprintf(out, "\n</head>\n<body>\n");
	fprintf(out, "\n<h2>%s's addressbook</h2>\n", realname );
	fprintf(out, "<br><br>\n\n");

	fprintf(out, "<center><table border>\n");
	fprintf(out, "\n<tr><th>Name<th>E-mail address(es)<th>Phone</tr>\n\n");

	free(realname);
}

static void
html_export_write_tail(FILE *out)
{
	fprintf(out, "\n</table></center>\n");
	fprintf(out, "\n</body>\n</html>\n");
}
	
/*
 * end of html export filter
 */

/*
 * pine addressbook import filter
 */

static void
pine_fixbuf(char *buf)
{
	int i,j;

	for(i=0,j=0; j < strlen(buf); i++, j++)
		buf[i] = buf[j] == '\n' ? buf[++j] : buf[j];
}

static void
pine_convert_emails(char *s)
{
	int i;
	char *tmp;

	if( s == NULL || *s != '(' )
		return;

	for(i=0; s[i]; i++ )
		s[i] = s[i+1];

	if( ( tmp = strchr(s,')')) )
		*tmp=0;
	
	for(i=1; ( tmp = strchr(s, ',') ) != NULL ; i++, s=tmp+1 )
		if( i > 3 ) {
			*tmp = 0;
			break;	
		}

}

static void
pine_parse_buf(char *buf)
{
	list_item item;
	char *start = buf;
	char *end;
	char tmp[MAX_FIELD_LEN];
	int i, len, last;

	memset(&item, 0, sizeof(item) );
	
	for(i=0, last=0; !last ; i++) {
		if( ! (end = strchr(start, '\t')) )
			last=1;
		
		len = last ? strlen(start) : min(MAX_FIELD_LEN - 1,
				(int) (end-start) );
		
		switch( i ) {
			case 1: strncpy(tmp, start, len);
				tmp[len] = 0;
				item[NAME] = strdup(tmp);
				break;
			case 2: strncpy(tmp, start, len);
				tmp[len] = 0;
				item[EMAIL] = strdup(tmp);
				break;
			case 4: strncpy(tmp, start, len);
				tmp[len] = 0;
				item[NOTES] = strdup(tmp);
				break;
		}
		start = end + 1;
	}
	
	pine_convert_emails(item[EMAIL]);
	add_item2database(item);
}
		

#define LINESIZE	1024

static int
pine_parse_file(FILE *in)
{
	char line[LINESIZE];
	char *buf=NULL;
	char *ptr;
	int i;

	fgets(line, LINESIZE, in);	
	
	while(!feof(in)) {
		for(i=2;;i++) {
			buf = realloc(buf, i*LINESIZE);
			if(i==2)
				strcpy(buf, line);
			fgets(line, LINESIZE, in);
			ptr=(char *)&line;
			if(*ptr != ' ' || feof(in) )
				break;
			else
				while( *ptr == ' ') ptr++;
				
			strcat(buf, ptr);
		}
		if( *buf == '#' )
			continue;
		pine_fixbuf(buf);

		pine_parse_buf(buf);

		my_free(buf);
	}

	return 0;
}

/*
 * end of pine addressbook import filter
 */

/*
 * pine addressbook export filter
 *
 *  filter doesn't wrap the lines as it should but Pine seems to handle
 *  created files without problems - JH
 */

static int
pine_export_database(FILE *out)
{
	int i;

	for(i=0; i < items; i++ ) {
		fprintf(out, have_multiple_emails(i) ?
				"\t%s\t(%s)\t\t%s\n" : "\t%s\t%s\t\t%s\n",
				safe_str(database[i][NAME]),
				safe_str(database[i][EMAIL]),
				safe_str(database[i][NOTES])
				);
	}

	return 0;
}

/*
 * end of pine addressbook export filter
 */


/*
 * csv addressbook export filter
 */

static int
csv_export_database(FILE *out)
{
	int i, j;
	int csv_export_fields[] = {
		NAME,
		EMAIL,
		PHONE,
		NOTES,
		-1
	};

	for(i=0; i < items; i++ ) {
		for(j = 0; csv_export_fields[j] >= 0; j++) {
			fprintf(out, strchr(safe_str(database[i][csv_export_fields[j]]), ',') ?
				"\"%s\"" : "%s",
				safe_str(database[i][csv_export_fields[j]])
				);
			if(csv_export_fields[j+1] >= 0)
				fputc(',', out);
		}
		fputc('\n', out);
	}
		


	return 0;
}

/*
 * end of csv export filter
 */


/*
 * GnomeCard (VCard) addressbook export filter
 */

static int
gcrd_export_database(FILE *out)
{
	char emails[MAX_EMAILS][MAX_EMAIL_LEN];
	int i, j;
	char name[MAX_FIELD_LEN];

	for(i=0; i < items; i++ ) {
		fprintf(out, "BEGIN:VCARD\nFN:%s\n",
				safe_str(database[i][NAME]));

		get_surname(i, name);
	        for( j = strlen(database[i][NAME]) - 1; j >= 0; j-- ) {
	                if(database[i][NAME][j] == ' ')
	                        break;
	        } 
		fprintf(out, "N:%s;%.*s\n",
			safe_str(name),
			j,
			safe_str(database[i][NAME])
			); 

		if ( database[i][ADDRESS] )
			fprintf(out, "ADR:;;%s;%s;%s;%s;%s\n",
				safe_str(database[i][ADDRESS]),
				safe_str(database[i][CITY]),
				safe_str(database[i][STATE]),
				safe_str(database[i][ZIP]),
				safe_str(database[i][COUNTRY])
				);
		
		if ( database[i][PHONE] ) 
			fprintf(out, "TEL:%s\n", safe_str(database[i][PHONE]));
		
		if ( database[i][EMAIL] ) {
			split_emailstr(i, emails);
			for(j=0; j < MAX_EMAILS ; j++) {
				if ( *emails[j] ) 
					fprintf(out, "EMAIL;INTERNET:%s\n",
						emails[j]);
			}
		}
		
		if ( database[i][NOTES] ) 
			fprintf(out, "NOTE:%s\n", safe_str(database[i][NOTES]));

		fprintf(out, "END:VCARD\n\n");
		
	}

	return 0;
}

/*
 * end of GnomeCard export filter
 */


/*
 * mutt alias export filter
 */

static char *
mutt_alias_genalias(char *str)
{
	char *tmp, *pos;
	
	if( !str )
		return strdup("");

	tmp = strdup(str);

	if( ( pos = strchr(tmp, ' ') ) )
		*pos = 0;

	strlower(tmp);

	return tmp;	
}

static int
mutt_alias_export(FILE *out)
{
	int i;
	char email[MAX_EMAIL_LEN];
	char *alias = NULL;

	for(i=0; i<items; i++) {
		alias = mutt_alias_genalias(database[i][NAME]);
		get_first_email(email, i);
		fprintf(out, "alias %d_%s %s <%s>\n",
				i+1,
				alias,
				database[i][NAME],
				email);
		my_free(alias);
	}

	return 0;
}

/*
 * end of mutt alias export filter
 */


/*
 * printable export filter
 */

static int
text_export_database(FILE *out)
{
	char emails[MAX_EMAILS][MAX_EMAIL_LEN];
	int i, j;
	char *realname;

	realname = get_real_name(getenv("USER"));

	fprintf(out, "%s's addressbook\n\n", realname);

	free(realname);

	for(i=0; i < items; i++ ) {
		fprintf(out, "-----------------------------------------"
				"\n%s\n\n",
				safe_str(database[i][NAME]));

		if( *database[i][EMAIL] ) {
			split_emailstr(i, emails);
			for(j=0; j < MAX_EMAILS ; j++) {
				if ( *emails[j] )
					fprintf(out, "%s\n", emails[j]);
			}
			fputc('\n', out);
		}

		if ( database[i][ADDRESS] ) {
			for(j=2; j < 7; j++)
				if ( database[i][j] )
					fprintf(out, "%s\n", database[i][j]);
			fputc('\n', out);
		}
				
		for(j=PHONE; j <= MOBILEPHONE; j++)
			if ( database[i][j] )
				fprintf(out, "%s: %s\n", abook_fields[j].name, database[i][j]);
		fputc('\n', out);

		if ( database[i][NOTES] )
			fprintf(out, "%s\n\n", database[i][NOTES]);

	}
	fprintf(out, "-----------------------------------------\n");

	return 0;
}

/*
 * end of printable export filter
 */


/*
 * elm alias export filter
 */

static int
elm_alias_export(FILE *out)
{
	int i;
	char email[MAX_EMAIL_LEN];
	char *alias = NULL;

	for(i=0; i<items; i++) {
		alias = mutt_alias_genalias(database[i][NAME]);
		get_first_email(email, i);
		fprintf(out, "%d_%s = %s = %s\n",
				i+1,
				alias,
				database[i][NAME],
				email);
		my_free(alias);
	}

	return 0;
}

/*
 * end of elm alias export filter
 */

