
/*
 * database.c
 * by JH <jheinonen@bigfoot.com>
 *
 * Copyright (C) Jaakko Heinonen
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "abook_curses.h"
#include "abook.h"
#include "database.h"
#include "list.h"
#include "misc.h"
#include "options.h"
#include "filter.h"
#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif

static void	free_item(int i);


list_item *database = NULL;

int items = 0;

#define INITIAL_LIST_CAPACITY	30

int list_capacity = 0;

extern int first_list_item;
extern int curitem;
extern char *selected;

extern char *datafile;
extern char *rcfile;

/*
 * field definitions
 */

#include "edit.h"

/*
 * notes about adding fields:
 * 	- do not change any fields in TAB_CONTACT
 * 	- do not add fields to contact tab
 * 	- 6 fields per tab is maximum
 * 	- reorganize the field numbers in database.h
 */

struct abook_field abook_fields[ITEM_FIELDS] = {
	{"Name",	"name",		TAB_CONTACT},/* NAME */
	{"E-mails",	"email",	TAB_CONTACT},/* EMAIL */
	{"Address",	"address",	TAB_ADDRESS},/* ADDRESS */
	{"City",	"city",		TAB_ADDRESS},/* CITY */
	{"State/Province","state",	TAB_ADDRESS},/* STATE */
	{"ZIP/Postal Code","zip",	TAB_ADDRESS},/* ZIP */
	{"Country",	"country",	TAB_ADDRESS},/* COUNTRY */
	{"Home Phone",	"phone",	TAB_PHONE},/* PHONE */
	{"Work Phone",	"workphone",	TAB_PHONE},/* WORKPHONE */
	{"Fax",		"fax",		TAB_PHONE},/* FAX */
	{"Mobile",	"mobile",	TAB_PHONE},/* MOBILEPHONE */
	{"Nickname/Alias", "nick",      TAB_OTHER},/* NICK */
	{"URL",		"url",		TAB_OTHER},/* URL */
	{"Notes",	"notes",	TAB_OTHER},/* NOTES */
};


int
parse_database(FILE *in)
{
        char *line = NULL;
	char *tmp;
	int sec=0, i;
 	list_item item;

	memset(&item, 0, sizeof(item));
	
	for(;;) {
		line = getaline(in);
		if( feof(in) ) {
			if( item[NAME] && sec )
				add_item2database(item);
			else
				free_list_item(item);
			break;
		}

		if( !*line || *line == '\n' || *line == '#' ) {
			free(line);
			continue;
		} else if( *line == '[' ) {
			if( item[NAME] && sec )
				add_item2database(item);
			else
				free_list_item(item);
			memset(&item, 0, sizeof(item));
			sec = 1;
			if ( !(tmp = strchr(line, ']')))
				sec = 0; /*incorrect section lines are skipped*/
		} else if((tmp = strchr(line, '=') ) && sec ) {
			*tmp++ = '\0';
			for(i=0; i<ITEM_FIELDS; i++)
				if( !strcmp(abook_fields[i].key, line) )
					item[i] = strdup(tmp);
		}
		free(line);
	}

	free(line);
	return 0;
}

		

int
load_database(char *filename)
{
	FILE *in;

	if( database != NULL )
		close_database();

	if ( (in = abook_fopen(filename, "r")) == NULL )
		return -1;
	
	parse_database(in);

	if ( items == 0 )
		return 2;

	return 0;
}

int
write_database(FILE *out)
{
	int i,j;

	fprintf(out, "# abook addressbook file\n\n");
	fprintf(out, "[format]\n");
	fprintf(out, "program=" PACKAGE "\n");
	fprintf(out, "version=" VERSION "\n");
	fprintf(out, "\n\n");

	for( i = 0; i < items; i++ ) {
		fprintf(out, "[%d]\n", i);
		for(j=0; j<ITEM_FIELDS; j++) {
			if( database[i][j] != NULL )
				if( *database[i][j] )
					fprintf(out, "%s=%s\n",
						abook_fields[j].key, database[i][j]);
		}
		fputc('\n', out);
	}

	return 0;
}

int
save_database()
{
	FILE *out;

	if( (out = abook_fopen(datafile, "w")) == NULL )
		return -1;

	if( items < 1 ) {
		fclose(out);
		unlink(datafile);
		return 1;
	}

	
	write_database(out);
	
	fclose(out);
	
	return 0;
}

static void
free_item(int item)
{
	free_list_item(database[item]);
}

void
free_list_item(list_item item)
{
	int i;

	for(i=0; i<ITEM_FIELDS; i++)
		my_free(item[i]);
}

void
close_database()
{
	int i;
	
	for(i=0; i < items; i++)
		free_item(i);

	free(database);
	free(selected);

	database = NULL;
	selected = NULL;

	items = 0;
	first_list_item = curitem = -1;
	list_capacity = 0;
}

#define _MAX_FIELD_LEN(X)	(X == EMAIL ? MAX_EMAILSTR_LEN:MAX_FIELD_LEN)

inline static void
validate_item(list_item item)
{
	int i;
	char *tmp;
	
	if(item[EMAIL] == NULL)
		item[EMAIL] = strdup("");

	for(i=0; i<ITEM_FIELDS; i++)
		if( item[i] && (strlen(item[i]) > _MAX_FIELD_LEN(i) ) ) {
			tmp = item[i];
			item[i][_MAX_FIELD_LEN(i)-1] = 0;
			item[i] = strdup(item[i]);
			free(tmp);
		}
}


static void
adjust_list_capacity()
{
	if(list_capacity < 1)
		list_capacity = INITIAL_LIST_CAPACITY;
	else if(items >= list_capacity)
		list_capacity *= 2;
	else if(list_capacity / 2 > items)
		list_capacity /= 2;
	else
		return;

	database = abook_realloc(database,
			sizeof(list_item) * list_capacity);
	selected = abook_realloc(selected, list_capacity);
}

int
add_item2database(list_item item)
{
	if( item[NAME] == NULL || ! *item[NAME] ) {
		free_list_item(item);
		return 1;
	}

	if( ++items > list_capacity)
		adjust_list_capacity();

	validate_item(item);

	selected[LAST_ITEM] = 0;
	itemcpy(database[LAST_ITEM], item);

	return 0;
}

void
remove_items()
{
	int i, j;

	if( items < 1 || curitem < 0 )
		return;

	statusline_addstr("Remove selected item(s) (Y/n)");
	switch( getch() ) {
		case '\r':
		case 'y':
		case 'Y': break;
		default:
			  clear_statusline();
			  return;
	}

	if( ! selected_items() )
		selected[ curitem ] = 1;
	
	for( j = LAST_ITEM; j >= 0; j-- ) {
		if( selected[j] ) {
			free_item(j); /* added for .4 data_s_ */
			for( i = j; i < LAST_ITEM; i++ ) {
				itemcpy(database[ i ], database[ i + 1 ]);
				selected[ i ] = selected[ i + 1 ];
			}
			items--;	
		}
	}

	if( curitem > LAST_ITEM && items > 0 )
		curitem = LAST_ITEM;


	adjust_list_capacity();

	select_none();
	clear_statusline();	
	refresh_list();
}

char *
get_surname(char *s)
{
	int i, a;
	int len = strlen(s);
	char *name = strdup(s);

	for( a = 0, i = len - 1; i >= 0; i--, a++ ) {
		name[a] = s[i];
		if(name[a] == ' ')
			break;
	}

	name[ a ] = 0;

	revstr(name);

	return name;
}

static int
surnamecmp(const void *i1, const void *i2)
{
	int ret;
	list_item a,b;
	char *s1, *s2;

	itemcpy(a, i1);
	itemcpy(b, i2);

	s1 = get_surname(a[NAME]);
	s2 = get_surname(b[NAME]);

	if( !(ret = safe_strcmp(s1, s2)) )
		ret = safe_strcmp(a[NAME], b[NAME]);

	free(s1);
	free(s2);

	return ret;
}

static int
namecmp(const void *i1, const void *i2)
{
	list_item a, b;

	itemcpy(a, i1);
	itemcpy(b, i2);
	
	return safe_strcmp( a[NAME], b[NAME] );
}

void
sort_database()
{
	select_none();
	
	qsort((void *)database, items, sizeof(list_item), namecmp);

	refresh_screen();
}

void
sort_surname()
{
	select_none();

	qsort((void *)database, items, sizeof(list_item), surnamecmp);

	refresh_screen();
}

void
clear_database()
{

	statusline_addstr("Clear WHOLE database (y/N)");
	switch( getch() ) {
		case 'y':
		case 'Y': break;
		default:
			clear_statusline();
			return;
	}

	close_database();

	refresh_screen();
}

void
find(int next)
{
	int i;
	static char findstr[81];
	char tmp[81];

#ifdef DEBUG
	fprintf(stderr, "find(): findstr = |%s|\n", findstr);
#endif
	
	if(next) {
		if( !*findstr )
			return;
	} else {
		clear_statusline();
		statusline_addstr("/");
		statusline_getnstr(findstr, 67, 0);
		strupper(findstr);
		clear_statusline();
	}

	if(items < 1)
		return;

	for( i = (curitem < LAST_ITEM) && next ? curitem+1 : curitem;
			i < items; i++ ) {
		strcpy(tmp, database[i][NAME]);
		if( strstr(strupper(tmp), findstr) != NULL ) {
			curitem = i;
			refresh_list();
			break;
		}
	}
}


void
print_number_of_items()
{
	char *str =
		mkstr("     |%3d/%3d", selected_items(), items);

	mvaddstr(0, COLS-strlen(str), str);

	free(str);
}

void
read_database()
{
	if(items > 0) {
		statusline_addstr("Your current data will be lost - Press 'y' to continue");
		switch( getch() ) {
			case 'y':
			case 'Y': break;
			default: clear_statusline();
				 return;
		}
		clear_statusline();
	}

	load_database(datafile);
	refresh_list();
}


void
print_database()
{
	FILE *handle;
	char *command = options_get_str("print_command");

	statusline_addstr("Print addressbook? (y/N)");
	switch( getch() ) {
		case 'y':
		case 'Y':
			break;
		default: clear_statusline(); return;
	}
	clear_statusline();

	if( ! *command || (handle = popen(command, "w")) == NULL)
		return;

	fexport("text", handle);
	
	pclose(handle);
}

