
/*
 * database.c
 * by JH <jheinonen@bigfoot.com>
 *
 * Copyright (C) Jaakko Heinonen
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "abook_curses.h"
#include "abook.h"
#include "database.h"
#include "list.h"
#include "misc.h"
#include "options.h"
#include "filter.h"
#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif

static void	free_item(int i);


list_item *database = NULL;

int items = 0;

extern int first_list_item;
extern int curitem;
extern char *selected;

extern char *datafile;
extern char *rcfile;

/*
 * field definitions
 */

#include "edit.h"

/*
 * notes about adding fields:
 * 	- do not change any fields in TAB_CONTACT
 * 	- do not add fields to contact tab
 * 	- 6 fields per tab is maximum
 * 	- reorganize the field numbers in database.h
 */

struct abook_field abook_fields[ITEM_FIELDS] = {
	{"Name",	"name",		TAB_CONTACT},/* NAME */
	{"E-mails",	"email",	TAB_CONTACT},/* EMAIL */
	{"Address",	"address",	TAB_ADDRESS},/* ADDRESS */
	{"City",	"city",		TAB_ADDRESS},/* CITY */
	{"State/Province","state",	TAB_ADDRESS},/* STATE */
	{"ZIP/Postal Code","zip",	TAB_ADDRESS},/* ZIP */
	{"Country",	"country",	TAB_ADDRESS},/* COUNTRY */
	{"Home Phone",	"phone",	TAB_PHONE},/* PHONE */
	{"Work Phone",	"workphone",	TAB_PHONE},/* WORKPHONE */
	{"Fax",		"fax",		TAB_PHONE},/* FAX */
	{"Mobile",	"mobile",	TAB_PHONE},/* MOBILEPHONE */
	{"Nickname/Alias", "nick",      TAB_OTHER},/* NICK */
	{"URL",		"url",		TAB_OTHER},/* URL */
	{"Notes",	"notes",	TAB_OTHER},/* NOTES */
};



#define MAX_LINESIZE	400

int
parse_database(FILE *in)
{
        char line[MAX_LINESIZE];
	char *tmp;
	int sec=0, i;
 	list_item item;

	memset(&item, 0, sizeof(item));
	
	for(;;) {
		fgets(line, MAX_LINESIZE, in);
		if( feof(in) ) {
			if( item[NAME] && sec )
				add_item2database(item);
			else
				free_list_item(item);
			break;
		}
		trim_line(line);

		if( !*line || *line == '\n' || *line == '#' )
			continue;
		else if( *line == '[' ) {
			if( item[NAME] && sec )
				add_item2database(item);
			else
				free_list_item(item);
			memset(&item, 0, sizeof(item));
			sec = 1;
			if ( !(tmp = strchr(line, ']')))
				sec = 0; /*incorrect section lines are skipped*/
		} else if((tmp = strchr(line, '=') ) && sec ) {
			*tmp = '\0'; tmp++;
			for(i=0; i<ITEM_FIELDS; i++)
				if( !strcmp(abook_fields[i].key, line) )
					item[i] = strdup(tmp);
		}
	}

	return 0;
}

		

int
load_database(char *filename)
{
	FILE *in;

	if( database != NULL )
		close_database();

	if ( (in = abook_fopen(filename, "r")) == NULL )
		return -1;
	
	parse_database(in);

	if ( items == 0 )
		return 2;

	return 0;
}

int
write_database(FILE *out)
{
	int i,j;

	fprintf(out, "# abook addressbook file\n\n");
	fprintf(out, "[format]\n");
	fprintf(out, "program=" PACKAGE "\n");
	fprintf(out, "version=" VERSION "\n");
	fprintf(out, "\n\n");

	for( i = 0; i < items; i++ ) {
		fprintf(out, "[%d]\n", i);
		for(j=0; j<ITEM_FIELDS; j++) {
			if( database[i][j] != NULL )
				if( *database[i][j] )
					fprintf(out, "%s=%s\n",
						abook_fields[j].key, database[i][j]);
		}
		fputc('\n', out);
	}

	return 0;
}

int
save_database()
{
	FILE *out;

	if( (out = abook_fopen(datafile, "w")) == NULL )
		return -1;

	if( items < 1 ) {
		fclose(out);
		unlink(datafile);
		return 1;
	}

	
	write_database(out);
	
	fclose(out);
	
	return 0;
}

static void
free_item(int item)
{
	free_list_item(database[item]);
}

void
free_list_item(list_item item)
{
	int i;

	for(i=0; i<ITEM_FIELDS; i++)
		my_free(item[i]);
}

void
close_database()
{
	int i;
	
	for(i=0; i < items; i++)
		free_item(i);

	free(database);
	free(selected);

	database = NULL;
	selected = NULL;

	items = 0;
	first_list_item = curitem = -1;
}

#define _MAX_FIELD_LEN(X)	(X == EMAIL ? MAX_EMAILSTR_LEN:MAX_FIELD_LEN)

inline static void
validate_item(list_item item)
{
	int i;
	char *tmp;
	
	if(item[EMAIL] == NULL)
		item[EMAIL] = strdup("");

	for(i=0; i<ITEM_FIELDS; i++)
		if( item[i] && (strlen(item[i]) > _MAX_FIELD_LEN(i) ) ) {
			tmp = item[i];
			item[i][_MAX_FIELD_LEN(i)-1] = 0;
			item[i] = strdup(item[i]);
			free(tmp);
		}
}

int
add_item2database(list_item item)
{
	if( item[NAME] == NULL || ! *item[NAME] ) {
		free_list_item(item);
		return 1;
	}

	database = abook_realloc(database,
			sizeof(list_item) * ++items);
	selected = abook_realloc(selected, items);

	validate_item(item);

	selected[LAST_ITEM] = 0;
	itemcpy(database[LAST_ITEM], item);

	return 0;
}

void
remove_items()
{
	int i, j;

	if( items < 1 || curitem < 0 )
		return;

	statusline_addstr("Remove selected item(s) (Y/n)");
	switch( getch() ) {
		case '\r':
		case 'y':
		case 'Y': break;
		default:
			  clear_statusline();
			  return;
	}

	selected[ curitem ] = 1;
	
	for( j = LAST_ITEM; j >= 0; j-- ) {
		if( selected[j] ) {
			free_item(j); /* added for .4 data_s_ */
			for( i = j; i < LAST_ITEM; i++ ) {
				itemcpy(database[ i ], database[ i + 1 ]);
				selected[ i ] = selected[ i + 1 ];
			}
			items--;	
		}
	}

	if( curitem > LAST_ITEM && items > 0 )
		curitem = LAST_ITEM;


	if(items > 0) {
		database = abook_realloc(database,
				sizeof(list_item) * items);
		selected = abook_realloc(selected, items);
	} else
		close_database();

	select_none();
	clear_statusline();	
	refresh_list();
}

void
sort_surname()
{
	char s1[80], s2[80];
	int i, j, min;
        list_item t;

	select_none();

	for( i = 0; i < LAST_ITEM; i++) {
		min = i;
		for( j = i + 1; j < items; j++ ) {
			get_surname(j, s1);
			get_surname(min, s2);
#ifdef DEBUG
			fprintf(stderr, 
				"sort_surname(): 1 = %s 2 = %s\n", s1, s2);
#endif
			if( strcmp( s1, s2 ) < 0 )
				min = j;
		}
                itemcpy(t, database[min]);
		itemcpy(database[min], database[i]);
		itemcpy(database[i], t);
	}
	refresh_screen();
}

void
get_surname(int item, char *name)
{
	int i, a;

	for( a = 0, i = strlen(database[item][NAME]) - 1; i >= 0; i--, a++ ) {
		name[a] = database[item][NAME][i];
		if(name[a] == ' ')
			break;
	}

	name[ a ] = 0;

	revstr(name);
}


void
sort_database()
{
	int i, j, min;
	list_item t;

	select_none();
	
        for( i = 0; i < LAST_ITEM; i++) {
		min = i;
		for( j = i + 1; j < items; j++ )
			if( strcmp( database[j][NAME], database[min][NAME] ) < 0 )
				min = j;

		itemcpy(t, database[min]);
		itemcpy(database[min], database[i]);
		itemcpy( database[i], t);
	}
	refresh_screen();
}

void
clear_database()
{

	statusline_addstr("Clear WHOLE database (y/N)");
	switch( getch() ) {
		case 'y':
		case 'Y': break;
		default:
			clear_statusline();
			return;
	}

	close_database();

	refresh_screen();
}

void
find(int next)
{
	int i;
	static char findstr[81];
	char tmp[81];

#ifdef DEBUG
	fprintf(stderr, "find(): findstr = |%s|\n", findstr);
#endif
	
	if(next) {
		if( !*findstr )
			return;
	} else {
		clear_statusline();
		statusline_addstr("/");
		statusline_getnstr(findstr, 67, 0);
		strupper(findstr);
		clear_statusline();
	}

	if(items < 1)
		return;

	for( i = (curitem < LAST_ITEM) && next ? curitem+1 : curitem;
			i < items; i++ ) {
		strcpy(tmp, database[i][NAME]);
		if( strstr(strupper(tmp), findstr) != NULL ) {
			curitem = i;
			refresh_list();
			break;
		}
	}
}


void
print_number_of_items()
{
	char *str =
		mkstr("|%3d/%3d", selected_items(), items);

	mvaddstr(0, COLS-strlen(str), str);

	free(str);
}

void
read_database()
{
	if(items > 0) {
		statusline_addstr("Your current data will be lost - Press 'y' to continue");
		switch( getch() ) {
			case 'y':
			case 'Y': break;
			default: clear_statusline();
				 return;
		}
		clear_statusline();
	}

	load_database(datafile);
	refresh_list();
}


void
print_database()
{
	FILE *handle;
	char *command = options_get_str("print", "command");

	statusline_addstr("Print addressbook? (y/N)");
	switch( getch() ) {
		case 'y':
		case 'Y':
			break;
		default: clear_statusline(); return;
	}
	clear_statusline();

	if( ! *command || (handle = popen(command, "w")) == NULL) {
		free(command);
		return;
	}

	fexport("text", handle);
	
	pclose(handle);

	free(command);
}

