"""
The main Zwiki module. See README.txt.

(c) 1999-2003 Simon Michael <simon@joyful.com> for the zwiki community.
Wikiwikiweb formatting by Tres Seaver <tseaver@zope.com>
Parenting code and regulations by Ken Manheimer <klm@zope.com>
Initial Zope CMF integration by Chris McDonough <chrism@zope.com>
Full credits are at http://zwiki.org/ZwikiContributors .

This product is available under the GNU GPL.  All rights reserved, all
disclaimers apply, etc.
"""

from __future__ import nested_scopes
import os, sys, re, string, time, math, traceback
from DateTime import DateTime
from string import split,join,find,lower,rfind,atoi,strip
from types import *
from AccessControl import getSecurityManager, ClassSecurityInfo
import Acquisition
from App.Common import absattr, rfc1123_date, aq_base
import DocumentTemplate
from urllib import quote, unquote
import ZODB
import Globals
from Globals import MessageDialog, package_home
from Products.MailHost.MailHost import MailBase
from OFS.CopySupport import CopyError
from OFS.content_types import guess_content_type
from OFS.Document import Document
from OFS.DTMLDocument import DTMLDocument
from OFS.ObjectManager import BadRequestException
from OFS.SimpleItem import SimpleItem
import OFS.Image
#import Products.ZCatalog
#from Products.ZCatalog.CatalogBrains import AbstractCatalogBrain
from WWML import translate_WMML
import StructuredText
from Utils import DLOG, flattenDtmlParse
try:
    # start headings at level 2, not 3; will affect all rst clients
    # XXX not working
    import docutils.writers.html4zope
    docutils.writers.html4zope.default_level = 2
    import reStructuredText
except ImportError:
    reStructuredText = None
    DLOG('could not import reStructuredText')

from Products.ZWiki import __version__
from Defaults import DISABLE_JAVASCRIPT, LARGE_FILE_SIZE, AUTO_UPGRADE
import Permissions
from Regexps import url, bracketedexpr, doublebracketedexpr, footnoteexpr, \
     wikiname1, wikiname2, wikiname, wikilink, interwikilink, remotewikiurl, \
     protected_line, javascriptexpr, dtmlorsgmlexpr, zwikiidcharsexpr, \
     anywikilinkexpr, markedwikilinkexpr, localwikilink, spaceandlowerexpr, \
     untitledwikilinkexpr, htmlheaderexpr, htmlfooterexpr
from Utils import thunk_substituter, within_literal, html_quote, \
     html_unquote, myDocumentWithImages, ZOPEVERSION, withinSgmlOrDtml, \
     parseHeadersBody
#from Rendering import Rendering
#from Editing import Editing
from UI import UI
from Parents import ParentsSupport
from Diff import DiffSupport
from Mail import MailSupport
from CatalogAwareness import CatalogAwareness
from Tracker import TrackerSupport, ISSUE_FORM
from Regulations import RegulationsSupport
from CMF import CMFAwareness
from Fit import FitSupport
from Messages import MessagesSupport
from PurpleNumbers import PurpleNumbersSupport
# i18n support
from LocalizerSupport import LocalDTMLFile, _, N_
DTMLFile = LocalDTMLFile
del LocalDTMLFile


class ZWikiPage(
    DTMLDocument,
    UI,
    ParentsSupport,
    DiffSupport,
    MailSupport,
    CatalogAwareness,
    TrackerSupport,
    RegulationsSupport,
    CMFAwareness,
    FitSupport,
    MessagesSupport,
    PurpleNumbersSupport,
    ):
    """
    A ZWikiPage is essentially a DTML Document which knows how to render
    itself in various wiki styles, and can function inside or outside a
    CMF site. A lot of utility methods are provided to support
    wiki-building.

    Mixins are used to organize functionality into distinct modules.
    Initialization, rendering, editing and miscellaneous methods remain in
    the base class.

    RESPONSIBILITIES: (old)

      - render itself

      - provide edit/create forms, backlinks, table of contents

      - accept edit/create requests, with authentication

      - store metadata such as permissions, time, last author, parents etc

      - manage subscriber lists for self & parent folder

    """
    # supported page types
    PAGE_TYPES = {
        'msgstxprelinkdtmlfitissuehtml':'Structured Text',
        'msgrstprelinkfitissue'        :'reStructured Text',
        'msgwwmlprelinkfitissue'       :'WikiWikiWeb markup',
        'dtmlhtml'                     :'HTML',
        'plaintext'                    :'Plain text',
        }
    #ALL_PAGE_TYPES = PAGE_TYPES.keys()
    ALLOWED_PAGE_TYPES = ALL_PAGE_TYPES = [
        'msgstxprelinkdtmlfitissuehtml',
        'msgrstprelinkfitissue',
        'msgwwmlprelinkfitissue',
        'dtmlhtml',
        'plaintext',
        ]
    DEFAULT_PAGE_TYPE = ALL_PAGE_TYPES[0]
    # page types to allow in a wiki (see also Defaults.py)
    def allowedPageTypes(self):
        """List the wiki's "allowed" page types.

        These are the types offered in the edit form. This
        will be all supported types (suitably ordered), unless
        overridden by an allowed_page_types property, 
        """
        return filter(lambda x:strip(x),
                      getattr(self,'allowed_page_types',
                              self.ALLOWED_PAGE_TYPES))
    def defaultPageType(self):
        """This wiki's default page type."""
        allowedtypes = self.allowedPageTypes()
        if allowedtypes: return allowedtypes[0]
        else: return self.DEFAULT_PAGE_TYPE
    
    # old page types to auto-upgrade
    PAGE_TYPE_UPGRADES = {
        # early zwiki
        'Structured Text'           :'msgstxprelinkdtmlfitissuehtml',
        'structuredtext_dtml'       :'msgstxprelinkdtmlfitissuehtml',
        'HTML'                      :'dtmlhtml',
        'html_dtml'                 :'dtmlhtml',
        'Classic Wiki'              :'msgwwmlprelinkfitissue',
        'Plain Text'                :'plaintext',
        # pre-0.9.10
        'stxprelinkdtml'            :'msgstxprelinkdtmlfitissuehtml',
        'structuredtextdtml'        :'msgstxprelinkdtmlfitissuehtml',
        'dtmlstructuredtext'        :'msgstxprelinkdtmlfitissuehtml',
        'structuredtext'            :'msgstxprelinkdtmlfitissuehtml',
        'structuredtextonly'        :'msgstxprelinkdtmlfitissuehtml',
        'classicwiki'               :'msgwwmlprelinkfitissue',
        'htmldtml'                  :'dtmlhtml',
        'plainhtmldtml'             :'dtmlhtml',
        'plainhtml'                 :'dtmlhtml',
        # pre-0.17
        'stxprelinkdtmlhtml'        :'msgstxprelinkdtmlfitissuehtml',
        'issuedtml'                 :'msgstxprelinkdtmlfitissuehtml',
        # pre-0.19
        'stxdtmllinkhtml'           :'msgstxprelinkdtmlfitissuehtml',
        'dtmlstxlinkhtml'           :'msgstxprelinkdtmlfitissuehtml',
        'stxprelinkhtml'            :'msgstxprelinkdtmlfitissuehtml',
        'stxlinkhtml'               :'msgstxprelinkdtmlfitissuehtml',
        'stxlink'                   :'msgstxprelinkdtmlfitissuehtml',
        'wwmllink'                  :'msgwwmlprelinkfitissue',
        'wwmlprelink'               :'msgwwmlprelinkfitissue',
        'prelinkdtmlhtml'           :'dtmlhtml',
        'dtmllinkhtml'              :'dtmlhtml',
        'prelinkhtml'               :'dtmlhtml',
        'linkhtml'                  :'dtmlhtml',
        'textlink'                  :'plaintext',
        # pre-0.20
        'stxprelinkfitissue'        :'msgstxprelinkdtmlfitissuehtml',
        'stxprelinkfitissuehtml'    :'msgstxprelinkdtmlfitissuehtml',
        'stxprelinkdtmlfitissuehtml':'msgstxprelinkdtmlfitissuehtml',
        'rstprelinkfitissue'        :'msgrstprelinkfitissue',
        'wwmlprelinkfitissue'       :'msgwwmlprelinkfitissue',
        # pre-0.22
        'msgstxprelinkfitissuehtml' :'msgstxprelinkdtmlfitissuehtml',
        'html'                      :'dtmlhtml',
        }

    meta_type = "ZWiki Page"
    icon      = "misc_/ZWiki/ZWikiPage_icon"
    creator = ''
    creator_ip = ''
    creation_time = ''
    last_editor = ''
    last_editor_ip = ''
    last_edit_time = ''
    last_log = ''
    page_type = DEFAULT_PAGE_TYPE
    _prerendered = ''   # cached rendered text

    # properties visible in the ZMI
    # would rather append to the superclass' _properties here
    # DocumentTemplate.inheritedAttribute('_properties'),...) ?
    _properties=(
        {'id':'title', 'type': 'string', 'mode':'w'},
        {'id':'page_type', 'type': 'selection', 'mode': 'w',
         'select_variable': 'ALL_PAGE_TYPES'},
        {'id':'creator', 'type': 'string', 'mode': 'r'},
        {'id':'creator_ip', 'type': 'string', 'mode': 'r'},
        {'id':'creation_time', 'type': 'string', 'mode': 'r'},
        {'id':'last_editor', 'type': 'string', 'mode': 'r'},
        {'id':'last_editor_ip', 'type': 'string', 'mode': 'r'},
        {'id':'last_edit_time', 'type': 'string', 'mode': 'r'},
        {'id':'last_log', 'type': 'string', 'mode': 'r'},
        ) \
        + ParentsSupport._properties \
        + MailSupport._properties \
        + CatalogAwareness._properties

    security = ClassSecurityInfo()
    security.declareObjectProtected('View')
    # set some permissions for superclass methods
    security.declareProtected(Permissions.Edit, 'manage_upload')
    # needed ?
    security.declareProtected(Permissions.FTP, 'manage_FTPstat')
    security.declareProtected(Permissions.FTP, 'manage_FTPlist')
    # make sure this appears in the security screen
    security.declareProtected(Permissions.ChangeType, 'dummy')
    def dummy(self):
        pass

    ######################################################################
    # initialization

    def __init__(self, source_string='', mapping=None, __name__=''):
        """
        Initialise this instance, including it's CMF data if applicable.

        Ugly, but putting CMFAwareness before DTMLDocument in the
        inheritance order creates problems.
        """
        if self.supportsCMF():
            CMFAwareness.__init__(self,
                                  source_string=source_string,
                                  mapping=mapping,
                                  __name__=__name__,
                                  )
        else:
            DTMLDocument.__init__(self,
                                  source_string=source_string,
                                  mapping=mapping,
                                  __name__=__name__,
                                  )

    ######################################################################
    # rendering

    security.declareProtected(Permissions.View, '__call__')
    def __call__(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render this zwiki page, upgrading it on the fly if needed

        Similar situation to __init__
        """
        if AUTO_UPGRADE: self.upgrade(REQUEST)
        if self.supportsCMF() and self.inCMF():
            return apply(CMFAwareness.__call__,
                         (self,client,REQUEST,RESPONSE),kw)
        else:
            body = apply(self.render,(client,REQUEST,RESPONSE),kw)
            if RESPONSE is not None:
                RESPONSE.setHeader('Content-Type', 'text/html')
                #RESPONSE.setHeader('Last-Modified', rfc1123_date(self._p_mtime)) 
                #causes browser caching problems ? 
            return body

    def render(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render the body of this zwiki page according to it's page_type
        """
        rendermethod = getattr(self, 'render_'+self.page_type,
                               self.render_plaintext)
        return apply(rendermethod,(self, REQUEST, RESPONSE), kw)

    security.declareProtected(Permissions.View, 'clearCache')
    def clearCache(self,REQUEST=None):
        """
        forcibly clear out any cached render data for this page
        """
        self._prerendered = ''
        if hasattr(self,'_v_cooked'):
            delattr(self,'_v_cooked')
            delattr(self,'_v_blocks')
        if REQUEST:
            REQUEST.RESPONSE.redirect(self.page_url())

    def preRendered(self):
        """
        Get this page's pre-rendered data.
        """
        # cope with a non-existing or None attribute
        return getattr(self,'_prerendered','') or ''

    def preRender(self,clear_cache=0):
        """
        Make sure any applicable pre-rendering for this page has been done
        
        each render_* method knows how to do it's own prerendering..
        pass the pre_only flag in kw to preserve the standard dtml method
        signature
        if clear_cache is 1, blow away any cached data
        """
        if clear_cache:
            self.clearCache()
        return apply(self.render,(self, {}, None), {'pre_only':1})

    # make DTML use our pre-rendered data if available
    import thread
    security.declareProtected(Permissions.View, 'cook') # for testing
    def cook(self,
             cooklock=thread.allocate_lock(),
             ):
        cooklock.acquire()
        try:
            self._v_blocks=self.parse(self.preRendered() or self.read())
            self._v_cooked=None
        finally:
            cooklock.release()

    # built-in render methods (page types).  Each render method does as
    # much pre-rendering as it can and caches it in _prerendered.
    # updated when needed. Call render methods with pre_only=1 to do just
    # the pre-rendering step.
    # 
    # These seem to be merging. NB we are exploring a range of
    # combinations of rendering rules and strategies, while meeting the
    # constraints of DTML, Structured Text etc. and aiming for blazing
    # performance and low memory consumption. It's a bit of a dance.

    security.declareProtected(Permissions.View, 'render_msgstxprelinkdtmlfitissuehtml')
    def render_msgstxprelinkdtmlfitissuehtml(self, client=None, REQUEST={},
                                             RESPONSE=None, **kw):
        """
        Render this page with STX and all available zwiki bells and whistles.

        That is: format messages, do structured text (with pre-formatting)
        and wiki links (with pre-linking), execute any DTML (if allowed),
        execute any fit test tables, add an issue properties form (if
        indicated by the page name), and display the result as HTML.
        
        """
        # pre render stage - do all we can up front and save in _prerendered
        if not self.preRendered():
            get_transaction().note('prerender')
            t = self.documentPart() + self.formattedMessages()
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            if self.usingPurpleNumbers(): t = self.renderPurpleNumbersIn(t)
            t = self.findLinksIn(t)
            self._prerendered = t
        if kw.get('pre_only',0): return
        # final render stage - do dynamic stuff and add layout
        # optimization: don't call DTML if we don't need to, to avoid 
        # unnecessary _v_blocks data and reduce our memory footprint
        if self.dtmlAllowed() and self.hasDynamicContent():
            if not hasattr(self,'_v_blocks'): # XXX debug
                DLOG('parsing and caching DTML code for',self.pageName())
            t = apply(DTMLDocument.__call__,(self,client,REQUEST,RESPONSE),kw)
        else:
            t = self.preRendered()
        t = self.renderMarkedLinksIn(t)
        if self.hasFitTests(): t = self.runFitTestsIn(t)
        if self.isIssue(): t = self.addIssueFormTo(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_msgrstprelinkfitissue')
    def render_msgrstprelinkfitissue(self, client=None, REQUEST={},
                                     RESPONSE=None, **kw):
        """
        Render this page with RST and all compatible bells and whistles.

        That is: format messages, do restructured text (with
        pre-formatting) and wiki links (with pre-linking), execute any fit
        test tables, and add an issue properties form (if indicated by the
        page name).
        
        """
        # pre render stage
        if not self.preRendered():
            get_transaction().note('prerender')
            t = self.documentPart() + self.formattedMessages()
            t = self.applyLineEscapesIn(t)
            if reStructuredText:
                t = reStructuredText.HTML(t,report_level=0)#doesn't hide em
            else:
                t = "<pre>Error: could not import reStructuredText</pre>\n" + t
            if self.usingPurpleNumbers(): t = self.renderPurpleNumbersIn(t)
            t = self.findLinksIn(t)
            self._prerendered = t
        if kw.get('pre_only',0): return
        # final render stage
        t = self.preRendered()
        t = self.renderMarkedLinksIn(t)
        if self.hasFitTests(): t = self.runFitTestsIn(t)
        if self.isIssue(): t = self.addIssueFormTo(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_msgwwmlprelinkfitissue')
    def render_msgwwmlprelinkfitissue(self, client=None, REQUEST={},
                                      RESPONSE=None, **kw):
        """
        Render this page with WikiWikiWeb formatting and compatible features.

        (messages, wiki links, fit tests & issue properties).
        """
        # pre render stage
        if not self.preRendered():
            get_transaction().note('prerender')
            t = self.documentPart() + self.formattedMessages()
            t = html_quote(t)
            t = self.applyLineEscapesIn(t)
            t = translate_WMML(t)
            if self.usingPurpleNumbers(): t = self.renderPurpleNumbersIn(t)
            t = self.findLinksIn(t)
            self._prerendered = t
        if kw.get('pre_only',0): return
        # final render stage
        t = self.preRendered()
        t = self.renderMarkedLinksIn(t)
        if self.hasFitTests(): t = self.runFitTestsIn(t)
        if self.isIssue(): t = self.addIssueFormTo(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_dtmlhtml')
    def render_dtmlhtml(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render this page using DTML (if allowed) and HTML, nothing else.
        """
        # pre render stage
        if not self.preRendered():
            get_transaction().note('prerender')
            t = str(self.read())
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render stage
        if self.dtmlAllowed() and self.hasDynamicContent():
            t = apply(DTMLDocument.__call__,(self,client,REQUEST,RESPONSE),kw)
        else:
            t = self.preRendered()
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_plaintext')
    def render_plaintext(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render this page as plain text with no surprises.
        """
        # pre render
        if not self._prerendered:
            get_transaction().note('prerender')
            t = str(self.read())
            t = "<pre>\n" + html_quote(t) + "\n</pre>\n"
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self.preRendered()
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'stxToHtml')
    def stxToHtml(self, text):
        """
        Render some structured text into html, with our customizations.
        """
        text = str(text)        
        if ZOPEVERSION < (2,4):
            # final single-line paragraph becomes a heading if there are
            # trailing blank lines - strip them
            text = re.sub(r'(?m)\n[\n\s]*$', r'\n', text)

        # an initial single word plus period becomes a numeric bullet -
        # prepend a temporary marker to prevent
        # XXX use locale/wikichars from Regexps.py instead of A-z
        text = re.sub(r'(?m)^([ \t]*)([A-z]\w*\.)',
                      r'\1<!--NOSTX-->\2',
                      text)

        # :: quoting fails if there is whitespace after the :: - remove it
        text = re.sub(r'(?m)::[ \t]+$', r'::', text)

        # suppress stx footnote handling so we can do it our way
        text = re.sub(footnoteexpr,r'<a name="ref\1">![\1]</a>',text)
        text = re.sub(r'(?m)\[',r'[<!--NOSTX-->',text)

        # process STX.. 
        if ZOPEVERSION < (2,4):
            text = str(StructuredText.HTML(text,level=2))
        # with a few more tweaks in STX NG
        else:
            text = StructuredText.HTMLWithImages(
                myDocumentWithImages(StructuredText.Basic(str(text))),
                level=2)

        # clean up
        text = re.sub(r'(<|&lt;)!--NOSTX--(>|&gt;)', r'', text)

        # strip html & body added by some zope versions
        text = re.sub(
            r'(?sm)^<html.*<body.*?>\n(.*)</body>\n</html>\n',r'\1',text)

        return text

    security.declareProtected(Permissions.View, 'supportsStx')
    def supportsStx(self):
        """does this page do Structured Text formatting ?"""
        return re.search(r'(?i)(structuredtext|stx)',
                         self.page_type) is not None

    security.declareProtected(Permissions.View, 'supportsRst')
    def supportsRst(self):
        """does this page do ReStructured Text formatting ?"""
        return re.search(r'(?i)(rst)',
                         self.page_type) is not None

    security.declareProtected(Permissions.View, 'supportsWiki')
    def supportsWiki(self):
        """does this page do wiki linking ?"""
        return re.search(r'(?i)plain',self.page_type) is None

    security.declareProtected(Permissions.View, 'supportsHtml')
    def supportsHtml(self):
        """does this page support embedded HTML ?"""
        return re.search(r'(?i)html',self.page_type) is not None

    security.declareProtected(Permissions.View, 'supportsDtml')
    def supportsDtml(self):
        """does this page support embedded DTML ?"""
        return re.search(r'(?i)dtml',self.page_type) is not None

    security.declareProtected(Permissions.View, 'dtmlAllowed')
    def dtmlAllowed(self):
        """is executing embedded DTML allowed in this wiki ?"""
        return (
            #getattr(self,'use_dtml',0) and
            not hasattr(self,'no_dtml')
            )

    security.declareProtected(Permissions.View, 'hasDynamicContent')
    def hasDynamicContent(self):
        """does this page contain dynamic content ?"""
        return (self.supportsDtml() and
                re.search(r'(?i)(<dtml|&dtml)',self.read()) is not None)

    ######################################################################
    # linking & link rendering

    def wikinameLinksAllowed(self):
        """Are wikinames linked in this wiki ?"""
        return getattr(self,'use_wikiname_links',1)

    def bracketLinksAllowed(self):
        """Are bracketed freeform names linked in this wiki ?"""
        return getattr(self,'use_bracket_links',1)

    def doublebracketLinksAllowed(self):
        """Are wikipedia-style double bracketed names linked in this wiki ?"""
        return getattr(self,'use_doublebracket_links',1)

    def hasAllowedLinkSyntax(self,link):
        if (re.match(url,link) or
            (self.wikinameLinksAllowed() and
             re.match(wikiname,link)) or
            (self.bracketLinksAllowed() and
             re.match(bracketedexpr,link) and
             not re.match(doublebracketedexpr,link)) or
            (self.doublebracketLinksAllowed() and
             re.match(doublebracketedexpr,link))):
            return 1
        else:
            return 0

    def findLinksIn(self,text):
        """
        Find and mark links in text, for fast replacement later.

        Successor to _preLink. Instead of generating a list of text
        extents and link names, this simply marks the links in place to
        make them easy to find again.  Tries to be smart about finding
        links only where you want it to.
        """
        get_transaction().note('findlinks')
        markedtext = ''
        state = {'lastend':0,'inpre':0,'incode':0,'intag':0,'inanchor':0}
        lastpos = 0
        while 1:
            m = anywikilinkexpr.search(text,lastpos)
            if m:
                link = m.group()
                linkstart,linkend = m.span()
                if (link[0]=='!' or
                    not self.hasAllowedLinkSyntax(link) or
                    within_literal(linkstart,linkend-1,state,text) or
                    withinSgmlOrDtml((linkstart,linkend),text)):
                    # found the link pattern but it's escaped or disallowed or
                    # inside a STX quote or SGML tag - ignore (and strip the !)
                    if link[0] == '!': link=link[1:]
                    markedtext += text[lastpos:linkstart] + link
                else:
                    # a link! mark it and save it
                    markedtext += '%s<zwiki>%s</zwiki>' \
                                  % (text[lastpos:linkstart],link)
                lastpos = linkend
            else:
                # no more links - save the final text extent & quit
                markedtext += text[lastpos:]
                break
        return markedtext

    def renderMarkedLinksIn(self,text):
        """
        Render the links in text previously marked by findLinksIn.
        """
        return re.sub(markedwikilinkexpr,self.renderLink,text)

    def renderLinksIn(self,text):
        """
        Find and render all links in text.
        """
        t = self.applyLineEscapesIn(text)
        t = re.sub(anywikilinkexpr,
                   thunk_substituter(self.renderLink, t, 1),
                   t)
        return t

    security.declareProtected(Permissions.View, 'wikilink')
    wikilink = renderLinksIn # api alias

    security.declareProtected(Permissions.View, 'applyLineEscapesIn')
    def applyLineEscapesIn(self, text):
        """
        implement wikilink-escaping in lines in text which begin with !
        """
        return re.sub(protected_line, self.protectLine, text)
        
    def protectLine(self, match):
        """
        return the string represented by match with all it's wikilinks escaped
        """
        return re.sub(wikilink, r'!\1', match.group(1))

    # XXX poor caching
    def renderLink(self,link,allowed=0,state=None,text=''):
        """
        Render a link depending on current wiki state.

        Can be called three ways:
        - directly (link should be a string)
        - from re.sub (link will be a match object, state will be None)
        - from re.sub via thunk substituter (state will be a dictionary) (old)
        """
        # preliminaries
        if type(link) == StringType:
            text = self.preRendered()
        elif state == None:
            link = link.group()
            text = self.preRendered()
        else:
            match = link
            link = match.group()
            # we are being called from re.sub, using thunk_substituter to
            # keep state - do the within_literal and within sgml checks that
            # would normally be done in findLinksIn
            if (within_literal(match.start(),match.end()-1,state,text) or
                withinSgmlOrDtml(match.span(),text)):
                return link
        linkorig = link = re.sub(markedwikilinkexpr, r'\1', link)
        linknobrackets = re.sub(bracketedexpr, r'\1', link)

        # is this link escaped ?
        if link[0] == '!': return link[1:]

        # is it an interwiki link ?
        if re.match(interwikilink,link): return self.renderInterwikiLink(link)

        # is it something in brackets ?
        if re.match(bracketedexpr,link):
            # a STX footnote ? check for matching named anchor in the page text
            if re.search(r'(?s)<a name="ref%s"' % (re.escape(link)),text):
                return '<a href="%s#ref%s" title="footnote %s">[%s]</a>' \
                       % (self.page_url(),link,link,link)
            # or, an allowed bracketed freeform link syntax for this wiki ?
            if ((re.match(doublebracketedexpr,link) and
                 self.doublebracketLinksAllowed()) or
                (not re.match(doublebracketedexpr,link) and
                 self.bracketLinksAllowed())):
                # convert to a page id if possible, and continue
                p = self.pageWithFuzzyName(linknobrackets,ignore_case=1)
                if p: link = p.getId() # XXX poor caching
        
        # is it a bare URL ?
        if re.match(url,link):
            return '<a href="%s">%s</a>' % (link, link)

        # it must be a wikiname - are wikiname links allowed in this wiki ?
        if not (self.wikinameLinksAllowed() or
                re.match(bracketedexpr,linkorig)): #was processed above
            return link

        # we have a wikiname - does a matching page exist in this wiki ?
        # NB still need to find the page id, since it will be different if
        # international characters are enabled in wiki names but not page ids
        if self.pageWithNameOrId(link):
            if not self.pageWithId(link):
                link = self.pageWithNameOrId(link).getId() # XXX poor caching
            linktitle = '' #self.pageWithId(link).linkTitle()
            style='style="background-color:%s;"' \
                   % self.pageWithNameOrId(link).issueColour()#XXX poor caching
            return '<a href="%s/%s" title="%s" %s>%s</a>' \
                   % (self.wiki_url(),quote(link),linktitle,style,linknobrackets)

        # subwiki support: or does a matching page exist in the parent folder ?
        # XXX this is dumber than the above; doesn't handle i18n
        # characters, freeform names
        if (hasattr(self.folder(),'aq_parent') and
              hasattr(self.folder().aq_parent, link) and
              self.isZwikiPage(getattr(self.folder().aq_parent,link))): #XXX poor caching
            return '<a href="%s/../%s" title="page in parent wiki">../%s</a>'\
                   % (self.wiki_url(),quote(link),linkorig)

        # otherwise, provide a creation link
        return '%s<a class="new" href="%s/%s/editform?page=%s" title="create this page">?</a>' \
               % (linkorig, self.wiki_url(), quote(self.id()),
                  quote(linknobrackets))

    def renderInterwikiLink(self, link):
        """
        Render an occurence of interwikilink. link is a string.
        """
        if link[0] == '!': return link[1:]
        m = re.match(interwikilink,link)
        local, remote  = m.group('local'), m.group('remote')
        # check local is an allowed link syntax for this wiki
        if not self.hasAllowedLinkSyntax(local): return link
        local = re.sub(bracketedexpr, r'\1', local)
        # look for a RemoteWikiURL definition
        if hasattr(self.folder(), local): 
            m = re.search(remotewikiurl,getattr(self.folder(),local).text())
            if m:
                return '<a href="%s%s">%s:%s</a>' \
                       % (m.group('remoteurl'),remote,local,remote)
                       #XXX old html_unquote needed ? I don't think so
        # otherwise return unchanged
        return link

    security.declareProtected(Permissions.View, 'links')
    def links(self):
        """
        List the unique links occurring on this page - useful for cataloging.

        Includes urls & interwiki links but not structured text links.
        Extracts the marked links from prerendered data.  Does not
        generate this if missing - too expensive when cataloging ?
        """
        #if not self.preRendered(): self.preRender()
        links = []
        for l in re.findall(markedwikilinkexpr,self.preRendered()):
            if not l in links: links.append(l)
        return links

    security.declareProtected(Permissions.View, 'canonicalLinks')
    def canonicalLinks(self):
        """
        List the canonical id form of the local wiki links in this page.

        Useful for calculating backlinks. Extracts this information
        from prerendered data, does not generate this if missing.
        """
        clinks = []
        localwikilinkexpr = re.compile(localwikilink)
        for link in self.links():
            if localwikilinkexpr.match(link):
                if link[0] == r'[' and link[-1] == r']':
                    link = link[1:-1]
                clink = self.canonicalIdFrom(link)
                clinks.append(clink)
        return clinks

    ######################################################################
    # page naming and lookup

    security.declareProtected(Permissions.View, 'pageName')
    def pageName(self):
        """
        Return the name of this wiki page.

        This is normally in the title attribute, but use title_or_id
        to handle eg pages created in the ZMI.
        """
        return self.title_or_id()
    
    security.declarePublic('Title')
    def Title(self):
        return self.pageName()

    security.declareProtected(Permissions.View, 'canonicalIdFrom')
    def canonicalIdFrom(self,name):
        """
        Convert a free-form page name to a canonical url- and zope-safe id.

        Constraints for zwiki page ids:
        - it needs to be a legal zope object id
        - to simplify linking, we will require it to be a valid url
        - it should be unique for a given name (ignoring whitespace)
        - we'd like it to be as similar to the name and as simple to read
          and work with as possible
        - we'd like to encourage serendipitous linking between free-form
          and wikiname links & pages

        So this version
        - discards non-word-separating punctuation (')
        - converts remaining punctuation to spaces
        - capitalizes and joins whitespace-separated words into a wikiname
        - converts any non-zope-and-url-safe characters and _ to _hexvalue
        - if the above results in an id beginning with _, prepends X
          (XXX this breaks the uniqueness requirement, better ideas ?)

        performance-sensitive
        """
        # remove punctuation, preserving word boundaries.
        # ' is not considered a word boundary.
        name = re.sub(r"'",r"",name)
        name = re.sub(r'[%s]+'%re.escape(string.punctuation),r' ',name)
        
        # capitalize whitespace-separated words (preserving existing
        # capitals) then strip whitespace
        id = ' '+name
        id = spaceandlowerexpr.sub(lambda m:string.upper(m.group(1)),id)
        id = string.join(string.split(id),'')

        # quote any remaining unsafe characters (international chars)
        safeid = ''
        for c in id:
            if zwikiidcharsexpr.match(c):
                safeid = safeid + c
            else:
                safeid = safeid + '_%02x' % ord(c)

        # zope ids may not begin with _
        if len(safeid) > 0 and safeid[0] == '_':
            safeid = 'X'+safeid
        return safeid

    security.declareProtected(Permissions.View, 'canonicalId')
    def canonicalId(self):
        """
        Give the canonical id of this page.
        """
        return self.canonicalIdFrom(self.pageName())

    # XXX poor caching when attributes are accessed
    security.declareProtected(Permissions.View, 'pageObjects')
    def pageObjects(self):
        """
        Return a list of all pages in this wiki.
        """
        return self.folder().objectValues(spec=self.meta_type)

    def wikiPath(self):
        """
        This wiki's folder path, for filtering our pages from catalog results
        """
        return self.getPath()[:self.getPath().rfind('/')]

    security.declareProtected(Permissions.View, 'pages')
    def pages(self, **kw):
        """
        Look up metadata (brains) for some or all pages in this wiki.

        optimisation: prior to 0.22 this returned the actual page objects,
        but to help with caching efficiency it now uses the catalog, if
        possible.  The page metadata objects are catalog brains (search
        results) containing the catalog's metadata, or workalikes
        containing a limited number of fields and getObject().

        Any keyword arguments will be passed through to the catalog, eg
        for restricting the search, sorting etc. But if there is no
        catalog these will be ignored. By default, all pages in the wiki
        are returned.
        """
        if self.hasCatalogIndexesMetadata((['meta_type','path'], [])):
            # this catalog may index more than one wiki.. we want just the
            # results in our folder, for now..
            wikipath = self.wikiPath()
            def folderpath(s): return s[:s.rfind('/')]
            return filter(lambda x:folderpath(x.getPath())==wikipath,
                          self.searchCatalog(meta_type=self.meta_type,
                                             path=wikipath,
                                             **kw))
        else:
            results = []
            for p in self.pageObjects(): results.append(self.metadataFor(p))
            return results

    security.declareProtected(Permissions.View, 'pageIds')
    def pageIds(self):
        """
        Return a list of all page ids in this wiki.
        """
        #return self.folder().objectIds(spec=self.meta_type) # more robust ?
        return map(lambda x:x.id,self.pages())

    security.declareProtected(Permissions.View, 'pageNames')
    def pageNames(self):
        """
        Return a list of all page names in this wiki.
        """
        return map(lambda x:x.Title,self.pages())

    security.declareProtected(Permissions.View, 'pageIdsStartingWith')
    def pageIdsStartingWith(self,text):
        #from __future__ import nested_scopes
        #return filter(lambda x:x[:len(text)]==text,self.pageIds())
        ids = []
        for i in self.pageIds():
            if i[:len(text)] == text:
                ids.append(i)
        return ids

    security.declareProtected(Permissions.View, 'pageNamesStartingWith')
    def pageNamesStartingWith(self,text):
        #from __future__ import nested_scopes
        #return filter(lambda x:x[:len(text)]==text,self.pageNames())
        names = []
        for n in self.pageNames():
            if n[:len(text)] == text:
                names.append(n)
        return names

    security.declareProtected(Permissions.View, 'firstPageIdStartingWith')
    def firstPageIdStartingWith(self,text):
        return (self.pageIdsStartingWith(text) or [None])[0]

    security.declareProtected(Permissions.View, 'firstPageNameStartingWith')
    def firstPageNameStartingWith(self,text):
        return (self.pageNamesStartingWith(text) or [None])[0]

    security.declareProtected(Permissions.View, 'pageWithId')
    def pageWithId(self,id,url_quoted=0,ignore_case=0):
        """
        Return the page in this folder which has this id, or None.

        Can also do a case-insensitive id search,
        and optionally unquote id.
        """
        if url_quoted:
            id = unquote(id)
        f = self.folder()
        # NB: don't acquire
        if hasattr(f.aq_base,id) and self.isZwikiPage(f[id]): # poor caching
            return f[id]
        elif ignore_case:
            id = string.lower(id)
            for i in self.pageIds():
                if i.lower() == id:
                    return f[i]
        else:
            return None

    security.declareProtected(Permissions.View, 'pageWithName')
    def pageWithName(self,name,url_quoted=0):
        """
        Return the page in this folder which has this name, or None.

        page name may be different from page id, and if so is stored in
        the title property. Ie page name is currently defined as
        the value given by title_or_id().

        As of 0.17, page ids and names always follow the invariant
        id == canonicalIdFrom(name).
        """
        return (self.pageWithId(self.canonicalIdFrom(name),url_quoted))

    security.declareProtected(Permissions.View, 'pageWithNameOrId')
    def pageWithNameOrId(self,name,url_quoted=0):
        """
        Return the page in this folder with this as it's name or id, or None.
        """
        return (self.pageWithId(name,url_quoted) or 
                self.pageWithName(name,url_quoted))
        
    security.declareProtected(Permissions.View, 'pageWithFuzzyName')
    def pageWithFuzzyName(self,name,url_quoted=0,
                          allow_partial=0,ignore_case=1):
        """
        Return the page in this folder for which name is a fuzzy link, or None.

        A fuzzy link ignores whitespace, capitalization & punctuation.  If
        there are multiple fuzzy matches, return the page whose name is
        alphabetically first.  The allow_partial flag allows even fuzzier
        matching. As of 0.17 ignore_case is not used and kept only for
        backward compatibility.

        performance-sensitive
        """
        if url_quoted:
            name = unquote(name)
        p = self.pageWithName(name)
        if p: return p
        id = self.canonicalIdFrom(name)
        idlower = string.lower(id)
        ids = self.pageIds()
        # in case this is a BTreeFolder (& old zope ?), work around as per
        # IssueNo0535PageWithFuzzyNameAndBTreeFolder.. may not return
        # the alphabetically first page in this case
        try:
            ids.sort()
        except AttributeError:
            pass
        for i in ids:
            ilower = string.lower(i)
            if (ilower == idlower or 
                (allow_partial and ilower[:len(idlower)] == idlower)):
                return self.pageWithId(i)
        return None
        
    security.declareProtected(Permissions.View, 'backlinksFor')
    def backlinksFor(self, page):
        """
        Return metadata objects for all the pages that link to page.

        Optimisation: like pages(), this method used to return actual page
        objects but now returns metadata objects (catalog results if possible,
        or workalikes) to improve caching. 

        page may be a name or id, and need not exist in the wiki

        The non-catalog search is not too smart.
        """
        p = self.pageWithNameOrId(page)
        if p: id = p.id()
        else: id = page
        if self.hasCatalogIndexesMetadata(
            (['meta_type','path','canonicalLinks'], [])):
            return self.pages(canonicalLinks=id)
        else:
            # brute-force search (poor caching)
            # find both [page] and bare wiki links
            # XXX should check for fuzzy links
            # XXX should do a smarter search (eg use links())
            results = []
            linkpat = re.compile(r'\b(%s|%s)\b'%(page,id))
            for p in self.pageObjects():
                if linkpat.search(p.read()):
                    results.append(self.metadataFor(p))
            return results

    ######################################################################
    # editing methods

    def _checkPermission(self, permission, object):
        return getSecurityManager().checkPermission(permission,object)

    security.declareProtected(Permissions.View, 'create') 
    # really Permissions.Add, but keep our informative unauthorized message
    def create(self,page,text=None,type=None,title='',REQUEST=None,log='',
               leaveplaceholder=1, updatebacklinks=1, sendmail=1):
        """
        Create a new wiki page and redirect there if appropriate; can
        upload a file at the same time.  Normally edit() will call
        this for you.

        Assumes page has been url-quoted. If it's not a url-safe name, we
        will create the page with a url-safe id that's similar. We assume
        this id won't match anything already existing (zwiki would have
        linked it instead of offering to create it).

        Can handle a rename during page creation also. This seems less
        sensible than in edit(), but it helps support CMF/Plone.
        """
        # do we have permission ?
        if not self._checkPermission(Permissions.Add,self.folder()):
            raise 'Unauthorized', (
                _('You are not authorized to add ZWiki Pages here.'))

        name = unquote(page)
        id = self.canonicalIdFrom(name)

        # make a new (blank) page object, situate it
        # in the parent folder and get a hold of it's
        # normal acquisition wrapper
        # newid should be the same as id, but don't assume
        p = ZWikiPage(source_string='', __name__=id)
        newid = self.folder()._setObject(id,p)
        p = getattr(self.folder(),newid)

        p.title = name
        p._setCreator(REQUEST)
        p._setLastEditor(REQUEST)
        p._setLastLog(log)
        p._setOwnership(REQUEST)
        p.parents = [self.title_or_id()]

        # set the specified page type, otherwise use this wiki's default
        p.page_type = type or self.defaultPageType()

        # set initial page text as edit() would, with cleanups and dtml
        # validation
        p._setText(text or '',REQUEST)

        # if a file was submitted as well, handle that
        p._handleFileUpload(REQUEST)

        # plone support, etc: they might alter the page name in the
        # creation form! allow that. We do a full rename after all the
        # above to make sure everything gets handled properly.  We need to
        # commit first though, so p.cb_isMoveable() succeeds.
        # Renaming will do all the indexing we need.
        if title and title != p.title_or_id():
            get_transaction().note('rename during creation')
            get_transaction().commit()
            p._handleRename(title,leaveplaceholder,updatebacklinks,REQUEST,log)
        else:
            # we got indexed after _setObject,
            # but do it again with our text in place
            p.index_object()

        if p.usingRegulations():
            # initialize regulations settings.
            p.setRegulations(REQUEST,new=1)

        # if auto-subscription is enabled (folder-wide), subscribe the creator
        if p.autoSubscriptionEnabled():
           usernameoremail = (REQUEST and (
                              str(REQUEST.get('AUTHENTICATED_USER')) or
                              REQUEST.cookies.get('email')))
           if usernameoremail:
               if not self.isWikiSubscriber(usernameoremail):
                   p.subscribe(usernameoremail)
                   DLOG('auto-subscribing',usernameoremail,'to',p.id())

        # always mail out page creations, unless disabled
        # and give these a message id we can use for threading
        message_id = self.messageIdFromTime(p.creationTime())
        if sendmail:
            p.sendMailToSubscribers(p.read(),
                                    REQUEST=REQUEST,
                                    subjectSuffix='',
                                    subject='(new) '+log,
                                    message_id=message_id)

        # redirect browser if needed
        if REQUEST is not None:
            try:
                u = (REQUEST.get('redirectURL',None) or
                     REQUEST['URL2']+'/'+ quote(p.id()))
                REQUEST.RESPONSE.redirect(u)
            except KeyError:
                pass

    security.declareProtected(Permissions.Comment, 'comment')
    def comment(self, text='', username='', time='',
                note=None, use_heading=None, # not used
                REQUEST=None, subject_heading='', message_id=None,
                in_reply_to=None, exclude_address=None):
        """
        A handy method, like append but adds a standard rfc2822 message
        heading with the specified or default values.

        If mailout policy is "comments", we generate the mailout here
        (otherwise it's done by edit).

        Finally, if auto-subscription is in effect, we subscribe the
        poster to this page.

        The subject_heading argument is so named to avoid a clash with
        some existing zope or CMF subject attribute. The note and
        use_heading arguments remain only for backwards compatibility with
        old skin templates.

        Special testing support: if the subject is '[test]', we skip
        changing the page (except on TestPage).
        """
        # gather various bits and pieces
        oldtext = self.read()
        text = re.sub(r'\r\n',r'\n',text) # where did these appear from ?
        subject = subject_heading      # clashes with some zope attribute
        if not username:
            username = self.zwiki_username_or_ip(REQUEST)
            if re.match(r'^[0-9\.\s]*$',username): 
                username = ''
        # ensure message_id is defined at this point so page and mail-out
        # will use the same value and threading will work
        # also ensure it matches time where possible (!) may help debugging
        if time:
            dtime = DateTime(time)
        else:
            dtime = self.ZopeTime()
            time = dtime.rfc822()
        if not message_id:
            message_id = self.messageIdFromTime(dtime)

        # add message to page
        # testing support: only if subject is not [test] (except on TestPage)
        if subject != '[test]' or self.getId() == 'TestPage':
            self.append(self.makeMessageFrom(username,time,message_id,
                                             subject,in_reply_to,text),
                        REQUEST=REQUEST,
                        log=subject)

        # if mailout policy is comments only, send it now
        # (otherwise append did it) # XXX refactor
        # testing support: is confused, see Mail.py XXX
        if (getattr(self.folder(),'mailout_policy','comments')=='comments'
            and (text or subject)):
            # get the username in there
            if REQUEST: REQUEST.cookies['zwiki_username'] = username
            self.sendMailToSubscribers(text, 
                                       REQUEST,
                                       subjectSuffix="",
                                       subject=subject,
                                       message_id=message_id,
                                       in_reply_to=in_reply_to,
                                       exclude_address=exclude_address)

        # if auto-subscription is in effect, subscribe this user
        if self.autoSubscriptionEnabled():
           usernameoremail = (REQUEST and (
                              str(REQUEST.get('AUTHENTICATED_USER')) or
                              REQUEST.cookies.get('email')))
           if usernameoremail:
               if not (self.isSubscriber(usernameoremail) or
                       self.isWikiSubscriber(usernameoremail)):
                   self.subscribe(usernameoremail)
                   DLOG('auto-subscribing',usernameoremail,'to',self.id())

    security.declareProtected(Permissions.Comment, 'append')
    def append(self, text='', separator='\n\n', REQUEST=None, log=''):
        """
        Appends some text to an existing zwiki page by calling edit;
        may result in mail notifications to subscribers.
        """
        oldtext = self.read()
        text = str(text)
        if text:
            # usability hack: scroll to bottom after adding a comment
            if REQUEST:
                REQUEST.set('redirectURL',REQUEST['URL1']+'#bottom')
            self.edit(text=oldtext+separator+text, REQUEST=REQUEST,log=log)

    security.declarePublic('edit')      # check permissions at runtime
    def edit(self, page=None, text=None, type=None, title='', 
             timeStamp=None, REQUEST=None, 
             subjectSuffix='', log='', check_conflict=1, # temp (?)
             leaveplaceholder=1, updatebacklinks=1): 
        """
        General-purpose method for editing & creating zwiki pages.
        Changes the text and/or markup type of this (or the specified)
        page, or creates the specified page (name or id allowed) if it
        does not exist.
        
        Other special features:

        - Usually called from a time-stamped web form; we use
        timeStamp to detect and warn when two people attempt to work
        on a page at the same time. This makes sense only if timeStamp
        came from an editform for the page we are actually changing.

        - The username (authenticated user or zwiki_username cookie)
        and ip address are saved in page's last_editor, last_editor_ip
        attributes if a change is made

        - If the text begins with "DeleteMe", move this page to the
        recycle_bin subfolder.

        - If file has been submitted in REQUEST, create a file or
        image object and link or inline it on the current page.

        - May also cause mail notifications to be sent to subscribers

        - if title differs from page, assume it is the new page name and
        do a rename (the argument remains as "title" for backwards
        compatibility)

        This code has become more complex to support late page
        creation, but the api should now be more general & powerful
        than it was.  Doing all this stuff in one method simplifies
        the layer above (the skin) I think.
        """
        #self._validateProxy(REQUEST)   # XXX correct ? don't think so
                                        # do zwiki pages obey proxy roles ?

        if page: page = unquote(page)
        # are we changing this page ?
        if page is None:
            p = self
        # changing another page ?
        elif self.pageWithNameOrId(page):
            p = self.pageWithNameOrId(page)
        # or creating a new page
        else:
            return self.create(page,text,type,title,REQUEST,log)

        # ok, changing p. We may be doing several things here;
        # each of these handlers checks permissions and does the
        # necessary. Some of these can halt further processing.
        # todo: tie these in to mail notification, along with 
        # other changes like reparenting
        if check_conflict and self.checkEditConflict(timeStamp, REQUEST):
            return self.editConflictDialog()
        if check_conflict and hasattr(self,'wl_isLocked') and self.wl_isLocked():
            return self.davLockDialog()
        if p._handleDeleteMe(text,REQUEST,log):
            return
        p._handleEditPageType(type,REQUEST,log)
        p._handleEditText(text,REQUEST,subjectSuffix,log)
        p._handleFileUpload(REQUEST,log)
        p._handleRename(title,leaveplaceholder,updatebacklinks,REQUEST,log)
        if self.usingRegulations():
            p._handleSetRegulations(REQUEST)

        # update catalog if present
        try:
            p.index_object()
        except:
            # XXX show traceback in log
            DLOG('failed to index '+p.id())
            try:
                p.reindex_object()
            except:
                DLOG('failed to reindex '+p.id()+', giving up')
                pass

        # redirect browser if needed
        if REQUEST is not None:
            try:
                u = (REQUEST.get('redirectURL',None) or
                     REQUEST['URL2']+'/'+ quote(p.id()))
                REQUEST.RESPONSE.redirect(u)
            except KeyError:
                pass

    def _handleSetRegulations(self,REQUEST):
        if REQUEST.get('who_owns_subs',None) != None:
            # do we have permission ?
            if not self._checkPermission(Permissions.ChangeRegs,self):
                raise 'Unauthorized', (
                  _("You are not authorized to set this ZWiki Page's regulations."))
            self.setRegulations(REQUEST)
            self.preRender(clear_cache=1)
            #self._setLastEditor(REQUEST)

    def _handleEditPageType(self,type,REQUEST=None,log=''):
        # is the new page type valid and different ?
        if (type is not None and
            type != self.page_type):
            # do we have permission ?
            if not self._checkPermission(Permissions.ChangeType,self):
                raise 'Unauthorized', (
                    _("You are not authorized to change this ZWiki Page's type."))
            # is it one of the allowed types for this wiki ?
            #if not type in self.allowedPageTypes():
            #    raise 'Unauthorized', (
            #        _("Sorry, that's not one of the allowed page types in this wiki."))
            # change it
            self.page_type = type
            self.preRender(clear_cache=1)
            self._setLastEditor(REQUEST)
            self._setLastLog(log)

    def _setLastLog(self,log):
        """\
        Note log message, if provided.
        """
        if log and string.strip(log):
            log = string.strip(log)
            get_transaction().note('"%s"' % log)
            self.last_log = log
        else:
            self.last_log = ''


    def _handleEditText(self,text,REQUEST=None, subjectSuffix='', log=''):
        # is the new text valid and different ?
        if (text is not None and
            self._cleanupText(text) != self.read()):
            # do we have permission ?
            if (not
                (self._checkPermission(Permissions.Edit, self) or
                 (self._checkPermission(Permissions.Append, self)
                  and find(self._cleanupText(text),self.read()) == 0))):
                raise 'Unauthorized', (
                    _('You are not authorized to edit this ZWiki Page.'))

            # change it
            oldtext = self.read()
            self._setText(text,REQUEST)
            self._setLastEditor(REQUEST)
            self._setLastLog(log)

            # if mailout policy is all edits, do it here
            if getattr(self.folder(),'mailout_policy','')=='edits':
                self.sendMailToSubscribers(
                    self.textDiff(a=oldtext,b=self.read()),
                    REQUEST=REQUEST,
                    subject=log)

    def _handleDeleteMe(self,text,REQUEST=None,log=''):
        if not text or not re.match('(?m)^DeleteMe', text):
            return 0
        if (not
            (self._checkPermission(Permissions.Edit, self) or
             (self._checkPermission(Permissions.Append, self)
              and find(self._cleanupText(text),self.read()) == 0))):
            raise 'Unauthorized', (
                _('You are not authorized to edit this ZWiki Page.'))
        if not self._checkPermission(Permissions.Delete, self):
            raise 'Unauthorized', (
                _('You are not authorized to delete this ZWiki Page.'))
        self._setLastLog(log)
        self._recycle(REQUEST)

        if REQUEST:
            # redirect to first existing parent, or front page
            destpage = ''
            for p in self.parents:
                if hasattr(self.folder(),p):
                    destpage = p
                    break
            REQUEST.RESPONSE.redirect(self.wiki_url()+'/'+quote(destpage))
            # I used to think redirect did not return, guess I was wrong

        # return true to terminate edit processing
        return 1

    def _handleRename(self,newname,leaveplaceholder,updatebacklinks,
                      REQUEST=None,log=''):
        # rename does everything we need
        return self.rename(newname,
                           leaveplaceholder=leaveplaceholder,
                           updatebacklinks=updatebacklinks,
                           REQUEST=REQUEST)

    security.declareProtected(Permissions.Delete, 'delete')
    def delete(self,REQUEST=None):
        """
        delete (recycle) this page, if permissions allow
        """
        if not (self._checkPermission(Permissions.Delete, self) and
                self.requestHasSomeId(REQUEST)):
            raise 'Unauthorized', (
                _('You are not authorized to delete this ZWiki Page.'))
        # first, transfer any children to our (first) parent to avoid orphans
        for id in self.offspringIdsAsList(REQUEST):
            child = getattr(self.folder(),id)
            try:
                child.parents.remove(self.title_or_id())
            except ValueError:
                pass
            if self.parents:
                child.parents.append(self.parents[0])
            child.index_object()
        self._recycle(REQUEST)
        if getattr(self.folder(),'mailout_policy','') == 'edits':
            self.sendMailToSubscribers(
                'This page was deleted.\n',
                REQUEST=REQUEST,
                subjectSuffix='',
                subject='(deleted)')
        if REQUEST:
            # redirect to first existing parent or front page
            destpage = ''
            for p in self.parents:
                if hasattr(self.folder(),p):
                    destpage = p
                    break
            REQUEST.RESPONSE.redirect(self.wiki_url()+'/'+quote(destpage))

    def _recycle(self, REQUEST=None):
        """
        move this page to the recycle_bin subfolder, creating it if necessary.
        """
        # create recycle_bin folder if needed
        f = self.folder()
        if not hasattr(f,'recycle_bin'):
            f.manage_addFolder('recycle_bin', 'deleted wiki pages')
        # & move page there
        id=self.id()
        cb=f.manage_cutObjects(id)

        # update catalog if present
        # don't let manage_afterAdd catalog the new location..
        save = ZWikiPage.manage_afterAdd # not self!
        ZWikiPage.manage_afterAdd = lambda self,item,container: None
        f.recycle_bin.manage_pasteObjects(cb)
        ZWikiPage.manage_afterAdd = save

    security.declareProtected(Permissions.Rename, 'rename')
    def rename(self,pagename,leaveplaceholder=0,updatebacklinks=0,
               sendmail=1,REQUEST=None):
        """
        Rename this page, if permissions allow, with optional fixups:

        - leave a placeholder page

        - update links throughout the wiki. Warning, this may not be 100%
        reliable. It replaces all occurrences of the old page name
        beginning and ending with a word boundary. When changing between a
        wikiname and freeform name, it should do the right thing with
        brackets. It won't change a fuzzy freeform name though.

        - if called with the existing name, ensures that id conforms to
        canonicalId(title).
        """
        # anything to do ?
        oldname, oldid = self.title_or_id(), self.getId()
        newname, newid = pagename, self.canonicalIdFrom(pagename)
        if not newname or (newname == oldname and newid == oldid):
            return 
        # freeform links should have brackets
        if self.isWikiName(oldname): oldlink = oldname
        else: oldlink = '[%s]' % oldname
        if self.isWikiName(newname): newlink = newname
        else: newlink = '[%s]' % newname

        # check permission
        # XXX rename permission is also declared above.. permission
        # cleanup due here
        if not (self._checkPermission(Permissions.Rename, self) and
                self.requestHasSomeId(REQUEST)):
            raise 'Unauthorized', (
                _('You are not authorized to rename this ZWiki Page.'))

        # set title first, updating old pages with an empty title and
        # covering all other cases (including renaming a CMF-created page)
        self.title = newname

        # has the page name changed ?
        if newname != oldname:
            DLOG('renaming %s to %s' % (oldlink,newlink))
            # first, update our entry in the parents list of any children
            # any later problems will roll all this back (go zope!)
            # XXX poor caching
            for id in self.offspringIdsAsList(REQUEST):
                child = getattr(self.folder(),id)
                try: child.parents.remove(oldname)
                except ValueError: pass
                child.parents.append(newname)
                child.index_object()
            # reindex, unless we're going to be doing it later
            if newid == oldid:
                self.index_object()
            # update links to our old name on other pages
            if updatebacklinks:
                for p in self.backlinksFor(oldname):
                    # regexps may fail on large pages (IssueNo0395), carry on
                    # poor caching
                    try: p.getObject()._replaceLinks(oldlink,newlink,REQUEST)
                    except RuntimeError:
                        DLOG('failed to update links during rename')

        # has the page id changed ?
        if newid != oldid:
            # rename, but preserve timestamps - means we have to index twice
            creation_time,creator,creator_ip = \
              self.creation_time,self.creator,self.creator_ip
            self.folder().manage_renameObject(oldid,newid,REQUEST)
            self = getattr(self.folder(),newid) # XXX needed ? don't think so
            self.creation_time,self.creator,self.creator_ip = \
              creation_time,creator,creator_ip
            self.index_object()
            # our URL has changed, so leave a placeholder page
            if leaveplaceholder:
                try:
                    # XXX unicode problem for UnixMailbox
                    #self.create(oldid,_("This page was renamed to %s.\n") % (newlink))
                    self.create(oldid,"This page was renamed to %s.\n" % (newlink),
                                sendmail=0)
                except BadRequestException:
                    # special case: we'll end up here when first saving a
                    # page that was created via the CMF/Plone content
                    # management interface, but we won't be able to save a
                    # placeholder page since the canonical ID hasn't really
                    # changed
                    pass
            # update links to our old id on other pages
            if updatebacklinks:
                for p in self.backlinksFor(oldid):
                    # regexps may fail on large pages (IssueNo0395), carry on
                    # poor caching
                    try: p.getObject()._replaceLinks(oldid,newid,REQUEST)
                    except RuntimeError:
                        DLOG('failed to update links during rename')

        # mail out rename notification
        if (getattr(self.folder(),'mailout_policy','') == 'edits' and
            sendmail and newname != oldname):
            self.sendMailToSubscribers(
                'This page was renamed from %s to %s.\n'%(oldlink,newlink),
                REQUEST=REQUEST,
                subjectSuffix='',
                subject='(renamed)')

        # redirect to new url
        if REQUEST:
            REQUEST.RESPONSE.redirect(self.page_url())

    def isWikiName(self,name):
        """Is name a WikiName ?"""
        return re.match('^%s$' % wikiname,name) is not None

    def _replaceLinks(self,oldlink,newlink,REQUEST=None):
        """
        Replace occurrences of oldlink with newlink in my text.

        oldlink will include brackets if it's a freeform link.
        This tries not to do too much damage, but is pretty dumb.
        Maybe it should use the pre-linking information.
        """
        if self.isWikiName(oldlink): oldpat = r'\b%s\b' % oldlink
        else: oldpat = r'%s' % re.escape(oldlink)
        newtext = re.sub(oldpat, newlink, self.read())
        self.edit(text=newtext,REQUEST=REQUEST,log='links updated after rename')

    def _handleFileUpload(self,REQUEST,log=''):
        # is there a file upload ?
        if (REQUEST and
            hasattr(REQUEST,'file') and
            hasattr(REQUEST.file,'filename') and
            REQUEST.file.filename):     # XXX do something

            # figure out the upload destination
            if hasattr(self,'uploads'):
                uploaddir = self.uploads
            else:
                uploaddir = self.folder()

            # do we have permission ?
            if not (self._checkPermission(Permissions.Upload,uploaddir)):# or
                    #self._checkPermission(Permissions.UploadSmallFiles,
                    #                self.folder())):
                raise 'Unauthorized', (
                    _('You are not authorized to upload files here.'))
            if not (self._checkPermission(Permissions.Edit, self) or
                    self._checkPermission(Permissions.Append, self)):
                raise 'Unauthorized', (
                    _('You are not authorized to add a link on this ZWiki Page.'))
            # can we check file's size ?
            # yes! len(REQUEST.file.read()), apparently
            #if (len(REQUEST.file.read()) > LARGE_FILE_SIZE and
            #    not self._checkPermission(Permissions.Upload,
            #                        uploaddir)):
            #    raise 'Unauthorized', (
            #        _("""You are not authorized to add files larger than
            #        %s here.""" % (LARGE_FILE_SIZE)))

            # create the File or Image object
            file_id, content_type, size = \
                    self._createFileOrImage(REQUEST.file,
                                            title=REQUEST.get('title', ''),
                                            REQUEST=REQUEST)
            if file_id:
                # link it on the page and finish up
                self._addFileLink(file_id, content_type, size, REQUEST)
                self._setLastLog(log)
                self.index_object()
            else:
                # failed to create - give up (what about an error)
                pass

    def _createFileOrImage(self,file,title='',REQUEST=None,parent=None):
        # based on WikiForNow which was based on
        # OFS/Image.py:File:manage_addFile
        """
        Add a new File or Image object, depending on file's filename
        suffix. Returns a tuple containing the new id, content type &
        size, or (None,None,None).
        """
        # set id & title from filename
        title=str(title)
        id, title = OFS.Image.cookId('', title, file)
        if not id:
            return None, None, None

        # find out where to store files - in an 'uploads'
        # subfolder if defined, otherwise in the wiki folder
        if (hasattr(self.folder().aq_base,'uploads') and
            self.folder().uploads.isPrincipiaFolderish):
            folder = self.folder().uploads
        else:
            folder = parent or self.folder() # see create()

        if hasattr(folder,id) and folder[id].meta_type in ('File','Image'):
            pass
        else:
            # First, we create the file or image object without data
            if guess_content_type(file.filename)[0][0:5] == 'image':
                folder._setObject(id, OFS.Image.Image(id,title,''))
            else:
                folder._setObject(id, OFS.Image.File(id,title,''))

        # Now we "upload" the data.  By doing this in two steps, we
        # can use a database trick to make the upload more efficient.
        folder._getOb(id).manage_upload(file)

        return id, folder._getOb(id).content_type, folder._getOb(id).getSize()

    def _addFileLink(self, file_id, content_type, size, REQUEST):
        """
        Add a link to the specified file at the end of this page,
        unless a link already exists.
        If the file is an image and not too big, inline it instead.
        """
        if re.search(r'(src|href)="%s"' % file_id,self.text()): return

        if hasattr(self,'uploads'):
            filepath = 'uploads/'
        else:
            filepath = ''
        if content_type[0:5] == 'image' and \
           not (hasattr(REQUEST,'dontinline') and REQUEST.dontinline) and \
           size <= LARGE_FILE_SIZE :
            linktxt = '\n\n<img src="%s%s" />\n' % (filepath,file_id)
        else:
            linktxt = '\n\n<a href="%s%s">%s</a>\n' % (filepath,file_id,file_id)
        self._setText(self.read()+linktxt,REQUEST)
        self._setLastEditor(REQUEST)

    def _setOwnership(self, REQUEST=None):
        """
        set up the zope ownership of a new page appropriately

        depends on whether we are using regulations or not
        """
        if not self.usingRegulations():
            # To help control executable content, make sure the new page
            # acquires it's owner from the parent folder.
            self._deleteOwnershipAfterAdd()
        else:
            self._setOwnerRole(REQUEST)
            
    def _setText(self, text='', REQUEST=None):
        """
        Change the page text, after cleanups, and do pre-rendering.

        Also insert purple number NIDs if appropriate.
        """
        self.raw = self._cleanupText(text)
        if self.usingPurpleNumbers():
            self.addPurpleNumbers()
        self.preRender(clear_cache=1)
        # re-cook DTML's cached parse data if necessary
        # will prevent edit if DTML can't parse.. hopefully no auth trouble
        if self.dtmlAllowed() and self.hasDynamicContent(): self.cook()
        # try running the DTML, to prevent edits if DTML can't execute
        # got authorization problems, commit didn't help..
        #if self.supportsDtml() and self.dtmlAllowed():
        #    #get_transaction().commit()
        #    DTMLDocument.__call__(self,self,REQUEST,REQUEST.RESPONSE)

    def _cleanupText(self, t):
        """do some cleanup of a page's new text
        """
        # strip any browser-appended ^M's
        t = re.sub('\r\n', '\n', t)

        # convert international characters to HTML entities for safekeeping
        #for c,e in intl_char_entities:
        #    t = re.sub(c, e, t)
        # assume today's browsers will not harm these.. if this turns out
        # to be false, do some smarter checking here

        # here's the place to strip out any disallowed html/scripting elements
        # XXX there are updates for this somewhere on zwiki.org
        if DISABLE_JAVASCRIPT:
            t = re.sub(javascriptexpr,r'&lt;disabled \1&gt;',t)

        # strip out HTML document header/footer if added
        # XXX these can be expensive, for now just skip if there's a problem
        try:
            t = re.sub(htmlheaderexpr,'',t)
            t = re.sub(htmlfooterexpr,'',t)
        except RuntimeError:
            pass

        return t

    def _setLastEditor(self, REQUEST=None):
        """
        record my last_editor & last_editor_ip
        """
        if REQUEST:
            self.last_editor_ip = REQUEST.REMOTE_ADDR
            self.last_editor = self.zwiki_username_or_ip(REQUEST)
        else:
            # this has been fiddled with before
            # if we have no REQUEST, at least update last editor
            self.last_editor_ip = ''
            self.last_editor = ''
        self.last_edit_time = DateTime(time.time()).ISO()

    def _setCreator(self, REQUEST=None):
        """
        record my creator, creator_ip & creation_time
        """
        self.creation_time = DateTime(time.time()).ISO()
        if REQUEST:
            self.creator_ip = REQUEST.REMOTE_ADDR
            self.creator = self.zwiki_username_or_ip(REQUEST)
        else:
            self.creator_ip = ''
            self.creator = ''

    security.declareProtected(Permissions.View, 'checkEditConflict')
    def checkEditConflict(self, timeStamp, REQUEST):
        """
        Warn if this edit would be in conflict with another.

        Edit conflict checking based on timestamps -
        
        things to consider: what if
        - we are behind a proxy so all ip's are the same ?
        - several people use the same cookie-based username ?
        - people use the same cookie-name as an existing member name ?
        - no-one is using usernames ?

        strategies:
        0. no conflict checking

        1. strict - require a matching timestamp. Safest but obstructs a
        user trying to backtrack & re-edit. This was the behaviour of
        early zwiki versions.

        2. semi-careful - record username & ip address with the timestamp,
        require a matching timestamp or matching non-anonymous username
        and ip.  There will be no conflict checking amongst users with the
        same username (authenticated or cookie) connecting via proxy.
        Anonymous users will experience strict checking until they
        configure a username.

        3. relaxed - require a matching timestamp or a matching, possibly
        anonymous, username and ip. There will be no conflict checking
        amongst anonymous users connecting via proxy. This is the current
        behaviour.
        """
        username = self.zwiki_username_or_ip()
        if (timeStamp is not None and
            timeStamp != self.timeStamp() and
            (not hasattr(self,'last_editor') or
             not hasattr(self,'last_editor_ip') or
             username != self.last_editor or
             REQUEST.REMOTE_ADDR != self.last_editor_ip)):
            return 1
        else:
            return 0

    security.declareProtected(Permissions.View, 'timeStamp')
    def timeStamp(self):
        return str(self._p_mtime)
    
    security.declareProtected(Permissions.FTP, 'manage_FTPget')
    def manage_FTPget(self):
        """Get source for FTP download.
        """
        #candidates = list(self.allowedPageTypes())
        #types = "%s (alternatives:" % self.page_type
        #if self.page_type in candidates:
        #    candidates.remove(self.page_type)
        #for i in candidates:
        #    types = types + " %s" % i
        #types = types + ")"
        types = "%s" % self.page_type
        return "Wiki-Safetybelt: %s\nType: %s\nLog: \n\n%s" % (
            self.timeStamp(), types, self.read())

    security.declareProtected(Permissions.Edit, 'PUT')
    def PUT(self, REQUEST, RESPONSE):
        """Handle HTTP/FTP/WebDav PUT requests."""
        self.dav__init(REQUEST, RESPONSE)
        self.dav__simpleifhandler(REQUEST, RESPONSE, refresh=1)
        body=REQUEST.get('BODY', '')
        self._validateProxy(REQUEST)

        headers, body = parseHeadersBody(body)
        log = string.strip(headers.get('Log', headers.get('log', ''))) or None
        type = (string.strip(headers.get('Type', headers.get('type', '')))
                or None)
        if type is not None:
            type = string.split(type)[0]
            #if type not in self.allowedPageTypes():
            #    # Silently ignore it.
            #    type = None
        timestamp = string.strip(headers.get('Wiki-Safetybelt', '')) or None
        if timestamp and self.checkEditConflict(timestamp, REQUEST):
            RESPONSE.setStatus(423) # Resource Locked
            return RESPONSE

        #self._setText(body)
        #self._setLastEditor(REQUEST)
        #self.index_object()
        #RESPONSE.setStatus(204)
        #return RESPONSE
        try:
            self.edit(text=body, type=type, timeStamp=timestamp,
                      REQUEST=REQUEST, log=log, check_conflict=0)
        except 'Unauthorized':
            RESPONSE.setStatus(401)
            return RESPONSE
        RESPONSE.setStatus(204)
        return RESPONSE

    security.declareProtected(Permissions.Edit, 'manage_edit')
    def manage_edit(self, data, title, REQUEST=None):
        """Do standard manage_edit kind of stuff, using our edit."""
        #self.edit(text=data, title=title, REQUEST=REQUEST, check_conflict=0)
        #I think we need to bypass edit to provide correct permissions
        self.title = title
        self._setText(data,REQUEST)
        self._setLastEditor(REQUEST)
        self.reindex_object()
        if REQUEST:
            message="Content changed."
            return self.manage_main(self,REQUEST,manage_tabs_message=message)

    ######################################################################
    # miscellaneous

    def size(self):
        """
        Give the size of this page's text.
        """
        return len(self.text())

    def cachedSize(self):
        """
        Give the total size of this page's text plus cached render data.
        """
        return self.size() + len(self.preRendered()) + self.cachedDtmlSize()

    def cachedDtmlSize(self):
        """
        Estimate the size of this page's cached DTML parse data.
        """
        strings = flattenDtmlParse(getattr(self,'_v_blocks',''))
        strings = filter(lambda x:type(x)==type(''), strings)
        return len(join(strings,''))

    def summary(self,size=100):
        """
        Give a short summary of this page's content.
        """
        return html_quote(self.text()[:size])

    def metadataFor(self,page):
        """
        Make a catalog-style "brain" object for page's principal meta data.

        To be used as a substitute when there is no catalog.
        """
        class PageBrain(SimpleItem):
            def __init__(self,obj): self._obj = obj
            def getObject(self): return self._obj
        b = PageBrain(page)
        for attr in [
            'id',
            'Title',       # preferred page name attribute
            'title_or_id', # but legacy templates use this
            'size',
            'cachedSize',
            'issueColour',
            #'canonicalId',
            'parents',
            'links',
            'page_url',    # legacy backlinks templates
            #'linkTitle',   # expect these
            ]:
            setattr(b,attr,absattr(getattr(page,attr)))
            #setattr(b,attr+'__roles__',None)
        b.linkTitle = '' #XXX just leave this blank for now so tests pass
        return b

    # XXX don't know how to make real brains work
    #def metadataFor(self,page):
    #    """
    #    Return a "brain" object containing page's principal meta data.
    #
    #    Catalog-style. We generate these ourselves when there is no catalog.
    #    """
    #    class PageBrain(AbstractCatalogBrain):
    #        __record_schema__ = {
    #            'id':1,
    #            'title_or_id':2,
    #            'size':3,
    #            'cachedSize':4,
    #            'issueColour':5,
    #            'parents':6,
    #            'links':7,
    #            #'canonicalId':,
    #            #'Title':,      # need this to mimic default CMF catalog ?
    #            }
    # from Catalog.py:
    #    scopy['data_record_id_']=len(self.schema.keys())
    #    scopy['data_record_score_']=len(self.schema.keys())+1
    #    scopy['data_record_normalized_score_']=len(self.schema.keys())+2
    #
    #    return PageBrain(page).__of__(self.folder())
    #
    #def __getitem__(self,key):
    #    """
    #    Make self['anyattribute'] work, for setting up PageBrains ?
    #
    #    XXX Hopefully not a security problem.
    #
    #    Not enough.
    #    """
    #    return getattr(self,key)

    security.declareProtected(Permissions.View, 'isZwikiPage')
    def isZwikiPage(self,object):
        return getattr(object,'meta_type',None) == self.meta_type

    security.declareProtected(Permissions.View, 'zwiki_version')
    def zwiki_version(self):
        """
        Return the zwiki product version.
        """
        return __version__

    # expose these darn things for dtml programmers once and for all!
    # XXX safe ?
    security.declareProtected(Permissions.View, 'htmlquote')
    def htmlquote(self, text):
        return html_quote(text)

    security.declareProtected(Permissions.View, 'htmlunquote')
    def htmlunquote(self, text):
        return html_unquote(text)

    security.declareProtected(Permissions.View, 'urlquote')
    def urlquote(self, text):
        return quote(text)

    security.declareProtected(Permissions.View, 'urlunquote')
    def urlunquote(self, text):
        return unquote(text)

    security.declareProtected(Permissions.View, 'zwiki_username_or_ip')
    def zwiki_username_or_ip(self, REQUEST=None):
        """
        search REQUEST for an authenticated member or a zwiki_username
        cookie or a username passed in by mailin.py

        XXX added REQUEST arg at one point when sending mail
        notification in append() was troublesome - still needed ?
        refactor
        """
        username = None
        REQUEST = REQUEST or getattr(self,'REQUEST',None)
        if REQUEST:
            username = REQUEST.get('MAILIN_USERNAME',None)
            if not username:
                user = REQUEST.get('AUTHENTICATED_USER')
                if user:
                    username = user.getUserName()
                if not username or username == str(user.acl_users._nobody):
                    username = REQUEST.cookies.get('zwiki_username',None)
                    if not username:
                        username = REQUEST.REMOTE_ADDR
        return username or ''

    security.declareProtected(Permissions.View, 'requestHasSomeId')
    def requestHasSomeId(self,REQUEST):
        """
        Check REQUEST has either a non-anonymous user or a username cookie.
        """
        username = self.zwiki_username_or_ip(REQUEST)
        if (username and username != REQUEST.REMOTE_ADDR):
            return 1
        else:
            return 0

    security.declareProtected(Permissions.View, 'text')
    def text(self, REQUEST=None, RESPONSE=None):
        # see also backwards compatibility section
        """
        return this page's raw text
        (a permission-free version of document_src)
        also fiddle the mime type for web browsing
        """
        if RESPONSE is not None:
            RESPONSE.setHeader('Content-Type', 'text/plain')
            #RESPONSE.setHeader('Last-Modified', rfc1123_date(self._p_mtime))
        return self.read()

    # cf _setText, IssueNo0157
    _old_read = DTMLDocument.read
    security.declareProtected(Permissions.View, 'read')
    def read(self):
        return re.sub('<!--antidecapitationkludge-->\n\n?','',
                      self._old_read())

    def __repr__(self):
        return ("<%s %s at 0x%s>"
                % (self.__class__.__name__, `self.id()`, hex(id(self))[2:]))

    security.declareProtected(Permissions.View, 'page_url')
    def page_url(self):
        """return the url path for this wiki page"""
        return self.wiki_url() + '/' + quote(self.id())

    security.declareProtected(Permissions.View, 'wiki_url')
    def wiki_url(self):
        """return the base url path for this wiki"""
        try: return self.folder().absolute_url()
        except KeyError,AttributeError: return '' # for debugging/testing

    security.declareProtected(Permissions.View, 'creationTime')
    def creationTime(self):
        """
        Return our creation time as a DateTime, guessing if necessary
        """
        try: return DateTime(self.creation_time)
        except (AttributeError,DateTime.SyntaxError):
            return DateTime('2001/1/1')

    security.declareProtected(Permissions.View, 'lastEditTime')
    def lastEditTime(self):
        """
        Return our last edit time as a DateTime, guessing if necessary
        """
        try: return DateTime(self.last_edit_time)
        except (AttributeError,DateTime.SyntaxError):
            return DateTime('2001/1/1')

    security.declareProtected(Permissions.View, 'folder')
    def folder(self):
        """
        return this page's containing folder

        We used to use self.aq_parent everywhere, now
        self.aq_inner.aq_parent to ignore acquisition paths.
        Work for pages without a proper acquisition wrapper too.
        """
        return getattr(getattr(self,'aq_inner',self),'aq_parent',None)

    security.declareProtected(Permissions.View, 'age')
    def age(self):
        """
        return a string describing the approximate age of this page
        """
        return self.asAgeString(self.creation_time)

    security.declareProtected(Permissions.View, 'lastEditInterval')
    def ageInDays(self):
        """
        return the number of days since page creation
        """
        return int(self.getPhysicalRoot().ZopeTime() -
                   self.creationTime())

    security.declareProtected(Permissions.View, 'lastEditInterval')
    def lastEditInterval(self):
        """
        return a string describing the approximate interval since last edit
        """
        return self.asAgeString(self.last_edit_time)

    security.declareProtected(Permissions.View, 'lastEditInterval')
    def lastEditIntervalInDays(self):
        """
        return the number of days since last edit
        """
        return int(self.getPhysicalRoot().ZopeTime() -
                   self.lastEditTime())

    security.declareProtected(Permissions.View, 'asAgeString')
    def asAgeString(self,time):
        """
        return a string describing the approximate elapsed period since time

        time may be a DateTime or suitable string. Returns a blank string
        if there was a problem. Based on the dtml version in ZwikiTracker.
        """
        if not time:
            return 'some time'
        if type(time) is StringType:
            time = DateTime(time)
        # didn't work on a page in CMF, perhaps due to skin acquisition magic
        #elapsed = self.ZopeTime() - time
        elapsed = self.getPhysicalRoot().ZopeTime() - time
        hourfactor=0.041666666666666664
        minutefactor=0.00069444444444444447
        secondsfactor=1.1574074074074073e-05
        days=int(math.floor(elapsed))
        weeks=days/7
        months=days/30
        years=days/365
        hours=int(math.floor((elapsed-days)/hourfactor))
        minutes=int(math.floor((elapsed-days-hourfactor*hours)/minutefactor))
        seconds=int(round((
            elapsed-days-hourfactor*hours-minutefactor*minutes)/secondsfactor))
        if years:
            s = "%d year%s" % (years, years > 1 and 's' or '')
        elif months:
            s = "%d month%s" % (months, months > 1 and 's' or '')
        elif weeks:
            s = "%d week%s" % (weeks, weeks > 1 and 's' or '')
        elif days:
            s = "%d day%s" % (days, days > 1 and 's' or '')
        elif hours:
            s = "%d hour%s" % (hours, hours > 1 and 's' or '')
        elif minutes:
            s = "%d minute%s" % (minutes, minutes > 1 and 's' or '')
        else:
            s = "%d second%s" % (seconds, seconds > 1 and 's' or '')
        return s

    security.declareProtected(Permissions.View, 'linkTitle')
    def linkTitle(self,prettyprint=0):
        """
        return a suitable value for the title attribute of links to this page

        with prettyprint=1, format it for use in the standard header.
        """
        return self.linkTitleFrom(self.last_edit_time,
                                  self.last_editor,
                                  prettyprint=prettyprint)

    # please clean me up
    security.declareProtected(Permissions.View, 'linkTitleFrom')
    def linkTitleFrom(self,last_edit_time=None,last_editor=None,prettyprint=0):
        """
        make a link title string from these last_edit_time and editor strings
        
        with prettyprint=1, format it for use in the standard header.
        """
        interval = self.asAgeString(last_edit_time)
        if not prettyprint:
            s = "last edited %s ago" % (interval)
        else:
            try:
                assert self.REQUEST.AUTHENTICATED_USER.has_permission(
                    'View History',self)
                #XXX do timezone conversion ?
                lastlog = self.lastlog()
                if lastlog: lastlog = ' ('+lastlog+')'
                s = '<b><u>l</u></b>ast edited <a href="%s/diff" title="show last edit%s" accesskey="l">%s</a> ago' % \
                    (self.page_url(), lastlog, interval)
            except:
                s = 'last edited %s ago' % (interval)
        if (last_editor and
            not re.match(r'^[0-9\.\s]*$',last_editor)):
            # escape some things that might cause trouble in an attribute
            editor = re.sub(r'"',r'',last_editor)
            if not prettyprint:
                s = s + " by %s" % (editor)
            else:
                s = s + " by <b>%s</b>" % (editor)
        return s
    
    ######################################################################
    # backwards compatibility

    security.declarePublic('upgradeAll') # check folder permission at runtime
    def upgradeAll(self,pre_render=1,upgrade_messages=0,REQUEST=None):
        """
        Clear cache, upgrade and pre-render all pages

        Normally pages are upgraded/pre-rendered as needed.  An
        administrator may want to call this, particularly after a zwiki
        upgrade, to minimize later delays and to ensure all pages have
        been rendered by the latest code.

        Requires 'Manage properties' permission on the folder.
        Commit every so often an attempt to avoid memory/conflict errors.
        Has problems doing a complete run in large wikis, or when other
        page accesses are going on ?
        """
        if not self._checkPermission(Permissions.manage_properties,
                                     self.folder()):
            raise 'Unauthorized', (
             _('You are not authorized to upgrade all pages.') + \
             _('(folder -> Manage properties)'))
        try: pre_render = int(pre_render)
        except: pre_render = 0
        if pre_render:
            DLOG('upgrading and prerendering all pages:')
        else:
            DLOG('upgrading all pages:')
        n = 0
        # poor caching (ok)
        for p in self.pageObjects():
            n = n + 1
            p.upgrade(REQUEST)
            p.upgradeId(REQUEST)
            if pre_render:
                p.preRender(clear_cache=1)
            if upgrade_messages:
                p.upgradeMessages(REQUEST)
            DLOG('upgraded page #%d %s'%(n,p.id()))
            if n % 10 == 0:
                DLOG('committing')
                get_transaction().commit()
            # last pages will get committed as this request ends
        DLOG('%d pages processed' % n)

    #security.declarePublic('upgradeId')
    security.declareProtected(Permissions.View, 'upgradeId')
    def upgradeId(self,REQUEST=None):
        """
        Make sure a page's id conforms with it's title, renaming as needed.

        Does not leave a placeholder, so may break incoming links.
        Presently too slow for auto-upgrade, so let people call this
        directly or via upgradeAll (good luck :( )

        updatebacklinks=1 is used even though it's slow, because it's less
        work than fixing up links by hand afterward.

        With legacy pages (not new ones), it may happen that there's a
        clash between two similarly-named pages mapping to the same
        canonical id. In this case we just log the error and move on.
        """
        id, cid = self.getId(), self.canonicalId()
        if id != cid:
            oldtitle = title = self.title_or_id()
            # as a special case, preserve tracker issue numbers in the title
            m = re.match(r'IssueNo[0-9]+$',id)
            if m:
                title = '%s %s' % (m.group(),self.title)
            try:
                self.rename(title,updatebacklinks=1,sendmail=0,REQUEST=REQUEST)
            except CopyError:
                DLOG('failed to rename "%s" (%s) to "%s" (%s) - id clash ?' \
                     % (oldtitle,id,title,self.canonicalIdFrom(title)))

    #security.declarePublic('upgrade')
    security.declareProtected(Permissions.View, 'upgrade')
    def upgrade(self,REQUEST=None):
        """
        Upgrade an old page instance (and possibly the parent folder).

        Called as needed, ie at view time (set AUTO_UPGRADE=0 in
        Default.py to prevent this).  You could also call this on every
        page in your wiki to do a batch upgrade. Affects
        bobobase_modification_time. If you later downgrade zwiki, the
        upgraded pages may not work so well.
        """
        # Note that the objects don't get very far unpickling, some
        # by-hand adjustment via command-line interaction is necessary
        # to get them over the transition, sigh. --ken
        # not sure what this means --SM

        # What happens in the zodb when class definitions change ? I think
        # all instances in the zodb conform to the new class shape
        # immediately on refresh/restart, but what happens to
        # (a) old _properties lists ? not sure, assume they remain in
        # unaffected and we need to add the new properties
        # and (b) old properties & attributes no longer in the class
        # definition ?  I think these lurk around, and we need to delete
        # them.

        changed = 0

        # As of 0.17, page ids are always canonicalIdFrom(title); we'll
        # rename to conform with this where necessary
        # too slow!
        # changed = self.upgradeId()

        # fix up attributes first, then properties
        # don't acquire while doing this
        realself = self
        self = self.aq_base

        # migrate a WikiForNow _st_data attribute
        if hasattr(self, '_st_data'):
            self.raw = self._st_data
            del self._st_data
            changed = 1

        # upgrade old page types
        if self.page_type in self.PAGE_TYPE_UPGRADES.keys():
            self.page_type = self.PAGE_TYPE_UPGRADES[self.page_type]
            # clear render cache; don't bother prerendering just now
            self.clearCache()
            changed = 1

        # Pre-0.9.10, creation_time has been a string in custom format and
        # last_edit_time has been a DateTime. Now both are kept as
        # ISO-format strings. Might not be strictly necessary to upgrade
        # in all cases.. will cause a lot of bobobase_mod_time
        # updates.. do it anyway.
        if not self.last_edit_time:
            self.last_edit_time = self.bobobase_modification_time().ISO()
            changed = 1
        elif type(self.last_edit_time) is not StringType:
            self.last_edit_time = self.last_edit_time.ISO()
            changed = 1
        elif len(self.last_edit_time) != 19:
            try: 
                self.last_edit_time = DateTime(self.last_edit_time).ISO()
                changed = 1
            except DateTime.SyntaxError:
                # can't convert to ISO, just leave it be
                pass

        # If no creation_time, just leave it blank for now.
        # we shouldn't find DateTimes here, but check anyway
        if not self.creation_time:
            pass
        elif type(self.creation_time) is not StringType:
            self.creation_time = self.creation_time.ISO()
            changed = 1
        elif len(self.creation_time) != 19:
            try: 
                self.creation_time = DateTime(self.creation_time).ISO()
                changed = 1
            except DateTime.SyntaxError:
                # can't convert to ISO, just leave it be
                pass

        # _wikilinks, _links and _prelinked are no longer used
        for a in (
            '_wikilinks',
            '_links',
            '_prelinked',
            ):
            #if hasattr(self.aq_base,a): #XXX why doesn't this work
            if hasattr(self,a):
                delattr(self,a)
                self.clearCache()
                changed = 1 

        # update _properties
        # keep in sync with _properties above. Better if we used that as
        # the template (efficiently)
        oldprops = { # not implemented
            'page_type'     :{'id':'page_type','type':'string'},
            }
        newprops = {
            #'page_type'     :{'id':'page_type','type':'selection','mode': 'w',
            #                  'select_variable': 'ZWIKI_PAGE_TYPES'},
            'creator'       :{'id':'creator','type':'string','mode':'r'},
            'creator_ip'    :{'id':'creator_ip','type':'string','mode':'r'},
            'creation_time' :{'id':'creation_time','type':'string','mode':'r'},
            'last_editor'   :{'id':'last_editor','type':'string','mode':'r'},
            'last_editor_ip':{'id':'last_editor_ip','type':'string','mode':'r'},
            'last_edit_time':{'id':'creation_time','type':'string','mode':'r'},
            'last_log'      :{'id':'last_log', 'type': 'string', 'mode': 'r'},
            'NOT_CATALOGED' :{'id':'NOT_CATALOGED', 'type': 'boolean', 'mode': 'w'},
            }
        props = map(lambda x:x['id'], self._properties)
        for p in oldprops.keys():
            if p in props: # and oldprops[p]['type'] != blah blah blah :
                pass
                #ack!
                #self._properties = filter(lambda x:x['id'] != p,
                #                          self._properties)
                #changed = 1
                # XXX this does work in python 1.5 surely.. what's the
                # problem ?
        for p in newprops.keys():
            if not p in props:
                self._properties = self._properties + (newprops[p],)
                changed = 1

        # install issue properties if needed, ie if this page is being
        # viewed as an issue for the first time
        # could do this in isIssue instead
        if (self.isIssue() and not 'severity' in props):
            realself.manage_addProperty('category','issue_categories','selection')
            realself.manage_addProperty('severity','issue_severities','selection')
            realself.manage_addProperty('status','issue_statuses','selection')
            realself.severity = 'normal'
            changed = 1

        if changed:
            # bobobase_modification_time changed - put in a dummy user so
            # it's clear this was not an edit
            # no - you should be looking at last_edit_times, in which case
            # you don't want to see last_editor change for this.
            #self.last_editor_ip = ''
            #self.last_editor = 'UpGrade'
            # do a commit now so the current render will have the
            # correct bobobase_modification_time for display (many
            # headers/footers still show it)
            get_transaction().commit()
            # and log it
            DLOG('upgraded '+self.id())

        # finally, MailSupport does a bit more (merge here ?)
        realself._upgradeSubscribers()

    # some CMF compatibility methods for standard ZWikiPage
    SearchableText = text
    view = __call__
            
    # a few old API methods to help keep legacy DTML working
    wiki_page_url = page_url
    wiki_base_url = wiki_url
    editTimestamp = timeStamp
    checkEditTimeStamp = checkEditConflict
    src = text

Globals.InitializeClass(ZWikiPage)

# ZMI page creation form
manage_addZWikiPageForm = DTMLFile('dtml/zwikiPageAdd', globals())

def manage_addZWikiPage(self, id, title='', file='', REQUEST=None,
                        submit=None):
    """
    Add a ZWiki Page object with the contents of file.

    Usually zwiki pages are created by clicking on ? (via create); this
    allows them to be added in the standard ZMI way. These should give
    mostly similar results; refactor the two methods together if possible.
    """
    # create page and initialize in proper order, as in create.
    p = ZWikiPage(source_string='', __name__=id)
    newid = self._setObject(id,p)
    p = getattr(self,newid)
    p.title=title
    p._setCreator(REQUEST)
    p._setLastEditor(REQUEST)
    p._setOwnership(REQUEST)
    p.page_type = type or self.defaultPageType()
    if getattr(self,'allowed_page_types',None):
        p.page_type = self.allowed_page_types[0]
    else:
        p.page_type = ZWikiPage.DEFAULT_PAGE_TYPE
    text = file
    if type(text) is not StringType: text=text.read()
    p._setText(text or '',REQUEST)
    p.index_object()
    if p.usingRegulations():
        p.setRegulations(REQUEST,new=1)
    if REQUEST is not None:
        try: u=self.DestinationURL()
        except: u=REQUEST['URL1']
        if submit==" Add and Edit ": u="%s/%s" % (u,quote(id))
        REQUEST.RESPONSE.redirect(u+'/manage_main')
    return ''


#python notes
#
#   "The first line should always be a short, concise summary of the
#object's purpose.  For brevity, it should not explicitly state the
#object's name or type, since these are available by other means (except
#if the name happens to be a verb describing a function's operation).
#This line should begin with a capital letter and end with a period.
#
#   If there are more lines in the documentation string, the second line
#should be blank, visually separating the summary from the rest of the
#description.  The following lines should be one or more paragraphs
#describing the object's calling conventions, its side effects, etc."
#
#   "Data attributes override method attributes with the same name; to
#avoid accidental name conflicts, which may cause hard-to-find bugs in
#large programs, it is wise to use some kind of convention that
#...
#verbs for methods and nouns for data attributes."
