/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#include "base.h"
#include "proc.h"
#include "expr.h"
#include "tree.h"
#include "context.h"
#include "vars.h"
// #include "xmlparse.h" - moved to parser.h
#include "parser.h"

/*
  include direct.h for getcwd()
*/
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#elif defined(WIN32)
#include <direct.h>
#elif defined(__BORLANDC__)
#include <dir.h>
#else
#error "Can't find appropriate include (unistd.h & Co.)"
#endif

/*****************************************************************
R u l e I t e m   methods
*****************************************************************/

RuleItem::RuleItem(XSLElement *arule,double aprio, QName &_name,
                   QName *_mode)
:
rule(arule), priority(aprio), name(_name), mode(_mode)
{
};


//
//  ~RuleItem
//  has to dispose of the mode
//
RuleItem::~RuleItem()
{
    cdelete(mode);
}


/*****************************************************************
R u l e S L i s t   methods
*****************************************************************/

RuleSList::RuleSList(Processor* _proc)
:
SList<RuleItem*>(LIST_SIZE_LARGE), proc(_proc)
{};

RuleSList::~RuleSList()
{
    freeall(FALSE);
}

int RuleSList::compare(int first, int second)
{
    return fcomp((*this)[first] -> priority, (*this)[second] -> priority);
};


XSLElement *RuleSList::findByName(QName &what)
{
    int theNumber = number();
    for (int i=0; i < theNumber; i++)
        if (proc -> cmpQNames((*this)[i] -> name, what))
            return (*this)[i] -> rule;
    return NULL;
}

/*****************************************************************

  DataLineItem

*****************************************************************/

DataLineItem::~DataLineItem()
{
    // the following close should be checked for error!!
    if (_dataline && _dataline -> mode != DLMODE_CLOSED)
        _dataline -> close();
    cdelete(_dataline);
    cdelete(_tree);
}

/*****************************************************************

  DataLinesList

*****************************************************************/

int DataLinesList::findNum(Str &absoluteURI, Bool _isXSL,
    DLAccessMode _mode)
{
    int theNumber = number();
    for (int i = 0; i < theNumber; i++)
    {
        DataLineItem* item = (*this)[i];
        if ((item->_dataline -> fullUri == absoluteURI) &&
            (item->_isXSL == _isXSL) && 
            (item-> _dataline -> mode == _mode))
        return i;
    }
    return -1;
}

Tree *DataLinesList::getTree(Str &absoluteURI, Bool _isXSL, 
    DLAccessMode _mode)
{
    int n;
    if ((n = findNum(absoluteURI, _isXSL, _mode)) != -1)
        return (*this)[n] -> _tree;
    else
        return NULL;
}

eFlag DataLinesList::addLine(DataLine *d, Tree *t, Bool isXSL)
{
    DataLineItem *item = new DataLineItem;
    item -> _dataline = d;
    item -> _tree = t;
    item -> _isXSL = isXSL;
    append(item);
    return OK;
}

/*****************************************************************
*                                                                *
*   P r o c e s s o r   methods                                  *
*                                                                *
*****************************************************************/

// disable the MSVC warning "'this' used in the initializer list"

#ifdef WIN32
#pragma warning( disable : 4355 )
#endif

Processor::Processor() 
: 
situation_( this ), rules ( this ), theEmptyQName( this ), 
theArena(this, PROC_ARENA_SIZE),
dictionary( this, &theArena, PROC_DICTIONARY_LOGSIZE )

#ifdef WIN32
#pragma warning( default : 4355 )
#endif

{
//    pushNewHandler();

    theSchemeHandler = NULL;
    theMessageHandler = NULL;
    theSAXHandler = NULL;
    theMiscHandler = NULL;
    theSchemeUserData = theMessageUserData = 
        theSAXUserData = theMiscUserData = NULL;
    
    instanceData = NULL;
    situation = &situation_;
};

void Processor::prepareForRun()
{
    vars = new VarsList(this);
    input = styleSheet = NULL;
    dictionary.initialize();
    dictionary.insert("", stdPhrases[PHRASE_EMPTY]);
    dictionary.insert("xsl", stdPhrases[PHRASE_XSL]);
    dictionary.insert(theXSLTNamespace, stdPhrases[PHRASE_XSL_NAMESPACE]);
    dictionary.insert(theXMLNamespace, stdPhrases[PHRASE_XML_NAMESPACE]);
    dictionary.insert("*", stdPhrases[PHRASE_STAR]);
}

void Processor::freeNonArgDatalines()
{
    int i = 0;
    while(i < datalines.number())
    {
        if (datalines[i] -> _dataline -> scheme != URI_ARG)
            datalines.freerm(i, FALSE);
        else
        {
            // removing the tree here because the arena gets disposed
            // in cleanupAfterRun(). So only the arg buffers remain.
            cdelete(datalines[i] -> _tree);
            i++;
        }
    }
}

void Processor::cleanupAfterRun()
{
/*
These deletes are now subsumed in "datalines.freeall()":
    cdelete(input);
    cdelete(result);
    cdelete(styleSheet);
*/
    cdelete(vars);
    freeNonArgDatalines();
    if (!situation->isError())
    {
        assert(modes.isEmpty());
        assert(outputters_.isEmpty());
    }
    else
    {
        modes.freeall(FALSE);
        outputters_.freeall(FALSE);
    };
    rules.freeall(FALSE);
    situation->clear();
    // hope this works:
    dictionary.destroy();
    theArena.dispose();
}

Processor::~Processor()
{
//    popNewHandler();
    baseURIMappings.freeall(FALSE);
};

void my_getcwd(DStr &dir)
{
char buf[256];
#if defined(WIN32)
    _getcwd(buf, 256);
#else //if defined(__linux__) || defined(__BORLANDC__) || defined(__unix)
    getcwd(buf,256);
#endif
    dir = buf;
    if (!(dir == (const char*) "/"))
        dir += '/';
}

eFlag Processor::open(char *sheetURI, char *inputURI, char *resultURI)
{
    Str temp;
    DStr theBase;

    my_getcwd(theBase);
    theBase = findBaseURI(Str("file://") + theBase);

    E( readTreeFromURI(styleSheet, temp = sheetURI, theBase, TRUE) );
    E( readTreeFromURI(input, temp = inputURI, theBase, FALSE) );

    E( pushOutputterForURI(temp = resultURI, theBase) );
    E( outputter() -> eventBeginOutput() );

    Log1(situation, L1_EXECUTING, styleSheet -> name);
    double time_was = getMillisecs();
    situation->setCurrFile(styleSheet -> name);

    Context *c = new Context(this);
    c -> set(&(input -> root));

    /*
        in vars, global prebindings go to call level 0,
        global bindings to call level 1
    */
    vars -> startCall();
    E( styleSheet -> root.execute(c) );
    vars -> endCall();

    E( outputter() -> eventEndOutput() );
        
    // report info about the output document to the MiscHandler
    void *miscUserData;
    OutputDefinition *outDef = &(styleSheet -> outputDef);
    MiscHandler *miscHlr = getMiscHandler(&miscUserData);
    if (miscHlr)
        miscHlr -> documentInfo(
            miscUserData,
            this,   // processor
            outDef -> getValueStr(XSLA_MEDIA_TYPE), 
            outDef -> getValueStr(XSLA_ENCODING));

    E( popOutputter() );
    delete c;
    Log1(situation, L1_EXECUTION_DONE, getMillisecsDiff(time_was));
    return OK;
}


/*================================================================
getVarBinding 
    finds the current binding for the variable with the given name.
    The binding is an expression. If none is present, returns NULL.
================================================================*/

Expression *Processor::getVarBinding(QName &which)
{
    return vars -> getBinding(which);
}

/*================================================================
cmpQNames
    compares two given QNames based on the namespace URIs,
    allowing wildcard * as local name of first QName.
================================================================*/

Bool Processor::cmpQNames(const QName& first, const QName &second) const
{
    if (first.local == stdPhrase(PHRASE_STAR))
        return (Bool)(first.prefix == UNDEF_PHRASE || 
            (first.uri == second.uri));
    else
        return (Bool) (first == second);
}

/*================================================================
execute
================================================================*/

eFlag Processor::execute(Vertex *IP, Context *c)
{
    while (!c -> isFinished())
    {
        if (IP)
            E( IP -> execute(c) )
        else
            E( execApply(c) );
        c -> shift();
    };
    delete c;
    return OK;
}

eFlag Processor::execute(VertexList &IPlist, Context *c)
{
    // we may need to remove some variable bindings on each pass
    // through a for loop
    Vertex *theParent = IPlist.number()? IPlist[0] -> parent : NULL;
    XSLElement *theForParent;
    if (theParent && isXSLElement(theParent) && toX(theParent) -> op == XSL_FOR_EACH)
        theForParent = toX(theParent);
    else
        theForParent = NULL;
    
    while (c -> current())
    {
        E( IPlist.execute(c) );
        c -> shift();
        if (theForParent)
            theForParent -> removeBindings();
    };
    delete c;
    return OK;
}

eFlag Processor::execApply(Context *c)
{
    XSLElement *rule;
    E( bestRule(rule, c) );
    if (!rule)
        E( builtinRule(c) )
    else
        E( rule -> execute(c) );
    return OK;
}

/*================================================================
bestRule
    Finds the highest-priority rule that is satisfied by v. 
    Assumes that 'rules' is sorted by priority in reverse order.
    If several rules are satisfied, returns the one that occurs last. 
    If no rule is found, returns NULL.
================================================================*/

eFlag Processor::bestRule(XSLElement *& ret, Context *c)
{
    int i;
    int bestRule = -1,
        rulesNumber = rules.number();
    double bestPrio=0;
    Expression *pattern = NULL;
    QName 
        *currMode = this -> getCurrentMode(),
        *thisMode;
    XSLElement *therule;
    for (i = 0; i < rulesNumber; i++)
    {
        if (
            (bestRule != -1) &&
            (fcomp(rules[i] -> priority, bestPrio) == -1)  //current < best
            )
            break;
        Attribute *a = (therule = rules[i] -> rule) -> atts.find(XSLA_MATCH);
        if (a) 
            pattern = a -> expr;
        else 
            continue;

        thisMode = rules[i] -> mode;
        if ((!thisMode) ^ (!currMode))
            continue;
        if (thisMode && !((*thisMode) == (*currMode)))
            continue;
        // else both thisMode and currMode are NULL which is OK
        if (pattern)
        {
            Bool result;
            E( pattern -> matchesPattern(c, result) );
            if (result)
            {
                bestRule = i;
                bestPrio = rules[i] -> priority;
            }
        };
    };
    if (bestRule == -1) ret = NULL;
    else ret = rules[bestRule] -> rule;
    return OK;
};

eFlag Processor::builtinRule(Context *c)
{
    Vertex *v = c -> current();
    
    switch(v -> vt & VT_BASE)
    {
    case VT_ROOT:
    case VT_ELEMENT:
        //apply-templates
        {
            Expression *e = new Expression(NULL,this, EXF_LOCPATH);
            e -> setLS(
                AXIS_CHILD, 
                EXNODE_NODE);
            Context *newc;
            E( e -> createContext(newc = c) );
            E( execute(NULL, newc) );
            delete e; ///
        }; break;
    case VT_TEXT:
    case VT_ATTRIBUTE:
        //copy contents to output
        {
            DStr temp;
            E( v -> value(temp, c) );
            E( this -> outputter() -> eventData(temp) );
        }; break;
    case VT_PI:
    case VT_COMMENT:
        {};
    };
    return OK;
};

eFlag Processor::insertRule(XSLElement *tmpl)
{
    double prio;
    Attribute *a = tmpl -> atts.find(XSLA_PRIORITY);
    if (!a)
        prio = defaultPriority(tmpl);
    else
    {
        if (a -> cont.toDouble(prio))
            Err(situation, ET_BAD_NUMBER);
    };
    QName q(this), *mode = NULL;
    if (!!(a = tmpl -> atts.find(XSLA_NAME)))
        E( q.setLogical(a -> cont, &(tmpl -> namespaces), FALSE) );
    if (!q.getLocal().isEmpty() && rules.findByName(q))
        Err1(situation, ET_DUPLICATE_RULE_NAME,(char*)(q.getname()));
    if (!!(a = tmpl -> atts.find(XSLA_MODE)))
        E( (mode=new QName(this)) -> setLogical(a -> cont, 
            &(tmpl -> namespaces), FALSE) );
    
    rules.insert(new RuleItem(tmpl,prio,q,mode));
    return OK;
}

double Processor::defaultPriorityLP(Expression *lpath)
{
    assert(lpath && lpath -> functor == EXF_LOCPATH);
    assert(lpath -> args.number());
    if ((lpath -> args.number() > 1) || lpath -> args[0] -> step -> preds.number())
        return .5;
    else
    {
        switch (lpath -> args[0] -> step -> ntype)
        {
        case EXNODE_COMMENT:
        case EXNODE_TEXT:
        case EXNODE_PI:  //// pi('literal') should have priority 0.5
        case EXNODE_NODE:
            return -0.5;
            break;
        case EXNODE_NONE:
            {
                QName &qn = lpath -> args[0] -> step -> ntest;
                if (qn.local != stdPhrase(PHRASE_STAR))
                    return 0.0;
                else
                {   
                    if (qn.prefix == UNDEF_PHRASE)
                        return -0.5;
                    else
                        return -0.25;
                };
            }; break;
        default:
            return 0.5;
        };
    };
    return 0;   // BCC thinks we don't return enough
}

double Processor::defaultPriority(XSLElement *tmpl)
{
    Expression *e = tmpl -> getAttExpr(XSLA_MATCH);
    if (!e) 
        return PRIORITY_NOMATCH;
    switch(e -> functor)
    {
    case EXF_LOCPATH:
        {
            return defaultPriorityLP(e);
        }; break;
    case EXFO_UNION:
        {
            double max=0, priority;
            BOOL first = TRUE;
            int eArgsNumber = e -> args.number();
            for (int i=0; i < eArgsNumber; i++)
            {
                priority = defaultPriorityLP(e -> args[i]);
                if (first || (priority > max)) 
                    max = priority;
                first = FALSE;
            };
            return max;
        }; break;
    default:
        {
            assert(!"expression not a union or LP");
            return 0;   // dummy
        }
    };
    return 0;       // dummy for BCC
}


/*................................................................
getCurrentMode()
RETURN: pointer to the current mode which is a QName
................................................................*/

QName *Processor::getCurrentMode()
{
    return (modes.number() ? modes.last() : NULL);
}

/*................................................................
pushMode()
called before every mode-changing apply-templates
ARGS:
m       the mode to be placed on top of the mode stack
................................................................*/

void Processor::pushMode(QName *m)
{
    modes.append(m);
}

/*................................................................
popMode()
called after every mode-changing apply-templates
removes the current mode from the mode stack FREEING it
................................................................*/

void Processor::popMode()
{
    modes.freelast(FALSE);
}

eFlag Processor::addLineNoTree(DataLine *&d, Str &absolute, Bool isXSL)
{
    M( situation, d = new DataLine (this) );
    E( d -> open(absolute, DLMODE_READ) );
    E( datalines.addLine(d, NULL, isXSL) );
    return OK;
}

eFlag Processor::addLineParse(Tree *& newTree, Str &absolute, Bool isXSL)
{
    DataLine *d = new DataLine (this);
    E( d -> open(absolute, DLMODE_READ) );
    newTree = new Tree(absolute, isXSL, this);
    newTree -> dline = d;
    E( parse(newTree, d));
    E( datalines.addLine(d, newTree, isXSL) );
    E( d -> close() );
    return OK;
}

const Str& Processor::findBaseURI(const Str& unmappedBase)
{
    Str scheme, rest;
    uri2SchemePath(unmappedBase, scheme, rest);
    Str *mapping = baseURIMappings.find(scheme);
    if (mapping) return *mapping;
    mapping = baseURIMappings.find(""/**theEmptyString*/);
    if (mapping) return *mapping;
    return unmappedBase;
}

//
//          Processor::baseForVertex
//
//  returns the *translated* base URI in effect for a given vertex
//

const Str& Processor::baseForVertex(Element *v)
{
    return findBaseURI(NZ(v) -> ownerT -> name);
}


//
//          Processor::readTreeFromURI
//
//  reads in a tree from URI 'location' resolved against 'base' which is 
//  a *translated* base (one for which the 'hard base URIs' have been taken 
//  into account)
//

eFlag Processor::readTreeFromURI(Tree*& newTree, const Str& location, const Str& base,
      Bool isXSL)
{
    Str
        absolute;
    makeAbsoluteURI(location, base, absolute);
    newTree = datalines.getTree(absolute, isXSL, DLMODE_READ);
    if (!newTree)
        E( addLineParse(newTree, absolute, isXSL) );
    return OK;
}

/*
eFlag Processor::makeTreeForURI(Tree*& newTree, Str& location, Str& base)
{
    Str absolute;
    makeAbsoluteURI(location, base, absolute);
    if (datalines.getTree(absolute, FALSE, DLMODE_WRITE))
        Err1(E1_CANNOT_WRITE, absolute);
    newTree = new Tree(absolute, FALSE);
    DataLine *d = new DataLine;
    newTree -> dline = d;
    E( d -> open(absolute, DLMODE_WRITE) );
    E( datalines.addLine(d, newTree, FALSE) );
    return OK;
}
*/

eFlag Processor::pushOutputterForURI(Str& location, Str& base)
{
    Str absolute;
    makeAbsoluteURI(location, base, absolute);
    if (datalines.getTree(absolute, FALSE, DLMODE_WRITE))
        Err1(situation, E1_CANNOT_WRITE, absolute);
    DataLine *d;
    M( situation, d = new DataLine (this) );
    E( d -> open(absolute, DLMODE_WRITE) );
    // FIXME: the NULL tree in the following
    E( datalines.addLine(d, NULL, FALSE) );
    OutputterObj *newOut;
    M( situation, newOut = new OutputterObj(this) );
    E( newOut -> setOptions(d, &(styleSheet -> outputDef)) );
    if (theSAXHandler)
        E( newOut -> setOptionsSAX(theSAXHandler, theSAXUserData) );
    outputters_.append(newOut);
    return OK;
}

eFlag Processor::pushTreeConstructer(TreeConstructer *& newTC, Tree *t)
{
    M( situation, newTC = new TreeConstructer(this) );
    E( newTC -> parseUsingSAX(t) );
    outputters_.append(NZ(newTC -> getOutputter()));
    return OK;
}

eFlag Processor::pushOutputter(OutputterObj* out_)
{
    outputters_.append(out_);
    return OK;
}

eFlag Processor::popOutputter()
{
    outputters_.freelast(FALSE);
    return OK;
}

eFlag Processor::popTreeConstructer(TreeConstructer *theTC)
{
    outputters_.deppend();
    delete theTC;
    return OK;
}


/*****************************************************************
findStylesheet()
finds a xsl:stylesheet child of the given daddy. Returns NULL
if not found.
*****************************************************************/

/*
XSLElement *findStylesheet(Daddy &d)
{
    Vertex *w;
    int dContentsNumber = d.contents.number();
    for (int i = 0; i < dContentsNumber; i++)
    {
        if (isXSLElement(w = d.contents[i]) && 
            (toX(w) -> op == XSL_STYLESHEET || toX(w) -> op == XSL_TRANSFORM))
            return toX(w);
    };
    return NULL;
}
*/

Element *findStylesheet(Daddy &d)
{
    Vertex *w;
    int dContentsNumber = d.contents.number();
    for (int i = 0; i < dContentsNumber; i++)
    {
        if (isElement(w = d.contents[i]))
        {
            const QName& wName = toE(w) -> name;
            if (!strcmp(wName.getUri(), theXSLTNamespace) && /* _PH_ */
              ((wName.getLocal() == xslOpNames[XSL_STYLESHEET]) ||
               (wName.getLocal() == xslOpNames[XSL_TRANSFORM])))
            return toE(w);
        }
    };
    return NULL;
}


Bool getWhDelimString(char *&list, Str& firstPart)
{
    skipWhite(list);
    if (!*list) return FALSE;
    char *list_was = list;
    for(; *list && !isWhite(*list); list++);
    firstPart.nset(list_was, (int)(list - list_was));
    return TRUE;
}

/*****************************************************************
Processor::processVertexAfterParse()
Performs any necessary actions on a vertex immediately after it is
parsed into the tree. The vertex is still 'current'. Typically, this
function calls popVertex.
ARGS:
    v   the vertex to be processed
    t   the containing tree
The function operates only on XSL vertices, namely:
xsl:include - replaces the vertex by the newly constructed tree
xsl:output  - files the information into the tree's structures
...
Other vertices are just popped off the stack.
*****************************************************************/

eFlag Processor::processVertexAfterParse(Vertex *v, Tree *t, TreeConstructer* tc)
{
    XSL_OP theOp;
    if (isXSLElement(v))
    {
        XSLElement *x = toX(v);
        situation->pushCurrV(v);
        switch(theOp = x -> op)
        {
        case XSL_INCLUDE:
            {
                Attribute *a = NZ( x -> atts.find(XSLA_HREF) );
                Tree *srcTree;
		        const Str& base = findBaseURI(t -> name);
                E( readTreeFromURI(srcTree, a -> cont, base, FALSE) );
                
                Element *theSheet = findStylesheet(srcTree -> root);
                if (!theSheet)
                    Warn1(situation, W_NO_STYLESHEET, (char*)(a -> cont));
                t -> dropCurrentElement(v);

                E( tc -> parseUsingSAXForAWhile() );
                if (theSheet)
                    E( theSheet -> contents.copy() );
                E( tc -> parseUsingSAXForAWhileDone() );
            }; break;
        case XSL_OUTPUT:
            {
                int i, attsNumber = x -> atts.number();
                Attribute *theAtt;
                for (i = 0; i < attsNumber; i++)
                {
                    theAtt = toA(x -> atts[i]);
                    switch(theAtt -> op)
                    {
                    case XSLA_METHOD:
                        {
                            QName q(this);
                            E( q.setLogical(theAtt -> cont, 
                                &(x -> namespaces), FALSE) );
                            E( t -> outputDef.setItemQName(XSLA_METHOD, q, TRUE) );
                        }; break;
                    case XSLA_CDATA_SECT_ELEMS:
                        {
                            QName q(this);
                            Bool someRemains;
                            Str listPart;
                            char *p = theAtt -> cont;
                            do
                            {
                                someRemains = getWhDelimString(p, listPart);
                                if (someRemains)
                                {
                                    E( q.setLogical(listPart, 
                                        &(x -> namespaces), TRUE) );
                                    E( t -> outputDef.setItemQName(XSLA_CDATA_SECT_ELEMS, 
                                        q, TRUE) );
                                };
                            }
                            while (someRemains);
                        }; break;
                    default:
                            E( t -> outputDef.setItemStrCheck(theAtt -> op, theAtt -> cont) );
                    };
                }
            t -> popVertex();
            }; break;
        case XSL_NAMESPACE_ALIAS:
          {
             Phrase style,result;
             dict().insert(NZ( x -> atts.find(XSLA_STYLESHEET_PREFIX)) -> cont,style);
             dict().insert(NZ( x -> atts.find(XSLA_RESULT_PREFIX)) -> cont,result);
             aliases.appendConstruct(style,result);
             t -> popVertex();
          }; break;
        case XSL_TEMPLATE:
            {
                this -> insertRule(x);
                t -> popVertex();
            }; break;
        default:
            t -> popVertex();
        }
        situation->popCurrent();
    }
    else
        t -> popVertex();
    return OK;
}

eFlag Processor::parse(Tree *t, DataLine *d)
{
    Log1(situation, L1_PARSING, t -> name);
    double time_was = getMillisecs();
    TreeConstructer tc(this);
    eFlag retval = tc.parseDataLineUsingExpat(t, d);
    if (!retval)
    {
        Log1(situation, L1_PARSE_DONE, getMillisecsDiff(time_was));
    }
    return retval;
}

eFlag Processor::getArg(const char* name, char*& buffer, Bool isUTF16)
{
    Str temp, *value = argList.find(temp = (char*)name);
    if (!value)
        Err1(situation, E1_ARG_NOT_FOUND,(char*) name);
    buffer = (char*) *value;
    return OK;
}

//
//
//
//      plugin handler stuff
//
//
//

eFlag Processor::setHandler(HandlerType type, void *handler, void *userData)
{
    void **whereHandler, **whereUserData;
    switch(type)
    {
    case HLR_SCHEME: 
        {
            whereHandler = (void **)&theSchemeHandler;
            whereUserData = &theSchemeUserData;
        }; break;
    case HLR_MESSAGE: 
        {
            whereHandler = (void **)&theMessageHandler;
            whereUserData = &theMessageUserData;
        }; break;
    case HLR_SAX: 
        {
            whereHandler = (void **)&theSAXHandler;
            whereUserData = &theSAXUserData;
        }; break;
    case HLR_MISC: 
        {
            whereHandler = (void **)&theMiscHandler;
            whereUserData = &theMiscUserData;
        }; break;
    default: 
        Err1(situation, E1_INVALID_HLR_TYPE, (int) type);
    }
    if (*whereHandler)
    {
        if (handler)
            Warn1(situation, W1_HLR_REGISTERED, hlrTypeNames[type])
        else
        {
            *whereHandler = NULL;
            *whereUserData = NULL;
        }
    }
    else
    {
        if (handler)
        {
            *whereHandler = handler;
            *whereUserData = userData;
        }
        else
            Warn1(situation, W1_HLR_NOT_REGISTERED, hlrTypeNames[type])
    }
    return OK;
}


SchemeHandler *Processor::getSchemeHandler(void **udata)
{
    if (udata)
        *udata = theSchemeUserData;
    return theSchemeHandler;
}

MessageHandler *Processor::getMessageHandler(void **udata)
{
    if (udata)
        *udata = theMessageUserData;
    return theMessageHandler;
}

SAXHandler *Processor::getSAXHandler(void **udata)
{
    if (udata)
        *udata = theSAXUserData;
    return theSAXHandler;
}

MiscHandler *Processor::getMiscHandler(void **udata)
{
    if (udata)
        *udata = theMiscUserData;
    return theMiscHandler;
}

void *Processor::getHandlerUserData(HandlerType type, void *handler)
{
    switch(type)
    {
    case HLR_SCHEME: return theSchemeUserData;
    case HLR_MESSAGE: return theMessageUserData;
    case HLR_MISC: return theMiscUserData;
    default: return theSAXUserData;
    }
}

void Processor::setHardEncoding(const Str& hardEncoding_)
{
    hardEncoding = hardEncoding_;
}

const Str& Processor::getHardEncoding() const
{
    return hardEncoding;
};


/*****************************************************************
copyArg
  called if the result document's location uses the arg:
  scheme. Returns information about the associated named buffer.
  if not found, returns -1 in argOrdinal and NULL in newCopy.
ARGS
  argName       the name of the arg
RETURNS
  *argOrdinal    the ordinal number of this arg. This is the number
                of the call to useArg() which defined the arg.
  newCopy       pointer to a new copy of the arg (allocated via
                malloc())
*****************************************************************/

void Processor::copyArg(const Str& argName, int* argOrdinal,
    char*& newCopy)
{
    Str absolute;
    int lineNo;
    if ((makeAbsoluteURI((Str&)argName, "arg:/", absolute) != URI_ARG)
        || (lineNo = datalines.findNum(absolute, FALSE, DLMODE_WRITE)) == -1)
    {
        newCopy = NULL;
        *argOrdinal = -1;
        return;
    }
    DynBlock *block = NZ( datalines[lineNo] -> _dataline -> getOutBuffer() );
    newCopy = block -> compactToBuffer();

    //  set *argOrdinal
    *argOrdinal = argList.findNum((char *)absolute + 4);    // skip 'arg:'

}

eFlag Processor::useArg(char *name, char *val)
{
    assert(name);
    DStr nameStr;
    if (*name != '/')
        nameStr = "/";
    nameStr += name;
    if (argList.find(nameStr))
        Err1(situation, E1_DUPLICATE_ARG, nameStr);
    StrStr *p = new StrStr;
    p -> key = nameStr;
    if (val)
        p -> value = val;
    else
        p -> value.empty();
    argList.append(p);
    return OK;
}

eFlag Processor::useGlobalParam(char *name, char *val)
{
    assert(name);
    QName q(this);
    q.setLocal(name);
    Expression *expr = new Expression(NULL,this,EXF_ATOM);
    expr -> setAtom(val);
    vars -> addPrebinding(q, expr);
    return OK;
}

void Processor::setHardBaseURI(const char* hardBase)
{
  addBaseURIMapping(""/**theEmptyString*/, (const Str) hardBase);
}

void Processor::addBaseURIMapping(const Str& scheme, const Str& mapping)
{
    int ndx = baseURIMappings.findNum(scheme);
    if (ndx != -1)
        baseURIMappings.freerm(ndx, FALSE);
    if (!mapping.isEmpty())
        baseURIMappings.appendConstruct(scheme, mapping);
}

eFlag Processor::freeResultArgs()
{
    datalines.freeall(FALSE);
    argList.freeall(FALSE);
    return OK;
}

Arena *Processor::getArena() 
{
    return &theArena;
}

Str Processor::getAliasedName(const QName& name, NamespaceStack& currNamespaces)
{
   DStr s;

   if (name.hasPrefix())
   {
      Phrase newPrefix = UNDEF_PHRASE;
      Str myUri = name.getUri();
      int i;
      for (i = 0; i < aliases.number(); i++)
      {
        if (*currNamespaces.getUri(dictionary.getKey(aliases[i]->key))
            == myUri)
        {
           newPrefix = aliases[i]->value;
        };
      };
      if (newPrefix == UNDEF_PHRASE)
      {
        s += name.getPrefix();
      } else {
        Str newPrefixStr = dictionary.getKey(newPrefix);
        s += newPrefixStr;
        currNamespaces.appendConstruct(name.getPrefix(),name.getUri(),TRUE);
        currNamespaces.appendConstruct(newPrefixStr,
                                       *currNamespaces.getUri(newPrefixStr));
      };
      if (!s.isEmpty()) // could be empty e.g. for the default namespace
        s += ":";
   };
   s += name.getLocal();
   return s;
}
