/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s): Marc Lehmann <pcg@goof.com>
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#include "expr.h"
#include "verts.h"
#include <float.h>
#include <math.h>
#include "context.h"
#include "tree.h"
#include "utf8.h"

//#include "proc.h"
//#include <ctype.h>

// period is not here as it can also start a number
char tokenShort[]=  
".. :: // != <= >= "
"(  )  [  ]  @  ,  "
"|  +  -  =  <  >  "
"/  *  "
"\x0\x0\x0";

ExToken tokenShortX[]=
{
    TOK_DPERIOD, TOK_DCOLON, TOK_DSLASH, TOK_NEQ, TOK_LE, TOK_GE,
        TOK_LPAREN, TOK_RPAREN, TOK_LBRACKET, TOK_RBRACKET, TOK_ATSIGN, TOK_COMMA,
        TOK_VERT, TOK_PLUS, TOK_MINUS, TOK_EQ, TOK_LT, TOK_GT,
        TOK_SLASH, TOK_STAR
};

/*================================================================
    function info table
================================================================*/

struct FuncInfoItem 
{
    const char *name;
    ExFunctor func;
    ExType type;
} 
funcInfoTable[] =
{
    // XPath functions - NODESET category
    {"last",EXFF_LAST,EX_NUMBER},
    {"position",EXFF_POSITION,EX_NUMBER},
    {"count",EXFF_COUNT,EX_NUMBER},
    {"id",EXFF_NONE,EX_NODESET},
    {"local-name",EXFF_LOCAL_NAME,EX_STRING},
    {"namespace-uri",EXFF_NAMESPACE_URI,EX_STRING},
    {"name",EXFF_NAME,EX_STRING},

    // XPath - STRING category
    {"string",EXFF_STRING,EX_STRING},
    {"concat",EXFF_CONCAT,EX_STRING},
    {"starts-with",EXFF_STARTS_WITH,EX_BOOLEAN},
    {"contains",EXFF_CONTAINS,EX_BOOLEAN},
    {"substring-before",EXFF_SUBSTRING_BEFORE,EX_STRING},
    {"substring-after",EXFF_SUBSTRING_AFTER,EX_STRING},
    {"substring",EXFF_SUBSTRING,EX_STRING},
    {"string-length",EXFF_STRING_LENGTH,EX_NUMBER},
    {"normalize-space",EXFF_NORMALIZE_SPACE,EX_STRING},
    {"translate",EXFF_TRANSLATE,EX_STRING},

    // XPath - BOOLEAN category
    {"boolean",EXFF_BOOLEAN,EX_BOOLEAN},
    {"not",EXFF_NOT,EX_BOOLEAN},
    {"true",EXFF_TRUE,EX_BOOLEAN},
    {"false",EXFF_FALSE,EX_BOOLEAN},
    {"lang",EXFF_LANG,EX_BOOLEAN},

    // XPath - NUMBER category
    {"number", EXFF_NUMBER, EX_NUMBER},
    {"sum", EXFF_SUM, EX_NUMBER},
    {"floor", EXFF_FLOOR, EX_NUMBER},
    {"ceiling", EXFF_CEILING, EX_NUMBER},
    {"round", EXFF_ROUND, EX_NUMBER},

    // XSLT core
    {"document",EXFF_DOCUMENT,EX_NODESET},
    {"key",EXFF_KEY,EX_NODESET},
    {"format-number",EXFF_FORMAT_NUMBER, EX_STRING},
    {"current",EXFF_CURRENT, EX_NODESET},
    {"unparsed-entity-uri",EXFF_UNPARSED_ENTITY_URI, EX_STRING},
    {"generate-id",EXFF_GENERATE_ID,EX_STRING},
    {"system-property",EXFF_SYSTEM_PROPERTY, EX_STRING},
    {NULL, EXFF_NONE, EX_UNKNOWN}
};

Str getFuncName(ExFunctor functor)
{
    return funcInfoTable[functor - EXF_FUNCTION - 1].name;
}

/**********************************************************
TokenItem
**********************************************************/

void TokenItem::speak(DStr &s, SpeakMode mode)
{
    switch(tok)
    {
    case TOK_VAR:   // remove leading $
        s.nadd(firstc + 1, len - 1);
        break;
    case TOK_LITERAL:// remove enclosing quotes or dquotes
        s.nadd(firstc + 1, len - 2);
        break;
    default:
        s.nadd(firstc,len);
    };
//    s += '\0'; - thrown out
};

/**********************************************************

  Tokenizer

**********************************************************/

Tokenizer::Tokenizer(Processor* proc_)
:
items(LIST_SIZE_EXPR_TOKENS)
{
    proc = proc_;
    situation = proc->situation;
};

Tokenizer::~Tokenizer()
{
    items.freeall(FALSE);
}

eFlag Tokenizer::tokenize(const Str &astring)
{
    char *p;
    TokenItem item;
    string = astring;
    p = (char *) string;

    E( getToken(p, item, TOK_NONE) );
    ExToken prevToken = item.tok;
    while ((item.tok != TOK_END) && (item.tok != TOK_NONE))
    {
        items.append(new TokenItem(item));
        E( getToken(p, item, prevToken) );        
        prevToken = item.tok;
    };
    
    if (item.tok == TOK_NONE)
    {
        DStr itemStr;
        item.speak(itemStr, SM_OFFICIAL);
        Err1(situation, ET_BAD_TOKEN,itemStr);
    }
    else
        items.append(new TokenItem(item));

    return OK;
}

void Tokenizer::speak(DStr &s, SpeakMode mode)
{
    if (mode & SM_DESCRIBE)
        s += "[tokenizer]";
    if (!(mode & SM_CONTENTS))
        return;
    int i, itemsNumber = items.number();
    for (i = 0; i < itemsNumber; i++)
    {
        s += "token ";
        s += (int) items[i] -> tok;
        s += " ";
        items[i] -> speak(s,mode);
        s += " len=";
        s += items[i] -> len;
        s += "\n";
    };
};

/*================================================================
namerTable
a table of tokens which have the effect that the following
name is recognized as a NCName (rather than operator name) and * is
recognized as a wildcard (rather than multiplication operator).
    The table should end with TOK_NONE (which is of this type too).
================================================================*/

static ExToken namerTable[] = {
    TOK_ATSIGN, TOK_DCOLON, TOK_LPAREN, TOK_LBRACKET,
        // operators:
    TOK_OR, TOK_AND, TOK_EQ, TOK_NEQ, TOK_LT, TOK_GT, TOK_LE, TOK_GE,
    TOK_PLUS, TOK_MINUS, TOK_MINUS1, TOK_MULT, TOK_DIV, TOK_MOD, TOK_VERT,
        // slashes are operators too but not for us
    TOK_SLASH, TOK_DSLASH, TOK_COMMA,
        // TOK_NONE (terminator)
    TOK_NONE};

/*================================================================
Bool isNamer()
returns True iff the token is contained in the namerTable table.
================================================================*/

static Bool isNamer(ExToken tok)
{
    int i;
    if (tok == TOK_NONE) return TRUE;
    for (i = 0; (namerTable[i] != tok) && 
        (namerTable[i] != TOK_NONE); i++);
    return (Bool) (namerTable[i] == tok);
}

/*================================================================
ExToken tryShort()
looks up a few characters at p in the tokenShort table containing
the (up-to-3-char) symbols
RETURNS the token identified, or TOK_NONE if no match is found
================================================================*/

ExToken Tokenizer::tryShort(char*& p, ExToken prevToken)
{
    int i;
    char* t;
    ExToken result;
    
    for (i=0, t=tokenShort; *t; i++,t+=3)
        if (*p==*t)
        if ((t[1] == ' ') || (t[1] == p[1])) break;
    if (*t)
    {
        p += ((t[1] == ' ')? 1 : 2);
        result = tokenShortX[i];
        if (result == TOK_STAR)
            result = (isNamer(prevToken)? TOK_NAME : TOK_MULT);
        if ((result == TOK_MINUS) && isNamer(prevToken))
            result = TOK_MINUS1;
    }
    else result = TOK_NONE;
    return result;
}

/*================================================================
eFlag lookToken()
sets 'ret' to the following token, but does not change the pointer p
================================================================*/

eFlag Tokenizer::lookToken(ExToken &ret, char* p, ExToken prevToken)
{
    // getToken_() changes p but this is passed by value so
    // remains unchanged

    E( getToken_(ret, p, prevToken) );
    return OK;
}

/*================================================================
Bool findChar
sets p to address FOLLOWING the next occurence of c in p
(to skip a character reference, use findChar(p,';'))
RETURNS false iff the next occurence was not found
================================================================*/

static Bool findChar(char*& p, char c)
{
    while (*p && (*p != c)) p++;
    if (*p)
    {
        p++;
        return TRUE;
    }
    else
        return FALSE;
}

/*================================================================
findSame
sets p to address FOLLOWING the next occurence of *p in p
RETURNS false iff another occurence was not found
================================================================*/

static Bool findSame(char*& p)
{
    char first = *p++;
    return findChar(p, first);
};

eFlag Tokenizer::getToken_(ExToken &ret, char*& p, ExToken prevToken)
{
    ExToken tok;
    char c;
    
    skipWhite(p);
    if (!*p)
    {
        ret = TOK_END; 
        return OK;
    }
    else
    {
        // the following may also translate * into TOK_NAME
        if ((tok = tryShort(p, prevToken)) != TOK_NONE)
        {
            ret = tok;
            return OK;
        };
        switch (c = *p)
        {
        case '$': 
            {
                // call getName with prevToken=TOK_NONE
                // to ensure that e.g. 'and' in '$and' is treated as a name
                E( getName(ret,++p,TOK_NONE) );
                if (ret != TOK_NONE)
                    ret = TOK_VAR;
                else 
                    Err(situation, ET_BAD_VAR);
            };
            break;
        case '\"':
        case '\'': 
            if(!findSame(p))
                Err(situation, ET_INFINITE_LITERAL)
            else
                ret = TOK_LITERAL;
            break;
        case '&': assert(0);  //DBG: do not process entity references so far
            break;
        case '.':
            if (utf8IsDigit(utf8CharCode(p+1)))
            {
                E( getNumber(p) );
                ret = TOK_NUMBER;
            }
            else {
                p++; 
                ret = TOK_PERIOD;
            };
            break;
        default:
            {
                if (utf8IsDigit(utf8CharCode(p)))
                {
                    E( getNumber(p) );
                    ret = TOK_NUMBER;
                }
                else
                {
                    if (utf8IsLetter(utf8CharCode(p)) || (*p == '_') || (*p == ':'))
                    {
                        // the following call finds TOK_NAME, TOK_FNAME,
                        // TOK_AXISNAME,
                        // as well as TOK_AND etc. (based on prev token)
                        E( getName(ret, p, prevToken) ); 
                    }
                    else 
                    {
                        Str temp;
                        temp.nset(p, 1);
                        Err1(situation, ET_BAD_TOKEN,temp); //unknown token
                    }
                }
            };  //default
        };      //switch
    };          //else
    return OK;
};              //getToken_

eFlag Tokenizer::getToken(char*& p, TokenItem& item, ExToken prevToken)
{
    ExToken t;
    skipWhite(p);
    item.firstc = p;
    E( getToken_(t, p, prevToken) );
    item.len = (long)(p - item.firstc);
    item.tok = t;
    return OK;
}

eFlag Tokenizer::getNumber(char*& p)
{
    Bool wasDot = FALSE;
    while ((*p) && (utf8IsDigit(utf8CharCode(p))) || (*p == '.'))
    {
        if (*p == '.')
            if (wasDot)
            Err(situation, ET_BAD_NUMBER)
            else wasDot = TRUE;
        p += utf8SingleCharLength(p);
    };
    return OK;
};

/*================================================================
getWordOp
checks whether the sequence at p of given length is an operator name
RETURNS the appropriate token if so; TOK_NONE otherwise
================================================================*/

static ExToken getWordOp(char *p, int length)
{
    if (length > 3) return TOK_NONE;
    if (length < 2) length = 2;
    if (!strncmp(p,"or",length)) return TOK_OR;
    if (length < 3) length = 3;
    if (!strncmp(p,"and",length)) return TOK_AND;
    if (!strncmp(p,"div",length)) return TOK_DIV;
    if (!strncmp(p,"mod",length)) return TOK_MOD;
    return TOK_NONE;
}

static Bool isNodeTest(char *p, int length)
{
    const char *q;
    int qlen;
    for (int i = 0; (q = exNodeTypeNames[i]) != NULL; i++)
    {
        if (!strncmp(q,p,
            (length < (qlen = strlen(q))? qlen : length)))
            break;
    };
    return (Bool)(q != NULL);
}

#define nameCharExtended(CH, PTR) ((CH = utf8CharCode(PTR))!= 0) &&\
        (utf8IsNameChar(CH) || strchr(".-_:*",CH))

int nameLength(char* from)
{
    char *q = from;
    int length = 0;
    unsigned long c;
    while(nameCharExtended(c,q)) {
       q += utf8SingleCharLength(q);
       length++;
    }
    return length;
}

eFlag Tokenizer::getName(ExToken &ret, char*& p, ExToken prevToken)
{
    char *former = p;
    unsigned long c;
    BOOL wasColon = FALSE;
    
    if ((!utf8IsLetter(utf8CharCode(p))) && (*p != '_'))
    {
        ret = TOK_NONE;
        return OK;
    }

    while (nameCharExtended(c,p))
    {
        if (c == ':')
        {
            if (wasColon)
            {
                // identify the bad qname;
                Str theName;
                theName.nset(former, nameLength(former));
                Err1(situation, E1_EXTRA_COLON, theName);
            }
            else
            {
                switch(*(p+1))
                {
                case ':':
                    {
                        ret = TOK_AXISNAME;
                        return OK;
                    };
                case '*':
                    {
                        ret = TOK_NAME;
                        p += 2;
                        return OK;
                    };
                default:
                    wasColon = TRUE;
                };
            };
        }
        else if (c == '*')
        {
            if ((p - former) && *(p - 1) != ':')   // the first condition could be dropped
            {
                ret = TOK_NAME;
                return OK;
            }
        }
        p += utf8SingleCharLength (p);
    }

    if (!wasColon && !isNamer(prevToken))
    {
        if ((ret = getWordOp(former, (int) (p - former))) != TOK_NONE)
            return OK;
    };

    ExToken next;

    // look at following token with prev=TOK_NAME
    E( lookToken(next,p,TOK_NAME) );
    switch(next)
    {
    case TOK_DCOLON:
        ret = TOK_AXISNAME;
        break;
    case TOK_LPAREN:
        {
            if (isNodeTest(former, (int) (p - former)))
                ret = TOK_NTNAME;
            else
                ret = TOK_FNAME;
        }; break;
    default:
        ret = TOK_NAME;
    };
    return OK;
};

/*================================================================
int findTop()
    finds the first top-level occurence of 'token' starting with
    position 'from'. If there is none, return value points at TOK_END.
================================================================*/

int Tokenizer::findTop(ExToken token, int from)
{
    int level = 0;
    ExToken ct;
    int i;
    for (i = from; 
        ((ct = items[i] -> tok) != TOK_END) && (level || (ct != token));
        i++)
        {
            if ((ct == TOK_LPAREN) || (ct == TOK_LBRACKET))
                level++;
            if ((ct == TOK_RPAREN) || (ct == TOK_RBRACKET))
                level--;
        }
    return i;
}


/*================================================================
eFlag getDelim()
given a position in 'pos', finds the corresponding next token.
If the left token is ( or [, looks for the matching right paren/bracket,
Otherwise looks for the occurence of the same token. 
Returns pos pointing at the matching token, or at TOK_END if there 
is none. (In case of failed reverse search, returns -1.)
================================================================*/

eFlag Tokenizer::getDelim(int &pos, Bool reverse /*=FALSE*/)
{
    ExToken first, second, tok;
    int level = 0,
        i = pos;
    
    switch(first = items[pos] -> tok)
    {
    case TOK_LBRACKET: 
        second = TOK_RBRACKET; 
        break;
    case TOK_LPAREN: 
        second = TOK_RPAREN; 
        break;
    case TOK_RBRACKET: 
        second = TOK_LBRACKET; 
        break;
    case TOK_RPAREN: 
        second = TOK_LPAREN; 
        break;
    default: 
        second = first;
    }
    
    i += reverse? -1 : 1;
    
    while ((i >= 0) && ((tok = items[i] -> tok) != TOK_END))
    {
        if (tok == second)
        {
            if (!level)
            {
                pos = i;
                return OK;
            }
            else level--;
        }
        else if (tok == first) level++;
        i += reverse? -1 : 1;
    }
    pos = i;
    return OK;
}

/*================================================================
stripParens()
given the left and right end positions of a tokenizer fragment, 
shifts them inwards to strip any outer parentheses.
================================================================*/

void Tokenizer::stripParens(int &left, int &right)
{
    int left0 = left;
    if (items[right]->tok == TOK_END)
        right--;
//    assert(left <= right);
    while ((items[left]->tok == TOK_LPAREN)
        && (items[right]->tok == TOK_RPAREN))
    {
        left0 = left;
        getDelim(left0);
        if (left0 == right)
        {
            left++; 
            right--;
        }
        else break;
    };
}

/*****************************************************************
*                                                                *
      L o c S t e p 
*                                                                *
*****************************************************************/

LocStep::LocStep(Processor* _proc,
                 ExAxis _axis       /*=AXIS_NONE*/, 
                 ExNodeType _ntype  /*=EXNODE_NONE*/)
: preds(LIST_SIZE_1), proc(_proc), ntest(_proc)
{
    set(_axis, _ntype);
    positional = FALSE;
    badPreds = 0;
}

LocStep::~LocStep()
{
    preds.freeall(FALSE);
}

void LocStep::set(ExAxis _axis, ExNodeType _ntype)
{
    ax = _axis;
    ntype = _ntype;
}

void LocStep::speak(DStr &s, SpeakMode mode)
{
    if (!(mode & SM_CONTENTS)) return;
    switch(ax)
    {
    case AXIS_CHILD:
//    case AXIS_DESC_OR_SELF:
    case AXIS_ROOT:
        break;
    case AXIS_ATTRIBUTE:
        s += '@';
        break;
    default:
        {
            s += axisNames[ax];
            s += "::";
        };
    };
    if((ntype != EXNODE_NONE) //&& (ax != AXIS_DESC_OR_SELF)
        && (ax != AXIS_ROOT))
    {
        s += exNodeTypeNames[ntype];
        s += "()";
    }
    else
        ntest.speak(s,mode);
    int i, predsNumber = preds.number();
    for (i = 0; i < predsNumber; i++)
    {
        s += '[';
        preds[i] -> speak(s,mode);
        s += ']';
    };
};

eFlag LocStep::parse(Tokenizer& tokens, int& pos, Element *ownerV)
{
    int right;
    int &i = pos;
    DStr temp;
    ExToken tok;
    
    tok = tokens.items[i++]->tok;
    if (tok == TOK_END)
        Err(tokens.situation, ET_EMPTY_PATT);
    switch (tok)
    {
    case TOK_PERIOD:
        ax = AXIS_SELF;
        ntype = EXNODE_NODE;
        return OK;
        break;
    case TOK_DPERIOD:
        ax = AXIS_PARENT;
        ntype = EXNODE_NODE;
        return OK;
        break;
    case TOK_ATSIGN:
        {
            ax = AXIS_ATTRIBUTE;
            tok = tokens.items[i++]->tok;
        };
        break;
    case TOK_STAR:
        {
            ax = AXIS_CHILD;
        };
        break;
    case TOK_AXISNAME:
        {
            tokens.items[i-1] -> speak(temp, SM_OFFICIAL);
            if ((ax = (ExAxis) lookup(temp, axisNames)) == AXIS_NONE)
                Err1(tokens.situation, E1_UNKNOWN_AXIS, temp);
            i++;
            tok = tokens.items[i++] -> tok;
        };
        break;
    case TOK_NAME:
    case TOK_NTNAME:
        {
            ax = AXIS_CHILD;
        };
        break;
    default:
        Err(tokens.situation, ET_EXPR_SYNTAX); // axis name or node-test expected
    };
    
    // axis has been determined; tok must now be a name or a node-test
    temp.empty();
    if ((tok != TOK_NAME) && (tok != TOK_NTNAME))
        Err(tokens.situation, ET_EXPR_SYNTAX); // axis name or node-test expected
    tokens.items[i-1] -> speak(temp,SM_OFFICIAL);
    ntype = EXNODE_NONE;

    if (tok == TOK_NTNAME)
    {
        ntype = (ExNodeType) lookup(temp, exNodeTypeNames);
        if (tokens.items[i++]->tok != TOK_LPAREN)
            Err(tokens.situation, ET_LPAREN_EXP);
        if (tokens.items[i++]->tok != TOK_RPAREN)
            Err(tokens.situation, ET_RPAREN_EXP);
    }
    else
        // set the QName from prefix:uri using
        // the namespace declarations in 'ownerV'
        // (passed through from the containing attribute)
        // without using the default namespace
        E( ntest.setLogical(temp, &(NZ(ownerV) -> namespaces), FALSE) ); 

    while ((tokens.items[i] -> tok) == TOK_LBRACKET)
    {
        badPreds = 0;
        E( tokens.getDelim(right = i) );
        if (tokens.items[right] -> tok == TOK_END)
            Err(tokens.situation, ET_RBRACKET_EXP)
        else 
        {
            Expression *ex = new Expression(ownerV, tokens.proc);
            E( ex -> parse(tokens,i+1,right-1) );
            // find out about the use of last() and position()
            int exPositionalType = ex -> optimizePositional();
            if (exPositionalType)
            {
                positional = TRUE;
                if (exPositionalType == 2)
                    badPreds++;
                // find sufficient position bounds
                ex -> optimizePositionBounds();
            }
            preds.append(ex);
        };
        i = right + 1;
    };

    return OK;  // pointing at the first char that does not start a predicate

};      //  end LocStep::parse()


Bool LocStep::matchesWithoutPreds(Vertex *v)
{
    // removed the following:
    // assert(v);
    // (because the parent-of-root calls etc.)
    if (!v)
        return FALSE;
    VTYPE ty = (VTYPE) ((v -> vt) & VT_BASE);
    switch(ntype)
    {
    case EXNODE_NODE:
//        if (!((ty == VT_ATTRIBUTE) || (ty == VT_ROOT)))
//            return TRUE;
        break;
    case EXNODE_TEXT:
        if (ty != VT_TEXT)
            return FALSE;
        break;
    case EXNODE_PI:
        if (ty != VT_PI) 
            return FALSE;
        break;
    case EXNODE_COMMENT:
        if (ty != VT_COMMENT)
            return FALSE;
        break;
    case EXNODE_NONE:
        if ((ty == VT_TEXT) || (ty == VT_COMMENT) || (ty == VT_ROOT))
            return FALSE;
        break;
    };

    switch(ax)
    {
    case AXIS_ROOT: 
        return (Bool) (ty == VT_ROOT); 
        break;
    case AXIS_ATTRIBUTE: 
        if (ty != VT_ATTRIBUTE) return FALSE; 
        break;
    case AXIS_NAMESPACE: 
        assert(!"namespace axis");
/*  the following is OK but we can't get the name later:
        if (ty != VT_NAMESPACE) return FALSE; 
        break;
*/
    case AXIS_CHILD: 
    case AXIS_DESCENDANT:
    case AXIS_DESC_OR_SELF:
    case AXIS_ANCESTOR:
    case AXIS_ANC_OR_SELF:
    case AXIS_FOLL_SIBLING:
    case AXIS_PREC_SIBLING:
        switch (ty)
        {
        case VT_ATTRIBUTE:
        case VT_NAMESPACE:
            return FALSE;
        case VT_ROOT:
            switch(ax)
            {
            case AXIS_DESC_OR_SELF:
            case AXIS_ANCESTOR:
            case AXIS_ANC_OR_SELF:
                break;
            default:
                return FALSE;
            };
        };
        break;
    case AXIS_PARENT:
    case AXIS_SELF:
        break;
    default: assert(0); //should be handled in parse
    };

    if (ntype != EXNODE_NONE)
        return TRUE;

    const QName &hisname = v -> getName();
    // FIXME: must get the name of 
    
    //compare names; no predicates so far
    return (proc -> cmpQNames(ntest,hisname));
}


eFlag LocStep::shift(Vertex *&v, Vertex *baseV)
{
    int i, ord;
    Vertex *w = NULL;       // the result
    switch(ax)
    {
    case AXIS_ATTRIBUTE:
        {
            assert(!v || isElement(baseV) && baseV == v -> parent);
            if (!isElement(baseV)) break; // i.e. v==NULL and baseV isn't daddy
            for (ord = v ? v -> ordinal + 1 : 0; ord < toE(baseV) -> atts.number(); ord++)
            {
                if (matchesWithoutPreds(w = toE(baseV) -> atts[ord])) break;
                else w = NULL;
            }
        }; break;
        
    case AXIS_CHILD:
        {
            assert(!v || isDaddy(baseV) && baseV == v -> parent);
            if (!isDaddy(baseV)) break;
            for (ord = v ? v -> ordinal + 1 : 0; ord < toD(baseV) -> contents.number(); ord++)
            {
                if (matchesWithoutPreds(w = toD(baseV) -> contents[ord])) break;
                else w = NULL;
            }
        }; break;
        
    case AXIS_NAMESPACE:
        {
            assert(!v || isElement(baseV) && baseV == v -> parent);
            if (!isElement(baseV)) break;
            for (ord = v ? v -> ordinal + 1 : 0; ord < toE(baseV) -> namespaces.number(); ord++)
            {
                if (matchesWithoutPreds(w = toE(baseV) -> namespaces[ord])) break;
                else w = NULL;
            }
        }; break;
        
    case AXIS_ROOT:     // technically not an axis
        {
            if (!v)
            {
                if (isRoot(baseV)) w = baseV;
                else 
                {
                    if (isRoot(baseV -> parent)) w = baseV -> parent;
                    else w = &(toE(baseV -> parent) -> ownerT -> root);
                }
            }
        }; break;
        
    case AXIS_SELF:
        {
            if (!v && matchesWithoutPreds(baseV))
                w = baseV;
        }; break;
        
    case AXIS_PARENT:
        {
            if (!v && matchesWithoutPreds(baseV -> parent))
                w = baseV -> parent;
        }; break;
        
    case AXIS_ANCESTOR:
    case AXIS_ANC_OR_SELF:
        {
            if (v) 
                w = v -> parent;
            else
                w = (ax == AXIS_ANCESTOR) ? baseV -> parent : baseV;
            for (; w; w = w -> parent)
                if (matchesWithoutPreds(w)) break;
        }; break;
        
    case AXIS_FOLL_SIBLING:
        {
            if (isRoot(baseV) || isNS(baseV) || isAttr(baseV))
                break;
            Daddy *theParent = toD(NZ(baseV -> parent));
//          changing to fix a bug
//          for (ord = v ? v -> ordinal + 1 : 0; ord < theParent -> contents.number(); ord++)
            for (ord = v ? v -> ordinal + 1 : baseV -> ordinal + 1; ord < theParent -> contents.number(); ord++)
            {
                if (matchesWithoutPreds(w = theParent -> contents[ord])) break;
                else w = NULL;
            }
        }; break;
        
    case AXIS_PREC_SIBLING:
        {
            if (isRoot(baseV) || isNS(baseV) || isAttr(baseV))
                break;
            Daddy *theParent = toD(NZ(baseV -> parent));
//          changing to fix a bug
//          for (ord = v ? v -> ordinal - 1 : theParent -> contents.number() - 1; ord >= 0; ord--)
            for (ord = v ? v -> ordinal - 1 : baseV -> ordinal - 1; ord >= 0; ord--)
            {
                if (matchesWithoutPreds(w = theParent -> contents[ord])) break;
                else w = NULL;
            }
        }; break;
        
    case AXIS_DESCENDANT:
    case AXIS_DESC_OR_SELF:
        {
            if (!v)
            {
                if (ax == AXIS_DESC_OR_SELF && matchesWithoutPreds(baseV))
                {
                    w = baseV;
                    break;
                }
                else
                    v = baseV;
            }
            
            // find next descendant
            do
            {
                if (isDaddy(v) && cast(Daddy*,v) -> contents.number())
                    v = cast(Daddy*,v) -> contents[0];
                else
                {
                    while (v != baseV)
                    {
                        i = v -> ordinal;
                        v = v -> parent;
                        if (i < cast(Daddy*,v) -> contents.number()-1)
                        {
                            v = cast(Daddy*,v) -> contents[i+1];
                            break;
                        }
                    }
                };
                if (v != baseV && matchesWithoutPreds(v))
                {
                    w = v;
                    break;
                };
            }
            while(v != baseV);
        }; break;
    default:
        assert(!"axes following, preceding: not yet implemented"); 
        };
    v = w;
    return OK;
};

/*
eFlag LocStep::createContextNoPreds(Context *&c, int baseIndex)
{
    int i;
    Vertex 
        *baseV = (*c)[baseIndex],
        *v;
    Vertex *currentNode = c->getCurrentNode();
    c = new Context;
    c->setCurrentNode(currentNode);

    switch(ax)
    {
    case AXIS_ATTRIBUTE:
        if (isElement(baseV))
        {
            Element *baseE = cast(Element*, baseV);
            int attsNumber = baseE -> atts.number(); 
            for (i = 0; i < attsNumber; i++)
                if (matchesWithoutPreds(v = baseE -> atts[i]))
                    c -> append(v);
        };
        break;
    case AXIS_NAMESPACE:
        if (isElement(baseV))
        {
            Element *baseE = cast(Element*, baseV);
            int attsNumber = baseE -> atts.number(); 
            for (i = 0; i < attsNumber; i++)
                if (matchesWithoutPreds(v = baseE -> atts[i]))
                    c -> append(v);
        };
        break;
    case AXIS_CHILD:
        if (isDaddy(baseV))
        {
            Daddy *baseD = cast(Daddy*, baseV);
            int contentsNumber = baseD -> contents.number();
            for (i = 0; i < contentsNumber; i++)
                if (matchesWithoutPreds(v = baseD -> contents[i]))
                   c -> append(v);
        };
        break;
    case AXIS_ROOT: // technically not an axis
        c -> append(&(proc -> input -> root));
        break;
    case AXIS_SELF:
        if (matchesWithoutPreds(baseV))
            c -> append(baseV);
        break;
    case AXIS_PARENT:
        if (matchesWithoutPreds(baseV -> parent))
            c -> append(baseV -> parent);
        break;
    case AXIS_ANCESTOR:
    case AXIS_ANC_OR_SELF:
        for (v = (ax == AXIS_ANCESTOR ? baseV -> parent : baseV); v; v = v -> parent)
            if (matchesWithoutPreds(v))
                c -> append(v);
        break;
    case AXIS_DESCENDANT:
    case AXIS_DESC_OR_SELF:
        {
            if ((ax == AXIS_DESC_OR_SELF) && matchesWithoutPreds(baseV))
                c -> append(baseV);                    
            // find next descendant
            // curr = findDescendant(curr, expr -> step); or something like that
            v = baseV;
            do
            {
                if (isDaddy(v) && cast(Daddy*,v) -> contents.number())
                    v = cast(Daddy*,v) -> contents[0];
                else
                {
                    while (v -> stamp != baseV -> stamp)
                    {
                        i = v -> ordinal;
                        v = v -> parent;
                        if (i < cast(Daddy*,v) -> contents.number()-1)
                        {
                            v = cast(Daddy*,v) -> contents[i+1];
                            break;
                        }
                    }
                };
                if (v -> stamp == baseV -> stamp)
                    break;
                else
                    if (matchesWithoutPreds(v))
                        c -> append(v);
            }
            while(TRUE);
        }; break;
    case AXIS_FOLL_SIBLING:
        {
            if (isRoot(currentNode) || isNS(currentNode) || isAttr(currentNode))
                break;
            Daddy *theParent = toD(NZ(currentNode -> parent));
            int i, contentsNumber = theParent -> contents.number();
            Vertex *w;
            for (i = currentNode -> ordinal + 1; i < contentsNumber; i++)
            {
                if (matchesWithoutPreds(w = theParent -> contents[i]))
                    c -> append(w);
            }
        }; break;
    case AXIS_PREC_SIBLING:
        {
            if (isRoot(currentNode) || isNS(currentNode) || isAttr(currentNode))
                break;
            Daddy *theParent = toD(NZ(currentNode -> parent));
            int i; 
            Vertex *w;
            for (i = currentNode -> ordinal - 1; i >= 0; i--)
            {
                if (matchesWithoutPreds(w = theParent -> contents[i]))
                    c -> append(w);
            }
        }; break;

    default: assert(!"axes following, preceding, namespace: not yet implemented"); 
    };

    // revert some axes:
    switch(ax)
    {
    case AXIS_ANCESTOR:
    case AXIS_PREC_SIBLING:
    case AXIS_PRECEDING:
    case AXIS_ANC_OR_SELF:
        {
            int n = c -> getSize();
            for (i = 0; i < n / 2; i++)
                c->swap(i, n-i-1);
        }
    }
    return OK;
}
*/

/**********************************************************
N u m b e r
**********************************************************/

Number::Number()
{
    *this = 0.0;
}

Number::Number(double y)
{
    *this = y;
}

Number& 
Number::operator= (double y)
{
    x = y;
    return *this;
};

Number& 
Number::operator= (const Str &s)
{
    char *endptr, *startptr = s;
    skipWhite(startptr);
    if (*startptr)
    {
        x = strtod(startptr, &endptr);
        if (*endptr)
            setNaN();
    }
    else
        setNaN();
    return *this;
};

Number::operator double() 
{
    return x;
}

Bool Number::operator== (double y)
{
    if (isNaN() || isnan__(y))
        return FALSE;
    if (isInf() || isinf__(y))
        return isInf() && isinf__(y) && !((x > 0) ^ (y > 0));
    double d = x - y;
    return (Bool)((d < EPS) && (d > -EPS));
}

Bool Number::operator== (Number& y)
{
    if (isNaN()) return FALSE;
    return (Bool)(operator== ((double) y));
}


Bool Number::operator< (double y) 
{
    return (Bool)(x < y);
}

Bool Number::operator< (Number& y)
{
    return (Bool)(x < (double) y);
}

Bool Number::operator> (double y)
{
    return (Bool)(x > y);
}

Bool Number::operator> (Number& y)
{
    return (Bool)(x > (double) y);
}

Bool Number::isNaN()
{
    return isnan__(x);
}

Bool Number::isInf()
{
    return isinf__(x);
};

void Number::setNaN()
{
    // divide by zero using a variable, to fool too clever compilers
    // 'volatile' suggested by Dirk Siebnich
    volatile int zero = 0;
    x = 0.0 / zero;
}

int Number::round()
{
    if (isNaN() || isInf())
        return 0;
    else return (int)(floor(x + 0.5)); // FIXME: ignoring the 'negative zero'
}


//________________________________________________________________


/*****************************************************************
|                                                                |
    E x p r e s s i o n
|                                                                |
*****************************************************************/

Expression::Expression(Element *_ownerV,
                       Processor* proc_,
                       ExFunctor _functor   /* = EXF_NONE   */
                       )
:args(1)
{
    proc = proc_;
    functor = _functor;
    ownerV = _ownerV;
    switch(functor)
    {
    case EXF_LOCSTEP:
        {
            step = new LocStep (proc);
            type = EX_NODESET;
        }; break;
    case EXF_LOCPATH:
        {
            type = EX_NODESET;
        }; break;
    case EXF_STRINGSEQ:
        type = EX_STRING;
        break;
    case EXF_NONE:
        {
            type = EX_UNKNOWN;
        }; break;
    };
    hasPath = FALSE;
    isPattern = FALSE;
    pTree = NULL;
    // the following sets patomnodeset, note that e.g. patomnumber
    // is with it in a union
    patomnodeset = NULL;
    usesLast = FALSE;
    positional = FALSE;
    optimizePositionFrom = optimizePositionTo = 0; // see header
}

void Expression::setLS(ExAxis _axis, ExNodeType _ntype)
{
    assert(functor == EXF_LOCPATH);
    Expression *ls = new Expression(ownerV, proc,
        EXF_LOCSTEP);
    args.append(ls);
    ls -> step -> set(_axis, _ntype);    
}

//
//  Expression destructor
//  
Expression::~Expression()
{
    clearContent();
}

//
//  deleteZ
//  this macro is only used in Expression::clearContent()
//  deletes a pointer, assert-checking it is non-NULL, setting it to a NULL
//
//    commenting this out because of harmless assertion faults, BUT
//    in processing Sablot-0-33.xml, there are number Expr's with 
//    NULL patomnumber - why?
// #define deleteZ( PTR ) { delete NZ( PTR ); PTR = NULL; }
//    setting it to cdelete (delete if nonzero and set to 0)
#define deleteZ( PTR )    { cdelete( PTR ); } 


//
//  clearContent()
//  called when setting the expression to a new value
//  to dispose of any existing contents.
//
void Expression::clearContent()
{
    args.freeall(FALSE);
    switch(functor)
    {
    case EXF_ATOM:
        {
            switch(type)
            {
            case EX_NODESET:
                deleteZ ( patomnodeset );
                break;
            case EX_STRING:
                deleteZ ( patomstring );
                break;
            case EX_NUMBER:
                deleteZ ( patomnumber );
                break;
            };
        }; break;
    case EXF_LOCSTEP:
        deleteZ ( step );
        break;
    case EXF_VAR:
        deleteZ ( pName );
        break;
    }
    cdelete( pTree );
};



/*================================================================
speak
    writes the expression to string 's'. Formatting is specified
    by 'mode'.
================================================================*/

void Expression::speak(DStr &s, SpeakMode mode)
{
    int i, argsNumber = args.number();
    switch(functor)
    {
    case EXF_ATOM:
        {
            s += tostring();
            return;
        }; break;
    case EXF_LOCSTEP:
        {
            step -> speak(s, mode);
        }; break;
    case EXF_LOCPATH:
        {
            for(i = 0; i < argsNumber; i++)
            {
                args[i] -> speak(s, mode);
                if (i < argsNumber-1) 
                    s += "/";
                else if ((argsNumber == 1) && 
                    (args[0] -> step -> ax == AXIS_ROOT))
                    s += "/";
            }
        }; break;
    default:
        {
            s += (DStr("\nfunctor ") + (int) functor + "\n--------ARGS:\n");
            for (i = 0; i < argsNumber; i++)
            {
                s += DStr("(") + (i+1) + ")   ";
                args[i] -> speak(s,mode);
                s += "\n";
            };
            s += "--------ARGS end\n";
        }
    }
}

/*================================================================
matches
    returns TRUE iff the current vertex of c satisfies the
    Expression's condition.
    PROBLEM:
    perhaps this should also return an error in case the expression is
    not of type nodeset?
================================================================*/

eFlag Expression::matchesPattern(Context *c, Bool &result)
{
    assert(type == EX_NODESET);
    if (functor == EXF_LOCPATH)
    {
        E(matchesSinglePath(c -> current(), args.number() - 1, result));
        return OK;
    }
    if (functor == EXFO_UNION)
    {
        int j, argsNumber = args.number();
        for (j = 0; j < argsNumber; j++)
        {
            E( args[j] -> matchesPattern(c, result) );
            if (result)
                RetOK( result, TRUE );
        };
    }
    RetOK(result, FALSE);
}

eFlag Expression::trueFor(Context *c, Bool& result)
{
    Expression ex(ownerV,proc);
    E( eval(ex,c) );
    switch(ex.type)
    {
    case EX_NUMBER:
        result = (Bool) (ex.tonumber() == (double) (c -> getPosition() + 1));
        break;
    default:
        result = ex.tobool();
    }
    return OK;
}


Bool Expression::tobool()
{
    assert(functor == EXF_ATOM);
    switch(type)
    {
    case EX_NUMBER:
        return (Bool) !(*patomnumber == 0.0 || patomnumber -> isNaN());
        break;
    case EX_STRING:
        return (Bool) !(patomstring -> isEmpty());
        break;
    case EX_BOOLEAN:
        return atombool;
        break;
    case EX_NODESET:
        return (Bool) !!(patomnodeset -> getSize());
        break;
    default: assert(0);
    };
    return FALSE;   //just to return something
}

Str Expression::tostring()
{
    assert(functor == EXF_ATOM);
    switch(type)
    {
    case EX_NUMBER:
        if (patomnumber -> isNaN())
            return "NaN";
        else
        {
            if (!patomnumber -> isInf())
            {
                Str s;
                s = (double)(*patomnumber);
                return s;
            }
            else if (*patomnumber > 0.0)
                return "+Infinity";
            else return "-Infinity";
        }
        break;
    case EX_STRING:
        return *patomstring;
        break;
    case EX_BOOLEAN:
        return (atombool ? (char *)"true" : (char *)"false");
        break;
    case EX_NODESET:
        if (!patomnodeset -> getSize())
            return "";
        else 
        {
            // FIXME: the following should be checked for error (wrapped in E):
            DStr s;
            patomnodeset -> current() -> value(s, patomnodeset);
            return s;
        }
        break;
    default: assert(0);
    };
    return "";
}

char *Expression::tostringCharPtr()
{
    assert((functor == EXF_ATOM) && (type == EX_STRING));
    return (char*)(*NZ(patomstring));
}

const Str& Expression::tostringRef() const
{
    assert((functor == EXF_ATOM) && (type == EX_STRING));
    return (*NZ(patomstring));
}

Number Expression::tonumber()
{
    assert(functor == EXF_ATOM);
    Number n;
    switch(type)
    {
    case EX_NUMBER:
        n = *patomnumber;
        break;
    case EX_STRING:
        n = *patomstring;
        break; 
    case EX_BOOLEAN:
        n = (atombool ? 1.0 : 0.0);
        break;
    case EX_NODESET:
        {
            // to avoid the following, tostring() must return const Str&:
            Str s = tostring();
            n = s;
        }
        break;
    default: assert(0);
    };
    return n;
}

Context& Expression::tonodeset()
{
    assert((functor == EXF_ATOM) && (type == EX_NODESET));
    return *(patomnodeset -> copy());
}

const Context& Expression::tonodesetRef()
{
    assert((functor == EXF_ATOM) && (type == EX_NODESET));
    return *patomnodeset;
}

eFlag Expression::patternOK()
{
    int i,
        argsNumber = args.number();
    assert(functor == EXFO_UNION || functor == EXF_LOCPATH);
    switch(functor)
    {
    case EXF_LOCPATH:
        {
            for (i = 0; i < argsNumber; i++)
            {
                LocStep *ls = args[i] -> step;
                switch (ls -> ax)
                {
                case AXIS_CHILD:
                case AXIS_ATTRIBUTE:
                case AXIS_ROOT:
                    break;
                case AXIS_DESC_OR_SELF:
                    if (ls -> ntype != EXNODE_NODE)
                        Err(proc -> situation, E_BAD_PATTERN);
                    break;
                default:
                    Err(proc -> situation, E_BAD_AXIS_IN_PATTERN);
                }
            }
        }; break;
    case EXFO_UNION:
        {
            for (i=0; i < argsNumber; i++)
                E(args[i] -> patternOK());
        }; break;
    default:
        assert(!"patternOK()");
    };
    return OK;
}

eFlag Expression::parse(const DStr &s,
                        Bool _isPattern /* = FALSE  */)
{
    isPattern = _isPattern;
    Tokenizer t ( proc );
    E( t.tokenize(s) );
    E( parse(t, 0, t.items.number() - 1) );
    if (_isPattern)
        E( patternOK() );
    return OK;
}

/*================================================================
Bool isOp()
returns True if the given token is an operator, in which case
'precedence' is set to its precedence
================================================================*/

Bool Expression::isOp(ExToken token, int &precedence)
{
    Bool is = TRUE;
    switch(token)
    {
    case TOK_OR:
        precedence = 0;
        break;
    case TOK_AND:
        precedence = 1;
        break;
    case TOK_EQ:
    case TOK_NEQ:
        precedence = 2;
        break;
    case TOK_LT:
    case TOK_GT:
    case TOK_LE:
    case TOK_GE:
        precedence = 3;
        break;
    case TOK_PLUS:
    case TOK_MINUS:
        precedence = 4;
        break;
    case TOK_MULT:
    case TOK_DIV:
    case TOK_MOD:
        precedence = 5;
        break;
    case TOK_MINUS1:
        precedence = 6;
        break;
    case TOK_VERT:
        precedence = 7;
        break;
    default:
        {
            is = FALSE;
            precedence = -1;
        };
    };
    return is;
}

/*================================================================
void getFunctionInfo()
returns function code and type for the function with given name
    if no such builtin function, returns EXFF_NONE
================================================================*/

void getFunctionInfo(const Str &s, ExFunctor &code, ExType &type)
{
    char *p = (char *) s;
    int i;
    for (i = 0; funcInfoTable[i].name; i++)
    {
        if (!strcmp(funcInfoTable[i].name,p))
            break;
    };
    code = funcInfoTable[i].func;
    type = funcInfoTable[i].type;
}

struct OpItem
{
    ExFunctor fu;
    ExType ty;
    int arity;
} 
opTable[] =
{
    {EXFO_OR, EX_BOOLEAN, 3},
    {EXFO_AND, EX_BOOLEAN, 3},
    {EXFO_EQ, EX_BOOLEAN, 2},
    {EXFO_NEQ, EX_BOOLEAN, 2},
    {EXFO_LT, EX_BOOLEAN, 2},
    {EXFO_GT, EX_BOOLEAN, 2},
    {EXFO_LE, EX_BOOLEAN, 2},
    {EXFO_GE, EX_BOOLEAN, 2},
    {EXFO_PLUS, EX_NUMBER, 2},
    {EXFO_MINUS2, EX_NUMBER, 2},
    {EXFO_MULT, EX_NUMBER, 2},
    {EXFO_MOD, EX_NUMBER, 2},
    {EXFO_DIV, EX_NUMBER, 2},
    {EXFO_MINUS1, EX_NUMBER, 1},
    {EXFO_UNION, EX_NODESET, 3}
};

/*================================================================
eFlag parseLP()
================================================================*/

eFlag Expression::parseLP(Tokenizer& tokens, int &pos, 
                     Bool dropRoot /*=FALSE*/)
{
    assert(functor == EXF_LOCPATH);
    ExToken tok;
    BOOL getaway = FALSE;
    Expression *ls;
    int& i = pos;
    Bool slashPending = FALSE,
        nameWas = FALSE;
    
    tok = tokens.items[i] -> tok;
    if (tok == TOK_END)
        Err(proc -> situation, ET_EMPTY_PATT);
    if ((tok == TOK_SLASH) || (tok== TOK_DSLASH))
    {
        if (!dropRoot)
        {
            args.append(ls = new Expression(ownerV,proc,EXF_LOCSTEP));
            ls -> step -> set(AXIS_ROOT,EXNODE_NODE);
        }
        if (tok == TOK_SLASH)
            i++;
    }
    
    while (!getaway)
    {
        tok = tokens.items[i] -> tok;
        switch(tok)
        {
        case TOK_NAME:
        case TOK_NTNAME:
        case TOK_AXISNAME:
        case TOK_ATSIGN:
        case TOK_PERIOD:
        case TOK_DPERIOD:
            {
                args.append(ls = new Expression(ownerV,proc,EXF_LOCSTEP));
                E( ls -> step -> parse(tokens,i,ownerV) );
                slashPending = FALSE;
                nameWas = TRUE;
            };
            break;
        case TOK_DSLASH:
            {
                args.append(ls = new Expression(ownerV,proc,EXF_LOCSTEP));
                ls -> step -> set(AXIS_DESC_OR_SELF, EXNODE_NODE);
            };
            // no break here?
        case TOK_SLASH:
            {
                if (slashPending)
                    Err(proc -> situation, ET_EXPR_SYNTAX);
                slashPending = TRUE;
                i++;
                if (tokens.items[i] -> tok == TOK_END) 
                    Err(proc -> situation, ET_EMPTY_PATT);
            };
            break;
        case TOK_VERT:
        case TOK_END:
        default: getaway = TRUE;
        };
    };
    if (slashPending && nameWas)
        Err(proc -> situation, ET_EMPTY_PATT);

    return OK;
}


/*================================================================
eFlag parseBasic()
    parses the basic expression in tokenizer 't' between positions
    'from' and 'to' inclusive. The basic expression is guaranteed
    to contain no operators (except for / and //) nor outer 
    parentheses.
================================================================*/

eFlag Expression::parseBasic(Tokenizer &t, int from, int to)
{
    Expression *e, *lp;
    // find the start of the filtering predicates
    int fstart, fright, fleft;
    ExToken tok;

    switch(t.items[from] -> tok)
    {
    case TOK_VAR:
    case TOK_LITERAL:
    case TOK_NUMBER:
        fstart = from + 1;
        break;
    case TOK_FNAME:
        {
            t.getDelim(fstart = from + 1);
            fstart++;
        };
        break;
    case TOK_LPAREN:
        {
            t.getDelim(fstart = from);
            fstart++;
        };
        break;
    default:
        fstart = -1;
    };

//#pragma Msg("adding '+1':")
    if ((fstart != -1) && (fstart <= to))
    {
        switch(t.items[fstart] -> tok)
        {
        case TOK_LBRACKET:
        case TOK_SLASH:
        case TOK_DSLASH:
            {
                // parse the filtered expression into args[0]
                e = new Expression(ownerV,proc);
                E( e -> parse(t, from, fstart - 1));
                args.append(e);
                //
                functor = EXF_FILTER;
                type = EX_NODESET;
                // e = new Expression(ownerV);

                fleft = fstart;
                while (t.items[fleft] -> tok == TOK_LBRACKET)
                {
                    t.getDelim(fright = fleft);
                    if ((t.items[fright] -> tok == TOK_END) || (fright > to))
                        Err(proc -> situation, ET_RBRACKET_EXP);
                    if (fleft + 1 == fright)
                        Err(proc -> situation, ET_EXPR_SYNTAX);
                    E( (e = new Expression(ownerV,proc)) -> parse(t,
                        fleft + 1, fright - 1) );
                    args.append(e);
                    fleft = fright + 1;
                };
                if (((tok = t.items[fleft] -> tok) == TOK_SLASH)
                    || (tok == TOK_DSLASH))
                {
                    E( (lp = new Expression(ownerV, proc, EXF_LOCPATH)) -> parseLP(
                        t,fleft,TRUE) );
                    hasPath = TRUE;
                    args.append(lp);
                };
                if (fleft != to + 1)
                    Err(proc -> situation, ET_EXPR_SYNTAX);
                return OK;
            };
            break;
        }
    };

    DStr temp;
    tok = t.items[from] -> tok;
    t.items[from] -> speak(temp,SM_OFFICIAL);
    if ((tok == TOK_VAR) || (tok == TOK_LITERAL)
        || (tok == TOK_NUMBER))
    {
        switch(t.items[from] -> tok)
        {
        case TOK_VAR:
            {
                functor = EXF_VAR;
                type = EX_UNKNOWN;
                E( (pName = new QName(proc)) -> setLogical(temp, 
                    &(NZ(ownerV) -> namespaces), FALSE) );
            }; break;
        case TOK_LITERAL:
            {
                functor = EXF_ATOM;
                type = EX_STRING;
                patomstring = new Str(temp);
            }; break;
        case TOK_NUMBER:
            {
                functor = EXF_ATOM;
                type = EX_NUMBER;
                *(patomnumber = new Number) = temp;
            }; break;
        };
        if (to != from)
            Err(proc -> situation, ET_EXPR_SYNTAX);
    }
    else
    {
        if (tok == TOK_FNAME)
        {
            ExFunctor funcNo;
            ExType funcType;
            getFunctionInfo(temp,funcNo,funcType);
            if (funcNo != EXFF_NONE)
            {
                functor = funcNo;
                type = funcType;
            }
            else
            {
                functor = EXF_OTHER_FUNC;
                E( (pName = new QName(proc)) -> setLogical(temp,
                    &(NZ(ownerV) -> namespaces), FALSE) );
                type = EX_UNKNOWN;
            };
            int i = from+1,
                j;
            assert(t.items[i] -> tok == TOK_LPAREN);
            i++;
            // original loop test:
            // while (t.items[j = t.findTop(TOK_COMMA,i)] -> tok != TOK_END)
            while (((j = t.findTop(TOK_COMMA,i)) <= to) && (t.items[j] -> tok != TOK_END))
            {
                switch(t.items[j-1] -> tok)
                {   
                case TOK_COMMA:
                case TOK_LPAREN:
                    Err(proc -> situation, ET_EXPR_SYNTAX);
                };
                args.append(e = new Expression(ownerV,proc));
                E( e -> parse(t,i,j-1) );
                i = j+1;
            };

            if ((t.items[j = t.findTop(TOK_RPAREN,i)]->tok == TOK_END) || (j > to))
                Err(proc -> situation, ET_RPAREN_EXP);
            if(t.items[j-1] -> tok == TOK_COMMA)
                Err(proc -> situation, ET_EXPR_SYNTAX);
            if (j > i)  // if any args
            {
                args.append(e = new Expression(ownerV,proc));
                E( e -> parse(t,i,j-1) );
            }
            if (to != j)
                Err(proc -> situation, ET_EXPR_SYNTAX);
        } // end "tok == TOK_FNAME"
        else
        {   // it must be a LocPath
            type = EX_NODESET;
            functor = EXF_LOCPATH;
            int howfar = from;
            E( parseLP(t, howfar) );
            if (howfar != to + 1)
                Err(proc -> situation, ET_EXPR_SYNTAX);
        }
    }
    return OK;
}

/*================================================================
eFlag parse()
    translates the given token list into an expression (a tree of
    'Expression' objects plus some leaves).
INPUT
    t           a tokenizer whose tokenize() method has been called
    from,to     first and last position in the token list the parsing
                applies to (i.e. a complex expression will parse the
                subexpressions with the same tokenizer but different
                limits)
================================================================*/

eFlag Expression::parse(Tokenizer& t, int from, int to)
// isOp, skipParens
{
    int i;
    ExToken 
        token, 
        mintoken = TOK_NONE;
    int precedence,
        minprec = 999,
        minndx = -1,
        leftmost,
        arity;

    if (from > to)
        Err(proc -> situation, ET_EXPR_SYNTAX);

    t.stripParens(from,to);
    // search from right to left (left-associativity)
    for (i = to; i >= from; i--)
    {
        switch(token = t.items[i] -> tok)
        {
        case TOK_RPAREN:
        case TOK_RBRACKET:
            {
                // reverse search:
                E( t.getDelim(i,TRUE) ); // i is decremented at loop end
                if (i == -1)
                    Err(proc -> situation, ET_LPARCKET_EXP);
            };
            break;
        default: 
            {
                if (isOp(token, precedence) && (precedence < minprec))
                {
                    minprec = precedence;
                    minndx = leftmost = i;
                    mintoken = token;
//                    if (token == TOK_OR) break;
                }
                else 
                    if (token == mintoken)
                        leftmost = i;
            };
        };
    };

    //minndx now points to the rightmost lowest-precedence operator
    // leftmost points at its leftmost occurence

    if (minndx == -1)
        E( parseBasic(t, from, to) )
    else 
    {
        int tablendx = t.items[minndx] -> tok - TOKGROUP_OPERATORS;
        functor = opTable[tablendx].fu;
        type = opTable[tablendx].ty;
        arity = opTable[tablendx].arity;
        Expression *e = new Expression(ownerV,proc);

        args.append(e);
        switch(arity)
        {
        case 1: 
            {
            if (minndx != from)
                Err(proc -> situation, ET_EXPR_SYNTAX)
            else
                E( e -> parse(t,from+1,to) );
            };
            break;
        case 2:
            {
                E( e -> parse(t,from,minndx - 1) );
                args.append(e = new Expression(ownerV,proc));
                E( e -> parse(t,minndx + 1, to) );
            };
            break;
        default:
            {
                E( e -> parse(t,from,leftmost - 1) );
                int another = leftmost, 
                    lastone = leftmost;
                // tom 24-10-00
                // the following fails for "x and not(x and x)"
                // t.getDelim(another);
                // changing it to:
                another = t.findTop(t.items[another]->tok, another+1);
                while((another <= to) && (t.items[another]->tok != TOK_END))
                {
                    args.append(e = new Expression(ownerV,proc));
                    E( e -> parse(t, lastone + 1, another - 1));
                    lastone = another;

                    // tom 14-11-00
                    // t.getDelim(another);     failed too, for "x and x and (x and x)"
                    another = t.findTop(t.items[another]->tok, another+1);
                };
                args.append(e = new Expression(ownerV,proc));
                E( e -> parse(t,lastone + 1, to) );
            };
        };
    }
    return OK;
}


void Expression::setAtom(Context *c)
{
    clearContent();
    functor = EXF_ATOM;
    type = EX_NODESET;
    patomnodeset = c;
}

void Expression::setAtom(const Number& n)
{
    clearContent();
    functor = EXF_ATOM;
    type = EX_NUMBER;
    *(patomnumber = new Number) = (Number&) n;
}

void Expression::setAtom(Bool b)
{
    clearContent();
    functor = EXF_ATOM;
    type = EX_BOOLEAN;
    atombool = b;
}

void Expression::setAtom(const DStr &s)
{
    clearContent();
    functor = EXF_ATOM;
    type = EX_STRING;
    patomstring = new Str(s);
}

/*================================================================
setFragment
    sets the expression to point to a 'result tree fragment' - a newly
    constructed tree - whose address it returns
================================================================*/

Tree *Expression::setFragment()
{
    functor = EXF_FRAGMENT;
    type = EX_NODESET;
    return pTree = new Tree("RTF",FALSE, proc); // not an XSL tree
}

#define funcIsOperator(f) ((EXFO_OR <= f) && (f <= EXFO_Z))
#define funcIsBuiltin(f) ((EXF_FUNCTION <= f) && (f <= EXFF_NONE))

eFlag Expression::eval(Expression &retxpr, Context *c)
{
    // puts(isPattern ? "T" : "F");
    assert(!isPattern && "evaluating pattern!");
    Context *newc;
    switch(functor)
    {
    case EXF_ATOM:
        {
            //cannot use retxpr = *this !!!
            switch(type)
            {
            case EX_STRING:
                retxpr.setAtom(*patomstring);
                break;
            case EX_NUMBER:
                retxpr.setAtom(*patomnumber);
                break;
            case EX_NODESET:
                retxpr.setAtom(patomnodeset -> copy());
                break;
            case EX_BOOLEAN:
                retxpr.setAtom(atombool);
                break;
            default: assert(0);
            }
        }; break;
    case EXF_VAR:
        {
            Expression *ex = proc -> getVarBinding(*pName);
            if (!ex)
                Err(proc -> situation, ET_VARIABLE_NOT_FOUND);
            E( ex -> eval(retxpr, c) );
        }; break;
    case EXF_LOCPATH:
    case EXFO_UNION:   
        {
            assert(c && "context is null!");
            E( createContext(newc = c) );
            // assign newc directly without copying
            retxpr.setAtom(newc -> copy()); 
            delete newc; 
        }; break;
    case EXF_OTHER_FUNC: // other function
        {
            Err1(proc -> situation, ET_FUNC_NOT_SUPPORTED,pName -> getname());
            // would go like:
            // E( callByName(retxpr, *pName, args, c) );
        }; break;
    case EXF_FILTER:
        {
            assert(c && "context is null!");
            E( createContext(newc = c, c -> getPosition()) );
            retxpr.setAtom(newc -> copy());
            delete newc;
        }; break;
    case EXF_STRINGSEQ:
        {
            DStr result;
            Expression temp(ownerV,proc);
            int i,
                argsNumber = args.number();
            for (i = 0; i < argsNumber; i++)
            {
                E( args[i] -> eval(temp, c) );
                result += (temp.tostring());
            };
            retxpr.setAtom(result);
        }; break;
    case EXF_FRAGMENT:
        {
            newc = new Context(proc);
            newc -> set(&(pTree -> root));
            retxpr.setAtom(newc -> copy());
            delete newc;
        }; break;
    default: 
        {
            int i, 
                argsNumber = args.number();
            // was: ExprList atoms(argsNumber); (should be "log" if anything)
            ExprList atoms(LIST_SIZE_1);
            Expression *ex;
            for (i = 0; i < argsNumber; i++)
            {
                ex = new Expression(ownerV,proc);
                E( args[i]->eval(*ex, c) );
                atoms.append(ex);
            };
            if (funcIsOperator(functor))
                    E( callOp(retxpr, atoms) )           //an operator
            else 
                if (funcIsBuiltin(functor))
                    E( callFunc(retxpr, atoms, c) )    //a core XPath function
                else
                    Err1(proc -> situation, ET_FUNC_NOT_SUPPORTED,pName -> getname());
            atoms.freeall(FALSE);
        };
    };
    return OK;
}

template<class T>
Bool hardCompare(ExFunctor op, T b1, T b2)
{
        Str p,q;
    switch(op)
    {
    case EXFO_EQ: return (Bool) (b1 == b2); break;
    case EXFO_NEQ: return (Bool) !(b1 == b2); break;
    case EXFO_LT: return (Bool) (b1 < b2); break;
    case EXFO_GT: return (Bool) (b2 < b1); break;
    case EXFO_LE: return (Bool) ((b1 < b2) || (b1 == b2)); break;
    case EXFO_GE: return (Bool) ((b2 < b1) || (b1 == b2)); break;
    default: assert(0);
    }
    return FALSE; //just to return something
}

ExFunctor _invertOp(ExFunctor op) 
{
    switch(op)
    {
    case EXFO_EQ: return EXFO_EQ; break;
    case EXFO_NEQ: return EXFO_NEQ; break;
    case EXFO_LT: return EXFO_GT; break;
    case EXFO_GT: return EXFO_LT; break;
    case EXFO_LE: return EXFO_GE; break;
    case EXFO_GE: return EXFO_LE; break;
    default: assert(!"_invertOp"); return EXF_NONE; // to return something
    }
}

Bool Expression::compareCC(ExFunctor op, const Context &c1, const Context &c2)
{
    DStr s1, s2;
    Context *c1prime = ((Context&) c1).copy(),
        *c2prime = ((Context&) c2).copy();
    Bool resulting = FALSE;
    c1prime->reset();
    while (c1prime->current())
    {
        c2prime->reset();
        while(c2prime->current())
        {
            E( c1prime->current() -> value(s1, c1prime) );
            E( c2prime->current() -> value(s2, c2prime) );
            if (hardCompare(op,s1,s2))
            {
                resulting = TRUE;
                break;
            }
            c2prime->shift();
        };
        c1prime->shift();
    };
    delete c1prime;
    delete c2prime;
    return resulting;
}

Bool Expression::compareCS(ExFunctor op, const Context &c1, const DStr &s2)
{
    DStr s1;
    Bool resulting = FALSE;
    Context *c = ((Context&) c1).copy();

    c -> reset();
    while(c -> current())
    {
        E( c -> current() -> value(s1, c) );
        if (hardCompare(op, s1, s2))
        {
            resulting = TRUE;
            break;
        }
        c -> shift();
    };
    delete c;
    return resulting;
}

Bool Expression::compareCN(ExFunctor op, const Context &c1, const Number& n2)
{
    Number n1;
    DStr s1;
    Context *c = ((Context&) c1).copy();
    Bool resulting = FALSE;

    c -> reset();
    while(c -> current())
    {
        E( c -> current() -> value(s1, c) );
        n1 = s1;
        if (hardCompare(op, n1, (Number)n2))
        {
            resulting = TRUE;
            break;
        }
        c -> shift();
    };
    delete c;
    return resulting;
}

eFlag Expression::compare(Bool &result, Expression &other, ExFunctor op)
// Both *this and other are assumed to be ATOMS.
{
    assert(functor == EXF_ATOM);
    assert(other.functor == EXF_ATOM);

    ExType histype = other.type;
    //
    if (type == EX_NODESET)
    {
        if (other.type == EX_BOOLEAN)
            result = hardCompare(op, tobool(), other.tobool());
        else
        {
            Context& mynodeset = tonodeset();
            switch(other.type)
            {
            case EX_NODESET:
                result = compareCC(op, mynodeset, other.tonodesetRef());
                break;
            case EX_STRING:
                result = compareCS(op, mynodeset, other.tostring());
                break;
            case EX_NUMBER:
                result = compareCN(op, mynodeset, other.tonumber());
                break;
            default: assert(0);
            };
            delete &mynodeset;
        }
    }
    else 
    {
        if (histype == EX_NODESET)
            E( other.compare(result, *this, _invertOp(op)) )
        else
        {   // none of the two are nodesets
            if (type == EX_BOOLEAN || histype == EX_BOOLEAN)
                result = hardCompare(op, tobool(), other.tobool());
            else
            {
                  if (type == EX_NUMBER || histype == EX_NUMBER)
                      result = hardCompare(op, tonumber(), other.tonumber());
                  else
                {
                    if (type == EX_STRING || histype == EX_STRING)
                        result = hardCompare(op, tostring(), other.tostring());
                    else
                        assert(0);
                }
            }
        }; 
    }
    return OK;
}

eFlag Expression::callOp(Expression& retxpr, ExprList& atoms)
{
    int i,
        atomsNumber = atoms.number();
    switch(functor)
    {
    case EXFO_OR:
    case EXFO_AND:
        {
            assert(atomsNumber > 1);
            Bool result;
            result = atoms[0] -> tobool();
            for (i = 1; i < atomsNumber; i++)
            {
                if (functor == EXFO_OR)
                {
                    if (atoms[i] -> tobool())
                    {
                        result = TRUE;
                        break;
                    }
                }
                else    //EXFO_AND
                {
                    if (!atoms[i] -> tobool())
                    {
                        result = FALSE;
                        break;
                    }
                };
            };
            retxpr.setAtom(result);
        }; break;
    case EXFO_EQ:
    case EXFO_NEQ:
    case EXFO_LT:
    case EXFO_LE:
    case EXFO_GT:
    case EXFO_GE:
        {
            assert(atomsNumber == 2);
            Bool result;
            E( atoms[0]->compare(result,*(atoms[1]),functor) );
            retxpr.setAtom(result);
        }; break;
    case EXFO_PLUS:
    case EXFO_MINUS2:
    case EXFO_MULT:
    case EXFO_DIV:
    case EXFO_MOD:
        {
            assert(atomsNumber > 1);
            double result;
            result = (double) (atoms[0] -> tonumber());
            for (i = 1; i < atomsNumber; i++)
            {
                switch(functor)
                {
                case EXFO_PLUS:
                    result += atoms[i] -> tonumber();
                    break;
                case EXFO_MINUS2:
                    result -= atoms[i] -> tonumber();
                    break;
                case EXFO_MULT:
                    result *= atoms[i] -> tonumber();
                    break;
                case EXFO_DIV:
                    result /= atoms[i] -> tonumber();
                    break;
                case EXFO_MOD:
                    {
                        double d = atoms[i] -> tonumber();
                        result = result - d * floor(result/d);
                    };
                    break;
                };
            };
            retxpr.setAtom(Number(result));
        }; break;
    case EXFO_MINUS1:
        {
            assert(atomsNumber == 1);
            retxpr.setAtom(Number(-(double)(atoms[0] -> tonumber())));
        }; break;
    };
    return OK;
}

/*================================================================
callFunc
    calls the built-in XPath or XSLT function as determined by 
    'this -> functor'. Returns an expression in 'retxpr'.
    'atoms' is a list of atomized arguments, 'c' is the current context
    necessary for certain functions.
================================================================*/

#define checkArgsCount(x) if (atomsNumber != x)\
    Err(proc -> situation, ET_BAD_ARGS_N);
#define checkArgsCountMax(x) if (atomsNumber > x)\
    Err(proc -> situation, ET_BAD_ARGS_N);
#define checkArgsCountMin(x) if (atomsNumber < x)\
    Err(proc -> situation, ET_BAD_ARGS_N);
#define checkArgsCountBetween(x,y) if ((atomsNumber < x) || \
    (atomsNumber > y)) Err(proc -> situation, ET_BAD_ARGS_N);

// only check for being a nodeset
#define checkIsNodeset(x) if (atoms[x] -> type != EX_NODESET)\
    Err(proc -> situation, ET_BAD_ARG_TYPE);

// Everything is a string, in a cosmic sense
#define checkIsString(x)
#define checkIsString2(x,y) 
#define checkIsNumber(x)

/*................
firstOccurence
Finds the first complete occurence of q in p; returns the 0-based starting
position, or -1 if not found
................*/

int firstOccurence(char *p, char *q)
{
    int i = 0,
        iCurr = -1,
        j = 0;
    do
    {
        i = ++iCurr; j = 0;
        while (p[i] && q[j] && (p[i] == q[j]))
        {
            i++; j++;
        };
    }
    while (p[i] && q[j]);
    if (q[j]) 
        return -1;
    else
        return iCurr;
}

/*................
getBetween
Returns in s the portion of the source string between form and to inclusive.
If to == -1 then copies the whole rest of the string.
................*/

void getBetween(Str& s, char *source, int from, int to)
{
    assert(source);
    int len = strlen(source);
    if (from < 0) from = 0;
    if (to >= len) to = len-1;
    if ((from >= len) || ((to != -1) && (from > to)))
        s = "";
    else
    {
        if (to == -1)
            s = source + from;
        else
            s.nset(source + from, to-from + 1);
    }
}

/*................
getCurrValue
Returns the string value of the current node
................*/

eFlag getCurrValue(Str &s, Context *c)
{
    Vertex *v;
    DStr temp;
    if (!!(v = c -> current()))
        E( v -> value(temp,c) )
    else
        s.empty();
    s = temp;
    return OK;
}


eFlag Expression::callFunc(Expression &retxpr, ExprList &atoms, Context *c)
{
    Vertex *v;
    int atomsNumber = atoms.number();
    switch(functor)
    {
    case EXFF_LAST:
        {
            checkArgsCount(0);
            retxpr.setAtom( Number(c -> getSize()) );
        }; break;
    case EXFF_POSITION:
        {
            checkArgsCount(0);
            retxpr.setAtom( Number(c -> getPosition() + 1) );
        }; break;
    case EXFF_COUNT:
        {
            checkArgsCount(1);
            retxpr.setAtom( 
                Number(atoms[0] -> tonodesetRef().getSize()) );
        }; break;
    case EXFF_LOCAL_NAME:
    case EXFF_NAMESPACE_URI:
    case EXFF_NAME:
        {
            checkArgsCountMax(1);
            DStr s;
            if (!atomsNumber)
                v = (c -> isFinished()) ? NULL : c -> current();
            else
            {
                checkIsNodeset(0);
                const Context& newc = atoms[0] -> tonodesetRef();
                v = (newc.isVoid()? NULL : newc.current());
            };
            if (v)
            {
                const QName& q = v -> getName();
                switch(functor)
                {
                case EXFF_NAME:
                    q.speak(s,SM_OFFICIAL); break;
                case EXFF_LOCAL_NAME:
                    s = q.getLocal(); break;
                case EXFF_NAMESPACE_URI:
                    s = q.getUri(); break;
                };
            };
            retxpr.setAtom(s);
        };
        break;

    case EXFF_STRING:
        {
            Str s;
            checkArgsCountMax(1);
            if (!atomsNumber)
                E( getCurrValue(s, c) )
            else
                s = atoms[0] -> tostring();
            retxpr.setAtom(s);
        }; break;

    case EXFF_CONCAT:
        {
            checkArgsCountMin(2);
            DStr s;
            for (int k = 0; k < atomsNumber; k++)
            {
                checkIsString(k);
                s += atoms[k] -> tostring();
            };
            retxpr.setAtom(s);
        }; break;

    case EXFF_STARTS_WITH:
        {
            checkArgsCount(2);
            checkIsString2(0,1);
            retxpr.setAtom((Bool) !firstOccurence(
                // changing tostringCharPtr() to tostring():
                atoms[0] -> tostring(),
                atoms[1] -> tostring()));
        }; break;

    case EXFF_CONTAINS:
        {
            checkArgsCount(2);
            checkIsString2(0,1);
            retxpr.setAtom((Bool) (firstOccurence(
                // changing tostringCharPtr() to tostring():
                atoms[0] -> tostring(),
                atoms[1] -> tostring()) != -1));
        }; break;

    case EXFF_SUBSTRING_BEFORE:
    case EXFF_SUBSTRING_AFTER:
        {
            Str s;
            const Str& 
                theBigger = atoms[0] -> tostring(),
                theSmaller = atoms[1] -> tostring();
            checkArgsCount(2);
            checkIsString2(0,1);
            int where = firstOccurence(
                // changing tostringCharPtr() to tostring():
                theBigger,
                theSmaller);
            if (where == -1)
                s.empty();
            else
            {
                if (functor == EXFF_SUBSTRING_BEFORE)
                {
                    if (where == 0)
                        s.empty();
                    else
                        getBetween(s, theBigger, 0, where-1);
                }
                else
                    getBetween(s, theBigger, where + theSmaller.length(), -1);
            };
            retxpr.setAtom(s);
        }; break;

    case EXFF_SUBSTRING:
        {
            checkArgsCountBetween(2,3); 
            checkIsString(0); checkIsNumber(1);
            /* useless test causing a warning in MSVC:

                if (atomsNumber == 3)
                checkIsNumber(2);
            */
            Str s;
            Number from_ = atoms[1] -> tonumber() - 1;
            if (!from_.isNaN() && !from_.isInf())
            {
                int from = from_.round(), 
                    to = -1;
                if (atomsNumber > 2)
                {
                    // use length in 3rd argument
                    Number len = atoms[2] -> tonumber();
                    if (len <= 0 || len.isNaN())
                        to = -2;
                    else if (!len.isInf())
                        to = from + len.round() - 1;  // otherwise it remains -1
                }
                getBetween(s, atoms[0] -> tostring(), from, to);
            }
            retxpr.setAtom(s);
        }; break;

    case EXFF_STRING_LENGTH:
        {
            checkArgsCountBetween(0,1);
            if (atomsNumber)
            {
                checkIsString(0);
                retxpr.setAtom(Number(atoms[0] -> tostring().length()));
            }
            else
            {
                Str s;
                E( getCurrValue(s, c) );
                retxpr.setAtom(Number(s.length()));
            }
        }; break;

    case EXFF_NORMALIZE_SPACE:
        {
            checkArgsCountBetween(0,1);
            Str s;
            if (atomsNumber)
            {
                checkIsString(0);
                s = atoms[0] -> tostring();
            }
            else
                E( getCurrValue(s, c) );
            char *p = (char*) s;
            DStr stripped;
            skipWhite(p);
            while(*p)
            {
                if (isWhite(*p))
                {
                    skipWhite(p);
                    if (*p)
                        stripped += ' ';
                    p--;
                }
                else
                    stripped += *p;
                p++;
            }
            retxpr.setAtom(stripped);
        }; break;

    case EXFF_TRANSLATE:
        {
            checkArgsCount(3);
            checkIsString2(0,1);
            checkIsString(2);

            DStr resulting;
            Str baseStr = atoms[0] -> tostring(),
                srcStr = atoms[1] -> tostring(),
                destStr = atoms[2] -> tostring();
            char *q,
                // changing tostringCharPtr() to tostring():
                *p = baseStr,
                *src = srcStr,
                *dest = destStr;
            int ndx,
                destlen = strlen(dest);
            while(*p)
            {
                q = strchr(src, *p);
                if (q)
                {
                    ndx = (int)(q-src);
                    if (ndx < destlen)
                        resulting += dest[ndx];
                }
                else
                    resulting += *p;
                p++;
            };
            retxpr.setAtom(resulting);
        }; break;

    case EXFF_BOOLEAN:
        {
            checkArgsCount(1);
            retxpr.setAtom(atoms[0] -> tobool());
        }; break;

    case EXFF_NOT:
        {
            checkArgsCount(1);
            retxpr.setAtom((Bool)!(atoms[0] -> tobool()));
        }; break;

    case EXFF_TRUE:
        {
            checkArgsCount(0);
            retxpr.setAtom(TRUE);
        }; break;

    case EXFF_FALSE:
        {
            checkArgsCount(0);
            retxpr.setAtom(FALSE);
        }; break;

    case EXFF_LANG:
        {
            checkArgsCount(1);
            checkIsString(0);
            Warn(proc -> situation, W_UNSUPP_LANG);
            retxpr.setAtom(FALSE);
        }; break;

    case EXFF_NUMBER:
        {
            checkArgsCountMax(1);
            Number n;
            if (!atomsNumber)
            {
                Str s;
                E( getCurrValue(s, c) );
                n = s;
            }
            else
                n = atoms[0] -> tonumber();
            retxpr.setAtom(n);
        }; break;

    case EXFF_SUM:
        {
            DStr s;
            Number n, sum = 0;
            checkArgsCount(1);
            checkIsNodeset(0);
            Context *newc = &(atoms[0] -> tonodeset());
            newc -> reset();
            while (!newc -> isFinished())
            {
                E( newc -> current() -> value(s, newc) );
                n = s;
                if (n.isNaN())
                {
                    sum.setNaN(); 
                    break;
                };
                sum = sum + n;
                newc -> shift();
            };
            delete newc;
            retxpr.setAtom(sum);
        }; break;

    case EXFF_FLOOR:
    case EXFF_CEILING:
    case EXFF_ROUND:
        {
            checkArgsCount(1);
            checkIsNumber(0);
            Number n = atoms[0] -> tonumber();
            switch(functor)
            {
            case EXFF_FLOOR:
                n = floor((double)n); break;
            case EXFF_CEILING:
                n = ceil((double)n); break;
            case EXFF_ROUND:
                n = floor((double)n + .5); break;
            };
            retxpr.setAtom(n);
        }; break;

    case EXFF_DOCUMENT:
        {
            checkArgsCount(1);
            DStr location;
            Tree *newtree;
            Context *newc = new Context(proc);

            // Current node doesn't change
            newc->setCurrentNode (c->getCurrentNode());

            if (atoms[0] -> type == EX_NODESET)
            {
                const Context& ctxt = atoms[0] -> tonodesetRef();
                int ctxtNumber = ctxt.getSize();
                for (int k = 0; k < ctxtNumber; k++)
                {
                    E( ctxt[k] -> value(location, c) );
                    E( proc -> readTreeFromURI(newtree, location, 
                        // the following used to be NZ(ownerV) -> ownerT -> name
                        proc -> baseForVertex(NZ(ownerV)), FALSE) );
                    // check for duplicities and correct URI sorting!
                    newc -> append(&(newtree -> root));
                };
            }
            else
            {
                location = atoms[0] -> tostring();
                E( proc -> readTreeFromURI(newtree, location, 
                    proc -> baseForVertex(NZ(ownerV)), FALSE) );
                newc -> append(&(newtree -> root));
            }
            retxpr.setAtom(newc);
        }; break;

    case EXFF_GENERATE_ID:
        {
            DStr s;
            switch(atomsNumber)
            {
            case 0:
                v = (c -> isFinished() ? NULL : c -> current());
                break;
            case 1:
                {
                    checkIsNodeset(0);
                    const Context& newc = atoms[0] -> tonodesetRef();
                    v = (newc.isVoid()? NULL : newc.current());
                }; break;
            default:
                Err(proc -> situation, ET_BAD_ARGS_N);
            };
            if (v)
            {
                s = "i__";
                s += v -> stamp;
            }
            retxpr.setAtom(s);
        }; break;

    case EXFF_SYSTEM_PROPERTY:
        {
            checkArgsCount(1);
            checkIsString(0);
            QName q(proc);
            q.setLogical(atoms[0] -> tostring(), &(NZ(ownerV) -> namespaces), FALSE);
            if (q.uri == proc -> stdPhrase(PHRASE_XSL_NAMESPACE))
            {
                if (q.getLocal() == (const char*) "version")
                    retxpr.setAtom(Number(1.0));
                else if (q.getLocal() == (const char*) "vendor")
                    retxpr.setAtom(Str("Ginger Alliance"));
                else if (q.getLocal() == (const char*) "vendor-url")
                    retxpr.setAtom(Str("www.gingerall.com"));
                else
                  retxpr.setAtom("");
                /* retxpr.setAtom(*theEmptyString); */
            }
            else
              retxpr.setAtom("");
              /* retxpr.setAtom(*theEmptyString); */
        }; break;

    case EXFF_CURRENT:
        {
            Context *newc = new Context(proc);
            newc -> set (c -> getCurrentNode());
            retxpr.setAtom (newc);
        }; break;

    default:
        Err1(proc -> situation, ET_FUNC_NOT_SUPPORTED, getFuncName(functor));
    }
    return OK;
}

/*
eFlag Expression::createLPContext(Context *&c, int baseNdx)
{
    assert(functor == EXF_LOCPATH);
    Context *myctxt;
    if (c -> isVoid()) 
    {
        c = new Context;
        return OK;
    };
    ( myctxt = c ) -> setCurrentNode ( c->getCurrentNode() );
    E( args[0] -> createContext(myctxt, baseNdx) );
    E( createLPContextSum(1, c = myctxt) );
    delete myctxt;
//#pragma Msg("createLPContext")
//    if (c -> getSize() >= 10)
//    printf("created LPContext of size %d\n",c -> getSize());
    return OK;
}
*/

eFlag Expression::createLPContext(Context *&c, int baseNdx, Vertex *givenGlobalCurrent /* = NULL */)
{
    assert(functor == EXF_LOCPATH);
    Context *theResult = new Context(proc),
        info(proc);
    info.setCurrentNode(givenGlobalCurrent ? givenGlobalCurrent : c -> current());
    theResult -> setCurrentNode(c -> getCurrentNode());
    E( createLPContextLevel(0, args.number(), c -> current(), info, theResult) );
    theResult -> sort();
    theResult -> uniquize();
    c = theResult;
    return OK;
}

/*
 *  createLPContextLevel
 *  ranges over all nodes that satisfy the stepLevel-th step, calling self
 *  recursively until the last one is reached. The vertices satisfying the last
 *  step are added to theResult. 
 *  Base is passed from the preceding step and used for expression
 *  evaluation. info holds the 'globally current' vertex.
 *  The purpose of this routine is to generate a context without having to
 *  also generate the intermediate contexts for each step. Also, some of the predicates
 *  may be known to use last(), in which case we first have to compute the number
 *  of nodes that reach such a predicate at all. 
 */

eFlag Expression::createLPContextLevel(
    int stepLevel, int stepsCount, Vertex *base,
    Context &info, Context *theResult)
{
    assert(functor == EXF_LOCPATH);
    int i, j, init,
        predsCount = args[stepLevel] -> step -> preds.number(),
        lastBad = -1;     // last bad predicate, or the step itself

    // keep a stack of positions, one for each predicate
    List<int> reached(predsCount),  // serves as position for next pred
        totalReached(predsCount);   // serves as size for next (bad) pred

    // there will be as many dry (size-counting) runs as there are bad preds
    Bool dryRun = TRUE,
        quitThisRound = FALSE, quitThisVertex = FALSE;
    for (i = 0; i <= predsCount; i++)
    {
        if (i == predsCount)
            dryRun = FALSE;
        if (!dryRun || args[stepLevel] -> step -> preds[i] -> usesLast)
        {
            // initialize the size arrays: 
            // append base values for preds past the last bad one, 
            // up to this bad one (incl.)
            for (init = 0; init <= lastBad; init++)
                reached[init] = 0;
            for (init = lastBad + 1; init <= i; init++)
            {
                reached.append(0);
                totalReached.append(-1);     // -1 just for safety
            };

            // locally current vertex 
            Vertex *locCurr = NULL;

            quitThisRound = FALSE;
            do
            {
                // shift the locally current vertex
                E( args[stepLevel] -> step -> shift(locCurr, base) );
                if (locCurr)
                {
                    // was: if (lastBad < 0) ++reached[0];
                    if ((lastBad < 0) || !dryRun) ++reached[0];
                    quitThisVertex = FALSE;
                    for (j = 0; j < i; j++)
                    {
                        Bool satisfies;
                        info.deppendall();
                        info.setVirtual(locCurr, reached[j] - 1, totalReached[j]);

                        Expression *thisPred = 
                            args[stepLevel] -> step -> preds[j];
                        // find whether we're in position bounds for this pred
                        switch(thisPred -> inBounds(reached[j] - 1))
                        {
                        case 0:
                            {
                                E( thisPred -> trueFor(&info, satisfies) );
                                if (satisfies)
                                    ++reached[j + 1];
                                else
                                    quitThisVertex = TRUE;
                            }; break;
                        case -1:
                            break;
                        case 1:
                            quitThisRound = TRUE;
                            break;
                        };
                        if (quitThisVertex || quitThisRound)
                            break;
                    };
                    if (j == i && !dryRun)     // passed all preds
                    {
                        if (stepLevel < stepsCount - 1)
                            E( createLPContextLevel(
                                stepLevel + 1, stepsCount,
                                locCurr, info, theResult))
                        else
                            theResult -> append(locCurr);

                    }   // if ! dryRun
                }       // if locCurr
            } while (locCurr && !quitThisRound);
            // move all data collected to safe places
            for (init = lastBad + 1; init <= i; init++)
                totalReached[init] = reached[init];
            lastBad = i;
        }               // if bad predicate
    }                   // for, over all preds
    return OK;
}


eFlag Expression::createLPContextSum(Context *&c, Vertex *globalCurrent /* = NULL */)
{
    assert(functor == EXF_LOCPATH);
    Context 
        *newc = new Context(proc), 
        *newc2, *returnedc;
    int cNumber = c -> getSize();
    for (int j = 0; j < cNumber; j++)
    {
        E( createLPContext(returnedc = c, j, globalCurrent) );
        newc2 = newc -> swallow(returnedc);
        delete newc;
        newc = newc2;
        delete returnedc;
    }
    c = newc;
    return OK;
}


/*................................................................
createContext()
    creates a context for this expression, based on its functor.
................................................................*/


eFlag Expression::createContext(Context *& c, int baseNdx /* = -1 */)
{
    Context *newc;
    int i, j, 
        argsNumber = args.number();
    if (baseNdx == -1)
        baseNdx = c -> getPosition();
    switch(functor)
    {
    case EXF_VAR:
        {
            Expression *deref = proc -> getVarBinding(*pName);
            if (!deref)
                Err(proc -> situation, ET_VARIABLE_NOT_FOUND);
            Vertex * current_node = c -> getCurrentNode();
            E( deref -> createContext(c, baseNdx) );
            c->setCurrentNode (current_node);
        };
        break;
    case EXF_ATOM:
        {
            if (type != EX_NODESET)
                Err(proc -> situation, ET_CONTEXT_FOR_BAD_EXPR);
            c = patomnodeset -> copy();
        }; break;
    case EXFO_UNION:
        {
            assert(baseNdx != -1);  // meaningful only for a locpath
            Context 
                *csummand,
                *newc2;
            assert(argsNumber);
            E( args[0] -> createContext(newc = c, baseNdx) );
            for (i = 1; i < argsNumber; i++)
            {
                E( args[i] -> createContext(csummand = c, baseNdx) );
                newc2 = newc -> swallow(csummand);
                delete newc;
                newc = newc2;
                delete csummand;
            }
            // clean up c!
            (c = newc) -> reset();
        }
        break;
    case EXF_LOCPATH:
        {
            E( createLPContext(c, baseNdx) );
        }
        break;
    case EXF_FILTER:
        {
            assert(baseNdx != -1);  // meaningful only for a locpath
            E( args[0] -> createContext(newc = c, baseNdx) );

            /// tom 21-10-00
            /// wild guess:
            newc -> setCurrentNode(c -> getCurrentNode());

            Context *filteredc;
            for (i = 1; i < argsNumber - (int) hasPath; i++)
            {
                filteredc = new Context(proc);
                newc -> reset();
                Bool istrue;
                int newcNumber = newc -> getSize();
                for (j = 0; j < newcNumber; j++)
                {
                    E(args[i] -> trueFor(newc, istrue));
                    if (istrue)
                        filteredc -> append((*newc)[j]);
                    newc -> shift();
                };
                delete newc;
                newc = filteredc;
                if (!newc -> getSize()) break;
            };
            if (hasPath)
            {
                E( args[argsNumber-1] -> createLPContextSum(filteredc = newc, newc -> getCurrentNode()) );
                delete newc;
                newc = filteredc;
            }
            c = newc;
        }
        break;
    case EXF_LOCSTEP:
        {
            assert(step);
            assert(baseNdx != -1);  // meaningful only for a locpath

            /////////
            // E( step -> createContextNoPreds(newc = c, baseNdx) );    - done as follows:
            Context *newc = new Context(proc);
            Vertex *curr = NULL;
            do
            {
                E( step -> shift(curr, (*c)[baseNdx]) );
                if (curr)
                    newc -> append(curr);
            } 
            while (curr);
            /////////

            Context *filteredc;
            int stepPredsNumber = step -> preds.number();
            for (i = 0; i < stepPredsNumber; i++)
            {
                filteredc = new Context(proc);
                newc -> reset();
                Bool istrue;
                int newcNumber = newc -> getSize();
                for (j = 0; j < newcNumber; j++)
                {
                    E( step -> preds[i] -> trueFor(newc,istrue) );
                    if (istrue)
                        filteredc -> append((*newc)[j]);
                    newc -> shift();
                };
                delete newc;
                newc = filteredc;
                if (!newc -> getSize()) break;
            };
            c = newc;
        };
        break;
    default:
        if (funcIsBuiltin(functor))
        {
            Expression resolved(ownerV,proc);
            E( eval(resolved, c) );
            E( resolved.createContext(c, baseNdx) );
        }
        else
        Err(proc -> situation, ET_CONTEXT_FOR_BAD_EXPR);
    };
    return OK;
}


eFlag Expression::matchesSingleStep(Vertex *v, Bool &result)
{
    assert(functor == EXF_LOCSTEP);
    if (!NZ(step) -> matchesWithoutPreds(v))
        RetOK(result, FALSE);
    if (!step -> preds.number())
        RetOK(result, TRUE);
    if (!v -> parent)
        RetOK(result, FALSE);
    if (!step -> positional)
    {
        Context *c = new Context(proc);
        c -> set(v);
        Bool stillOK = TRUE;
        for (int i = 0; i < step -> preds.number() && stillOK; i++)
            E(step -> preds[i] -> trueFor(c,stillOK));
        delete c;
        RetOK(result,stillOK);
    }
    else // positional case
    {
        Context *c = new Context(proc),*newc;
        c -> set(v -> parent);
        E( createContext(newc = c, 0) );
        result = (newc -> contains(v));
        delete c;
        delete newc;
    }
    return OK;
}


eFlag Expression::matchesSinglePath(Vertex *v, int lastIndex, Bool& result)
{
    assert(functor == EXF_LOCPATH);
    // Vertex *v = c -> current();
    int i;
    Vertex *w = v;
    // for (i = args.number()-1; i >= 0; i--)
    for (i = lastIndex; i >= 0; i--)
    {
        if (!w) 
            RetOK(result, FALSE);
        switch(args[i] -> step -> ax)
        {
        case AXIS_ROOT:
            if (i)
                assert(!"root not first");
            E( args[i] -> matchesSingleStep(w, result) );
            if (!result) RetOK(result, FALSE);
            break;
        case AXIS_CHILD:
        case AXIS_ATTRIBUTE:
            {
                E( args[i] -> matchesSingleStep(w, result) );
                if (!result) RetOK(result, FALSE);
                w = w -> parent;
            };
            break;
        case AXIS_DESC_OR_SELF:
            {
                E( args[i] -> matchesSingleStep(w, result) );
                if (!result) RetOK(result, FALSE);
                Vertex *previous = w;
                while (previous)
                {
                    E(matchesSinglePath(previous, i-1, result));
                    if (result)
                        return OK;
                    else
                        previous = previous -> parent;
                };
                RetOK( result, FALSE );
            }
            break;
        default:
            assert(!"bad axis in pattern");
        }
    }
    result = TRUE;
    return OK;
}

/*
 *  optimizePositional()
 *  called for a predicate of a locstep
 *  returns 2 if the predicate uses the last() function
 *    so the size of the context has to be determined
 *  returns 1 if the predicate only uses position()
 *          0 if neither
 */

int Expression::optimizePositional()
{
    int result = 0;
    switch(functor)
    {
    case EXFF_LAST:
        result = 2; break;
    case EXFF_POSITION:
        result = 1; break;
    case EXF_ATOM:
    case EXF_VAR:
    case EXF_LOCPATH:
        /* result = 0; */ break;
    case EXF_FILTER:
    case EXF_STRINGSEQ:
    case EXF_FRAGMENT:
    case EXF_LOCSTEP:
        assert(!"invalid predicate type");
        break;
    default: // all the functions and operators, including EXF_OTHER_FUNC
        {
            int sub = 0;
            for (int i = 0; i < args.number(); i++)
            {
                if (!!(sub = args[i] -> optimizePositional()))
                {
                    result = sub;
                    if (result == 2) break;
                }
            }
        }
    }
    usesLast = (result == 2);
    positional = (result >= 1);
    return result; 
}

/*
 *  optimizePositionBounds()
 *  called for a predicate of a locstep
 *  returns the range of positions that need to be examined for a context
 *  e.g. the predicate in foo[1] will return both set to 1.
 *  the positions returned are 1-based, 0 means "no restriction"
 */

void Expression::optimizePositionBounds()
{
    int from = 0, to = 0;
    switch(functor)
    {
    case EXF_ATOM:
        {
            if (type == EX_NUMBER)
                from = to = NZ(patomnumber) -> round();    // bad values like NaN return 0 which is OK.
        }; break;
    case EXFO_EQ:
    case EXFO_LE:
    case EXFO_LT:
    case EXFO_GE:
    case EXFO_GT:
        {
            if (args[0] -> functor == EXFF_POSITION && 
                args[1] -> functor == EXF_ATOM && args[1] -> type == EX_NUMBER)
            {
                int bound = args[1] -> patomnumber -> round();
                switch(functor)
                {
                case EXFO_EQ: from = to = bound; break;
                case EXFO_LE: to = bound; break;
                case EXFO_LT: to = bound - 1; break;
                case EXFO_GE: from = bound; break;
                case EXFO_GT: from = bound + 1; break;
                }
            }
        }; break;
    }
    optimizePositionFrom = from;
    optimizePositionTo = to;
}

int Expression::inBounds(int position) const
{
    if (optimizePositionTo && position > optimizePositionTo-1)
        return 1;
    if (optimizePositionFrom && position < optimizePositionFrom-1)
        return -1;
    return 0;
}
