/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#if !defined(OutputHIncl)
#define OutputHIncl

#include "datastr.h"
#include "utf8.h"

class DataLine;     // see uri.h

enum OutputMethod
{
    OUTPUT_XML,
    OUTPUT_HTML,
    OUTPUT_TEXT,
    OUTPUT_UNKNOWN
};

enum EscMode
{
    ESCAPING_NONE,
    ESCAPING_URI,
    ESCAPING_ATTR,
//    ESCAPING_CDATA, - use ESCAPING_NONE
    ESCAPING_LT_AMP,
    ESCAPING_HTML_URI,
    ESCAPING_HTML_ATTR
};

#define THE_NAMESPACE_SEPARATOR '`'

// the following must match the size of outputStringAtts[] - 1
// as defined in output.cpp
#define STRING_ITEMS_COUNT 8

//
//  OutputDef
//  stores the output-controlling information for an XSLT tree
//

#define OUTPUT_BUFFER_LIMIT     1024
#define OUTPUT_BUFFER_SIZE      OUTPUT_BUFFER_LIMIT + 64

#define HTML_SPECIAL            TRUE
#define NOT_HTML_SPECIAL        FALSE

class OutputDefinition
{
public:
    OutputDefinition(Processor *proc_);
    ~OutputDefinition();
    eFlag setItemStrCheck(XSL_ATT itemId, const Str& value);
    eFlag setItemStrIfNot(XSL_ATT itemId, const Str& value);
    eFlag setItemStrForce(XSL_ATT itemId, const Str& value);
    eFlag setDefaults();
    eFlag setItemQName(XSL_ATT itemId, const QName& value, Bool doCheck);
    const Str& getValueStr(XSL_ATT itemId) const;
    const QName& getValueQName(XSL_ATT itemId) const;
    Bool askQNameList(XSL_ATT itemId, const QName &what) const;
    int getStatus(XSL_ATT itemId) const;
    OutputMethod getMethod() const;
    Encoding getEncoding() const;
private:
    eFlag setItemStr_(XSL_ATT itemId, const Str& value, Bool doCheck, Bool soft);
    Str stringItems[STRING_ITEMS_COUNT];
    QName
        method;
    QNameList
        cdataElems;
    DataLine *targetDataLine;
    Processor *proc;
};

//
//
//  PhysicalOutputterObj
//
//

class PhysicalOutputLayerObj
{
public:
    PhysicalOutputLayerObj(Processor *proc_);
    ~PhysicalOutputLayerObj();
    eFlag setOptions(DataLine *targetDataLine_, OutputDefinition *outDef_);
    eFlag outputElementStart(const Str& name, 
        const NamespaceStack& namespaces, const int namespace_index,
        const QNameStrList& atts, Bool isEmpty);
    eFlag outputElementEnd(const Str& name, Bool isEmpty);
    eFlag outputText(const Str& contents, Bool outputEscaping, Bool inHTMLSpecial);
    eFlag outputComment(const Str& contents);
    eFlag outputCDataSection(const Str& contents);
    eFlag outputPI(const Str& target, const Str& data);
    eFlag outputDTD(const Str& name, const Str& publicId, const Str& systemId);
    eFlag outputDone();
    eFlag setMethodByDefault(OutputMethod method_);
private:
    Processor *proc;
    DataLine *targetDataLine;
    OutputDefinition *outDef;
    OutputMethod method;
    char buffer[OUTPUT_BUFFER_SIZE],
        smallBuf[SMALL_BUFFER_SIZE];
    int curr;
    Encoding enc;
    eFlag sendOut(const char* data, int length, EscMode escapeMode);
    eFlag sendOutUntil(const char* & data, int length,
        EscMode escapeMode, const char* stoppingText);
    int writeCharacterRef(char* dest, const char *src, EscMode escapeMode);
    eFlag flushBuffer();
};

//
//
//  OutputterObj
//
//

enum OutputterState
{
    STATE_OUTSIDE = 0,
    STATE_IN_MARKUP,
    STATE_IN_ELEMENT,
    STATE_IN_ATTRIBUTE,
    STATE_IN_COMMENT,
    STATE_IN_PI,
    STATE_DONE,
    STATE_UNDEFINED
};

struct OutputHistoryItem
{
    int flags;
    int firstOwnNS;
};

typedef PList<OutputHistoryItem*> OutputHistory;

//
//  FrontMatter
//

enum FrontMatterKind
{
    FM_TEXT,
    FM_COMMENT,
    FM_PI
};

struct FrontMatterItem
{
    FrontMatterKind kind;
    Str string1, string2;
    Bool disableEsc;
};

class FrontMatter : public PList<FrontMatterItem*>
{
public:
    SituationObj* situation;
    eFlag appendConstruct(FrontMatterKind kind, const Str& string1,
        const Str& string2, Bool disableEsc);
};

class OutputterObj
{
public:
    OutputterObj(Processor *proc_);
    ~OutputterObj();
    eFlag setOptions(DataLine *targetDataLine_, OutputDefinition *outDef_);
    eFlag setOptionsSAX(SAXHandler *streaming_, void *userData);
    eFlag eventBeginOutput();
    eFlag eventElementStart(const QName& name);
    eFlag eventElementEnd(const QName& name);
    eFlag eventAttributeStart(const QName& name);
    eFlag eventAttributeEnd();
    eFlag eventCommentStart();
    eFlag eventCommentEnd();
    eFlag eventNamespace(const Str& prefix, const Str& uri);
    eFlag eventPIStart(const Str& name);
    eFlag eventPIEnd();
    eFlag eventData(const Str& data);
    eFlag eventDisableEscapingForNext();
    eFlag eventEndOutput();
private:
    eFlag reportCurrData();
    eFlag reportStartTag(Bool isEmpty);
    eFlag throwInMeta();
    void pushLevel(const QName& name);
//    void popLevel();
    int getLevel() 
    {
        return history.number();
    };
    int getFlags()
    {
        return (history.number()? history.last() -> flags : 0);
    }
    int getFirstOwnNS()
    {
        return (history.number()? history.last() -> firstOwnNS : 0);
    }
    eFlag reportXMLDeclIfMust();
    eFlag reportDTDIfMust(const QName& docElementName);
    eFlag reportFront();
    //
    PhysicalOutputLayerObj *physical;
    SAXHandler *mySAXHandler;
    void *mySAXUserData;
    OutputDefinition *outDef;
    OutputMethod method;
    OutputterState state;
    Bool outputEscaping;
    DStr currData;
    Str
        currPIName;
    QName
        currElement, 
        currAttName;
    NamespaceStack
        currNamespaces;
    QNameStrList
        currAtts;
    OutputHistory history;
    Bool noElementYet,
        noHeadYet;
    FrontMatter front;
    Processor *proc;
};

#endif  // OutputHIncl
