/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

/*****************************************************************

    c   o   n   t   e   x   t   .   c   p   p

*****************************************************************/

#include "context.h"
#include "tree.h"

/*****************************************************************
    C L i s t   methods
*****************************************************************/

CList::CList()
:
VertexList(LIST_SIZE_LARGE)
{
    refCount = 1;
}

void CList::incRefCount()
{
    refCount++;
}

int CList::decRefCount()
{
    return --refCount;
}

int CList::compare(Vertex *v1, Vertex *v2)
{
    assert(v1 && v2);
    if (v1 -> stamp < v2 -> stamp)
        return -1;
    else if (v1 -> stamp > v2 -> stamp)
        return 1;
    else return 0;
}

/*****************************************************************

    C o n t e x t    methods

******************************************************************/

Context::Context()
{
    array = new CList;
    currentNode = NULL;
    position = -1;
    virtualPosition = 0;
    virtualSize = -1;
}

Context::~Context()
{
    assert(array);
    if (!array -> decRefCount())
        delete array;
}

int Context::getPosition() const
{
    return position + virtualPosition;
}

int Context::getSize() const
{
    if (virtualSize != -1)
        return virtualSize;
    else
        return array -> number();
}

Vertex *Context::current() const
{
    if (!isFinished())
        return (*array)[position];
    else
        return NULL;
};

Vertex *Context::getCurrentNode() const
{
    if (currentNode)
        return currentNode;
    else
        return current();
}

void Context::setCurrentNode(Vertex * v)
{
    currentNode = v;
}

void Context::reset()
{
    if (!array -> number())
        position = -1;
    else
        position = 0;
}

Bool Context::isFinished() const
{
    return (Bool) !((position >= 0) && (position < array -> number()));
}

Bool Context::isVoid() const
{
    return (Bool) !(array -> number());
}

Vertex *Context::shift()
{
    if ((position >= 0) && (position < array -> number() - 1))
        return (*this)[++position];
    position = -1;
    return NULL;
}

void Context::set(Vertex *v)
{
    array -> append(v);
    reset();
}

void Context::setVirtual(Vertex *v, int virtualPosition_, int virtualSize_)
{
    assert(!array -> number() && "setVirtual() on nonvoid context");
    array -> append(v);
    virtualSize = virtualSize_;
    virtualPosition = virtualPosition_;
    reset();
}

Vertex *Context::operator[] (int n) const
{
    return (*array)[n];
}

void Context::deppendall()
{
    if (!array -> decRefCount())
        delete array;
    array = new CList;
    position = -1;
}

void Context::append(Vertex *v)
{
    array -> append(v);
    reset();
}

void Context::deppend()
{
    array -> deppend();
    if (position >= array -> number())
        position = array -> number() - 1;
}

Context* Context::copy()
{
    Context *newc = new Context;
    delete NZ(newc -> array);
    newc -> currentNode = currentNode;
    newc -> array = array;
    newc -> virtualPosition = virtualPosition;
    newc -> virtualSize = virtualSize;
    array -> incRefCount();
    newc -> reset();
    return newc;
}

void Context::swap(int i, int j)
{
    array -> swap(i, j);
}

Context *Context::swallow(Context *other)
{
    Context *result = new Context;
    int i = 0, j = 0,
        iLimit = array -> number(), jLimit = other -> array -> number();
    long vstamp, wstamp;
    Vertex *v, *w;
    while ((i < iLimit) && (j < jLimit))
    {
        vstamp = (v = (*array)[i]) -> stamp;
        wstamp = (w = (*(other -> array))[j]) -> stamp;
        if (v == w) //XXXX wrong! need consistent document ordering
//        if (vstamp == wstamp)
        {
            j++;
            continue;
        }
        if (vstamp < wstamp)
        {
            result -> append(v);
            i++;
        }
        else
        {
            result -> append(w);
            j++;
        }
    }
    while (i < iLimit)
        result -> append((*array)[i++]);
    while (j < jLimit)
        result -> append((*(other -> array))[j++]);
    deppendall();
    other -> deppendall();
    return result;
}

Bool Context::contains(const Vertex *v) const
{
    int i;
    for (i = 0; i < array -> number(); i++)
    {
        if (v == (*array)[i])
            return TRUE;
    }
    return FALSE;
}

void Context::uniquize()
{
    int i;
    for (i = array -> number() - 2; i >= 0; i--)
        if (!array -> compare((*array)[i], (*array)[i+1]))
            array -> rm(i);
}
