/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#include "CGarminDBTrack.h"
#include "CCentralResources.h"
#include "CGpx.h"
#include "CToolViewTrack.h"
#include "CToolViewTrackInfo.h"
#include "CToolViewTrackProfile.h"
#include "icons.h"

using namespace std;

#include <QtGui>

// ------------------------------------------------------------
// --------------------- CGarminDBTrack -----------------------
// ------------------------------------------------------------
CGarminDBTrack::CGarminDBTrack(QTabWidget * parent, QSplitter * splitter)
: QObject(parent)
, tab(parent)
{
    trackprofile = new CToolViewTrackProfile(splitter,this);
    trackprofile->hide();
    splitter->addWidget(trackprofile);

    trackinfo = new CToolViewTrackInfo(splitter,this);
    trackinfo->hide();
    splitter->addWidget(trackinfo);

    toolview = new CToolViewTrack(parent,this);
    toolview->show();
    parent->addTab(toolview,QPixmap(iconTrack16x16),"");
    parent->setTabToolTip(parent->indexOf(toolview), tr("Track View"));

}


CGarminDBTrack::~CGarminDBTrack()
{

}


void CGarminDBTrack::gainFocus()
{
    if(tab) {
        if(tab->currentWidget() != toolview) {
            tab->setCurrentWidget(toolview);
        }
    }
}


void CGarminDBTrack::downloadTracks()
{
    Garmin::IDevice * dev = 0;
    try
    {
        list<Garmin::Track_t> trks;

        dev = gpResources->device();
        if(dev) {
            dev->downloadTracks(trks);
        }

        std::list<Garmin::Track_t>::const_iterator trk = trks.begin();
        while(trk != trks.end()) {

            removeTrack("device|" + QString(trk->ident.c_str()));

            CGarminTrack * t = new CGarminTrack(this,(*trk));
            connect(t,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));
            tracks["device|" + QString(trk->ident.c_str())] = t;
            ++trk;
        }
    }
    catch(int e) {
        if(dev == 0) return;
        QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
        if(e == Garmin::errSync) {
            gpResources->resetDevice();
        }
    }

    emit sigTrackListChanged();
}


void CGarminDBTrack::loadGPX(CGpx& gpx)
{
    const QDomNodeList& trks = gpx.elementsByTagName("trk");
    uint N = trks.count();
    for(uint n = 0; n < N; ++n) {
        const QDomNode& trk = trks.item(n);

        CGarminTrack * t = new CGarminTrack(this);

        if(trk.namedItem("name").isElement()) {
            t->setName(trk.namedItem("name").toElement().text());
        }

        if(trk.namedItem("extension").isElement()) {
            const QDomNode& ext = trk.namedItem("extension");
            if(ext.namedItem("color").isElement()) {
                t->setColor(ext.namedItem("color").toElement().text().toUInt());
            }
        }

        const QDomNode& trkseg = trk.namedItem("trkseg");
        QDomElement trkpt = trkseg.firstChildElement("trkpt");

        while (!trkpt.isNull()) {
            CGarminTrack::TrkPt_t pt;

            QDomNamedNodeMap attr = trkpt.attributes();

            pt.lon = attr.namedItem("lon").nodeValue().toDouble();
            pt.lat = attr.namedItem("lat").nodeValue().toDouble();

            if(trkpt.namedItem("ele").isElement()) {
                double ele = trkpt.namedItem("ele").toElement().text().toDouble();
                if(ele < 0) {
                    pt.dpth = ele;
                }
                else {
                    pt.alt = ele;
                }
            }

            if(trkpt.namedItem("time").isElement()) {
                QDateTime time = QDateTime::fromString(trkpt.namedItem("time").toElement().text(),"yyyy-MM-dd'T'hh:mm:ss'Z'");
                time.setTimeSpec(Qt::UTC);
                pt.time = time.toTime_t() - gpResources->getUTCOffset();
            }

            if(trkpt.namedItem("extension").isElement()) {
                const QDomNode& ext = trkpt.namedItem("extension");
                if(ext.namedItem("flags").isElement()) {
                    pt.flags = ext.namedItem("flags").toElement().text().toUInt();
                    pt.flags &= ~CGarminTrack::eFocus;
                    pt.flags &= ~CGarminTrack::eSelected;
                    pt.flags &= ~CGarminTrack::eCursor;
                }
            }

            t->push_back(pt);
            trkpt = trkpt.nextSiblingElement("trkpt");
        }

        t->update();
        removeTrack("gpx|" + t->getName());
        tracks["gpx|" + t->getName()] = t;

        connect(t,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));

    }
    emit sigTrackListChanged();
}


void CGarminDBTrack::saveGPX(CGpx& gpx)
{
    QDomElement root = gpx.documentElement();
    QMap<QString,CGarminTrack*>::iterator track = tracks.begin();
    while(track != tracks.end()) {
        QDomElement gpxTrack = gpx.createElement("trk");
        root.appendChild(gpxTrack);

        QDomElement name = gpx.createElement("name");
        gpxTrack.appendChild(name);
        QDomText _name_ = gpx.createTextNode((*track)->getName());
        name.appendChild(_name_);

        QDomElement ext = gpx.createElement("extension");
        gpxTrack.appendChild(ext);

        QDomElement color = gpx.createElement("color");
        ext.appendChild(color);
        QDomText _color_ = gpx.createTextNode(QString::number((*track)->Garmin::Track_t::color));
        color.appendChild(_color_);

        QDomElement trkseg = gpx.createElement("trkseg");
        gpxTrack.appendChild(trkseg);

        QVector<CGarminTrack::TrkPt_t>::const_iterator trkpt = (*track)->begin();
        while(trkpt != (*track)->end()) {
            QDomElement gpxTrkpt = gpx.createElement("trkpt");
            trkseg.appendChild(gpxTrkpt);
            gpxTrkpt.setAttribute("lat",QString::number(trkpt->lat,'f',6));
            gpxTrkpt.setAttribute("lon",QString::number(trkpt->lon,'f',6));

            if(trkpt->alt != 1e25f || trkpt->dpth != 1e25f) {
                QDomElement ele = gpx.createElement("ele");
                gpxTrkpt.appendChild(ele);
                QDomText _ele_ = gpx.createTextNode(QString::number((trkpt->alt != 1e25f) ? trkpt->alt : -trkpt->dpth));
                ele.appendChild(_ele_);
            }
            if(trkpt->time != 0x000000000 && trkpt->time != 0xFFFFFFFF) {
                QDateTime t = QDateTime::fromTime_t(trkpt->time + gpResources->getUTCOffset()).toUTC();
                QDomElement time = gpx.createElement("time");
                gpxTrkpt.appendChild(time);
                QDomText _time_ = gpx.createTextNode(t.toString("yyyy-MM-dd'T'hh:mm:ss'Z'"));
                time.appendChild(_time_);
            }

            QDomElement extension = gpx.createElement("extension");
            gpxTrkpt.appendChild(extension);

            QDomElement flags = gpx.createElement("flags");
            extension.appendChild(flags);
            QDomText _flags_ = gpx.createTextNode(QString::number(trkpt->flags));
            flags.appendChild(_flags_);

            ++trkpt;
        }

        ++track;
    }
}


CGarminTrack* CGarminDBTrack::highlightedTrack()
{

    QMap<QString,CGarminTrack*>::iterator track = tracks.begin();
    while(track != tracks.end()) {
        if((*track)->isHighlighted()) return *track;
        ++track;
    }
    return 0;

}


void CGarminDBTrack::highlightTrack(const QString& key)
{
    QMap<QString,CGarminTrack*>::iterator track = tracks.begin();
    while(track != tracks.end()) {
        (*track)->setHighlight(false);
        ++track;
    }

    tracks[key]->setHighlight(true);
    emit sigTrackListChanged();
}


void CGarminDBTrack::removeTrack(const QString& key)
{
    if(!tracks.contains(key)) return;
    delete tracks.take(key);
    emit sigTrackListChanged();
}


void CGarminDBTrack::removeTracks(const QStringList& keys)
{
    QString key;
    foreach(key,keys) {
        if(!tracks.contains(key)) continue;
        delete tracks.take(key);
    }
    emit sigTrackListChanged();
}


void CGarminDBTrack::showTrackInfo()
{

    if(trackinfo->isHidden()) {
        trackprofile->show();
        trackinfo->show();
    }
    else {
        trackprofile->hide();
        trackinfo->hide();
    }

}


void CGarminDBTrack::clear()
{
    removeTracks(tracks.keys());
}


void CGarminDBTrack::slotTrackChanged()
{
    emit sigTrackListChanged();
}


void CGarminDBTrack::combine()
{
    QString key;
    QStringList keys;
    toolview->getSelectedTracks(keys);

    if(keys.size() < 2) {
        QMessageBox::warning(0,tr("Combine tracks ..."),tr("You must select at least two tracks to do this."), QMessageBox::Ok);
        return;
    }

    CGarminTrack * track = new CGarminTrack(this);
    foreach(key,keys) {
        *track += *tracks[key];
    }

    removeTrack("user|" + track->getName());
    tracks["user|" + track->getName()] = track;

    connect(track,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));
    emit sigTrackListChanged();
}


void CGarminDBTrack::split(quint32 idx)
{
    quint32 i;

    CGarminTrack * track = highlightedTrack();
    if(!track) {
        QMessageBox::warning(0,tr("Split track ..."),tr("You must select a track to do this."), QMessageBox::Ok);
        return;
    }

    QVector<CGarminTrack::TrkPt_t>::iterator trkpt = track->begin();

    CGarminTrack * track1 = new CGarminTrack(this);
    track1->setName(track->getSiblingName());
    for(i=0;( i <= idx) && (trkpt != track->end()); ++i) {
        track1->push_back(*trkpt++);
    }

    track1->update( );
    removeTrack("user|" + track1->getName());
    tracks["user|" + track1->getName()] = track1;
    connect(track1,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));

    CGarminTrack * track2 = new CGarminTrack(this);
    track2->setName(track->getSiblingName());
    for(; trkpt != track->end(); ++i) {
        track2->push_back(*trkpt++);
    }

    track2->update( );
    removeTrack("user|" + track2->getName());
    tracks["user|" + track2->getName()] = track2;
    connect(track2,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));

    emit sigTrackListChanged();
}


CGarminTrack* CGarminDBTrack::getTrack(const QString& key)
{
    if(tracks.contains(key)) {
        return tracks[key];
    }
    CGarminTrack * t = new CGarminTrack(this);
    t->setName(key.section('|',1,1));
    tracks[key] = t;

    connect(t,SIGNAL(sigChanged()),this,SLOT(slotTrackChanged()));
    emit sigTrackListChanged();
    return t;
}
