/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#ifndef IMAP_H
#define IMAP_H

#include <QtGui>


class IMap : public QObject
{
    public:
        IMap(QObject * parent);
        virtual ~IMap();

        /// render the map
        /**
            Set buffered to true for fast rendering (screen). This will
            draw the map into a pixmap buffer if it has to be updated. Else
            it will just use this buffer. For printing you will rather use
            the unbuffered drawing.
        */
        virtual void draw(QPainter& p,bool buffered);
        /// change the map's viewport size
        virtual void resize(const QSize& s);
        /// convert a point from the viewport (in pixel) to degrees
        virtual void convertPt2Deg(double& u, double& v);
        /// convert a point from the viewport (in pixel) to rad
        virtual void convertPt2Rad(double& u, double& v);
        /// convert a point from the viewport (in pixel) to meter
        virtual void convertPt2M(double& u, double& v);
        /// zoom out to fit complete map to screen
        virtual void world() = 0;
        /// zoom in, zoom out
        virtual void zoom(bool zoomIn, const QPoint& p) = 0;
        /// zoom to area
        virtual void zoom(QRect& r) = 0;
        /// move the map
        virtual void move(const QPoint& old, const QPoint& next);
        /// reduce details for faster drawing
        virtual void setEasyDraw(bool f);
        /// increase detail level
        virtual void incDetails() = 0;
        /// decrease detail level
        virtual void decDetails() = 0;
        /// show / hide city names
        virtual void toggleCityNames() = 0;
        /// show / hide points of interest
        virtual void togglePOIs() = 0;
        /// show / hide street name
        virtual void toggleStreetNames() = 0;

        virtual bool seeCityNames(){return showCityNames;}
        virtual bool seePOIs(){return showPOIs;}
        virtual bool seeStreetNames(){return showStreetNames;}

        inline qint32 u2px(double m)
        {
            return (qint32)(((m - x_off) / scale) + 0.5);
        }

        inline qint32 v2px(double m)
        {
            return (qint32)(((y_off - m) / scale) + 0.5);
        }

        inline double px2u(double px)
        {
            return (double)px * scale + x_off;
        }

        inline double px2v(double px)
        {
            return y_off - (double)px * scale;
        }

        inline double getScale(){return scale;}

    protected:
        /// render the map
        /**
            If p is non zero the map must be rendered to p rather than the
            internal pixmap buffer pixBottom;
        */
        virtual void drawBackground(QPainter * p = 0) = 0;

        inline virtual void calcWorldRect()
        {
            worldrect.setCoords(  px2u(rect.left()), px2v(rect.top())
                                , px2u(rect.right()), px2v(rect.bottom()));
            worldrect = worldrect.normalized();
        }
        /// canvas size
        QSize size;
        /// canvas rectangle in pixel
        QRectF rect;
        /// canvas rectangle in world coordinates [m]
        QRectF worldrect;

        /// offset of left border
        double x_off;
        /// offset of top border
        double y_off;
        /// the scalefactor;
        double scale;

        /// if true reduce details, to speed up drawing
        bool easyDraw;

        /// rectangle of scale bar at lower right corner
        QRect scaleIcon;
        /// rectangle to place current scale per 100px text
        QRect scaleTextRect;
        /// the actual scale per 100px text
        QString scaleText;

        /// the background pixmap to draw the map on
        QPixmap pixBottom;

        bool needRedraw;

        bool showCityNames;

        bool showPOIs;

        bool showStreetNames;


};

#endif //IMAP_H

