/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#include "CToolViewTrackInfo.h"
#include "CCentralResources.h"
#include "CGarminDBTrack.h"
#include "GeoMath.h"
#include "icons.h"

#include <IDevice.h>

#include <QtGui>

using namespace Garmin;
using namespace std;

CToolViewTrackInfo::CToolViewTrackInfo(QWidget * parent, CGarminDBTrack * db)
    : QWidget(parent)
    , originator(false)
{
    setupUi(this);

    connect(butApply,SIGNAL(pressed()),this,SLOT(slotApply()));
    connect(checkRemoveDelTrkPt,SIGNAL(clicked(bool)),this,SLOT(slotCheckRemove(bool)));
    connect(checkResetDelTrkPt,SIGNAL(clicked(bool)),this,SLOT(slotCheckReset(bool)));
    connect(listPoints,SIGNAL(itemSelectionChanged()),this,SLOT(slotPointSelectionChanged()));
    connect(listPoints,SIGNAL(itemClicked(QTreeWidgetItem*,int)),this,SLOT(slotPointSelection(QTreeWidgetItem*)));

    connect(db,SIGNAL(sigTrackListChanged()),this,SLOT(slotTrackSelectionChanged()));

    listPoints->header()->setResizeMode(0,QHeaderView::Interactive);
    for(int i=0; i< eMaxColumn - 1; ++i){
        listPoints->resizeColumnToContents(i);
    }

    QPixmap icon(16,8);
    for(int i=0; i < 17; ++i){
        icon.fill(CGarminTrack::colors[i]);
        comboColor->addItem(icon,"",QVariant(i));
    }

    connect(listPoints,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Purge"),this,SLOT(slotPurge()));

    setEnabled(false);
}

CToolViewTrackInfo::~CToolViewTrackInfo()
{

}


void CToolViewTrackInfo::slotCheckReset(bool checked)
{
    if(checked){
        checkRemoveDelTrkPt->setChecked(false);
    }
}

void CToolViewTrackInfo::slotCheckRemove(bool checked)
{
    if(checked){
        checkResetDelTrkPt->setChecked(false);
    }
}

void CToolViewTrackInfo::slotTrackSelectionChanged()
{
    if(originator) return;
    // reset tree widget
    listPoints->clear();

    track = gpResources->trackdb().highlightedTrack();
    if(track.isNull()){
        lineIdent->clear();
        setEnabled(false);
        return;
    }
    setEnabled(true);

    QTime time;
    time = time.addSecs(track->getTotalTime());
    lblTotalTime->setText(time.toString("HH:mm:ss"));


    lblTotalDist->setText(gpResources->distance2str(track->getTotalDistance()));

    if(track->getAvgSpeed() > 0){
        if(gpResources->doMetric()){
            if(track->getAvgSpeed() < 10.0){
                lblAvgSpeed->setText(QString("%1 km/h").arg(track->getAvgSpeed(),0,'f',2));
            }
            else{
                lblAvgSpeed->setText(QString("%1 km/h").arg(track->getAvgSpeed(),0,'f',0));
            }
        }
        else{
            lblAvgSpeed->setText(QString("%1 ml/h").arg(track->getAvgSpeed() * 0.6213699,0,'f',2));
        }
    }
    else{
        lblAvgSpeed->setText(" - ");
    }

    comboColor->setCurrentIndex(((Garmin::Track_t*)track)->color > 16 ? 4 : ((Garmin::Track_t*)track)->color);
    lineIdent->setText(track->getName());

    quint32 ptcnt = 0;
    QTreeWidgetItem * focus = 0;
    QVector<CGarminTrack::TrkPt_t>::const_iterator trackpoint = track->begin();
    while(trackpoint != track->end()){

        // add trackpoint to tree widget
        QString str;
        QTreeWidgetItem * item = new QTreeWidgetItem(listPoints);
        item->setData(0, Qt::UserRole, trackpoint->idx);

        // gray shade deleted items
        if(trackpoint->flags & CGarminTrack::eDeleted){
            item->setFlags(item->flags() & ~Qt::ItemIsEnabled);
        }

        // temp. store item of user focus
        if(trackpoint->flags & CGarminTrack::eFocus){
            focus = item;
        }

        // point number
        item->setText(eNum,QString::number(++ptcnt));

        // timestamp
        if(trackpoint->time != 0x00000000 && trackpoint->time != 0xFFFFFFFF){
            QDateTime time = QDateTime::fromTime_t(trackpoint->time + gpResources->getUTCOffset());
            time.setTimeSpec(Qt::LocalTime);
            str = time.toString();
        }
        else{
            str = "-";
        }

        item->setText(eTime,str);

        // altitude
        if(trackpoint->alt != 1e25f  || trackpoint->dpth != 1e25f){

            if(gpResources->doMetric()){
                if(trackpoint->alt != 1e25f){
                    str = QString::number(trackpoint->alt,'f',0) + "m";
                }
                else if(trackpoint->dpth != 1e25f){
                    str = QString::number(-trackpoint->dpth,'f',0) + "m";
                }
            }
            else{
                if(trackpoint->alt != 1e25f){
                    str = QString::number(trackpoint->alt * 3.28084,'f',0) + "ft";
                }
                else if(trackpoint->dpth != 1e25f){
                    str = QString::number(-trackpoint->dpth  * 3.28084,'f',0) + "ft";
                }
            }
        }
        else {
            str = "-";
        }

        item->setText(eAltitude,str);


        if(gpResources->doMetric()){
            if(trackpoint->delta < 10.0) {
                str.sprintf("%1.2f m",trackpoint->delta);
            }
            else{
                str.sprintf("%1.0f m",trackpoint->delta);
            }
        }
        else{
            str.sprintf("%1.0f ft",trackpoint->delta * 3.28084);
        }
        item->setText(eDelta,str);
        str.sprintf("%1.0f\260",trackpoint->azimuth);
        item->setText(eAzimuth,str);

        str = gpResources->distance2str(trackpoint->distance);
        item->setText(eDistance,str);

        // speed
        if(trackpoint->speed > 0){
            if(gpResources->doMetric()){
                if(trackpoint->speed < 10.0){
                    str.sprintf("%1.2f km/h",trackpoint->speed);
                }
                else{
                    str.sprintf("%1.0f km/h",trackpoint->speed);
                }
            }
            else{
                str.sprintf("%1.2f ml/h",trackpoint->speed * 0.6213699);
            }
        }
        else{
            str = "-";
        }
        item->setText(eSpeed,str);

        // position
        qint32 degN,degE;
        double minN,minE;
        GPS_Math_Deg_To_DegMin(trackpoint->lat, &degN, &minN);
        GPS_Math_Deg_To_DegMin(trackpoint->lon, &degE, &minE);

        QString lat,lng;
        lat = degN < 0 ? "S" : "N";
        lng = degE < 0 ? "W" : "E";
        str.sprintf("%s%02d\260 %06.3f %s%03d\260 %06.3f",lat.toUtf8().data(),abs(degN),minN,lng.toUtf8().data(),abs(degE),minE);

        item->setText(ePosition,str);

        item->setTextAlignment(eNum,Qt::AlignLeft);
        item->setTextAlignment(eAltitude,Qt::AlignRight);
        item->setTextAlignment(eDelta,Qt::AlignRight);
        item->setTextAlignment(eAzimuth,Qt::AlignRight);
        item->setTextAlignment(eDistance,Qt::AlignRight);
        item->setTextAlignment(eSpeed,Qt::AlignRight);

        ++trackpoint;
    }

    // adjust column sizes to fit
    listPoints->header()->setResizeMode(0,QHeaderView::Interactive);
    for(int i=0; i< eMaxColumn - 1; ++i){
        listPoints->resizeColumnToContents(i);
    }

    // scroll to item of user focus
    if(focus){
        listPoints->setCurrentItem(focus);
        listPoints->scrollToItem(focus);
    }

    lblNoPoints->setText(QString::number(ptcnt));
}

void CToolViewTrackInfo::slotPointSelection(QTreeWidgetItem * item)
{
    if(track.isNull()) return;
    originator = true;
    track->setPointOfFocus(item->data(0,Qt::UserRole).toInt());
    originator = false;
}

void CToolViewTrackInfo::slotPointSelectionChanged()
{
    if(track.isNull()) return;

    // reset previous selections
    QVector<CGarminTrack::TrkPt_t>::iterator trkpt = track->begin();
    while(trkpt != track->end()){
        trkpt->flags &= ~CGarminTrack::eSelected;
        ++trkpt;
    }

    // set eSelected flag for selected points
    QList<QTreeWidgetItem*> items = listPoints->selectedItems();
    QList<QTreeWidgetItem*>::const_iterator item = items.begin();
    while(item != items.end()){
        quint32 idxTrkPt = (*item)->data(0,Qt::UserRole).toUInt();
        (*track)[idxTrkPt].flags |= CGarminTrack::eSelected;
        ++item;
    }
    originator = true;
    track->update();
    originator = false;
}

void CToolViewTrackInfo::slotApply()
{
    if(track.isNull()) return;

    originator = true;

    if(checkResetDelTrkPt->isChecked()){
        QVector<CGarminTrack::TrkPt_t>::iterator pt = track->begin();
        while(pt != track->end()){
            pt->flags &= ~CGarminTrack::eDeleted;
            ++pt;
        }
        checkResetDelTrkPt->setChecked(false);
        originator = false;
    }

    if(checkRemoveDelTrkPt->isChecked()){
        QMessageBox::warning(0,tr("Remove track points ...")
                            ,tr("You are about to remove purged track points permanently. If you press 'yes', all information will be lost.")
                            ,QMessageBox::Yes|QMessageBox::No);
        QVector<CGarminTrack::TrkPt_t>::iterator pt = track->begin();
        while(pt != track->end()){
            if(pt->flags & CGarminTrack::eDeleted){
                pt = track->erase(pt);
            }
            else{
                ++pt;
            }
        }
        checkRemoveDelTrkPt->setChecked(false);
        originator = false;
    }

    track->setName(lineIdent->text());
    track->setColor(comboColor->currentIndex());
    track->update(true);
    originator = false;
}


void CToolViewTrackInfo::keyPressEvent(QKeyEvent * e)
{
    if(track.isNull()) return;

    if(e->key() == Qt::Key_Delete){
      slotPurge();
    }
    else{
        QWidget::keyPressEvent(e);
    }
}

void CToolViewTrackInfo::slotPurge()
{
  QList<QTreeWidgetItem*> items = listPoints->selectedItems();
  QList<QTreeWidgetItem*>::const_iterator item = items.begin();
  while(item != items.end()){
    quint32 idxTrkPt = (*item)->data(0,Qt::UserRole).toUInt();
    if((*track)[idxTrkPt].flags & CGarminTrack::eDeleted){
      (*track)[idxTrkPt].flags &= ~CGarminTrack::eDeleted;
    }
    else{
      (*track)[idxTrkPt].flags |= CGarminTrack::eDeleted;
    }

    ++item;
  }
  track->update();
}

void CToolViewTrackInfo::slotContextMenu(const QPoint& pos)
{
  if(!listPoints->selectedItems().isEmpty()){
    QPoint p = listPoints->mapToGlobal(pos);
    p.setY(p.y() + listPoints->header()->height());
    contextMenu->exec(p);
  }
}

